package com.ejie.y41e.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.apache.commons.lang.StringUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.security.UserCredentials;
import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.dao.rwmap.Y41ePreguntaRowMapper;
import com.ejie.y41e.model.Y41ePregunta;
import com.ejie.y41e.utils.Y41eSqlUtils;

/**
 * Y41ePreguntaDaoImpl
 *  
 */

@Repository
@Transactional
public class Y41ePreguntaDaoImpl extends Y41eGenericoDaoImpl<Y41ePregunta> implements Y41ePreguntaDao {

    public static final String TABLA="Y41E04S01";
    public static final String SECUENCIA="Y41E04Q00";
    public static final String ID_PREGUNTA_004 ="ID_PREGUNTA_004";
    public static final String DESC_PUBL_ES_004 ="DESC_PUBL_ES_004";
    public static final String DESC_PUBL_EU_004 ="DESC_PUBL_EU_004";
    public static final String DESC_INT_ES_004 ="DESC_INT_ES_004";
    public static final String DESC_INT_EU_004 ="DESC_INT_EU_004";
    public static final String ACTIVA_004 ="ACTIVA_004";
    public static final String OBLIGATORIO_004 ="OBLIGATORIO_004";
    public static final String ID_TIPO_PREGUNTA_004 ="ID_TIPO_PREGUNTA_004";
    public static final String FEC_ALTA_004 ="FEC_ALTA_004";
    public static final String FEC_ULTMODIF_004 ="FEC_ULTMODIF_004";
    public static final String USU_ALTA_004 ="USU_ALTA_004";
    public static final String USU_MODIF_004 ="USU_MODIF_004";
    public static final String AVISO_004 ="AVISO_004";
    protected static final String[] ORDER_BY_WHITE_LIST = new String[] { ID_PREGUNTA_004, DESC_PUBL_ES_004, DESC_PUBL_EU_004, DESC_INT_ES_004,
            DESC_INT_EU_004, ACTIVA_004, OBLIGATORIO_004, ID_TIPO_PREGUNTA_004, FEC_ALTA_004, FEC_ULTMODIF_004, USU_ALTA_004, USU_MODIF_004, AVISO_004, 
            "DESC_ES_008", "DESC_EU_008"};

    /**
     * Y41ePreguntaDaoImpl
     */
    public Y41ePreguntaDaoImpl() {
        super(Y41ePregunta.class);
    }

    /*
     * ROW_MAPPERS
     */
    private final RowMapper<Y41ePregunta> rwMapPK = new RowMapper<Y41ePregunta>() {
        @Override
        public Y41ePregunta mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            return new Y41ePregunta(resultSet.getLong(ID_PREGUNTA_004)); 
        }
    };

    /**
     * rwPreguntaAviso
     */
    private final RowMapper<Y41ePregunta> rwPreguntaAviso = new RowMapper<Y41ePregunta>() {
        @Override
        public Y41ePregunta mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Y41ePregunta pregunta = new Y41ePregunta();
            pregunta.setDescPublEs(resultSet.getString(DESC_PUBL_ES_004));
            pregunta.setDescPublEu(resultSet.getString(DESC_PUBL_EU_004));
            pregunta.setDescIntEs(resultSet.getString(DESC_INT_ES_004));
            pregunta.setDescIntEu(resultSet.getString(DESC_INT_EU_004));
            pregunta.getResultado().getRespuesta().setDescEs(resultSet.getString("DESC_ES_009"));
            pregunta.getResultado().getRespuesta().setDescEu(resultSet.getString("DESC_EU_009"));
            return pregunta;
        }
    };

    /*
     * OPERACIONES CRUD
     */

    /**
     * Inserts a single row in the Y41ePregunta table.
     *
     * @param pregunta Y41ePregunta
     * @return Y41ePregunta
     */
    @Override
    public Y41ePregunta add(Y41ePregunta pregunta) {

        pregunta.setIdPregunta(this.getNextVal(SECUENCIA));

        StringBuilder insert = new StringBuilder(Y41eSqlUtils.INSERT)
                .append(TABLA).append(" (")
                .append(ID_PREGUNTA_004).append(",")
                .append(DESC_PUBL_ES_004).append(",")
                .append(DESC_PUBL_EU_004).append(",")
                .append(DESC_INT_ES_004).append(",")
                .append(DESC_INT_EU_004).append(",")
                .append(ACTIVA_004).append(",")
                .append(OBLIGATORIO_004).append(",")
                .append(ID_TIPO_PREGUNTA_004).append(",")
                .append(FEC_ALTA_004).append(",")
                .append(FEC_ULTMODIF_004).append(",")
                .append(USU_ALTA_004).append(",")
                .append(USU_MODIF_004).append(",")
                .append(AVISO_004).append(")")
                .append("VALUES (?,")
                .append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append("?,?,?, SYSDATE, SYSDATE,?,?,?)");

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        final UserCredentials  credentials = (UserCredentials) authentication.getCredentials();

        this.getJdbcTemplate().update(insert.toString(), pregunta.getIdPregunta(), pregunta.getDescPublEs(), pregunta.getDescPublEu(), pregunta.getDescIntEs(), pregunta.getDescIntEu(),
                pregunta.getActiva(), StringUtils.isBlank(pregunta.getObligatorio())?Y41eConstants.NO:pregunta.getObligatorio(), 
                        pregunta.getTipoPregunta().getIdTipoPregunta(), credentials.getNif(),credentials.getNif(),
                        StringUtils.isBlank(pregunta.getAviso())?Y41eConstants.NO:pregunta.getAviso());

        return pregunta;
    }

    /**
     * Updates a single row in the Y41ePregunta table.
     *
     * @param pregunta Y41ePregunta
     * @return Y41ePregunta
     */
    @Override
    public Y41ePregunta update(Y41ePregunta pregunta) {
        StringBuilder update = new StringBuilder(Y41eSqlUtils.UPDATE)
                .append(TABLA).append(" SET ")
                .append(DESC_PUBL_ES_004).append("=").append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(DESC_PUBL_EU_004).append("=").append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(DESC_INT_ES_004).append("=").append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(DESC_INT_EU_004).append("=").append(Y41eSqlUtils.SUBSTR_500).append(",")
                .append(ACTIVA_004).append("=?,")
                .append(OBLIGATORIO_004).append("=?,")
                .append(ID_TIPO_PREGUNTA_004).append("=?,")
                .append(FEC_ULTMODIF_004).append("= SYSDATE,")
                .append(USU_MODIF_004).append("=?,")
                .append(AVISO_004).append("=?")
                .append(Y41eSqlUtils.WHERE)
                .append(ID_PREGUNTA_004).append("=?");

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        final UserCredentials  credentials = (UserCredentials) authentication.getCredentials();

        this.getJdbcTemplate().update(update.toString(), pregunta.getDescPublEs(), pregunta.getDescPublEu(), pregunta.getDescIntEs(), pregunta.getDescIntEu(),
                pregunta.getActiva(), StringUtils.isBlank(pregunta.getObligatorio())?Y41eConstants.NO:pregunta.getObligatorio(), pregunta.getTipoPregunta().getIdTipoPregunta(),
                        credentials.getNif(), StringUtils.isBlank(pregunta.getAviso())?Y41eConstants.NO:pregunta.getAviso(), pregunta.getIdPregunta());

        return pregunta;
    }

    /**
     * Removes a single row in the Y41ePregunta table.
     *
     * @param bloque Y41ePregunta
     * @return
     */
    @Override
    public void remove(Y41ePregunta bloque) {
        StringBuilder delete = new StringBuilder(Y41eSqlUtils.DELETE).append(TABLA).append(Y41eSqlUtils.WHERE).append(ID_PREGUNTA_004).append("=?");
        this.getJdbcTemplate().update(delete.toString(), bloque.getIdPregunta());
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getSelect()
     */
    @Override
    protected String getSelect() {
        StringBuilder select = new StringBuilder("SELECT ")
                .append(ID_PREGUNTA_004).append(",")
                .append(DESC_PUBL_ES_004).append(",")
                .append(DESC_PUBL_EU_004).append(",")
                .append(DESC_INT_ES_004).append(",")
                .append(DESC_INT_EU_004).append(",")
                .append(ACTIVA_004).append(",")
                .append(OBLIGATORIO_004).append(",")
                .append(ID_TIPO_PREGUNTA_004).append(",")
                .append(FEC_ALTA_004).append(",")
                .append(FEC_ULTMODIF_004).append(",")
                .append(USU_ALTA_004).append(",")
                .append(USU_MODIF_004).append(",")
                .append(AVISO_004).append(",")
                .append("DESC_ES_008").append(",")
                .append("DESC_EU_008");
        return select.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getFrom()
     */
    @Override
    protected String getFrom() {
        StringBuilder from = new StringBuilder(" FROM ").append(TABLA)
                .append(" JOIN Y41E08S01 ON ").append(ID_TIPO_PREGUNTA_004).append("=").append("ID_TIPO_PREGUNTA_008");
        return from.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getRwMap()
     */
    @Override
    protected RowMapper<Y41ePregunta> getRwMap() {
        return new Y41ePreguntaRowMapper();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getPK()
     */
    @Override
    protected String getPK() {
        return ID_PREGUNTA_004;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getRwMapPK()
     */
    @Override
    protected RowMapper<Y41ePregunta> getRwMapPK() {
        return this.rwMapPK;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWherePK(java.lang.Object, java.util.List)
     */
    @Override
    protected String getWherePK(Y41ePregunta bean, List<Object> params) {
        StringBuilder wherePK = new StringBuilder();
        wherePK.append(Y41eSqlUtils.WHERE).append(ID_PREGUNTA_004).append("=?");
        params.add(bean.getIdPregunta());
        return wherePK.toString();

    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWhere(java.lang.Object, java.util.List)
     */
    @Override
    protected String getWhere(Y41ePregunta bean, List<Object> params) {
        StringBuilder where = new StringBuilder(Y41ePreguntaDaoImpl.STRING_BUILDER_INIT);

        if (bean  != null){
            where.append(Y41eSqlUtils.generarWhereIgual(ID_PREGUNTA_004, bean.getIdPregunta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_PUBL_ES_004, bean.getDescPublEs(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_PUBL_EU_004, bean.getDescPublEu(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_INT_ES_004, bean.getDescIntEs(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_INT_EU_004, bean.getDescPublEu(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ACTIVA_004, bean.getActiva(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(OBLIGATORIO_004, bean.getObligatorio(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_TIPO_PREGUNTA_004, bean.getTipoPregunta().getIdTipoPregunta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ALTA_004, bean.getFecAlta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ULTMODIF_004, bean.getFecUltModif(), params));
        }

        return where.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWhereLike(java.lang.Object, java.lang.Boolean, java.util.List, java.lang.Boolean)
     */
    @Override
    protected String getWhereLike(Y41ePregunta bean, Boolean startsWith, List<Object> params, Boolean search) {
        Locale locale = LocaleContextHolder.getLocale();
        StringBuilder where = new StringBuilder(Y41ePreguntaDaoImpl.STRING_BUILDER_INIT);

        if (bean  != null){
            where.append(Y41eSqlUtils.generarWhereIgual(ID_PREGUNTA_004, bean.getIdPregunta(), params));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_PUBL_ES_004, bean.getDescPublEs(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_PUBL_EU_004, bean.getDescPublEu(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_INT_ES_004, bean.getDescIntEs(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_INT_EU_004, bean.getDescPublEu(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereIgual(ACTIVA_004, bean.getActiva(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(OBLIGATORIO_004, bean.getObligatorio(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_TIPO_PREGUNTA_004, bean.getTipoPregunta().getIdTipoPregunta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ALTA_004, bean.getFecAlta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ULTMODIF_004, bean.getFecUltModif(), params));

            where.append(Y41eSqlUtils.generarWhereLikeNormalizado("DESC_PUBL_"+ locale.getLanguage().toUpperCase() +"_004", bean.getDescPubl(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado("DESC_INT_"+ locale.getLanguage().toUpperCase() +"_004", bean.getDescInt(), params, startsWith));

            if(bean.getIdBloque()!=null && bean.getIdBloque()>0){
                where.append(" AND NOT EXISTS (SELECT 1 FROM Y41E05S01 WHERE ID_PREGUNTA_005= ID_PREGUNTA_004 AND ID_BLOQUE_005=?)");
                params.add(bean.getIdBloque());
            }

            if(bean.getSelectedAll() != null) {
                if(bean.getSelectedAll()){
                    where.append(Y41eSqlUtils.generarWhereNotIn(ID_PREGUNTA_004, bean.getSelectedIds(), params));
                }else{
                    where.append(Y41eSqlUtils.generarWhereIn(ID_PREGUNTA_004, bean.getSelectedIds(), params));
                }
            }
        }
        return where.toString();
    }

    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id de la pregunta.
     * @param idPregunta Long
     * @return El área o áreas a la que pertenece la encuesta
     */
    @Override
    public Integer validarPreguntaEnviada(Long idPregunta){
        StringBuilder select = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        select.append("SELECT COUNT(1)")
        .append(" FROM Y41E01S01")
        .append(" JOIN Y41E10S01 ON ID_ENCUESTA_001=ID_ENCUESTA_010")
        .append(" JOIN Y41E03S01 ON ID_ENCUESTA_003= ID_ENCUESTA_001")
        .append(" JOIN Y41E05S01 ON ID_BLOQUE_005= ID_BLOQUE_003")
        .append(" WHERE FEC_RESPUESTA_010 IS NOT NULL ")
        .append(" AND ID_PREGUNTA_005=?");

        return this.getJdbcTemplate().queryForObject(select.toString(), Integer.class, idPregunta);
    }

    /**
     * Comprueba que la pregunta pertenece a una encuesta.
     * @param idPregunta Long
     * @return El área o áreas a la que pertenece la encuesta
     */
    @Override
    public Integer validarPreguntaEncuesta(Long idPregunta){
        StringBuilder select = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        select.append("SELECT COUNT(1)")
        .append(" FROM Y41E01S01")
        .append(" JOIN Y41E03S01 ON ID_ENCUESTA_003= ID_ENCUESTA_001")
        .append(" JOIN Y41E05S01 ON ID_BLOQUE_005= ID_BLOQUE_003")
        .append(" WHERE ID_PREGUNTA_005=?");

        return this.getJdbcTemplate().queryForObject(select.toString(), Integer.class, idPregunta);
    }

    /**
     * Comprueba que el bloque no esté asociado a una encuesta enviada.
     * @param idBloque Long
     * @return Integer
     */
    @Override
    public Integer validarDesasociarEnviada(Long idBloque){
        StringBuilder select = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        select.append("SELECT COUNT(1)")
        .append(" FROM Y41E01S01")
        .append(" JOIN Y41E03S01 ON ID_ENCUESTA_003= ID_ENCUESTA_001")
        .append(" JOIN Y41E10S01 ON ID_ENCUESTA_010 = ID_ENCUESTA_001")
        .append(" WHERE ID_BLOQUE_003=?")
        .append(" AND FEC_RESPUESTA_010 IS NOT NULL");

        return this.getJdbcTemplate().queryForObject(select.toString(), Integer.class, idBloque);
    }

    /**
     * Comprueba que el bloque no esté asociado a una encuesta.
     * @param idBloque Long
     * @return Integer
     */
    @Override
    public Integer validarDesasociarEncuesta(Long idBloque){
        StringBuilder select = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        select.append("SELECT COUNT(1)")
        .append(" FROM Y41E01S01")
        .append(" JOIN Y41E03S01 ON ID_ENCUESTA_003= ID_ENCUESTA_001")
        .append(" WHERE ID_BLOQUE_003=?");

        return this.getJdbcTemplate().queryForObject(select.toString(), Integer.class, idBloque);
    }

    /**
     * Obtiene el suggest de bloques
     * @param pregunta Y41ePregunta
     * @param startsWith boolean
     * @return List<Y41ePregunta>
     */
    @Override
    public List<Y41ePregunta> findAllSuggest(Y41ePregunta pregunta, boolean startsWith){
        List<Object> params = new ArrayList<Object>();
        StringBuilder sqlSuggestPregunta = new StringBuilder();
        sqlSuggestPregunta.append(this.getSelect()); 
        sqlSuggestPregunta.append(this.getFrom());
        sqlSuggestPregunta.append(Y41eSqlUtils.DEFAULT_WHERE);
        sqlSuggestPregunta.append(this.getWhereLike(pregunta, startsWith, params, false));
        return this.getJdbcTemplate().query(sqlSuggestPregunta.toString(), this.getRwMap(), params.toArray());
    }

    /**
     * Añade las respuestas que generan aviso
     * @param bean Y41ePregunta
     */
    @Override
    public void addRespuestasAviso(Y41ePregunta bean){
        //obtener el máximo orden
        StringBuilder sqlAvisos = new StringBuilder(Y41ePreguntaDaoImpl.STRING_BUILDER_INIT);
        sqlAvisos.append(Y41eSqlUtils.INSERT)
        .append("Y41E13S01").append(" (")
        .append("ID_PREGUNTA_013").append(",")
        .append("ID_TIPO_RESPUESTA_013").append(")")
        .append("VALUES (?,?)");

        final List<Object[]> paramsInsert = new ArrayList<Object[]>();
        List<Object> auxInsert = new ArrayList<Object>();
        for(String respuesta:bean.getRespuestasAviso().split(",")){
            auxInsert.clear();
            auxInsert.add(bean.getIdPregunta());
            auxInsert.add(respuesta);
            paramsInsert.add(auxInsert.toArray().clone());
        }
        this.getJdbcTemplate().batchUpdate(sqlAvisos.toString(), paramsInsert);
    }

    /**
     * Obtiene las respuestas con aviso
     * @param bean Y41ePregunta
     * @return String[]
     */
    @Override
    public String getRespuestasAviso(Y41ePregunta bean){
        StringBuilder sqlRespAviso = new StringBuilder(Y41ePreguntaDaoImpl.STRING_BUILDER_INIT);
        sqlRespAviso.append(Y41eSqlUtils.SELECT)
        .append(" LISTAGG(ID_TIPO_RESPUESTA_013, ',') WITHIN GROUP (ORDER BY ID_PREGUNTA_013) ID_TIPO_RESPUESTA_013")
        .append(Y41eSqlUtils.FROM)
        .append(" Y41E13S01")
        .append(Y41eSqlUtils.WHERE)
        .append(" ID_PREGUNTA_013=?");

        List<Object> params = new ArrayList<Object>();
        params.add(bean.getIdPregunta());

        return this.getJdbcTemplate().queryForObject(sqlRespAviso.toString(), params.toArray(), String.class);
    }

    /**
     * Eliminar los avisos de la pregunta
     * @param bean Y41ePregunta
     */
    @Override
    public void removeRespuestasAviso(Y41ePregunta bean){
        StringBuilder sql = new StringBuilder(Y41ePreguntaDaoImpl.STRING_BUILDER_INIT);
        sql.append(Y41eSqlUtils.DELETE)
        .append(" Y41E13S01")
        .append(Y41eSqlUtils.WHERE)
        .append(" ID_PREGUNTA_013=?");
        List<Object> params = new ArrayList<Object>();
        params.add(bean.getIdPregunta());

        this.getJdbcTemplate().update(sql.toString(), params.toArray());
    }

    /**
     * Devuelve la pregunta si tiene configurado avisos
     * @param idPregunta Long
     * @param idTipoRespuesta Long
     * @return Y41ePregunta
     */
    @Override
    public Y41ePregunta findPreguntaAviso(Long idPregunta, Long idTipoRespuesta){
        StringBuilder select = new StringBuilder(Y41ePreguntaDaoImpl.STRING_BUILDER_INIT);
        select.append(Y41eSqlUtils.SELECT)
        .append(DESC_INT_ES_004).append(",")
        .append(DESC_INT_EU_004).append(",")
        .append(DESC_PUBL_ES_004).append(",")
        .append(DESC_PUBL_EU_004).append(",")
        .append("DESC_ES_009, DESC_EU_009")
        .append(Y41eSqlUtils.FROM)
        .append(TABLA)
        .append(Y41eSqlUtils.JOIN).append(" Y41E13S01 ON ID_PREGUNTA_013 = ").append(ID_PREGUNTA_004)
        .append(Y41eSqlUtils.JOIN).append(" Y41E09S01 ON ID_TIPO_RESPUESTA_009= ID_TIPO_RESPUESTA_013 ")
        .append(Y41eSqlUtils.WHERE).append(ID_PREGUNTA_004).append("=?")
        .append(Y41eSqlUtils.AND).append("ID_TIPO_RESPUESTA_013=?")
        .append(Y41eSqlUtils.AND).append(AVISO_004).append("='").append(Y41eConstants.SI).append("'");

        List<Y41ePregunta> beanList = this.getJdbcTemplate().query(select.toString(), this.rwPreguntaAviso, idPregunta, idTipoRespuesta);
        return DataAccessUtils.uniqueResult(beanList);
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getOrderBy()
     */
    @Override
    protected String[] getOrderBy() {
        return Y41ePreguntaDaoImpl.ORDER_BY_WHITE_LIST;
    }

    /**
     * StringBuilder initilization value
     */
    public static final int STRING_BUILDER_INIT = 4096;
}
