package com.ejie.y41e.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.apache.commons.lang.StringUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.security.UserCredentials;
import com.ejie.y41e.common.Y41eConstants;
import com.ejie.y41e.dao.rwmap.Y41eEncuestaRowMapper;
import com.ejie.y41e.dao.rwmap.Y41eGraficoRowMapper;
import com.ejie.y41e.model.Y41eEncuesta;
import com.ejie.y41e.model.Y41eGrafico;
import com.ejie.y41e.model.Y41eResultados;
import com.ejie.y41e.utils.Y41eSqlUtils;
import com.ejie.y41e.utils.Y41eUtils;

/**
 * Y41eEncuestaDaoImpl
 *  
 */

@Repository
@Transactional
public class Y41eEncuestaDaoImpl extends Y41eGenericoDaoImpl<Y41eEncuesta> implements Y41eEncuestaDao {

    public static final String TABLA="Y41E01S01";
    public static final String SECUENCIA="Y41E01Q00";
    public static final String ID_ENCUESTA_001 = "ID_ENCUESTA_001";
    public static final String DESC_PUBL_ES_001 = "DESC_PUBL_ES_001";
    public static final String DESC_PUBL_EU_001 = "DESC_PUBL_EU_001";
    public static final String DESC_INT_ES_001 = "DESC_INT_ES_001";
    public static final String DESC_INT_EU_001 = "DESC_INT_EU_001";
    public static final String ID_AREA_001 = "ID_AREA_001";
    public static final String ID_ORIGEN_001 = "ID_ORIGEN_001";
    public static final String ACTIVA_001 = "ACTIVA_001";
    public static final String ENVIADA_001 = "ENVIADA_001";
    public static final String FEC_ENVIO_001 = "FEC_ENVIO_001";
    public static final String FEC_VIG_INICIO_001 = "FEC_VIG_INICIO_001";
    public static final String FEC_VIG_FIN_001 = "FEC_VIG_FIN_001";
    public static final String PERIODO_RESP_001 = "PERIODO_RESP_001";
    public static final String FEC_ALTA_001 = "FEC_ALTA_001";
    public static final String FEC_ULTMODIF_001 = "FEC_ULTMODIF_001";
    public static final String USU_ALTA_001 = "USU_ALTA_001";
    public static final String USU_MODIF_001 = "USU_MODIF_001";
    public static final String PARAM_001 = "PARAM_001";
    protected static final String[] ORDER_BY_WHITE_LIST = new String[] { ID_ENCUESTA_001, DESC_PUBL_ES_001, DESC_PUBL_EU_001, DESC_INT_ES_001,
            DESC_INT_EU_001, ID_AREA_001, ID_ORIGEN_001, ACTIVA_001, ENVIADA_001, FEC_ENVIO_001, FEC_VIG_INICIO_001, FEC_VIG_FIN_001,
            PERIODO_RESP_001, FEC_ALTA_001, FEC_ULTMODIF_001, USU_ALTA_001, USU_MODIF_001};

    /**
     * Y41eEncuestaDaoImpl
     */
    public Y41eEncuestaDaoImpl() {
        super(Y41eEncuesta.class);
    }

    /**
     * rwMapPK
     */
    private final RowMapper<Y41eEncuesta> rwMapPK = new RowMapper<Y41eEncuesta>() {
        @Override
        public Y41eEncuesta mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            return new Y41eEncuesta(
                    resultSet.getLong(ID_ENCUESTA_001)
                    ); 
        }
    };

    /**
     * rwExcelEncuesta
     */
    private final RowMapper<Y41eResultados> rwExcelEncuesta = new RowMapper<Y41eResultados>() {
        @Override
        public Y41eResultados mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Y41eResultados result = new Y41eResultados();
            result.getEnvio().getEncuesta().setIdEncuesta(resultSet.getLong("ID_ENCUESTA_001"));
            result.getEnvio().getEncuesta().setDescPublEs(resultSet.getString("DESC_PUBL_ES_001"));
            result.getEnvio().getEncuesta().setDescPublEu(resultSet.getString("DESC_PUBL_EU_001"));
            result.getBloque().setIdBloque(resultSet.getLong("ID_BLOQUE_002"));
            result.getBloque().setDescPublEs(resultSet.getString("DESC_PUBL_ES_002"));
            result.getBloque().setDescPublEu(resultSet.getString("DESC_PUBL_EU_002"));
            result.getPregunta().setIdPregunta(resultSet.getLong("ID_PREGUNTA_004"));
            result.getPregunta().setDescPublEs(resultSet.getString("DESC_PUBL_ES_004"));
            result.getPregunta().setDescPublEu(resultSet.getString("DESC_PUBL_EU_004"));
            result.getPregunta().setRespuestas(resultSet.getString("RESPUESTAS"));
            return result;
        }
    };

    /**
     * rwMapEncuestaInternet
     */
    private final RowMapper<Y41eEncuesta> rwMapEncuestaInternet = new RowMapper<Y41eEncuesta>() {
        @Override
        public Y41eEncuesta mapRow(ResultSet resultSet, int rowNum) throws SQLException {
            Y41eEncuesta encuesta = new Y41eEncuesta();
            encuesta.setIdEncuesta(resultSet.getLong(ID_ENCUESTA_001));
            encuesta.setDescPublEs(resultSet.getString(DESC_PUBL_ES_001));
            encuesta.setDescPublEu(resultSet.getString(DESC_PUBL_EU_001));
            encuesta.setActiva(resultSet.getString(ACTIVA_001));
            encuesta.setFecVigInicio(resultSet.getDate(FEC_VIG_INICIO_001));
            encuesta.setFecVigFin(resultSet.getDate(FEC_VIG_FIN_001));
            encuesta.setPeriodoResp(resultSet.getLong(PERIODO_RESP_001));
            encuesta.setFecEnvio(resultSet.getDate(FEC_ENVIO_001));
            encuesta.setIdEnvio(resultSet.getLong("ID_ENVIO_010"));
            encuesta.setRespondida(resultSet.getString("RESPONDIDA"));
            return encuesta;
        }
    };

    /*
     * OPERACIONES CRUD
     */

    /**
     * Inserts a single row in the Y41eEncuesta table.
     *
     * @param encuesta Y41eEncuesta
     * @return Y41eEncuesta
     */
    @Override
    public Y41eEncuesta add(Y41eEncuesta encuesta) {

        encuesta.setIdEncuesta(this.getNextVal(SECUENCIA));

        StringBuilder insert = new StringBuilder(Y41eSqlUtils.INSERT)
                .append(TABLA).append(" (")
                .append(ID_ENCUESTA_001).append(",")
                .append(DESC_PUBL_ES_001).append(",")
                .append(DESC_PUBL_EU_001).append(",")
                .append(DESC_INT_ES_001).append(",")
                .append(DESC_INT_EU_001).append(",")
                .append(ID_AREA_001).append(",")
                .append(ID_ORIGEN_001).append(",")
                .append(ACTIVA_001).append(",")
                .append(ENVIADA_001).append(",")
                .append(FEC_ENVIO_001).append(",")
                .append(FEC_VIG_INICIO_001).append(",")
                .append(FEC_VIG_FIN_001).append(",")
                .append(PERIODO_RESP_001).append(",")
                .append(FEC_ALTA_001).append(",")
                .append(FEC_ULTMODIF_001).append(",")
                .append(USU_ALTA_001).append(",")
                .append(USU_MODIF_001).append(",")
                .append(PARAM_001).append(")")
                .append("VALUES (?,")
                .append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append("?,?,?,'N',?,?,?,?, SYSDATE, SYSDATE,?,?,?)");

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        final UserCredentials  credentials = (UserCredentials) authentication.getCredentials();

        this.getJdbcTemplate().update(insert.toString(), encuesta.getIdEncuesta(), encuesta.getDescPublEs(), encuesta.getDescPublEu(), encuesta.getDescIntEs(), encuesta.getDescIntEu(),
                encuesta.getArea().getIdArea(), encuesta.getOrigen().getIdOrigen(), StringUtils.isBlank(encuesta.getActiva())?Y41eConstants.NO:encuesta.getActiva(), 
                        encuesta.getFecEnvio(), encuesta.getFecVigInicio(), encuesta.getFecVigFin(), encuesta.getPeriodoResp(), credentials.getNif(),credentials.getNif(), encuesta.getUrl());

        return encuesta;
    }

    /**
     * Updates a single row in the Y41eEncuesta table.
     *
     * @param encuesta Y41eEncuesta
     * @return Y41eEncuesta
     */
    @Override
    public Y41eEncuesta update(Y41eEncuesta encuesta) {
        StringBuilder update = new StringBuilder(Y41eSqlUtils.UPDATE)
                .append(TABLA).append(" SET ")
                .append(DESC_PUBL_ES_001).append("=").append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(DESC_PUBL_EU_001).append("=").append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(DESC_INT_ES_001).append("=").append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(DESC_INT_EU_001).append("=").append(Y41eSqlUtils.SUBSTR_250).append(",")
                .append(ID_AREA_001).append("=?,")
                .append(ID_ORIGEN_001).append("=?,")
                .append(ACTIVA_001).append("=?,")
                .append(FEC_VIG_INICIO_001).append("=?,")
                .append(FEC_VIG_FIN_001).append("=?,")
                .append(PERIODO_RESP_001).append("=?,")
                .append(FEC_ULTMODIF_001).append("=SYSDATE,")
                .append(USU_MODIF_001).append("=?")
                .append(Y41eSqlUtils.WHERE)
                .append(ID_ENCUESTA_001).append("=?");

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        final UserCredentials  credentials = (UserCredentials) authentication.getCredentials();

        this.getJdbcTemplate().update(update.toString(), encuesta.getDescPublEs(), encuesta.getDescPublEu(), encuesta.getDescIntEs(), encuesta.getDescIntEu(),
                encuesta.getArea().getIdArea(), encuesta.getOrigen().getIdOrigen(), StringUtils.isBlank(encuesta.getActiva())?Y41eConstants.NO:encuesta.getActiva(),
                        encuesta.getFecVigInicio(), encuesta.getFecVigFin(), encuesta.getPeriodoResp(),
                        credentials.getNif(), encuesta.getIdEncuesta());

        return encuesta;
    }

    /**
     * Removes a single row in the Y41eEncuesta table.
     *
     * @param encuesta Y41eEncuesta
     * @return
     */
    @Override
    public void remove(Y41eEncuesta encuesta) {
        StringBuilder delete = new StringBuilder(Y41eSqlUtils.DELETE).append(TABLA).append(Y41eSqlUtils.WHERE).append(ID_ENCUESTA_001).append("=?");
        this.getJdbcTemplate().update(delete.toString(), encuesta.getIdEncuesta());
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getSelect()
     */
    @Override
    protected String getSelect() {
        StringBuilder select = new StringBuilder("SELECT ")
                .append(ID_ENCUESTA_001).append(",")
                .append(DESC_PUBL_ES_001).append(",")
                .append(DESC_PUBL_EU_001).append(",")
                .append(DESC_INT_ES_001).append(",")
                .append(DESC_INT_EU_001).append(",")
                .append(ID_AREA_001).append(",")
                .append(ID_ORIGEN_001).append(",")
                .append(ACTIVA_001).append(",")
                .append(ENVIADA_001).append(",")
                .append(FEC_ENVIO_001).append(",")
                .append(FEC_VIG_INICIO_001).append(",")
                .append(FEC_VIG_FIN_001).append(",")
                .append(PERIODO_RESP_001).append(",")
                .append(FEC_ALTA_001).append(",")
                .append(FEC_ULTMODIF_001).append(",")
                .append(USU_ALTA_001).append(",")
                .append(USU_MODIF_001).append(",")
                .append("DESC_ES_006").append(",")
                .append("DESC_EU_006").append(",")
                .append("DESC_ES_007").append(",")
                .append("DESC_EU_007").append(",")
                .append("DECODE(NVL(RESPONDIDA,0), 0, 'N','S') RESPONDIDA").append(",")
                .append(PARAM_001);
        return select.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getFrom()
     */
    @Override
    protected String getFrom() {
        StringBuilder from = new StringBuilder(" FROM ").append(TABLA)
                .append(" LEFT JOIN Y41E06S01 ON ").append(ID_ORIGEN_001).append("=").append("ID_ORIGEN_006")
                .append(" LEFT JOIN Y41E07S01 ON ").append(ID_AREA_001).append("=").append("ID_AREA_007")
                .append(" LEFT JOIN (SELECT COUNT(1) RESPONDIDA, ID_ENCUESTA_010 FROM Y41E10S01 GROUP BY ID_ENCUESTA_010)")
                .append(" ON ID_ENCUESTA_010=").append(ID_ENCUESTA_001);
        return from.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getRwMap()
     */
    @Override
    protected RowMapper<Y41eEncuesta> getRwMap() {
        return new Y41eEncuestaRowMapper();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getPK()
     */
    @Override
    protected String getPK() {
        return ID_ENCUESTA_001;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getRwMapPK()
     */
    @Override
    protected RowMapper<Y41eEncuesta> getRwMapPK() {
        return this.rwMapPK;
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWherePK(java.lang.Object, java.util.List)
     */
    @Override
    protected String getWherePK(Y41eEncuesta bean, List<Object> params) {
        StringBuilder wherePK = new StringBuilder();
        wherePK.append(Y41eSqlUtils.WHERE).append(ID_ENCUESTA_001).append("=?");
        params.add(bean.getIdEncuesta());
        return wherePK.toString();

    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWhere(java.lang.Object, java.util.List)
     */
    @Override
    protected String getWhere(Y41eEncuesta bean, List<Object> params) {
        StringBuilder where = new StringBuilder(Y41eEncuestaDaoImpl.STRING_BUILDER_INIT);

        if (bean  != null){
            where.append(Y41eSqlUtils.generarWhereIgual(ID_ENCUESTA_001, bean.getIdEncuesta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_PUBL_ES_001, bean.getDescPublEs(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_PUBL_EU_001, bean.getDescPublEu(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_INT_ES_001, bean.getDescIntEs(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(DESC_INT_EU_001, bean.getDescPublEu(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_AREA_001, bean.getArea().getIdArea(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_ORIGEN_001, bean.getOrigen().getIdOrigen(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ACTIVA_001, bean.getActiva(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ENVIADA_001, bean.getEnviada(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ENVIO_001, bean.getFecEnvio(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_VIG_INICIO_001, bean.getFecVigInicio(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_VIG_FIN_001, bean.getFecVigFin(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(PERIODO_RESP_001, bean.getPeriodoResp(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ALTA_001, bean.getFecAlta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ULTMODIF_001, bean.getFecUltModif(), params));
        }

        return where.toString();
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getWhereLike(java.lang.Object, java.lang.Boolean, java.util.List, java.lang.Boolean)
     */
    @Override
    protected String getWhereLike(Y41eEncuesta bean, Boolean startsWith, List<Object> params, Boolean search) {
        Locale locale = LocaleContextHolder.getLocale();
        StringBuilder where = new StringBuilder(Y41eEncuestaDaoImpl.STRING_BUILDER_INIT);
        if (bean  != null){
            where.append(Y41eSqlUtils.generarWhereIgual(ID_ENCUESTA_001, bean.getIdEncuesta(), params));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_PUBL_ES_001, bean.getDescPublEs(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_PUBL_EU_001, bean.getDescPublEu(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_INT_ES_001, bean.getDescIntEs(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado(DESC_INT_EU_001, bean.getDescPublEu(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_AREA_001, bean.getArea().getIdArea(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ID_ORIGEN_001, bean.getOrigen().getIdOrigen(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(PERIODO_RESP_001, bean.getPeriodoResp(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ALTA_001, bean.getFecAlta(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(FEC_ULTMODIF_001, bean.getFecUltModif(), params));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado("DESC_PUBL_"+ locale.getLanguage().toUpperCase() +"_001", bean.getDescPubl(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereLikeNormalizado("DESC_INT_"+ locale.getLanguage().toUpperCase() +"_001", bean.getDescInt(), params, startsWith));
            where.append(Y41eSqlUtils.generarWhereIgual(ACTIVA_001, bean.getActiva(), params));
            where.append(Y41eSqlUtils.generarWhereIgual(ENVIADA_001, bean.getEnviada(), params));
            where.append(Y41eSqlUtils.generarWhereMayorIgual(FEC_ENVIO_001, bean.getFecEnvioInicio(), params));
            where.append(Y41eSqlUtils.generarWhereMenorIgual(FEC_ENVIO_001, bean.getFecEnvioFin(), params));
            where.append(Y41eSqlUtils.generarWhereMayorIgual(FEC_VIG_INICIO_001, bean.getFecVigInicio(), params));
            where.append(Y41eSqlUtils.generarWhereMenorIgual(FEC_VIG_FIN_001, bean.getFecVigFin(), params));

            if(StringUtils.isNotBlank(bean.getPeriodoRespuesta())){
                where.append(Y41eSqlUtils.AND).append(" SYSDATE ");
                if(Y41eConstants.NO.equals(bean.getPeriodoRespuesta())){
                    where.append(" NOT ");
                }
                where.append(" BETWEEN ").append(FEC_ENVIO_001).append(Y41eSqlUtils.AND).append(FEC_ENVIO_001).append("+").append(PERIODO_RESP_001);
            }

        }
        return where.toString();
    }

    /**
     * Comprueba que no existe ninguna encuesta enviada o respondida para el id de la encuesta.
     * 
     * @param idEncuesta Long
     * @return String el area a la que pertence la encuesta 
     *            
     */
    @Override
    public String validarEncuesta(Long idEncuesta){
        StringBuilder select = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        select.append("SELECT COUNT(1) ")
        .append(Y41eSqlUtils.FROM)
        .append(TABLA)
        .append(" LEFT JOIN Y41E10S01 ON ID_ENCUESTA_001=ID_ENCUESTA_010")
        .append(" WHERE FEC_RESPUESTA_010 IS NOT NULL ")
        .append(" AND ID_ENCUESTA_001=?");

        return this.getJdbcTemplate().queryForObject(select.toString(), String.class, idEncuesta);
    }

    /**
     * Crea una encuesta como copia de la que recibe por parámetro
     * @param y41eEncuesta Y41eEncuesta
     * @return Y41eEncuesta
     */
    @Override
    public Y41eEncuesta copiarEncuesta(Y41eEncuesta y41eEncuesta){
        Y41eEncuesta encuesta = new Y41eEncuesta();
        encuesta.setIdEncuesta(this.getNextVal(SECUENCIA));

        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        final UserCredentials  credentials = (UserCredentials) authentication.getCredentials();

        StringBuilder sqlCopiar = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        sqlCopiar.append(Y41eSqlUtils.INSERT).append(TABLA).append(" (");
        sqlCopiar.append(ID_ENCUESTA_001).append(",").append(DESC_PUBL_ES_001).append(",");
        sqlCopiar.append(DESC_PUBL_EU_001).append(",").append(DESC_INT_ES_001).append(",");
        sqlCopiar.append(DESC_INT_EU_001).append(",").append(ID_AREA_001).append(",");
        sqlCopiar.append(ID_ORIGEN_001).append(",").append(ACTIVA_001).append(",");
        sqlCopiar.append(ENVIADA_001).append(",").append(FEC_ENVIO_001).append(",");
        sqlCopiar.append(FEC_VIG_INICIO_001).append(",").append(FEC_VIG_FIN_001).append(",");
        sqlCopiar.append(PERIODO_RESP_001).append(",").append(FEC_ALTA_001).append(",");
        sqlCopiar.append(FEC_ULTMODIF_001).append(",").append(USU_ALTA_001).append(",");
        sqlCopiar.append(USU_MODIF_001).append(",");
        sqlCopiar.append(PARAM_001).append(")");
        sqlCopiar.append(Y41eSqlUtils.SELECT).append(encuesta.getIdEncuesta()).append(",")
        .append(Y41eSqlUtils.SUBSTR).append("(").append(DESC_PUBL_ES_001).append(Y41eSqlUtils.LENGTH_0_244).append("|| '").append(Y41eConstants.SUFIJO_COPIA).append("',")
        .append(Y41eSqlUtils.SUBSTR).append("(").append(DESC_PUBL_EU_001).append(Y41eSqlUtils.LENGTH_0_244).append("|| '").append(Y41eConstants.SUFIJO_COPIA).append("',")
        .append(Y41eSqlUtils.SUBSTR).append("(").append(DESC_INT_ES_001).append(Y41eSqlUtils.LENGTH_0_244).append("|| '").append(Y41eConstants.SUFIJO_COPIA).append("',")
        .append(Y41eSqlUtils.SUBSTR).append("(").append(DESC_INT_EU_001).append(Y41eSqlUtils.LENGTH_0_244).append("|| '").append(Y41eConstants.SUFIJO_COPIA).append("',")
        .append(ID_AREA_001).append(",").append(ID_ORIGEN_001).append(",'").append(Y41eConstants.SI)
        .append("','").append(Y41eConstants.NO).append("', NULL, NULL, NULL, NULL,")
        .append("SYSDATE, SYSDATE, '").append(credentials.getNif()).append("', '").append(credentials.getNif()).append("', '").append(Y41eUtils.generarCodigo()).append("'")
        .append(Y41eSqlUtils.FROM).append(TABLA).append(Y41eSqlUtils.WHERE).append(ID_ENCUESTA_001).append("=?");

        this.getJdbcTemplate().update(sqlCopiar.toString(), y41eEncuesta.getIdEncuesta());

        return encuesta;
    }

    /**
     * Obtiene las preguntas de la encuesta
     * @param resultado Y41eResultados
     * @return List<Y41eGrafico>
     */
    @Override
    public List<Y41eGrafico> getGraficosEncuesta(Y41eResultados resultado){
        StringBuilder sqlGrafico = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        List<Object> params = new ArrayList<Object>();
        sqlGrafico.append(Y41eSqlUtils.SELECT).append("ID_BLOQUE_002,DESC_PUBL_ES_002, DESC_PUBL_EU_002, DESC_INT_ES_002, DESC_INT_EU_002,")
        .append("ID_PREGUNTA_004, DESC_PUBL_ES_004, DESC_PUBL_EU_004, DESC_INT_ES_004, DESC_INT_EU_004, GRAFICO_008")
        .append(Y41eSqlUtils.FROM).append(" Y41E05S01 ")
        .append(Y41eSqlUtils.JOIN).append(" Y41E03S01 ON ID_BLOQUE_003=ID_BLOQUE_005")
        .append(Y41eSqlUtils.JOIN).append(" Y41E02S01 ON ID_BLOQUE_003=ID_BLOQUE_002")
        .append(Y41eSqlUtils.JOIN).append(" Y41E04S01 ON ID_PREGUNTA_004=ID_PREGUNTA_005")
        .append(Y41eSqlUtils.JOIN).append(" Y41E08S01 ON ID_TIPO_PREGUNTA_008=ID_TIPO_PREGUNTA_004")
        .append(Y41eSqlUtils.JOIN).append(" Y41E10S01 ON ID_ENCUESTA_010=ID_ENCUESTA_003")
        .append(Y41eSqlUtils.WHERE).append(" ID_ENCUESTA_003=?");

        params.add(resultado.getEnvio().getEncuesta().getIdEncuesta());
        sqlGrafico.append(Y41eSqlUtils.generarWhereMayorIgual("FEC_RESPUESTA_010", resultado.getEnvio().getFecRespuestaInicio(), params));
        sqlGrafico.append(Y41eSqlUtils.generarWhereMenorIgual("FEC_RESPUESTA_010", resultado.getEnvio().getFecRespuestaFin(), params));

        sqlGrafico.append(Y41eSqlUtils.GROUP_BY)
        .append(" ID_BLOQUE_002,DESC_PUBL_ES_002, DESC_PUBL_EU_002, DESC_INT_ES_002, DESC_INT_EU_002, ")
        .append(" ID_PREGUNTA_004, DESC_PUBL_ES_004, DESC_PUBL_EU_004, DESC_INT_ES_004, DESC_INT_EU_004, GRAFICO_008, ORDEN_BLOQ_003, ORDEN_PREG_005")
        .append(Y41eSqlUtils.ORDER_BY)
        .append(" ORDEN_BLOQ_003 ASC, ORDEN_PREG_005 ASC");

        return this.getJdbcTemplate().query(sqlGrafico.toString(), new Y41eGraficoRowMapper() , params.toArray());

    }

    /**
     * Obtiene los datos para generar el excel de encuesta
     * @param y41eEncuesta
     * @return List<Y41eResultados>
     */
    @Override
    public List<Y41eResultados> getExcelEncuesta(Y41eEncuesta y41eEncuesta){
        List<Object> params = new ArrayList<Object>();
        StringBuilder sqlResultados = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        sqlResultados.append(Y41eSqlUtils.SELECT)
        .append(ID_ENCUESTA_001).append(",")
        .append(DESC_PUBL_ES_001).append(",")
        .append(DESC_PUBL_EU_001).append(",")
        .append(" ID_BLOQUE_002, DESC_PUBL_ES_002, DESC_PUBL_EU_002, ID_PREGUNTA_004, DESC_PUBL_ES_004, DESC_PUBL_EU_004,")
        .append("LISTAGG(DECODE(DESC_EU_009, DESC_ES_009, DESC_EU_009, DESC_EU_009|| '/' || DESC_ES_009), ' ') WITHIN GROUP (ORDER BY ID_TIPO_RESPUESTA_009, ID_TIPO_PREGUNTA_009) RESPUESTAS")
        .append(Y41eSqlUtils.FROM).append(TABLA)
        .append(Y41eSqlUtils.LEFT_JOIN).append(" Y41E03S01 ON ID_ENCUESTA_001=ID_ENCUESTA_003")
        .append(Y41eSqlUtils.LEFT_JOIN).append(" Y41E02S01 ON ID_BLOQUE_003=ID_BLOQUE_002")
        .append(Y41eSqlUtils.LEFT_JOIN).append(" Y41E05S01 ON ID_BLOQUE_003=ID_BLOQUE_005")
        .append(Y41eSqlUtils.LEFT_JOIN).append(" Y41E04S01 ON ID_PREGUNTA_004=ID_PREGUNTA_005")
        .append(Y41eSqlUtils.LEFT_JOIN).append(" Y41E09S01 ON ID_TIPO_PREGUNTA_009=ID_TIPO_PREGUNTA_004")
        .append(Y41eSqlUtils.WHERE).append(ID_ENCUESTA_001).append("=?")
        .append(Y41eSqlUtils.GROUP_BY)
        .append(ID_ENCUESTA_001).append(",").append(DESC_PUBL_ES_001).append(",").append(DESC_PUBL_EU_001).append(",")
        .append(" ID_BLOQUE_002, DESC_PUBL_ES_002, DESC_PUBL_EU_002, ID_PREGUNTA_004, DESC_PUBL_ES_004, DESC_PUBL_EU_004, ORDEN_BLOQ_003, ORDEN_PREG_005")
        .append(Y41eSqlUtils.ORDER_BY).append("ORDEN_BLOQ_003 ASC, ORDEN_PREG_005 ASC");

        params.add(y41eEncuesta.getIdEncuesta());

        return this.getJdbcTemplate().query(sqlResultados.toString(), rwExcelEncuesta ,params.toArray());
    }

    /**
     * Cambia la encuesta a enviada y actualiza la fecha de envio
     * @param encuesta Y41eEncuesta
     */
    @Override
    public void enviarEncuesta(Y41eEncuesta encuesta){
        List<Object> params = new ArrayList<Object>();
        StringBuilder sqlUpdate = new StringBuilder(Y41eResultadosDaoImpl.STRING_BUILDER_INIT);
        sqlUpdate.append(Y41eSqlUtils.UPDATE)
        .append(TABLA).append(" SET ")
        .append(ENVIADA_001).append("='").append(Y41eConstants.SI).append("',")
        .append(FEC_ENVIO_001).append("=SYSDATE")
        .append(Y41eSqlUtils.WHERE)
        .append(ID_ENCUESTA_001).append("=?");
        params.add(encuesta.getIdEncuesta());
        this.getJdbcTemplate().update(sqlUpdate.toString(), params.toArray());
    }

    /**
     * Obtiene la encuesta según el parámetro
     * @param param
     * @return
     */
    @Override
    public Y41eEncuesta getEncuestaWsByParam(String param){
        List<Object> params = new ArrayList<Object>();
        StringBuilder sql = new StringBuilder(Y41eSqlUtils.SELECT)
                .append(ID_ENCUESTA_001).append(",")
                .append(DESC_PUBL_ES_001).append(",")
                .append(DESC_PUBL_EU_001).append(",")
                .append(ACTIVA_001).append(",")
                .append(FEC_VIG_INICIO_001).append(",")
                .append(FEC_VIG_FIN_001).append(",")
                .append(FEC_ENVIO_001).append(",")
                .append(PERIODO_RESP_001).append(",")
                .append("ID_ENVIO_010").append(",")
                .append("NVL2(FEC_RESPUESTA_010, 'S', 'N') RESPONDIDA")
                .append(Y41eSqlUtils.FROM)
                .append(TABLA)
                .append(Y41eSqlUtils.JOIN).append(" Y41E10S01")
                .append(" ON ").append(ID_ENCUESTA_001).append("= ID_ENCUESTA_010")
                .append(Y41eSqlUtils.WHERE)
                .append("PARAM_010=?");
        params.add(param);

        List<Y41eEncuesta> beanList = this.getJdbcTemplate().query(sql.toString(), rwMapEncuestaInternet, params.toArray());
        return DataAccessUtils.uniqueResult(beanList);
    }


    /**
     * Obtiene la encuesta según el parámetro  en la tabla Y41E01T00
     * @param param
     * @return
     */
    @Override
    public Y41eEncuesta getEncuestaByParam(String param){
        List<Object> params = new ArrayList<Object>();
        StringBuilder sql = new StringBuilder(Y41eSqlUtils.SELECT)
                .append(ID_ENCUESTA_001).append(",")
                .append(DESC_PUBL_ES_001).append(",")
                .append(DESC_PUBL_EU_001).append(",")
                .append(ACTIVA_001).append(",")
                .append(FEC_VIG_INICIO_001).append(",")
                .append(FEC_VIG_FIN_001).append(",")
                .append(" NVL(FEC_ENVIO_001,FEC_ALTA_001) FEC_ENVIO_001" ).append(",")
                .append(PERIODO_RESP_001).append(",")
                .append(" NULL ID_ENVIO_010").append(",")
                .append("'N' RESPONDIDA")
                .append(Y41eSqlUtils.FROM)
                .append(TABLA)
                .append(Y41eSqlUtils.WHERE)
                .append("PARAM_001=?");
        params.add(param);

        List<Y41eEncuesta> beanList = this.getJdbcTemplate().query(sql.toString(), rwMapEncuestaInternet, params.toArray());
        return DataAccessUtils.uniqueResult(beanList);
    }

    /* (non-Javadoc)
     * @see com.ejie.y41e.dao.Y41eGenericoDaoImpl#getOrderBy()
     */
    @Override
    protected String[] getOrderBy() {
        return Y41eEncuestaDaoImpl.ORDER_BY_WHITE_LIST;
    }

    /**
     * StringBuilder initilization value
     */
    public static final int STRING_BUILDER_INIT = 4096;
}
