package com.ejie.y42b.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.simple.SimpleJdbcTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y42b.constantes.Y42bAgendaConstantes;
import com.ejie.y42b.model.T17Calle;
import com.ejie.y42b.model.T17Portal;

/**
 * * T17CalleDaoImpl  
 * 
 *  
 */
@Repository()
@Transactional()
public class T17CalleDaoImpl implements T17CalleDao {

	private SimpleJdbcTemplate jdbcTemplate;
	private RowMapper<T17Calle> rwMap = new RowMapper<T17Calle>() {
		public T17Calle mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {

			return new T17Calle(resultSet.getLong("ID"),
					resultSet.getString("DS_O"));
		}
	};

	private RowMapper<T17Calle> rwMapProvMun = new RowMapper<T17Calle>() {
		public T17Calle mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {

			T17Calle calle = new T17Calle();
			calle.setProvinciaId(resultSet.getString("PROVINCIA_ID"));
			calle.setMunicipioId(resultSet.getString("MUNICIPIO_ID"));
			return calle;
		}
	};

	private RowMapper<T17Calle> rwMapBusquedaCalle = new RowMapper<T17Calle>() {
		public T17Calle mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {

			T17Calle calle = new T17Calle();
			calle.setId(resultSet.getLong("ID"));
			calle.setTipoViaId(resultSet.getString("TIPO_VIA_ID"));
			calle.setDsO(resultSet.getString("DS_O"));

			return calle;

		}
	};

	/**
	 * Method use to set the datasource.
	 * 
	 * @param dataSource
	 *            DataSource
	 * 
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new SimpleJdbcTemplate(dataSource);
	}

	/**
	 * Finds a single row in the T17Calle table.
	 * 
	 * @param t17calle
	 *            calle
	 * @return T17Calle
	 */
	@Transactional(readOnly = true)
	public T17Calle find(T17Calle t17calle) {
		try {
			String query = "SELECT t1.ID ID, t1.DS_O DS_O "
					+ "FROM S17CALLE t1  " + "WHERE t1.ID = ?";
			return (T17Calle) this.jdbcTemplate.queryForObject(query,
					this.rwMap, t17calle.getId());
		} catch (EmptyResultDataAccessException e) {
			T17Calle t17calleResul = new T17Calle();
			t17calleResul.setDsO("");
			return t17calleResul;
		}
	}

	/**
	 * Finds a List of rows in the T17Calle table.
	 * 
	 * @param t17calle
	 *            calle
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<T17Calle> findAll(T17Calle t17calle, Pagination pagination) {
		StringBuilder where = new StringBuilder(Y42bAgendaConstantes.NUM_3000);
		List<Object> params = new ArrayList<Object>();
		where.append(" WHERE 1=1 	");
		StringBuilder query = new StringBuilder(
				"SELECT ID, (t1.tipo_via||'  '||t1.ds_o) AS DS_O  FROM S17CALLE t1  ");

		where.append(" AND t1.PROVINCIA_ID = ?");
		params.add(t17calle.getProvinciaId());

		where.append(" AND t1.MUNICIPIO_ID = ?");
		params.add(t17calle.getMunicipioId());

		query.append(where);

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}

		return (List<T17Calle>) this.jdbcTemplate.query(query.toString(),
				this.rwMap, params.toArray());
	}

	/**
	 * Counts rows in the T17Calle table.
	 * 
	 * @param t17calle
	 *            calle
	 * @return Long
	 */
	@Transactional(readOnly = true)
	public Long findAllCount(T17Calle t17calle) {

		StringBuilder where = new StringBuilder(Y42bAgendaConstantes.NUM_3000);
		List<Object> params = new ArrayList<Object>();
		where.append(" WHERE 1=1  ");

		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM  S17CALLE t1  ");
		if (t17calle != null && t17calle.getProvinciaId() != null) {
			where.append(" AND t1.PROVINCIA_ID = ?");
			params.add(t17calle.getId());
		}
		if (t17calle != null && t17calle.getMunicipioId() != null) {
			where.append(" AND t1.MUNICIPIO_ID = ?");
			params.add(t17calle.getMunicipioId());
		}

		query.append(where);

		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());

	}

	/**
	 * Finds rows in the T17Calle table using like.
	 * 
	 * @param t17calle
	 *            T17Calle
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<T17Calle> findAllLike(T17Calle t17calle, Pagination pagination,
			Boolean startsWith) {
		StringBuilder where = new StringBuilder(Y42bAgendaConstantes.NUM_3000);
		List<Object> params = new ArrayList<Object>();
		where.append(" WHERE 1=1 	");
		StringBuilder query = new StringBuilder(
				"SELECT t1.ID ID, t1.DS_O DS_O FROM S17CALLE t1  ");

		if (t17calle != null && t17calle.getDsO() != null) {
			where.append(" AND t1.ID like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(t17calle.getDsO() + "%");
			} else {
				params.add("%" + t17calle.getDsO() + "%");
			}
			where.append(" AND t1.DS_O IS NOT NULL");

		}

		query.append(where);

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}

		return (List<T17Calle>) this.jdbcTemplate.query(query.toString(),
				this.rwMap, params.toArray());
	}

	@Override()
	public T17Calle add(T17Calle t17calle) {
		return null;
	}

	@Override()
	public T17Calle update(T17Calle t17calle) {
		return null;
	}

	@Override()
	public void remove(T17Calle t17calle) {

	}

	/**
	 * Finds a List of distinct CP rows in the T17Portal table.
	 * 
	 * @param t17Portal
	 *            T17Portal
	 * @return T17Calle
	 */
	@Transactional(readOnly = true)
	public T17Calle findProvMun(T17Portal t17Portal) {
		StringBuilder query = new StringBuilder();
		List<Object> params = new ArrayList<Object>();

		query.append("SELECT t1.MUNICIPIO_ID, t1.PROVINCIA_ID FROM S17CALLE t1 WHERE ID=(");
		query.append("SELECT  * FROM ( ");
		query.append("SELECT  distinct(t2.CALLE_ID) CALLE_ID FROM S17PORTAL t2 ");
		query.append("WHERE 1=1");

		if (t17Portal != null && t17Portal.getCodigoPostal() != null) {
			query.append(" AND TRIM(t2.CODIGO_POSTAL) = ?");
			params.add(t17Portal.getCodigoPostal());
		}
		query.append(" ORDER BY t2.CALLE_ID ASC");
		query.append(") WHERE ROWNUM <= 1");
		query.append(")");

		return (T17Calle) this.jdbcTemplate.queryForObject(query.toString(),
				rwMapProvMun, params.toArray());
	}

	/**
	 * Finds a List of rows in the T17Calle table.
	 * 
	 * @param t17calle
	 *            T17Calle
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<T17Calle> findAllCalleLike(T17Calle t17calle,
			Pagination pagination, Boolean startsWith) {
		StringBuilder where = new StringBuilder(3000);
		List<Object> params = new ArrayList<Object>();

		StringBuilder query = new StringBuilder(
				"SELECT distinct(t1.id) as ID, t1.tipo_via AS TIPO_VIA_ID, t1.ds_O AS DS_O  FROM S17CALLE t1, S17_TIPO_VIA t2");

		where.append(" WHERE 1=1 	");
		where.append(" AND t1.PROVINCIA_ID = ?");
		params.add(t17calle.getProvinciaId());

		where.append(" AND t1.MUNICIPIO_ID = ?");
		params.add(t17calle.getMunicipioId());

		if (t17calle != null && t17calle.getTipoViaId() != null) {
			where.append(" AND t1.TIPO_VIA_ID = t2.ID");
			where.append(" AND t2.ID = ?");
			params.add(t17calle.getTipoViaId());
		}

		if (t17calle != null && t17calle.getDsO() != null) {
			where.append(" AND UPPER(t1.DS_O) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(t17calle.getDsO().toUpperCase() + "%");
			} else {
				params.add("%" + t17calle.getDsO().toUpperCase() + "%");
			}
			where.append(" AND t1.DS_O IS NOT NULL");
		}

		query.append(where);

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}

		List<T17Calle> miLista = this.jdbcTemplate.query(query.toString(),
				rwMapBusquedaCalle, params.toArray());

		return miLista;
	}

	/**
	 * Counts rows in the T17Calle table.
	 * 
	 * @param t17portal
	 *            T17Calle
	 * @return Long
	 */
	@Transactional(readOnly = true)
	public Long findAllCalleLikeCount(T17Calle t17calle, Boolean startsWith) {

		StringBuilder where = new StringBuilder(3000);
		List<Object> params = new ArrayList<Object>();

		StringBuilder query = new StringBuilder(
				"SELECT COUNT(distinct(t1.id)) FROM S17CALLE t1, S17_TIPO_VIA t2");

		where.append(" WHERE 1=1 	");
		where.append(" AND t1.PROVINCIA_ID = ?");
		params.add(t17calle.getProvinciaId());

		where.append(" AND t1.MUNICIPIO_ID = ?");
		params.add(t17calle.getMunicipioId());

		if (t17calle != null && t17calle.getTipoViaId() != null) {
			where.append(" AND t1.TIPO_VIA_ID = t2.ID");
			where.append(" AND t2.ID = ?");
			params.add(t17calle.getTipoViaId());
		}

		if (t17calle != null && t17calle.getDsO() != null) {
			where.append(" AND UPPER(t1.DS_O) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(t17calle.getDsO().toUpperCase() + "%");
			} else {
				params.add("%" + t17calle.getDsO().toUpperCase() + "%");
			}
			where.append(" AND t1.DS_O IS NOT NULL");
		}

		query.append(where);

		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());

	}

}
