package com.ejie.y41a.utils;

import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.io.Writer;
import java.math.BigDecimal;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.SQLException;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Locale;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;

import com.ejie.y41a.constantes.Y41aConstantes;
import com.ejie.y41a.utils.fecha.DateFormatSymbolsProvider_eu;
import com.ejie.y41a.utils.mail.Y41aMailMessage;
import com.ejie.y41a.utils.mail.Y41aMailSender;

public class Y41aUtils {
	private static Y41aUtils INSTANCE = null;
	private static final int posCIF = 23;
	private static final int MAXCADENABYTE = 1024;

	private static final Logger logger = LoggerFactory
			.getLogger(Y41aUtils.class);

	// Private constructor suppresses
	private Y41aUtils() {

	}

	// creador sincronizado para protegerse de posibles problemas multi-hilo
	// otra prueba para evitar instanciación múltiple
	private synchronized static void createInstance() {
		if (INSTANCE == null) {
			INSTANCE = new Y41aUtils();
		}
	}

	public static Y41aUtils getInstance() {
		if (INSTANCE == null)
			createInstance();
		return INSTANCE;
	}

	public static String ConvertToUTF8(String label) {
		String temp = "";

		try {
			temp = new String(label.getBytes("ISO-8859-1"), "UTF8");
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
		}

		return temp;
	}

	/**
	 * Parsea un string de fecha con el formato correspondiente al idioma del
	 * locale y devuelve la fecha.
	 * 
	 * @param dateString
	 * @param locale
	 * @return
	 * @throws Exception
	 */
	public static Date parseDate(String dateString, Locale locale)
			throws Exception {
		if (dateString != null && !dateString.equals("")) {
			DateFormat dateFormat = null;
			if (Y41aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				dateFormat = new SimpleDateFormat("dd/MM/yyyy");
			} else if (Y41aConstantes.EUSKERA.equals(locale.getLanguage())) {
				dateFormat = new SimpleDateFormat("yyyy/MM/dd");
			} else {
				throw new Exception(
						"Language date parse support not implemented for this language.");
			}
			return dateFormat.parse(dateString);
		}
		return null;
	}

	/**
	 * Parsea un string de fecha con el formato correspondiente al idioma del
	 * locale y devuelve la fecha. Recibe un tipo date
	 * 
	 * @param date
	 *            (Date)
	 * @param locale
	 * @return
	 * @throws Exception
	 */
	public static Date parseDate(Date date, Locale locale) throws Exception {
		if (date != null) {
			SimpleDateFormat formatter = null;
			if (Y41aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				formatter = new SimpleDateFormat("dd/MM/yyyy");
			} else if (Y41aConstantes.EUSKERA.equals(locale.getLanguage())) {
				formatter = new SimpleDateFormat("yyyy/MM/dd");
			} else {
				throw new Exception(
						"Language date parse support not implemented for this language.");
			}
			return formatter.parse(formatter.format(date));

		}
		return null;
	}

	/**
	 * 
	 * @param strFecha
	 * @return
	 */
	public static Date parseDate(String strFecha) {
		SimpleDateFormat formatoDelTexto = new SimpleDateFormat("yyyy-MM-dd");
		formatoDelTexto.setLenient(false);
		// String strFecha = "2007-12-25";
		Date fecha = null;
		try {
			if (strFecha != null) {
				if (strFecha.length() == 10) {
					fecha = formatoDelTexto.parse(strFecha);
				}
			}

		} catch (ParseException ex) {
			logger.error("Fecha incorrecta: " + strFecha);

			ex.printStackTrace();
		}
		return fecha;
	}

	/**
	 * 
	 * @param fecha
	 * @param lang
	 * @return
	 */
	public static String parseDateToString(Date fecha, String lang) {
		String fechaFormateada = "";
		DecimalFormat df = new DecimalFormat("00");

		if (fecha != null) {
			Calendar calendar = Calendar.getInstance();
			calendar.setTime(fecha);

			int dia = calendar.get(Calendar.DAY_OF_MONTH); // dia del mes
			int mes = calendar.get(Calendar.MONTH) + 1; // mes, de 0 a 11
			int anio = calendar.get(Calendar.YEAR); // año

			if (Y41aConstantes.EUSKERA.equals(lang)) {
				fechaFormateada = anio + "/" + df.format(mes) + "/"
						+ df.format(dia);
			} else {
				fechaFormateada = df.format(dia) + "/" + df.format(mes) + "/"
						+ anio;
			}
		}
		return fechaFormateada;
	}

	/**
	 * Convierte un OutputStream a String
	 * 
	 * @param rd
	 * @return
	 * @throws IOException
	 */
	public static String convertOutputStreamToString(OutputStream os)
			throws IOException {

		return "";
	}

	/**
	 * Convierte un InputStream a String
	 * 
	 * @param is
	 * @return
	 * @throws IOException
	 */
	public static String convertInputStreamToString(InputStream is)
			throws IOException {
		/*
		 * To convert the InputStream to String we use the Reader.read(char[]
		 * buffer) method. We iterate until the Reader return -1 which means
		 * there's no more data to read. We use the StringWriter class to
		 * produce the string.
		 */
		if (is != null) {
			Writer writer = new StringWriter();

			char[] buffer = new char[1024];
			try {
				Reader reader = new BufferedReader(new InputStreamReader(is,
						"UTF-8"));
				int n;
				while ((n = reader.read(buffer)) != -1) {
					writer.write(buffer, 0, n);
				}
			} finally {
				is.close();
			}
			return writer.toString();
		} else {
			return "";
		}
	}

	/**
	 * Convierte un InputStream a byte[]
	 * 
	 * @param is
	 * @return byte
	 * @throws IOException
	 */
	public static byte[] convertInputStreamToByteArray(InputStream is)
			throws IOException {
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		int reads = is.read();
		while (reads != -1) {
			baos.write(reads);
			reads = is.read();
		}
		return baos.toByteArray();

	}

	/**
	 * Obtiene longitud de cualquier inputStream
	 * 
	 * @param is
	 * @return
	 * @throws IOException
	 */
	public static int getLongitudInputStream(InputStream is) {
		int n = 0;
		if (is != null) {
			try {
				while ((n = is.read()) != -1) {
					n++;
				}
			} catch (IOException e) {
				e.printStackTrace();
			} finally {
				try {
					is.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
		return n;
	}

	/**
	 * Muestra la etiqueta parseada para una estructura json - arbol
	 * 
	 * @param lblEs
	 * @param lblEu
	 * @return
	 */
	public static String printTreeLabel(String lblEs, String lblEu) {
		return isLanguage(lblEs, lblEu);
	}

	/**
	 * Muestra la etiqueta en el idioma correspondiente
	 * 
	 * @param lblEs
	 * @param lblEu
	 * @return
	 */
	private static String isLanguage(String lblEs, String lblEu) {
		// Idioma
		Locale locale = LocaleContextHolder.getLocale();

		if (Y41aConstantes.EUSKERA.equals(locale.getLanguage())) {
			return filtroDB(lblEu);
		} else {
			return filtroDB(lblEs);
		}

	}

	public static String filtroDB(String input) {

		logger.info(input);
		if (input == null) {
			return "";
		} else {
			input = input.replaceAll("\"", "");
		}

		return input;
	}

	/**
	 * Filtra caracteres antes de ingreso en DB
	 * 
	 * @param label
	 * @return
	 */
	public static String _filtroDB(String input) {

		logger.debug("**********************************");
		logger.debug(input);
		logger.debug("**********************************");
		// Comprueba que no contenga caracteres prohibidos
		/*
		 * El caracter '+' representa una o más veces El caracter '^' dentro de
		 * los corchetes es un NOT, permite encontrar cualquier carácter que NO
		 * se encuentre dentro del grupo indicado
		 */
		// Pattern p = Pattern.compile("[^A-Za-z0-9\\.\\@_\\-~#]+");
		Pattern p = Pattern.compile("[^A-Za-z0-9@ _()-]");
		Matcher m = p.matcher(input);
		StringBuffer sb = new StringBuffer();
		boolean resultado = m.find();
		boolean caracteresIlegales = false;
		while (resultado) {
			caracteresIlegales = true;
			m.appendReplacement(sb, "");
			resultado = m.find();
		}

		// Añade el ultimo segmento de la entrada a la cadena
		m.appendTail(sb);

		if (caracteresIlegales) {
			logger.info("La cadena contenía caracteres ilegales que han sido suprimidos");
		}

		return sb.toString();
	}

	/**
	 * Calcula la letra correspondiente a un DNI
	 * 
	 * @param nif
	 *            NIF a partir del cual se calculará su correspondiente letra
	 * @return Letra calculada para el DNI del NIF
	 */
	public static char calcularLetraNIF(String nif) {
		String cadena = "TRWAGMYFPDXBNJZSQVHLCKE";
		int dni = 0;
		int pos = 0;

		try {
			dni = Integer.parseInt(nif.substring(0, nif.length() - 1));
			pos = dni % Y41aUtils.posCIF;
			return cadena.charAt(pos);
		} catch (NumberFormatException nfe) {
			return '!';
		}
	}

	/**
	 * Devuelve true si el string vienen informado y false si el string es nulo
	 * o vacio
	 * 
	 * @param stringIn
	 *            Dato a evaluar
	 * @return boolean true o false
	 */
	public static boolean isFilled(String stringIn) {
		if (stringIn == null || stringIn.equals("")) {
			return false;
		}
		return true;
	}

	/**
	 * Transforma un campo Clob a String
	 * 
	 * @param clob
	 *            Clob la cual se desea transformar
	 * @return String Devuelve la cadena resultante de las transformaciones.
	 * @throws SQLException
	 *             SQLException
	 * @throws IOException
	 *             IOException
	 */
	public static String clobToString(Clob clob) // throws SQLException,
													// IOException
	{

		try {
			if (clob != null) {
				Reader reader = clob.getCharacterStream();
				CharArrayWriter writer = new CharArrayWriter();

				int i = 1;

				while ((i = reader.read()) != -1) {
					writer.write(i);
				}

				return new String(writer.toCharArray());
			} else {
				return null;
			}
		} catch (SQLException sqle) {
			return null;
		} catch (IOException ioe) {
			return null;
		}
	}

	/**
	 * Transforma un campo Blob a String
	 * 
	 * @param blob
	 *            Blob la cual se desea transformar
	 * @return String Devuelve la cadena resultante de las transformaciones.
	 * @throws SQLException
	 *             SQLException
	 * @throws IOException
	 *             IOException
	 */
	public static String blobToString(Blob blob) throws SQLException,
			IOException {

		if (blob != null) {
			InputStream inStream = blob.getBinaryStream();

			int count;
			java.io.ByteArrayOutputStream bo = new java.io.ByteArrayOutputStream();
			byte[] b = new byte[Y41aUtils.MAXCADENABYTE];
			try {
				while ((count = inStream.read(b, 0, Y41aUtils.MAXCADENABYTE)) > 0) {
					bo.write(b, 0, count);
				}
			} finally {
				bo.close();
			}

			return new String(bo.toString());
		} else {
			return null;
		}
	}

	/**
	 * Recoge una hora HH:mm:ss y la deja como HH:mm
	 * 
	 * @param String
	 *            hora
	 * 
	 * @return String hora
	 * 
	 */
	public static String horaToHora(String hora) {
		String hora5char = "";
		if (hora.length() > 5) {
			hora5char = hora.substring(0, 5);
		}
		return hora5char;
	}

	/**
	 * Se envia un mail
	 * 
	 * @param smtp
	 *            Servidor smtp
	 * @param asunto
	 *            Asunto del mensaje
	 * @param body
	 *            Cuerpo del mensaje
	 * @param from
	 *            Email quien envia el mensaje
	 * @param to
	 *            Email a quien va dirigido el mensaje
	 * @param cco
	 *            Email copia oculta
	 * @param attachments
	 *            Ficheros adjuntos en el mensaje
	 * @return boolean True si el mensaje se envio correctamente y False en caso
	 *         contrario
	 */
	public static boolean sendMail(String smtp, String asunto, String body,
			String from, String to, String cco, Collection attachments) {

		try {

			Y41aMailMessage eMailMessage = new Y41aMailMessage();
			Y41aMailSender sendMessage = new Y41aMailSender();
			eMailMessage.setFrom(from);
			eMailMessage.setEmailServer(smtp);
			eMailMessage.setSubject(asunto);
			eMailMessage.setNotDeliveryAddress("");
			eMailMessage.setMsgBodyText(body);

			// para introducir los destinatarios
			String[] destinatarios = to.toString().split(";");
			for (int i = 0; i < destinatarios.length; i++) {
				if (!"".equals(destinatarios[i])) {
					eMailMessage.addTo(destinatarios[i]);
				}
			}

			// Cambiado para que vayan con copia oculta
			String[] copiaoculta = cco.toString().split(";");
			for (int i = 0; i < copiaoculta.length; i++) {
				if (!"".equals(copiaoculta[i])) {
					eMailMessage.addBCC(copiaoculta[i]);
				}
			}

			eMailMessage.setAttachments((ArrayList) attachments);

			sendMessage.send(eMailMessage);
		} catch (Exception e) {
			logger.error("Excepcion correo: --> ", e);
			return false;
		}
		return true;
	}

	/**
	 * Transforma una fecha String a java.sql.Date.
	 * 
	 * @param fecha
	 *            String con la fecha a convertir
	 * @param formato
	 *            String con el formato de la fecha
	 * @return java.sql.Date fecha convertida
	 * @exception ParseException
	 *                Se produce la excepción cuando la fecha está mal
	 *                formateada.
	 */
	public static java.sql.Date formatearStringToDateFormato(String fecha,
			String formato) throws ParseException {
		SimpleDateFormat formatoDelTexto = new SimpleDateFormat(formato);
		java.sql.Date dFecha = null;

		dFecha = new java.sql.Date(formatoDelTexto.parse(fecha).getTime());

		return dFecha;
	}

	/**
	 * Transforma una fecha String a java.sql.Date.
	 * 
	 * @param fecha
	 *            String con la fecha a convertir
	 * @return java.sql.Date fecha convertida
	 * @exception ParseException
	 *                Se produce la excepción cuando la fecha está mal
	 *                formateada.
	 */
	public static java.sql.Date formatearStringToDateFormato(String fecha) {

		if (fecha == null || "".equals(fecha)) {
			return null;
		}

		String formato = "";
		Locale locale = LocaleContextHolder.getLocale();
		if (Y41aConstantes.EUSKERA.equals(locale.getLanguage())) {
			formato = Y41aConstantes.FORMATO_FECHA_EU;
		} else {
			formato = Y41aConstantes.FORMATO_FECHA_ES;
		}
		SimpleDateFormat formatoDelTexto = new SimpleDateFormat(formato);
		java.sql.Date dFecha = null;
		try {
			dFecha = new java.sql.Date(formatoDelTexto.parse(fecha).getTime());
		} catch (ParseException e) {
			dFecha = null;
		}

		return dFecha;
	}

	/**
	 * Transforma un java.sql.Date a la mascara de formato de fecha recibida
	 * como parámetro.
	 * 
	 * @param fecha
	 *            String
	 * @param formato
	 *            String con el formato a parsear
	 * @return String fecha formateada
	 * @exception ParseException
	 *                Se produce la excepción cuando la fecha está mal
	 *                formateada.
	 */
	public static String formatearDateToString(java.util.Date fecha,
			String formato) throws ParseException {
		DateFormat sdf = null;

		if (fecha == null) {
			return null;
		} else {
			sdf = new SimpleDateFormat(formato);
		}

		return sdf.format(fecha);
	}

	/**
	 * Transforma un java.sql.Date a la mascara de formato de fecha recibida
	 * como parámetro.
	 * 
	 * @param fecha
	 *            String
	 * @param formato
	 *            String con el formato a parsear
	 * @param idioma
	 *            String con el idioma
	 * @return String fecha formateada
	 * @exception ParseException
	 *                Se produce la excepción cuando la fecha está mal
	 *                formateada.
	 */
	public static String formatearDateToString(java.util.Date fecha,
			String formato, String idioma) throws ParseException {

		SimpleDateFormat sdf = null;

		if (fecha == null) {
			return null;
		} else {
			sdf = new SimpleDateFormat(formato, new Locale(idioma));
		}

		return sdf.format(fecha);
	}

	/**
	 * Transforma un java.sql.Date a la mascara de formato de fecha recibida
	 * como parámetro.
	 * 
	 * @param fecha
	 *            String
	 * @param formato
	 *            String con el formato a parsear
	 * @param idioma
	 *            String con el idioma
	 * @return String fecha formateada
	 * @exception ParseException
	 *                Se produce la excepción cuando la fecha está mal
	 *                formateada.
	 */
	public static String formatearDateToStringEuskera(java.util.Date fecha,
			String formato, String idioma) throws ParseException {

		SimpleDateFormat sdf = null;
		DateFormatSymbolsProvider_eu dateFormatSymbolsProvider_eu = new DateFormatSymbolsProvider_eu();

		if (fecha == null) {
			return null;
		} else {
			sdf = new SimpleDateFormat(formato,
					dateFormatSymbolsProvider_eu
							.getInstance(new Locale(idioma)));
		}

		return sdf.format(fecha);
	}

	/**
	 * Normaliza datos de entrada de formularios a 0 - Para combos normalmente
	 * 
	 * 
	 * @param input
	 *            input
	 * @return "" vacio
	 */
	public static String normalizarCero(String input) {

		String rt = "";
		if (input == null) {
			rt = Y41aConstantes.CERO;
		} else if ("".equals(input)) {
			rt = Y41aConstantes.CERO;
		} else {
			rt = input;
		}

		return rt;
	}

	/**
	 * Normaliza datos de entrada de formularios a "" - excepto combos
	 * 
	 * @param input
	 *            String
	 * @return String
	 */
	public static String normalizarVacio(String input) {

		String rt = Y41aConstantes.VACIO;
		if (input != null) {
			if (!"null".equals(input)) {
				rt = input;
			}
		}

		return rt;
	}

	/**
	 * Convierte un Long en String
	 * 
	 * @param input
	 *            Long
	 * @return String
	 */
	public static String convertLong2String(Long input) {

		if (input != null) {
			return input.toString();
		} else {
			return Y41aConstantes.VACIO;
		}
	}

	/**
	 * Convierte un BigDecimal en String
	 * 
	 * @param input
	 *            BigDecimal
	 * @return String
	 */
	public static String convertBigDecimal2String(BigDecimal input) {

		if (input != null) {
			return input.toString();
		} else {
			return Y41aConstantes.VACIO;
		}
	}

	/**
	 * Normaliza datos de entrada de formularios a null en caso de cadenas
	 * vacias
	 * 
	 * @param input
	 *            String
	 * @return String
	 */
	public static String normalizarNulo(String input) {

		if (input != null) {
			if ("".equals(input)) {
				return null;
			} else {
				return input;
			}
		} else {
			return null;
		}
	}

	/**
	 * Normaliza datos de entrada de formularios a null en caso de cadenas
	 * vacias
	 * 
	 * @param input
	 *            String
	 * @return BigDecimal
	 */

	public static BigDecimal normalizarBigDecimal(String input) {

		if (input != null) {
			if ("".equals(input)) {
				return null;
			} else {
				return new BigDecimal(input);
			}
		} else {
			return null;
		}
	}

	/**
	 * Método que dada 1 cadena de texto html elimina todas las etiquetas HTML
	 * de que consta
	 * 
	 * @param cadena
	 *            String la cadena de texto html
	 * @return String la cadena sin etiquetas html
	 */
	public static String eliminarTags(String cadena) {
		while (true) {
			int izdaTag = cadena.indexOf("<");
			if (izdaTag < 0)
				return cadena;
			int derTag = cadena.indexOf(">", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + " "
					+ cadena.substring(derTag + 1);
		}
	}

	/**
	 * Método que dada 1 cadena de texto html elimina todas los &xxxxx; de que
	 * consta
	 * 
	 * @param cadena
	 *            String la cadena de texto html
	 * @return String la cadena sin etiquetas html
	 */
	public static String eliminarAmpersand(String cadena) {
		while (true) {
			int izdaTag = cadena.indexOf("&");
			if (izdaTag < 0)
				return cadena;
			int derTag = cadena.indexOf(";", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + " "
					+ cadena.substring(derTag + 1);
		}
	}

	/**
	 * Método que dada 1 cadena de texto html sustituye los <li xxxxxxxx>
	 * de que consta
	 * 
	 * @param cadena
	 *            String la cadena de texto html
	 * @return String la cadena sin etiquetas <li>
	 */
	public static String sustituirTagsLi(String cadena) {
		while (true) {
			int izdaTag = cadena.indexOf("<li");
			if (izdaTag < 0)
				return cadena;
			int derTag = cadena.indexOf(">", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + "&bullet;"
					+ cadena.substring(derTag + 1);
		}
	}

	/**
	 * Método que dada 1 cadena de texto html sustituye los <span de subrayado
	 * 
	 * @param cadena
	 *            String la cadena de texto html
	 * @return String la cadena sin etiquetas <span >
	 */
	public static String sustituirTagsSpanUnderline(String cadena) {
		while (true) {
			int izdaTag = cadena
					.indexOf("<span style=\"text-decoration: underline;\">");
			if (izdaTag < 0)
				return cadena;
			int derTag = cadena.indexOf(">", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + "&subrayadoini;"
					+ cadena.substring(derTag + 1);

			izdaTag = cadena.indexOf("</span>", izdaTag);
			if (izdaTag < 0)
				return cadena;
			derTag = cadena.indexOf(">", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + "&subrayadofin;"
					+ cadena.substring(derTag + 1);

		}
	}

	/**
	 * Método que dada 1 cadena de texto html sustituye los <span de tachado
	 * 
	 * @param cadena
	 *            String la cadena de texto html
	 * @return String la cadena sin etiquetas <span >
	 */
	public static String sustituirTagsSpanStrike(String cadena) {
		while (true) {
			int izdaTag = cadena
					.indexOf("<span style=\"text-decoration: line-through;\">");
			if (izdaTag < 0)
				return cadena;
			int derTag = cadena.indexOf(">", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + "&tachadoini;"
					+ cadena.substring(derTag + 1);

			izdaTag = cadena.indexOf("</span>", izdaTag);
			if (izdaTag < 0)
				return cadena;
			derTag = cadena.indexOf(">", izdaTag);
			if (derTag < 0)
				return cadena;
			cadena = cadena.substring(0, izdaTag) + "&tachadofin;"
					+ cadena.substring(derTag + 1);

		}
	}

	/**
	 * MÃ©todo que obtiene la extension de un fichero
	 * 
	 * @param filename
	 *            String
	 * @return String extension del fichero
	 */
	public static String getExtensionFile(String filename) {
		int index = filename.lastIndexOf('.');
		if (index == -1) {
			return "";
		} else {
			return filename.substring(index + 1);
		}
	}

	/**
	 * Checks if is numeric.
	 * 
	 * @param cadena
	 *            the cadena
	 * @return true, if is numeric
	 */
	public static boolean isNumeric(String cadena) {
		try {
			Integer.parseInt(cadena);
			return true;
		} catch (NumberFormatException nfe) {
			return false;
		}
	}

	/**
	 * Format double2decimales ycoma.
	 * 
	 * @param valor
	 *            the valor
	 * @return the string
	 */
	public static String formatDouble2decimalesYcoma(double valor) {
		DecimalFormat decim = new DecimalFormat("#.00");
		String s = decim.format(valor);
		s = s.replace('\u002e', '\u002c');
		return s;
	}

	/**
	 * Método que indica si una cadena es nula o cadena vacia
	 * 
	 * @param valor
	 *            a comprobar
	 * @return booleano indicando el resultado
	 */
	public static boolean esNullOCadenaVacia(String valor) {
		return (valor == null || "".equals(valor));
	}

	// public static void main(String args[]) {
	// String cadena = "<li><strong><em><span >uno</span></em></strong></li>";
	// //
	// "<p><strong>&Eacute;sta l&iacute;nea en negrita.</strong></p><p><em>&Eacute;sta otra en cursiva.</em></p><p><span style=\"text-decoration: underline;\">&Eacute;sta subrayada.</span></p><p><span style=\"text-decoration: line-through;\"><span style=\"text-decoration: underline;\"><em><strong>Y &eacute;sta todo lo anterior y tachada</strong></em></span></span></p><p style=\"text-align: right;\">Alineada a la derecha</p><p style=\"text-align: left;\">Y un lista en colores</p><ul><li><span style=\"color: #339966;\">VERDE</span></li><li><span style=\"color: #0000ff;\">AZUL</span></li><li><span style=\"color: #ff0000;\">ROJO</span></li></ul>";
	//
	// System.out.println(Y41aUtils.sustituirTagsLi(cadena));
	// // 0));
	// }

}
