package com.ejie.y41a.service;

import java.util.Date;
import java.util.List;
import java.util.Locale;

import javax.annotation.Resource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.stereotype.Service;

import com.ejie.x38.dto.Pagination;
import com.ejie.y41a.constantes.Y41aConstantes;
import com.ejie.y41a.model.Consulta;
import com.ejie.y41a.model.ConsultaBusqueda;
import com.ejie.y41a.model.SolicitudBusqueda;
import com.ejie.y41a.utils.Y41aUtils;

/**
 *  
 */
@Service(value = "excelGeneratorService")
public class Y41aExcelGeneratorServiceImpl implements Y41aExcelGeneratorService {

	@Autowired
	private ConsultaService consultaService;

	@Autowired
	private DenunciaService denunciaService;;

	@Resource
	ReloadableResourceBundleMessageSource appMessageSource;

	/**
	 * Genera un tabla html para devolver como excel los elementos seleccionados
	 * en consultas
	 * 
	 * @param consulta
	 *            Consulta
	 * @return String
	 */
	public String getConsultasExcel(Consulta consulta, Date fechaDesde,
			Date fechaHasta) {
		StringBuffer xslString = new StringBuffer(Y41aConstantes.NUM_40);
		xslString.append("<table border='1'>");
		xslString.append(this.getConsultasListTable(consulta, fechaDesde,
				fechaHasta));
		xslString.append("</table>");
		return xslString.toString();
	}

	/**
	 * Genera un tabla html para devolver como excel los elementos seleccionados
	 * en solicitudes
	 * 
	 * @param solicitudBusqueda
	 *            SolicitudBusqueda
	 * @return String
	 */
	public String getSolicitudesExcel(SolicitudBusqueda solicitudBusqueda,
			Date fechaPresentacionDesde, Date fechaPresentacionHasta,
			Date fechaRegistroDesde, Date fechaRegistroHasta,
			Date fechaDescisionDesde, Date fechaDescisionHasta) {
		StringBuffer xslString = new StringBuffer(Y41aConstantes.NUM_40);
		xslString.append("<table border='1'>");
		xslString.append(this.getSolicitudesListTable(solicitudBusqueda,
				fechaPresentacionDesde, fechaPresentacionHasta,
				fechaRegistroDesde, fechaRegistroHasta, fechaDescisionDesde,
				fechaDescisionHasta));
		xslString.append("</table>");
		return xslString.toString();
	}

	/**
	 * Obtiene tabla que contiene la lista de consultas
	 * 
	 * @param consulta
	 *            FiltroConsulta datos del filtro de búsqueda
	 * @param fechaDesde
	 *            Date rango inferior de fechas
	 * @param fechaHasta
	 *            Date rango superior de fechas
	 * @return String texto con el informe excel en formato HTML
	 */
	private String getConsultasListTable(Consulta consulta, Date fechaDesde,
			Date fechaHasta) {

		final Long ROWS = Long
				.valueOf(Y41aConstantes.EXCEL_NUMERO_MAXIMO_RESULTADOS);
		final String CAMPO_ORDENACION = "numeroConsulta";
		final String ORDEN = Y41aConstantes.ORDENACION_ASC.toLowerCase();

		StringBuffer xslString = new StringBuffer();

		// add header
		xslString.append(getConsultasHeaderRow());

		Pagination pagination = new Pagination();
		pagination.setRows(ROWS);
		pagination.setAscDsc(ORDEN);
		pagination.setSort(CAMPO_ORDENACION);
		pagination.setPage(Long.valueOf(1));
		List<ConsultaBusqueda> consultasBusqueda = this.consultaService
				.findAllLikeConsulta(consulta, pagination, false, fechaDesde,
						fechaHasta);

		for (ConsultaBusqueda consultaBusquedaAux : consultasBusqueda) {
			xslString.append(getConsultasRow(consultaBusquedaAux));
		}

		return xslString.toString();
	}

	/**
	 * Obtiene datos.
	 * 
	 * @param solicitudBusqueda
	 *            SolicitudBusqueda
	 * @return String
	 */
	private String getSolicitudesListTable(SolicitudBusqueda solicitudBusqueda,
			Date fechaPresentacionDesde, Date fechaPresentacionHasta,
			Date fechaRegistroDesde, Date fechaRegistroHasta,
			Date fechaDescisionDesde, Date fechaDescisionHasta) {
		StringBuffer xslString = new StringBuffer();

		final Long ROWS = Long
				.valueOf(Y41aConstantes.EXCEL_NUMERO_MAXIMO_RESULTADOS);
		final String CAMPO_ORDENACION = "numExpediente";
		final String ORDEN = Y41aConstantes.ORDENACION_ASC.toLowerCase();

		// add header
		xslString.append(getSolicitudesHeaderRow());

		Pagination pagination = new Pagination();
		pagination.setRows(ROWS);
		pagination.setAscDsc(ORDEN);
		pagination.setSort(CAMPO_ORDENACION);
		pagination.setPage(Long.valueOf(1));
		List<SolicitudBusqueda> solicitudesBusqueda = this.denunciaService
				.findAllLikeSolicitudes(solicitudBusqueda, pagination, false,
						fechaPresentacionDesde, fechaPresentacionHasta,
						fechaRegistroDesde, fechaRegistroHasta,
						fechaDescisionDesde, fechaDescisionHasta);

		for (SolicitudBusqueda solicitudBusquedaAux : solicitudesBusqueda) {
			xslString.append(getSolicitudesRow(solicitudBusquedaAux));
		}

		return xslString.toString();
	}

	/**
	 * Genera la cabecera del informe de la búsqueda de consultas
	 * 
	 * @return String texto con la fila de la cabecera formateada en HTML
	 */
	private String getConsultasHeaderRow() {
		// Obtener el mensaje dependiendo del idioma
		Locale locale = LocaleContextHolder.getLocale();
		StringBuffer header = new StringBuffer(Y41aConstantes.NUM_128);
		header.append("<tr style='BACKGROUND-COLOR: #808080;FONT-WEIGHT: bold;color: #FFFFFF'>");
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.numconsulta", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage("excel.fecha",
				null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.monitor", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.municipio", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.productoservicio", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.origen", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.pendientes", null, locale)));
		header.append("</tr>");
		return header.toString();
	}

	/**
	 * Obtiene los textos de cabecera para el informe de solicitudes
	 * 
	 * @return String
	 */
	private String getSolicitudesHeaderRow() {
		// Obtener el mensaje dependiendo del idioma
		Locale locale = LocaleContextHolder.getLocale();
		StringBuffer header = new StringBuffer();
		header.append("<tr style='BACKGROUND-COLOR: #808080;FONT-WEIGHT: bold;color: #FFFFFF'>");

		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.estado", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.numExpediente", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.fechapresentacion", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.personaconsumidora", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.apellido1", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.apellido2", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.municipio", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.establecimiento", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.ultimotramite", null, locale)));
		header.append(getHeaderField(appMessageSource.getMessage(
				"excel.decision", null, locale)));

		header.append("</tr>");
		return header.toString();
	}

	/**
	 * Genera la cabecera del informe excel
	 * 
	 * @param fieldName
	 *            String texto con el nombre de la columna
	 * @return String texto con el nombre de la columna formateado en HTML
	 */
	private static String getHeaderField(String fieldName) {
		StringBuffer field = new StringBuffer();
		field.append("<th>");
		field.append(Y41aUtils.normalizarVacio(fieldName));
		field.append("</th>");
		return field.toString();
	}

	/**
	 * Genera la fila de datos para una tupla recuperada de base de datos
	 * 
	 * @param consultaBusqueda
	 *            ConsultaBusqueda datos de una búsqueda de consultas
	 * @return String texto con la fila formateada en HTML
	 */
	private String getConsultasRow(ConsultaBusqueda consultaBusqueda) {

		Locale locale = LocaleContextHolder.getLocale();

		StringBuffer row = new StringBuffer();
		row.append("<tr valign=\"top\">");

		// número de consulta
		row.append(getField(Y41aUtils.normalizarVacio(consultaBusqueda
				.getNumeroConsulta())));

		// fecha de consulta
		row.append(getFieldFecha(Y41aUtils.normalizarVacio(Y41aUtils
				.parseDateToString(consultaBusqueda.getCofecha(),
						locale.getLanguage()))));

		// nombre de monitor
		row.append(getField(Y41aUtils.normalizarVacio(consultaBusqueda
				.getMonombre())));

		// nombre de municipio
		row.append(getField(Y41aUtils.normalizarVacio(consultaBusqueda
				.getNombreMunicipio())));

		// nombre de producto o servicio
		if (Y41aConstantes.CASTELLANO.equals(locale.getLanguage())) {
			row.append(getField(Y41aUtils.normalizarVacio(consultaBusqueda
					.getPsnombre())));
		} else {
			row.append(getField(Y41aUtils.normalizarVacio(consultaBusqueda
					.getPsnombree())));
		}

		// origen de la consulta
		if (consultaBusqueda.getCoinl37() != null
				&& consultaBusqueda.getCoinl37()
						.equals(Y41aConstantes.VALOR_SI)) {
			row.append(getField(appMessageSource.getMessage("excel.web", null,
					locale)));
		} else {
			row.append(getField(appMessageSource.getMessage("excel.oficina",
					null, locale)));
		}

		// estado de la consulta
		if ((consultaBusqueda.getCosituacion() != null && consultaBusqueda
				.getCosituacion().equals(Y41aConstantes.SITUACION_PENDIENTES))
				|| consultaBusqueda.getCosituacion() == null
				|| consultaBusqueda.getCosituacion().equals("")) {
			row.append(getField(Y41aConstantes.SI));
		} else {
			row.append(getField(Y41aUtils.normalizarVacio("")));
		}
		row.append("</tr>");
		return row.toString();
	}

	/**
	 * Genera la fila de datos para una tupla recuperada de base de datos
	 * 
	 * @param solicitudBusqueda
	 *            SolicitudBusqueda datos de una búsqueda de solicitudes
	 * @return String texto con la fila formateada en HTML
	 */
	private String getSolicitudesRow(SolicitudBusqueda solicitudBusqueda) {
		Locale locale = LocaleContextHolder.getLocale();

		StringBuffer row = new StringBuffer();
		row.append("<tr valign=\"top\">");

		// estado
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getEstado())));
		// numExpediente
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getNumExpediente())));

		// fecha presentación denuncia
		row.append(getFieldFecha(Y41aUtils.normalizarVacio(Y41aUtils
				.parseDateToString(solicitudBusqueda.getDefede(),
						locale.getLanguage()))));
		// dtnombre
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getDtnombre())));
		// dtapellido1
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getDtapellido1())));
		// dtapellido2
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getDtapellido2())));
		// municipio
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getDtnombreMunicipio())));

		// esnombre
		row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
				.getEsnombre())));

		// Nombre tramitador
		if (Y41aConstantes.CASTELLANO.equals(locale.getLanguage())) {
			row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
					.getTrnombre())));
		} else {
			row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
					.getTrnombree())));
		}

		// Nombre establecimiento
		if (Y41aConstantes.CASTELLANO.equals(locale.getLanguage())) {
			row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
					.getDenombre())));
		} else {
			row.append(getField(Y41aUtils.normalizarVacio(solicitudBusqueda
					.getDenombree())));
		}
		row.append("</tr>");
		return row.toString();
	}

	/**
	 * Formatea el contenido de una celda en HTML
	 * 
	 * @param fieldValue
	 *            String
	 * @return String
	 */
	private static String getField(final String fieldValue) {
		StringBuffer field = new StringBuffer();
		field.append("<td>");
		field.append(Y41aUtils.normalizarVacio(fieldValue));
		field.append("</td>");
		return field.toString();
	}

	/**
	 * Formatea el contenido de una celda de tipo fecha en HTML
	 * 
	 * @param fieldValue
	 *            String
	 * @return String
	 */
	private static String getFieldFecha(final String fieldValue) {
		StringBuffer field = new StringBuffer(Y41aConstantes.NUM_128);
		field.append("<td style=\"width:5em;\">");
		field.append("=T(\"");
		field.append(Y41aUtils.normalizarVacio(fieldValue));
		field.append("\")");
		field.append("</td>");
		return field.toString();
	}

}
