package com.ejie.aa83b.util;

import java.util.List;

import javax.xml.bind.JAXBException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.ejie.aa83b.util.Aa83BConstants;
import com.ejie.aa83b.util.Aa83bMessageUnparser;
import com.ejie.aa83b.util.Aa83bUtilidades;
import com.ejie.aa83b.util.Aa83bUtilProperties;
import com.ejie.aa83b.model.Aa83bDocumentoDokusi;
import com.ejie.schemas.t65b.T65BAttributeListListType;
import com.ejie.schemas.t65b.T65BAttributeListType;
import com.ejie.schemas.t65b.T65BAttributeType;
import com.ejie.schemas.t65b.T65BContentListType;
import com.ejie.schemas.t65b.T65BContentType;
import com.ejie.schemas.t65b.T65BDocumentAttrInfoListType;
import com.ejie.schemas.t65b.T65BDocumentAttrInfoType;
import com.ejie.schemas.t65b.T65BDocumentIDListType;
import com.ejie.schemas.t65b.T65BDocumentIDType;
import com.ejie.schemas.t65b.T65BFormatType;
import com.ejie.schemas.t65b.T65BStringListType;

/**
 * Clase con metodos para obtener objetos T65B a partir de objetos Documento
 * 
 * @author DS
 * 
 */
public class Aa83bDokusiUtil {

	private static final Logger logger = LoggerFactory
			.getLogger(Aa83bDokusiUtil.class);

	private static final com.ejie.schemas.t65b.ObjectFactory objectFactory = new com.ejie.schemas.t65b.ObjectFactory();

	private static final String ATRIBUTO_ACL_NAME = "acl_name";
	private static final String ATRIBUTO_TITLE = "title";
	private static final String ATRIBUTO_OBJECT_NAME = "object_name";
	private static final String ATRIBUTO_EJGV_SERIE = "ejgv_serie";
	private static final String ATRIBUTO_EJGV_ORIGEN = "ejgv_origen";
	private static final String ATRIBUTO_EJGV_TIPO_INDEXACION = "ejgv_tipo_indexacion";

	/**
	 * 
	 */
	private Aa83bDokusiUtil() {
	}

	/**
	 * A partir de un Documento, devuelve su documentID en un objeto
	 * T65BDocumentIDType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return T65BDocumentIDType
	 */
	public static T65BDocumentIDType getT65BDocumentIDAsObject(
			Aa83bDocumentoDokusi docDokusi) {
		Aa83bDokusiUtil.logger.trace("getT65BDocumentIDAsObject INI");
		T65BDocumentIDType objT65B = Aa83bDokusiUtil.objectFactory
				.createT65BDocumentIDType();

		objT65B.setId(docDokusi.getOid());
		// XXX(sin version) objT65B.setVersion(docDokusi.getVersion());

		return objT65B;
	}

	/**
	 * A partir de un Documento, devuelve su documentID en un String que
	 * representa un objeto T65BDocumentIDType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BDocumentIDAsString(
			Aa83bDocumentoDokusi docDokusi) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BDocumentIDAsString INI");
		Aa83bMessageUnparser<T65BDocumentIDType> unparser = new Aa83bMessageUnparser<T65BDocumentIDType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createDocumentID(Aa83bDokusiUtil
						.getT65BDocumentIDAsObject(docDokusi)));
		return resultado;
	}

	/**
	 * A partir de una lista de Documento, devuelve su contentType en un objeto
	 * T65BContentListType
	 * 
	 * @param documentos
	 *            List<Documento>
	 * @return T65BContentListType @ e
	 */
	public static T65BContentListType getT65BContentListTypeAsObject(
			List<Aa83bDocumentoDokusi> documentos) {
		Aa83bDokusiUtil.logger.trace("getT65BContentListTypeAsObject INI");
		T65BContentListType objResul = null;
		T65BContentType contentDoc = null;

		if (documentos != null && !documentos.isEmpty()) {
			objResul = new T65BContentListType();
			for (Aa83bDocumentoDokusi doc : documentos) {
				contentDoc = Aa83bDokusiUtil.getT65BContentTypeAsObject(doc);
				objResul.getContent().add(contentDoc);
			}
		}

		return objResul;
	}

	/**
	 * A partir de una lista de Documento, devuelve su contentType en un String
	 * que representa un objeto T65BContentListType
	 * 
	 * @param documentos
	 *            List<Documento>
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BContentListTypeAsString(
			List<Aa83bDocumentoDokusi> documentos) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BContentListTypeAsString INI");
		Aa83bMessageUnparser<T65BContentListType> unparser = new Aa83bMessageUnparser<T65BContentListType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createContentList(Aa83bDokusiUtil
						.getT65BContentListTypeAsObject(documentos)));
		return resultado;
	}

	/**
	 * A partir de un Documento, devuelve su Content en un objeto
	 * T65BContentType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return T65BContentType
	 */
	public static T65BContentType getT65BContentTypeAsObject(
			Aa83bDocumentoDokusi docDokusi) {
		Aa83bDokusiUtil.logger.trace("getT65BContentTypeAsObject INI");
		T65BContentType objContentType = Aa83bDokusiUtil.objectFactory
				.createT65BContentType();
		if (docDokusi.getRutaPif() != null
				&& !Aa83bUtilidades.isEmpty(docDokusi.getRutaPif())) {
			objContentType.setPifId(docDokusi.getRutaPif());
		}
		objContentType.setFormat(Aa83bDokusiUtil
				.getT65BFormatTypeAsObject(docDokusi));
		if (docDokusi.getTamanyo() != null) {
			objContentType.setLenght(Long.valueOf(docDokusi.getTamanyo()));
		}

		return objContentType;
	}

	/**
	 * A partir de un Documento, devuelve su Content en un String que representa
	 * un objeto T65BContentType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BContentTypeAsString(
			Aa83bDocumentoDokusi docDokusi) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BContentTypeAsString INI");
		Aa83bMessageUnparser<T65BContentType> unparser = new Aa83bMessageUnparser<T65BContentType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createContent(Aa83bDokusiUtil
						.getT65BContentTypeAsObject(docDokusi)));
		return resultado;
	}

	/**
	 * A partir de un Documento, devuelve su Format en un objeto T65BFormatType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return T65BFormatType @ e
	 */
	public static T65BFormatType getT65BFormatTypeAsObject(
			Aa83bDocumentoDokusi docDokusi) {
		Aa83bDokusiUtil.logger.trace("getT65BFormatTypeAsObject INI");
		T65BFormatType objFormat = Aa83bDokusiUtil.objectFactory
				.createT65BFormatType();
		if (docDokusi != null) {
			objFormat.setExtension(docDokusi.getExtension());
			objFormat.setMimeType(docDokusi.getContentType());
		}

		return objFormat;
	}

	/**
	 * A partir de un Documento, devuelve su Format en un String que representa
	 * un objeto T65BFormatType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BFormatTypeAsString(
			Aa83bDocumentoDokusi docDokusi) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BFormatTypeAsString INI");
		Aa83bMessageUnparser<T65BFormatType> unparser = new Aa83bMessageUnparser<T65BFormatType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createFormat(Aa83bDokusiUtil
						.getT65BFormatTypeAsObject(docDokusi)));
		return resultado;
	}

	/**
	 * A partir de un Documento, devuelve su lista de atributos (metadatos) en
	 * un objeto T65BAttributeListType. Los atributos informados son:
	 * <b>ACL</b>, <b>TITLE</b>, <b>OBJECT_NAME</b>, <b>EGJV_SERIE</b> y
	 * <b>EJGV_ORIGEN</b>.
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return T65BAttributeListType
	 */
	public static T65BAttributeListType getT65BAttibuteListTypeAsObject(
			Aa83bDocumentoDokusi docDokusi) {
		Aa83bDokusiUtil.logger.trace("getT65BAttibuteListTypeAsObject INI");
		T65BAttributeListType objAttributeListType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeListType();
		// ACL
		T65BAttributeType objAttributeType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeType();
		objAttributeType.setKey(Aa83bDokusiUtil.ATRIBUTO_ACL_NAME);
		objAttributeType.setValue(Aa83bUtilProperties
				.getProperty(Aa83bUtilProperties.DOKUSI_ACL));
		objAttributeListType.getAttribute().add(objAttributeType);
		// TITLE
		objAttributeType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeType();
		objAttributeType.setKey(Aa83bDokusiUtil.ATRIBUTO_TITLE);
		StringBuilder sbTitle = new StringBuilder(
				Aa83BConstants.STRING_BUILDER_INIT);
		if (docDokusi.getDescripcionEs().equals(docDokusi.getDescripcionEu())) {
			sbTitle.append(docDokusi.getDescripcionEs());
		} else {
			sbTitle.append(docDokusi.getDescripcionEs()).append("/")
					.append(docDokusi.getDescripcionEu());
		}
		objAttributeType.setValue(sbTitle.toString());
		objAttributeListType.getAttribute().add(objAttributeType);
		// OBJECT_NAME
		objAttributeType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeType();
		objAttributeType.setKey(Aa83bDokusiUtil.ATRIBUTO_OBJECT_NAME);
		objAttributeType.setValue(docDokusi.getTipoDocumento()
				.getTipoDocumental());
		objAttributeListType.getAttribute().add(objAttributeType);
		// EJGV_SERIE
		objAttributeType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeType();
		objAttributeType.setKey(Aa83bDokusiUtil.ATRIBUTO_EJGV_SERIE);
		StringBuilder sbSerie = new StringBuilder(
				Aa83BConstants.STRING_BUILDER_INIT);
		sbSerie.append(Aa83bUtilProperties.DOKUSI_SERIE_PREFIJO);
		sbSerie.append(docDokusi.getTipoDocumento().getSerie());
		objAttributeType.setValue(Aa83bUtilProperties.getProperty(sbSerie
				.toString()));
		objAttributeListType.getAttribute().add(objAttributeType);
		// EJGV_ORIGEN
		objAttributeType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeType();
		objAttributeType.setKey(Aa83bDokusiUtil.ATRIBUTO_EJGV_ORIGEN);
		StringBuilder sbOrigen = new StringBuilder(
				Aa83BConstants.STRING_BUILDER_INIT);
		sbOrigen.append(Aa83bUtilProperties.DOKUSI_ORIGEN_PREFIJO);
		sbOrigen.append(docDokusi.getOrigen());
		objAttributeType.setValue(Aa83bUtilProperties.getProperty(sbOrigen
				.toString()));
		objAttributeListType.getAttribute().add(objAttributeType);
		// EJGV_TIPO_INDEXACION con el valor none
		objAttributeType = Aa83bDokusiUtil.objectFactory
				.createT65BAttributeType();
		objAttributeType.setKey(Aa83bDokusiUtil.ATRIBUTO_EJGV_TIPO_INDEXACION);
		objAttributeType.setValue(Aa83bUtilProperties.DOKUSI_NONE);
		objAttributeListType.getAttribute().add(objAttributeType);

		return objAttributeListType;
	}

	/**
	 * A partir de un Documento, devuelve su lista de atributos (metadatos) en
	 * un String que representa un objeto T65BAttributeListType
	 * 
	 * @param docDokusi
	 *            Documento
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BAttibuteListTypeAsString(
			Aa83bDocumentoDokusi docDokusi) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BAttibuteListTypeAsString INI");
		Aa83bMessageUnparser<T65BAttributeListType> unparser = new Aa83bMessageUnparser<T65BAttributeListType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createAttributeList(Aa83bDokusiUtil
						.getT65BAttibuteListTypeAsObject(docDokusi)));
		return resultado;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de atributos
	 * (metadatos) en un objeto T65BAttributeListListType. Los atributos
	 * informados son: <b>ACL</b>, <b>TITLE</b>, <b>OBJECT_NAME</b>,
	 * <b>EGJV_SERIE</b> y <b>EJGV_ORIGEN</b>.
	 * 
	 * @param documentos
	 *            List<Documento>
	 * @return T65BAttributeListListType
	 */
	public static T65BAttributeListListType getT65BAttibuteListListTypeAsObject(
			List<Aa83bDocumentoDokusi> documentos) {
		Aa83bDokusiUtil.logger.trace("getT65BAttibuteListListTypeAsObject INI");
		T65BAttributeListListType objResul = null;
		T65BAttributeListType attributeListType = null;

		if (documentos != null && !documentos.isEmpty()) {
			objResul = new T65BAttributeListListType();
			for (Aa83bDocumentoDokusi doc : documentos) {
				attributeListType = Aa83bDokusiUtil
						.getT65BAttibuteListTypeAsObject(doc);
				objResul.getAttributeList().add(attributeListType);
			}
		}

		return objResul;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de atributos
	 * (metadatos) en un String que representa un objeto
	 * T65BAttributeListListType
	 * 
	 * @param documentos
	 *            List<Documento>
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BAttibuteListListTypeAsString(
			List<Aa83bDocumentoDokusi> documentos) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BAttibuteListListTypeAsString INI");
		Aa83bMessageUnparser<T65BAttributeListListType> unparser = new Aa83bMessageUnparser<T65BAttributeListListType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createAttributeListList(Aa83bDokusiUtil
						.getT65BAttibuteListListTypeAsObject(documentos)));
		return resultado;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de atributos de
	 * documento en un objeto T65BDocumentAttrInfoListType
	 * 
	 * @param listaDocs
	 *            List<Documento>
	 * @return T65BDocumentAttrInfoListType
	 */
	public static T65BDocumentAttrInfoListType getT65BDocumentAttrInfoListTypeAsObject(
			List<Aa83bDocumentoDokusi> listaDocs) {
		Aa83bDokusiUtil.logger
				.trace("getT65BDocumentAttrInfoListTypeAsObject INI");
		T65BDocumentAttrInfoListType objDocumentAttrInfoList = Aa83bDokusiUtil.objectFactory
				.createT65BDocumentAttrInfoListType();
		T65BDocumentAttrInfoType objDocumentAttrInfo = null;

		if (listaDocs != null && !listaDocs.isEmpty()) {
			for (Aa83bDocumentoDokusi doc : listaDocs) {
				objDocumentAttrInfo = Aa83bDokusiUtil.objectFactory
						.createT65BDocumentAttrInfoType();
				objDocumentAttrInfo.setDocumentId(Aa83bDokusiUtil
						.getT65BDocumentIDAsObject(doc));
				objDocumentAttrInfo.setDocumentalType(doc.getTipoDocumento()
						.getTipoDocumental());
				objDocumentAttrInfoList.getDocumentInfo().add(
						objDocumentAttrInfo);
			}
		}

		return objDocumentAttrInfoList;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de atributos de
	 * documento en un String que representa un objeto
	 * T65BDocumentAttrInfoListType
	 * 
	 * @param listaDocs
	 *            List<Documento>
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BDocumentAttrInfoListTypeAsString(
			List<Aa83bDocumentoDokusi> listaDocs) throws JAXBException {
		Aa83bDokusiUtil.logger
				.trace("getT65BDocumentAttrInfoListTypeAsString INI");
		Aa83bMessageUnparser<T65BDocumentAttrInfoListType> unparser = new Aa83bMessageUnparser<T65BDocumentAttrInfoListType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createDocumentAttrInfoList(Aa83bDokusiUtil
						.getT65BDocumentAttrInfoListTypeAsObject(listaDocs)));
		return resultado;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de atributos de
	 * documento en un objeto T65BDocumentAttrInfoListType
	 * 
	 * @param listaDocs
	 *            List<Documento>
	 * @return T65BDocumentIDListType
	 */
	public static T65BDocumentIDListType getT65BDocumentIDListTypeAsObject(
			List<Aa83bDocumentoDokusi> listaDocs) {
		Aa83bDokusiUtil.logger.trace("getT65BDocumentIDListTypeAsObject INI");
		T65BDocumentIDListType objDocumentIDList = Aa83bDokusiUtil.objectFactory
				.createT65BDocumentIDListType();
		T65BDocumentIDType objDocumentID = null;

		if (listaDocs != null && !listaDocs.isEmpty()) {
			for (Aa83bDocumentoDokusi doc : listaDocs) {
				objDocumentID = Aa83bDokusiUtil.objectFactory
						.createT65BDocumentIDType();
				objDocumentID.setId(doc.getOid());
				// XXX(sin version) objDocumentID.setVersion(doc.getVersion());
				objDocumentIDList.getDocumentID().add(objDocumentID);
			}
		}

		return objDocumentIDList;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de ID de documento
	 * en un String que representa un objeto T65BDocumentIDListType
	 * 
	 * @param listaDocs
	 *            List<Documento>
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BDocumentIDListTypeAsString(
			List<Aa83bDocumentoDokusi> listaDocs) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BDocumentIDListTypeAsString INI");
		Aa83bMessageUnparser<T65BDocumentIDListType> unparser = new Aa83bMessageUnparser<T65BDocumentIDListType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createDocumentIDList(Aa83bDokusiUtil
						.getT65BDocumentIDListTypeAsObject(listaDocs)));
		return resultado;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de Strings tipos
	 * documentales en un objeto T65BStringListType
	 * 
	 * @param listaDocs
	 *            List<Documento>
	 * @return T65BStringListType
	 */
	public static T65BStringListType getT65BStringListTypeAsObject(
			List<Aa83bDocumentoDokusi> listaDocs) {
		Aa83bDokusiUtil.logger.trace("getT65BStringTypeAsObject INI");
		T65BStringListType objStringList = Aa83bDokusiUtil.objectFactory
				.createT65BStringListType();

		if (listaDocs != null && !listaDocs.isEmpty()) {
			for (Aa83bDocumentoDokusi doc : listaDocs) {
				objStringList.getElement().add(
						doc.getTipoDocumento().getTipoDocumental());
			}
		}

		return objStringList;
	}

	/**
	 * A partir de una lista de Documento, devuelve su lista de Strings tipos
	 * documentales en un objeto T65BStringListType
	 * 
	 * @param listaDocs
	 *            List<Documento>
	 * @return String
	 * @throws JAXBException
	 *             e
	 */
	public static String getT65BStringListTypeAsString(
			List<Aa83bDocumentoDokusi> listaDocs) throws JAXBException {
		Aa83bDokusiUtil.logger.trace("getT65BStringListTypeAsString INI");
		Aa83bMessageUnparser<T65BStringListType> unparser = new Aa83bMessageUnparser<T65BStringListType>();
		String resultado = null;
		resultado = unparser.parse(Aa83bDokusiUtil.objectFactory
				.createStringList(Aa83bDokusiUtil
						.getT65BStringListTypeAsObject(listaDocs)));
		return resultado;
	}

}
