/*
 * Copyright 2012 E.J.I.E., S.A.
 *
 * Licencia con arreglo a la EUPL, Versión 1.1 exclusivamente (la «Licencia»);
 * Solo podrá usarse esta obra si se respeta la Licencia.
 * Puede obtenerse una copia de la Licencia en
 *
 * http://ec.europa.eu/idabc/eupl.html
 *
 * Salvo cuando lo exija la legislación aplicable o se acuerde por escrito,
 * el programa distribuido con arreglo a la Licencia se distribuye «TAL CUAL»,
 * SIN GARANTÍAS NI CONDICIONES DE NINGÚN TIPO, ni expresas ni implícitas.
 * Véase la Licencia en el idioma concreto que rige los permisos y limitaciones
 * que establece la Licencia.
 */

package com.ejie.aa21b.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa21b.model.NoraMunicipio;
import com.ejie.x38.dto.Pagination;

/**
 * * NoraMunicipioDaoImpl  
 * 
 *  
 */

@Repository
@Transactional
public class NoraMunicipioDaoImpl implements NoraMunicipioDao {
	private JdbcTemplate jdbcTemplate;
	private RowMapper<NoraMunicipio> rwMap = new RowMapper<NoraMunicipio>() {
		public NoraMunicipio mapRow(ResultSet resultSet, int rowNum)
				throws SQLException {
			return new NoraMunicipio(resultSet.getString("ID"),
					resultSet.getString("DSO"),
					resultSet.getString("PROVINCIAID"),
					resultSet.getString("COMARCAID"));
		}
	};

	/**
	 * Method use to set the datasource.
	 * 
	 * @param dataSource
	 *            DataSource
	 * @return
	 */
	@Resource
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/**
	 * Inserts a single row in the NoraMunicipio table.
	 * 
	 * @param municipio
	 *            Pagination
	 * @return NoraMunicipio
	 */
	public NoraMunicipio add(NoraMunicipio municipio) {

		String query = "INSERT INTO AA21B94S01( ID,DS_O,PROVINCIA_ID,COMARCA_ID)"
				+ "VALUES (?,?,?,?)";

		this.jdbcTemplate.update(query, municipio.getId(), municipio.getDsO(),
				municipio.getProvinciaId(), municipio.getComarcaId());
		return municipio;
	}

	/**
	 * Updates a single row in the NoraMunicipio table.
	 * 
	 * @param municipio
	 *            Pagination
	 * @return NoraMunicipio
	 */
	public NoraMunicipio update(NoraMunicipio municipio) {
		String query = "UPDATE AA21B94S01 SET  WHERE ID=? AND DS_O=? AND PROVINCIA_ID=? AND COMARCA_ID=?";
		this.jdbcTemplate.update(query, municipio.getId(), municipio.getDsO(),
				municipio.getProvinciaId(), municipio.getComarcaId());
		return municipio;
	}

	/**
	 * Finds a single row in the NoraMunicipio table.
	 * 
	 * @param municipio
	 *            Pagination
	 * @return NoraMunicipio
	 */
	@Transactional(readOnly = true)
	public NoraMunicipio find(NoraMunicipio municipio) {
		String query = "SELECT t1.ID ID, t1.DS_O DSO, t1.PROVINCIA_ID PROVINCIAID, t1.COMARCA_ID COMARCAID "
				+ "FROM AA21B94S01 t1  "
				+ "WHERE t1.ID = ?   AND t1.PROVINCIA_ID = ? ";
		return (NoraMunicipio) this.jdbcTemplate.queryForObject(query, rwMap,
				municipio.getId(), municipio.getProvinciaId());
	}

	/**
	 * Removes a single row in the NoraMunicipio table.
	 * 
	 * @param municipio
	 *            Pagination
	 * @return
	 */
	public void remove(NoraMunicipio municipio) {
		String query = "DELETE  FROM AA21B94S01 WHERE ID=? AND DS_O=? AND PROVINCIA_ID=? AND COMARCA_ID=?";
		this.jdbcTemplate.update(query, municipio.getId(), municipio.getDsO(),
				municipio.getProvinciaId(), municipio.getComarcaId());
	}

	/**
	 * Finds a List of rows in the NoraMunicipio table.
	 * 
	 * @param municipio
	 *            NoraMunicipio
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<NoraMunicipio> findAll(NoraMunicipio municipio,
			Pagination pagination) {
		StringBuffer where = new StringBuffer(3000);
		List<Object> params = new ArrayList<Object>();
		where.append(" WHERE 1=1 	");

		StringBuilder query = new StringBuilder(
				"SELECT  t1.ID ID,t1.DS_O DSO,t1.PROVINCIA_ID PROVINCIAID,t1.COMARCA_ID COMARCAID "
						+ "FROM AA21B94S01 t1 ");

		if (municipio != null && municipio.getId() != null) {
			where.append(" AND t1.ID = ?");
			params.add(municipio.getId());
		}
		if (municipio != null && municipio.getDsO() != null) {
			where.append(" AND t1.DS_O = ?");
			params.add(municipio.getDsO());
		}
		if (municipio != null && municipio.getProvinciaId() != null) {
			where.append(" AND t1.PROVINCIA_ID = ?");
			params.add(municipio.getProvinciaId());
		}
		if (municipio != null && municipio.getComarcaId() != null) {
			where.append(" AND t1.COMARCA_ID = ?");
			params.add(municipio.getComarcaId());
		}
		query.append(where);

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}
		return (List<NoraMunicipio>) this.jdbcTemplate.query(query.toString(),
				rwMap, params.toArray());
	}

	/**
	 * Counts rows in the NoraMunicipio table.
	 * 
	 * @param municipio
	 *            NoraMunicipio
	 * @return Long
	 */
	@Transactional(readOnly = true)
	public Long findAllCount(NoraMunicipio municipio) {

		StringBuffer where = new StringBuffer(3000);
		List<Object> params = new ArrayList<Object>();
		where.append(" WHERE 1=1  ");

		StringBuffer query = new StringBuffer(
				"SELECT COUNT(1) FROM  AA21B94S01 t1  ");
		if (municipio != null && municipio.getId() != null) {
			where.append(" AND t1.ID = ?");
			params.add(municipio.getId());
		}
		if (municipio != null && municipio.getDsO() != null) {
			where.append(" AND t1.DS_O = ?");
			params.add(municipio.getDsO());
		}
		if (municipio != null && municipio.getProvinciaId() != null) {
			where.append(" AND t1.PROVINCIA_ID = ?");
			params.add(municipio.getProvinciaId());
		}
		if (municipio != null && municipio.getComarcaId() != null) {
			where.append(" AND t1.COMARCA_ID = ?");
			params.add(municipio.getComarcaId());
		}
		query.append(where);
		return this.jdbcTemplate.queryForLong(query.toString(),
				params.toArray());
	}

	/**
	 * Finds rows in the NoraMunicipio table using like.
	 * 
	 * @param municipio
	 *            NoraMunicipio
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	@Transactional(readOnly = true)
	public List<NoraMunicipio> findAllLike(NoraMunicipio municipio,
			Pagination pagination, Boolean startsWith) {
		StringBuffer where = new StringBuffer(3000);
		List<Object> params = new ArrayList<Object>();
		where.append(" WHERE 1=1 	");

		StringBuilder query = new StringBuilder(
				"SELECT  t1.ID ID,t1.DS_O DSO,t1.PROVINCIA_ID PROVINCIAID,t1.COMARCA_ID COMARCAID "
						+ "FROM AA21B94S01 t1 ");

		if (municipio != null && municipio.getId() != null) {
			where.append(" AND UPPER(t1.ID) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(municipio.getId().toUpperCase() + "%");
			} else {
				params.add("%" + municipio.getId().toUpperCase() + "%");
			}
			where.append(" AND t1.ID IS NOT NULL");
		}
		if (municipio != null && municipio.getDsO() != null) {
			where.append(" AND UPPER(t1.DS_O) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(municipio.getDsO().toUpperCase() + "%");
			} else {
				params.add("%" + municipio.getDsO().toUpperCase() + "%");
			}
			where.append(" AND t1.DS_O IS NOT NULL");
		}
		if (municipio != null && municipio.getProvinciaId() != null) {
			where.append(" AND UPPER(t1.PROVINCIA_ID) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(municipio.getProvinciaId().toUpperCase() + "%");
			} else {
				params.add("%" + municipio.getProvinciaId().toUpperCase() + "%");
			}
			where.append(" AND t1.PROVINCIA_ID IS NOT NULL");
		}
		if (municipio != null && municipio.getComarcaId() != null) {
			where.append(" AND UPPER(t1.COMARCA_ID) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(municipio.getComarcaId().toUpperCase() + "%");
			} else {
				params.add("%" + municipio.getComarcaId().toUpperCase() + "%");
			}
			where.append(" AND t1.COMARCA_ID IS NOT NULL");
		}
		query.append(where);

		if (pagination != null) {
			query = pagination.getPaginationQuery(query);
		}
		return (List<NoraMunicipio>) this.jdbcTemplate.query(query.toString(),
				rwMap, params.toArray());
	}
}
