package com.ejie.aa80a.control;

import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.annotation.AuthenticationPrincipal;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;

import com.ejie.aa80a.exception.AjaxTimeoutException;
import com.ejie.aa80a.model.Combo;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Usuario;
import com.ejie.aa80a.model.UsuarioDetalles;
import com.ejie.aa80a.service.RecursoService;
import com.ejie.aa80a.service.UsuarioService;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;
import com.ejie.aa80a.util.exception.ValidacionException;

/**
 * Controlador principal
 * 
 *  
 */
@Controller()
public class MainController {

	private static final Logger logger = LoggerFactory.getLogger(MainController.class);

	@Autowired()
	@Qualifier(value = "messageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private UsuarioService usuarioService;

	@Autowired()
	private RecursoService recursoService;

	@Autowired()
	private Properties appConfiguration;

	/**
	 * Página de inicio.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/", method = RequestMethod.GET)
	public String index(Map<String, Object> model, @AuthenticationPrincipal() UsuarioDetalles usuario) {
		logger.debug("Usuario: {}", Utilidades.getDetallesUsuario());

		logger.debug("Usuario por parámetro: {}", usuario);

		return "redirect:/inicio";
	}

	/**
	 * Determina la url de inicio dependiendo del perfil del usuario logado.
	 * 
	 * @param usuario Datos del usuario logado
	 * 
	 * @return url de destino
	 */
	@RequestMapping(value = "/inicio", method = RequestMethod.GET)
	public String inicio(@AuthenticationPrincipal() UsuarioDetalles usuario) {
		String url = "";

		Collection<? extends GrantedAuthority> authorities = usuario.getAuthorities();

		List<String> roles = new ArrayList<String>();

		for (GrantedAuthority a : authorities) {
			roles.add(a.getAuthority());
		}

		if (Utilidades.esAdministrador(roles)) {
			url = "inicioCoordinador";
		} else if (Utilidades.esOficina(roles)) {
			url = "inicioOficina";
		} else if (Utilidades.esUsuario(roles)) {
			url = "inicioUsuario";
		} else {
			url = "accessDenied";
		}

		return url;
	}

	/**
	 * Página de inicio para un usuario.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/inicioUsuario", method = RequestMethod.GET)
	public String indexUsuario(Map<String, Object> model, @AuthenticationPrincipal() UsuarioDetalles usuario) {
		logger.debug("Usuario: {}", usuario);

		return "inicioUsuario";
	}

	/**
	 * Página de inicio para una oficina.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/inicioOficina", method = RequestMethod.GET)
	public String indexOficina(Map<String, Object> model, @AuthenticationPrincipal() UsuarioDetalles usuario) {
		logger.debug("Usuario: {}", usuario);

		return "inicioOficina";
	}

	/**
	 * Página de inicio para un coordinador.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/inicioCoordinador", method = RequestMethod.GET)
	public String indexCoordinador(Map<String, Object> model, @AuthenticationPrincipal() UsuarioDetalles usuario) {
		logger.debug("Usuario: {}", usuario);

		return "inicioCoordinador";
	}

	/**
	 * Página de modificación de datos de usuario.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/datosUsuario", method = RequestMethod.GET)
	public String datosUsuario(ModelMap model, @AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		// Carga de datos del usuario
		Usuario datosUsuario = this.usuarioService.findByUsuario(usuario.getUsername());

		String nombreRecurso = "";

		if (locale.equals(new Locale("eu"))) {
			nombreRecurso = usuario.getNombreRecursoEuskera();
		} else {
			nombreRecurso = usuario.getNombreRecursoCastellano();
		}

		model.addAttribute("datosUsuario", datosUsuario);
		model.addAttribute("nombreRecurso", nombreRecurso);

		return "datosUsuario";
	}

	/**
	 * Página para cambiar la password.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/cambiarPassword", method = RequestMethod.GET)
	public String cambiarPassword(ModelMap model, @AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		String nombreRecurso = "";

		if (locale.equals(new Locale("eu"))) {
			nombreRecurso = usuario.getNombreRecursoEuskera();
		} else {
			nombreRecurso = usuario.getNombreRecursoCastellano();
		}

		model.addAttribute("nombreRecurso", nombreRecurso);

		return "cambiarPassword";
	}

	/**
	 * Página de gestión de usuarios.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios", method = RequestMethod.GET)
	public String gestionUsuarios(ModelMap model, @AuthenticationPrincipal() UsuarioDetalles usuario) {
		logger.debug("Usuario: {}", usuario);

		return "gestionUsuarios";
	}

	/**
	 * Página de búsqueda de recursos.
	 * 
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/oficina/buscarRecurso", method = RequestMethod.GET)
	public String busquedaRecurso(ModelMap model, @AuthenticationPrincipal() UsuarioDetalles usuario) {
		logger.debug("Usuario: {}", usuario);

		return "buscarRecurso";
	}

	/**
	 * Página de modificación de los datos del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getDatosRecurso", method = RequestMethod.GET)
	public String getDatosRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getDatosRecurso(codigoRecurso);

			model.addAttribute("recurso", recurso);

			// Opción seleccionada en el menú
			model.addAttribute("datos", true);
			model.addAttribute("misDatos", true);

			// // Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

			// En caso de tratarse de un recurso con ficha reducida
			if (Constants.TIPO_RECURSOS_FICHA_REDUCIDAD.contains(recurso.getTipo())) {

				// en caso de tratarse de N3 - convention bureau, no muestro el campo imagen, ya que ese subtipo tiene
				// ficha base
				if (Constants.EMPRESAS_RELACIONADAS.equals(recurso.getTipo())
						&& Constants.N3_SUBTIPO_CONVENTION_BUREAU.equals(recurso.getSubtipo())) {
					model.addAttribute("fotoPrincipal", false); // no muestro el campo de la foto principal para
																// convention bureau
				} else {
					model.addAttribute("fotoPrincipal", true); // muestro el campo de la foto principal
				}

			} else {
				model.addAttribute("fotoPrincipal", false); // no muestro la foto principal
			}

			model.addAttribute("fotoPrincipalWidth", Constants.FICHA_REDUCIDA_FOTO_WIDTH); // ancho de la imagen de
			// la foto principal
			model.addAttribute("fotoPrincipalHeight", Constants.FICHA_REDUCIDA_FOTO_HEIGHT); // alto de la imagen de
			// la foto principal
			model.addAttribute("urlPortal", this.appConfiguration.getProperty("portal.host"));

			// monto el mensaje de aviso de la pantalla en función de tipo de recurso
			String mensajeAviso = "";
			if (Constants.ALOJAMIENTO.equals(recurso.getTipo())) {
				// si se trata de un alojamiento, los links del mensaje van en función del subtipo de alojamiento
				String linkSustanciales = this.messageSource.getMessage("recursos.avisoDatosGeneralesAlojamientos."
						+ recurso.getSubtipo() + ".sustanciales", null, locale);
				String linkNoSustanciales = this.messageSource.getMessage("recursos.avisoDatosGeneralesAlojamientos."
						+ recurso.getSubtipo() + ".noSustanciales", null, locale);
				mensajeAviso = this.messageSource.getMessage("recursos.avisoDatosGeneralesAlojamientos", new Object[] {
						linkSustanciales, linkNoSustanciales }, locale);

			} else if (Constants.EMPRESAS_RELACIONADAS.equals(recurso.getTipo())
					&& (Constants.N3_AGENCIA_VIAJES_MINORISTA.equals(recurso.getSubtipo())
							|| Constants.N3_AGENCIA_VIAJES_MAYORISTA.equals(recurso.getSubtipo()) || Constants.N3_AGENCIA_VIAJES_MINORISTA_MAYORISTA
								.equals(recurso.getSubtipo()))) {
				// en caso de ser una agencia de viajes
				String linkSustanciales = this.messageSource.getMessage(
						"recursos.avisoDatosGeneralesAlojamientos.agencia.sustanciales", null, locale);
				String linkNoSustanciales = this.messageSource.getMessage(
						"recursos.avisoDatosGeneralesAlojamientos.agencia.noSustanciales", null, locale);

				mensajeAviso = this.messageSource.getMessage("recursos.avisoDatosGeneralesAlojamientos", new Object[] {
						linkSustanciales, linkNoSustanciales }, locale);

			} else {// para el resto de casos, siempre el mismo mensaje
				mensajeAviso = this.messageSource.getMessage("recursos.avisoDatosGeneralesGlobal", null, locale);
			}
			model.addAttribute("mensajeAviso", mensajeAviso);

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "datosRecurso";
	}

	/**
	 * Página de modificación de las características y servicios del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getServiciosRecurso", method = RequestMethod.GET)
	public String getServiciosRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getServiciosRecurso(codigoRecurso);

			model.addAttribute("recurso", recurso);

			// Opción seleccionada en el menú
			model.addAttribute("servicios", true);
			model.addAttribute("misDatos", true);

			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "serviciosRecurso";
	}

	/**
	 * Página de modificación de la localización del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getLocalizacionRecurso", method = RequestMethod.GET)
	public String getLocalizacionRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getLocalizacionRecurso(codigoRecurso);

			model.addAttribute("recurso", recurso);

			// Opción seleccionada en el menú
			model.addAttribute("localizacion", true);
			model.addAttribute("misDatos", true);

			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "localizacionRecurso";
	}

	/**
	 * Página de modificación de precios del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getPreciosRecurso", method = RequestMethod.GET)
	public String getPreciosRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getPreciosRecurso(codigoRecurso);
			model.addAttribute("recurso", recurso);

			List<Combo> tipoPrecios = this.recursoService.getComboTiposPrecio(recurso.getSubtipo());

			LinkedHashMap<String, String> listTipoPrecios = new LinkedHashMap<String, String>();

			for (Combo tipo : tipoPrecios) {
				listTipoPrecios.put(tipo.getValue(), tipo.getText());
			}
			// para cargar combo de tipos de precio
			model.addAttribute("tiposPrecio", listTipoPrecios);
			model.addAttribute("codTipoPrecio", "");

			// Opción seleccionada en el menú
			model.addAttribute("precios", true);

			// constantes de tipos de precios
			model.addAttribute("ALOJAMIENTO", Constants.ALOJAMIENTO);
			model.addAttribute("TIPO_ALOJAMIENTO_HOTEL", Constants.TIPO_ALOJAMIENTO_HOTEL);
			model.addAttribute("TIPO_ALOJAMIENTO_CASA_RURAL", Constants.TIPO_ALOJAMIENTO_CASA_RURAL);
			model.addAttribute("TIPO_ALOJAMIENTO_AGROTURISMO", Constants.TIPO_ALOJAMIENTO_AGROTURISMO);

			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "preciosRecurso";
	}

	/**
	 * Página de modificación de archivos multimedia del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getMultimediaRecurso", method = RequestMethod.GET)
	public String getMultimediaRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getMultimediaRecurso(codigoRecurso);

			recurso.setRutaRecurso(Utilidades.getURLContenido(recurso.getCodigo(), recurso.getTipo(), "es"));

			String tamanioMultimedia = "";

			if (recurso.isEsTop()) {
				tamanioMultimedia = (String) Constants.MAPA_DATOS_IMAGEN_RECURSO.get(Constants.RECURSO_TOP);
			} else {
				tamanioMultimedia = (String) Constants.MAPA_DATOS_IMAGEN_RECURSO.get(recurso.getTipo());
			}

			model.addAttribute("multimediaWidth", tamanioMultimedia.split("x")[0]);
			model.addAttribute("multimediaHeight", tamanioMultimedia.split("x")[1]);

			model.addAttribute("USUARIO_GAURKOTU", Constants.USUARIO_GAURKOTU);

			model.addAttribute("recurso", recurso);

			model.addAttribute("urlPortal", this.appConfiguration.getProperty("portal.host"));

			// Opción seleccionada en el menú
			model.addAttribute("multimedia", true);

			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "multimediaRecurso";
	}

	/**
	 * Página de sugerencias del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getSugerenciasRecurso", method = RequestMethod.GET)
	public String getSugerenciasRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getSugerenciasRecurso(codigoRecurso);

			model.addAttribute("recurso", recurso);

			// Opción seleccionada en el menú
			model.addAttribute("sugerencias", true);
			model.addAttribute("misDatos", true);

			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "sugerenciasRecurso";
	}

	/**
	 * Página de ofertas del recurso.
	 * 
	 * @param codigo Código de recurso
	 * @param model Modelo
	 * @param usuario Detalles del usuario autenticado
	 * @param locale Locale actual
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/usuario/recursos/getOfertasRecurso", method = RequestMethod.GET)
	public String getOfertasRecurso(@RequestParam(value = "codigo") String codigo, ModelMap model,
			@AuthenticationPrincipal() UsuarioDetalles usuario, Locale locale) {
		logger.debug("Usuario: {}", usuario);

		try {
			long codigoRecurso = Long.parseLong(codigo);

			// Comprobamos si el usuario tiene acceso a ese recurso
			if (!Utilidades.tieneAcceso(usuario, codigoRecurso)) {
				logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", codigo);

				throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null, locale),
						this.messageSource.getMessage("error.mensajeRecursoNoAccesible", null, locale));
			}

			Recurso recurso = this.recursoService.getOfertasRecurso(codigoRecurso);

			String tamanioMultimedia = (String) Constants.MAPA_DATOS_IMAGEN_RECURSO.get(Constants.OFERTAS);
			model.addAttribute("multimediaWidth", tamanioMultimedia.split("x")[0]);
			model.addAttribute("multimediaHeight", tamanioMultimedia.split("x")[1]);

			model.addAttribute("recurso", recurso);

			model.addAttribute("urlPortal", this.appConfiguration.getProperty("portal.host"));

			// Opción seleccionada en el menú
			model.addAttribute("ofertas", true);

			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			model.addAttribute("editable", Utilidades.getEditable(recurso));

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null, locale),
					this.messageSource.getMessage("error.mensajeRecursoNoEncontrado", null, locale));
		}

		return "ofertasRecurso";
	}

	/**
	 * Página de gestión de notificaciones.
	 * 
	 * @param model Modelo
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/gestion/gestionNotificaciones", method = RequestMethod.GET)
	public String gestionNotificaciones(ModelMap model) {

		model.addAttribute("gestionNotificaciones", true);
		model.addAttribute("gestionSugerencias", false);
		model.addAttribute("urlPortal", this.appConfiguration.getProperty("portal.host"));
		model.addAttribute("estadoNotificacionVista", Constants.NOTIFICACION_VISTA);

		return "gestionNotificaciones";
	}

	/**
	 * Página de gestión de sugerencias.
	 * 
	 * @param model Modelo
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/gestion/gestionSugerencias", method = RequestMethod.GET)
	public String gestionSugerencias(ModelMap model) {

		model.addAttribute("gestionNotificaciones", false);
		model.addAttribute("gestionSugerencias", true);
		model.addAttribute("urlPortal", this.appConfiguration.getProperty("portal.host"));
		model.addAttribute("estadoSugerenciaVista", Constants.SUGERENCIA_VISTA);

		return "gestionSugerencias";
	}

	/**
	 * Página de accesso denegado.
	 * 
	 * @param model Modelo
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/accessDenied", method = RequestMethod.GET)
	public String accessDeniedPage(ModelMap model) {
		logger.debug("Usuario: {}", Utilidades.getDetallesUsuario());

		model.addAttribute("user", Utilidades.getDetallesUsuario());
		return "accessDenied";
	}

	/**
	 * Página de accesso denegado.
	 * 
	 * Salta cuando la sesión ha expirado y hacemos una petición AJAX
	 * 
	 * @param request request
	 * @param response response
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/accessDenied", method = RequestMethod.POST)
	public String accessDeniedPagePost(HttpServletRequest request, HttpServletResponse response) {
		String ajaxHeader = request.getHeader("X-Requested-With");

		if ("XMLHttpRequest".equals(ajaxHeader)) {
			logger.info("Llamada Ajax detectada, se lanza AjaxTimeoutException");

			Authentication auth = SecurityContextHolder.getContext().getAuthentication();
			if (auth != null) {
				new SecurityContextLogoutHandler().logout(request, response, auth);
			}

			throw new AjaxTimeoutException();
		}

		return "redirect:/login?timeout";
	}

	/**
	 * Página de error 404.
	 * 
	 * @param model Modelo
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/notFound", method = RequestMethod.GET)
	public String notFoundPage(ModelMap model) {
		logger.debug("Usuario: {}", Utilidades.getDetallesUsuario());

		model.addAttribute("user", Utilidades.getDetallesUsuario());
		return "notFound";
	}

	/**
	 * Página de login.
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/login", method = RequestMethod.GET)
	public String loginPage() {
		return "login";
	}

	/**
	 * Página de logout.
	 * 
	 * @param request request
	 * @param response response
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/logout", method = RequestMethod.GET)
	public String logoutPage(HttpServletRequest request, HttpServletResponse response) {
		Authentication auth = SecurityContextHolder.getContext().getAuthentication();
		if (auth != null) {
			new SecurityContextLogoutHandler().logout(request, response, auth);
		}
		return "redirect:/login?logout";
	}

	/**
	 * Página de timeout.
	 * 
	 * @param request request
	 * @param response response
	 * 
	 * @return vista
	 */
	@RequestMapping(value = "/timeout", method = RequestMethod.GET)
	public String timeoutPage(HttpServletRequest request, HttpServletResponse response) {
		logger.debug("Usuario: {}", Utilidades.getDetallesUsuario());

		String ajaxHeader = request.getHeader("X-Requested-With");

		if ("XMLHttpRequest".equals(ajaxHeader)) {
			logger.info("Llamada Ajax detectada, se lanza AjaxTimeoutException");

			Authentication auth = SecurityContextHolder.getContext().getAuthentication();
			if (auth != null) {
				new SecurityContextLogoutHandler().logout(request, response, auth);
			}

			throw new AjaxTimeoutException();
		}

		return "redirect:/login?timeout";
	}

}