package com.ejie.aa80a.control;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.servlet.http.HttpServletResponse;
import javax.validation.Valid;
import javax.xml.datatype.DatatypeConfigurationException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.oxm.XmlMappingException;
import org.springframework.stereotype.Controller;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RequestPart;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.ModelAndView;

import com.ejie.aa80a.model.Condicion;
import com.ejie.aa80a.model.FiltroNotificacion;
import com.ejie.aa80a.model.FiltroRecurso;
import com.ejie.aa80a.model.FiltroSugerencia;
import com.ejie.aa80a.model.FiltroUsuario;
import com.ejie.aa80a.model.Imagen;
import com.ejie.aa80a.model.ListaPaginadaServidor;
import com.ejie.aa80a.model.Localidad;
import com.ejie.aa80a.model.Multimedia;
import com.ejie.aa80a.model.Municipio;
import com.ejie.aa80a.model.Notificacion;
import com.ejie.aa80a.model.Oferta;
import com.ejie.aa80a.model.Precio;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Sugerencia;
import com.ejie.aa80a.model.Usuario;
import com.ejie.aa80a.service.FcaService;
import com.ejie.aa80a.service.NotificacionService;
import com.ejie.aa80a.service.RecursoService;
import com.ejie.aa80a.service.SugerenciaService;
import com.ejie.aa80a.service.UsuarioService;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;
import com.ejie.aa80a.util.exception.ParameterValidationException;
import com.ejie.aa80a.util.exception.ValidacionException;

/**
 * Controlador para las peticiones AJAX.
 * 
 *  
 */
@Controller()
public class AjaxController {

	private static final Logger logger = LoggerFactory.getLogger(AjaxController.class);

	@Autowired()
	@Qualifier(value = "messageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private UsuarioService usuarioService;

	@Autowired()
	private RecursoService recursoService;

	@Autowired()
	private NotificacionService notificacionService;

	@Autowired()
	private SugerenciaService sugerenciaService;

	@Autowired()
	private FcaService fcaService;

	/**
	 * Envío de un correo con una nueva contraseña.
	 * 
	 * @param params Parámetros de la request
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/recordarPassword", method = RequestMethod.POST)
	public @ResponseBody()
	String recordarPassword(@RequestBody() Map<String, String> params) {
		String email = params.get("email");

		return this.usuarioService.recordarPassword(email);
	}

	/**
	 * Guardar los datos del usuario.
	 * 
	 * @param usuario Objeto con los datos del usuario
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/guardarDatosUsuario", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarDatosUsuario(@RequestBody() @Valid() Usuario usuario) {
		return this.usuarioService.guardarUsuario(usuario);
	}

	/**
	 * Cambiar la password.
	 * 
	 * @param params Parámetros de la request
	 * @param locale Locale actual
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/guardarPassword", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarPassword(@RequestBody() Map<String, String> params, Locale locale) {
		String password = params.get("password");
		String nuevaPassword = params.get("nuevaPassword");
		String confirmarPassword = params.get("confirmarPassword");

		if (password == null) {
			throw new ParameterValidationException(this.messageSource.getMessage("error.passwordObligatoria", null,
					locale));
		}

		if (nuevaPassword == null || confirmarPassword == null) {
			throw new ParameterValidationException(this.messageSource.getMessage("error.nuevaPasswordObligatoria",
					null, locale));
		}

		if (!nuevaPassword.equals(confirmarPassword)) {
			throw new ParameterValidationException(
					this.messageSource.getMessage("error.passwordDistinta", null, locale));
		}

		return this.usuarioService.cambiarPassword(password, nuevaPassword);
	}

	// ******** GESTION USUARIOS INICIO **********

	/**
	 * Obtiene la lista de usuarios.
	 * 
	 * @param filtro Criterios de búsqueda
	 * 
	 * @return Lista de usuarios
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaUsuarios(@ModelAttribute() FiltroUsuario filtro) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.usuarioService.findByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.usuarioService.findByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Usuario>());
		}

		return resultado;
	}

	/**
	 * Obtiene los datos de un usuario.
	 * 
	 * @param userName Login del usuario
	 * 
	 * @return Objeto con los datos del usuario
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios/getUsuario", method = RequestMethod.POST)
	public @ResponseBody()
	Usuario getUsuario(@RequestBody() String userName) {
		return this.usuarioService.findByUsuario(userName);
	}

	/**
	 * Envío de un correo con una nueva contraseña.
	 * 
	 * @param usuario Objeto con los datos del usuario
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios/cambiarPassword", method = RequestMethod.POST)
	public @ResponseBody()
	String cambiarPassword(@RequestBody() FiltroUsuario usuario) {
		return this.usuarioService.recordarPassword(usuario.getEmail());
	}

	/**
	 * Obtiene los datos de un usuario.
	 * 
	 * @param idusuario Id del usuario
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios/eliminarUsuario", method = RequestMethod.POST)
	public @ResponseBody()
	String getUsuario(@RequestBody() Integer idusuario) {
		return this.usuarioService.eliminarUsuario(idusuario);
	}

	/**
	 * Guardar los datos del usuario.
	 * 
	 * @param usuario Objeto con los datos del usuario
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios/guardarUsuario", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarUsuario(@RequestBody() @Valid() Usuario usuario) {
		// Dependiendo de si viene con codigoUsuario o no, será un alta o una
		// modificación.

		if (usuario.getIdUsuario() != null) {
			return this.usuarioService.guardarUsuario(usuario);
		} else {
			return this.usuarioService.crearUsuario(usuario);
		}
	}

	/**
	 * Obtiene la lista de recursos restringidos a los tipos A1 y E1.
	 * 
	 * @param filtro Criterios de búsqueda
	 * 
	 * @return Lista de recursos
	 */
	@RequestMapping(value = "/gestion/gestionUsuarios/getListaRecursos", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaRecursos(@ModelAttribute() FiltroRecurso filtro) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.recursoService.findByCountCriteria(filtro, false);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.recursoService.findByCriteria(filtro, false));
		} else {
			resultado.setRows(new ArrayList<Recurso>());
		}

		return resultado;
	}

	// ******** GESTION USUARIOS FIN **********

	// ******** GESTION RECURSOS FIN **********

	/**
	 * Obtiene la lista de todos los recursos editables.
	 * 
	 * @param filtro Criterios de búsqueda
	 * 
	 * @return Lista de recursos
	 */
	@RequestMapping(value = "/oficina/recursos/getListaRecursos", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaRecursosEditables(@ModelAttribute() FiltroRecurso filtro) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.recursoService.findByCountCriteria(filtro, true);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.recursoService.findByCriteria(filtro, true));
		} else {
			resultado.setRows(new ArrayList<Recurso>());
		}

		return resultado;
	}

	/**
	 * Guarda los datos modificados de un recurso
	 * 
	 * @param recurso {@link Recurso} Objeto con los datos del recurso
	 * @param file fichero de la imagen de la foto principal del recurso (solo para recursos de ficha reducida)
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usario/recursos/guardarDatosRecurso", method = RequestMethod.POST)
	public @ResponseBody()
	// String guardarDatosRecurso(@RequestBody() @Valid() Recurso recurso) throws XmlMappingException,
	// DatatypeConfigurationException, IOException {
	String guardarDatosRecurso(@RequestPart(value = "object") @Valid() Recurso recurso,
			@RequestPart(value = "file", required = false) MultipartFile file) throws XmlMappingException,
			DatatypeConfigurationException, IOException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), recurso.getCodigo())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", recurso.getCodigo());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		if (!Utilidades.getEditable(recurso).isDatosBasicos()) { // Comprobamos si los datos del
																	// recurso son editables
			logger.warn("Se ha intentado editar un recurso no editable: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoEditable",
					null, LocaleContextHolder.getLocale()));
		}

		// En caso de tener que guardar tambien la imagen de la foto principal, almacenamos la imagen en el objeto
		// recurso
		if (file != null) {
			// oferta.getImagen().setLength(file.getBytes().length);
			recurso.getImagen().setBytes(file.getBytes());
			recurso.getImagen().setTipo(file.getContentType());

			recurso.getImagen().setBytes(Utilidades.recortarImagen(recurso.getImagen()));
			recurso.getImagen().setLength(recurso.getImagen().getBytes().length);
			recurso.getImagen().setNombre(Utilidades.escapeFileName(recurso.getImagen().getNombre()));
		}

		return this.recursoService.guardarDatosRecurso(recurso);
	}

	/**
	 * Guarda los servicios modificados de un recurso
	 * 
	 * @param recurso {@link Recurso} Objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usario/recursos/guardarServiciosRecurso", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarServiciosRecurso(@RequestBody() Recurso recurso) throws XmlMappingException,
			DatatypeConfigurationException, IOException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), recurso.getCodigo())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", recurso.getCodigo());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		// Comprobamos si los servicios del recurso son editables
		if (!Utilidades.getEditable(recurso).isServicios()) {
			logger.warn("Se ha intentado editar servicios no editables: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.serviciosNoEditables", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeServiciosNoEditables", null, LocaleContextHolder.getLocale()));
		}

		return this.recursoService.guardarServiciosRecurso(recurso);
	}

	/**
	 * Guarda las sugerencias introducidas para un recurso
	 * 
	 * @param recurso {@link Recurso} Objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 */
	@RequestMapping(value = "/usario/recursos/guardarSugerenciasRecurso", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarSugerenciasRecurso(@RequestBody() @Valid() Recurso recurso) {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), recurso.getCodigo())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", recurso.getCodigo());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		return this.recursoService.guardarSugerenciasRecurso(recurso);
	}

	/**
	 * Guarda la localización introducida para un recurso
	 * 
	 * @param recurso {@link Recurso} Objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usario/recursos/guardarLocalizacionRecurso", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarLocalizacionRecurso(@RequestBody() Recurso recurso) throws XmlMappingException,
			DatatypeConfigurationException, IOException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), recurso.getCodigo())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", recurso.getCodigo());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		// Comprobamos si la localización del recurso es editable
		if (!Utilidades.getEditable(recurso).isLocalizacion()) {
			logger.warn("Se ha intentado editar una localización no editable: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.localizacionNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeLocalizacionNoEditable", null, LocaleContextHolder.getLocale()));
		}

		return this.recursoService.guardarLocalizacionRecurso(recurso);
	}

	/**
	 * Obtiene la lista de todas las ofertas del recurso.
	 * 
	 * @param codigo Código del recurso
	 * 
	 * @return Lista de recursos
	 */
	@RequestMapping(value = "/usuario/recursos/getListaOfertas", method = RequestMethod.POST)
	public @ResponseBody()
	List<Oferta> getListaOfertas(@RequestParam(value = "codigo") String codigo) {
		List<Oferta> resultado = new ArrayList<Oferta>();

		try {
			long codigoRecurso = Long.parseLong(codigo);

			resultado = this.recursoService.findOfertasByCorec(codigoRecurso);
		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoEncontrado",
					null, LocaleContextHolder.getLocale()));
		}

		return resultado;
	}

	/**
	 * Guarda la oferta introducida.
	 * 
	 * @param oferta {@link Oferta} Objeto con los datos de la oferta
	 * @param file Contenido binario del archivo de imagen
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usuario/recursos/guardarOferta", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarOferta(@RequestPart(value = "object") @Valid() Oferta oferta,
			@RequestPart(value = "file", required = false) MultipartFile file) throws IOException, XmlMappingException,
			DatatypeConfigurationException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), oferta.getCodigoPadre())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", oferta.getCodigoPadre());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		// Comprobamos si podemos introducir ofertas para ese recurso
		if (!Constants.EDITABLES_MAP.get(oferta.getTipoPadre()).isOfertas()) {
			logger.warn("Se ha intentado introducir una oferta para un tipo no soportado: '{}'", oferta.getTipoPadre());

			throw new ValidacionException(this.messageSource.getMessage("error.ofertaNoSoportada", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeOfertaNoSoportada",
					null, LocaleContextHolder.getLocale()));
		}

		// Almacenamos la imagen en el objeto Oferta
		if (file != null) {
			// oferta.getImagen().setLength(file.getBytes().length);
			oferta.getImagen().setBytes(file.getBytes());
			oferta.getImagen().setTipo(file.getContentType());

			oferta.getImagen().setBytes(Utilidades.recortarImagen(oferta.getImagen()));
			oferta.getImagen().setLength(oferta.getImagen().getBytes().length);
			oferta.getImagen().setNombre(Utilidades.escapeFileName(oferta.getImagen().getNombre()));
		}

		return this.recursoService.guardarOfertaRecurso(oferta);
	}

	/**
	 * Obtiene los datos de una oferta.
	 * 
	 * @param codigoOferta Código de la oferta
	 * 
	 * @return Objeto {@link Oferta} con los datos de la oferta
	 */
	@RequestMapping(value = "/usuario/recursos/getOfertaRecurso", method = RequestMethod.POST)
	public @ResponseBody()
	Oferta getUsuario(@RequestBody() Long codigoOferta) {
		return this.recursoService.getOfertaRecurso(codigoOferta);
	}

	/**
	 * Elimina una oferta.
	 * 
	 * @param codigoOferta Código de la oferta
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usuario/recursos/eliminarOferta", method = RequestMethod.POST)
	public @ResponseBody()
	String eliminarOferta(@RequestBody() Long codigoOferta) throws XmlMappingException, DatatypeConfigurationException,
			IOException {
		return this.recursoService.eliminarOfertaRecurso(codigoOferta);
	}

	/**
	 * Obtiene la lista de los elementos multimedia tipo imagen de un recurso
	 * 
	 * @param codigo Código del recurso
	 * @return Lista de multimedias
	 */
	@RequestMapping(value = "/usuario/recursos/getListaMultimedias", method = RequestMethod.GET)
	public @ResponseBody()
	List<Multimedia> getListaMultimedia(@RequestParam(value = "codigo") String codigo) {
		List<Multimedia> resultado = new ArrayList<Multimedia>();

		try {
			long codigoRecurso = Long.parseLong(codigo);

			resultado = this.recursoService.findMultimediasByCorec(codigoRecurso);
		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoEncontrado",
					null, LocaleContextHolder.getLocale()));
		}

		return resultado;
	}

	/**
	 * Guarda el archivo multimedia introducido
	 * 
	 * @param multimedia {@link Multimedia} Objeto con los datos del archivo multimedia
	 * @param file fichero de la imagen
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usuario/recursos/guardarMultimedia", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarMultimedia(@RequestPart(value = "object") Multimedia multimedia,
			@RequestPart(value = "file") MultipartFile file) throws IOException, XmlMappingException,
			DatatypeConfigurationException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), multimedia.getCodRecurso())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'",
					multimedia.getCodRecurso());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		String tipo = multimedia.getTipoRecurso();
		// para los tipos N3 - Convention bureau (0009) si que está habilitado el multimedia ya que tiene ficha base con
		// galería
		if (Constants.EMPRESAS_RELACIONADAS.equals(tipo)) {
			// obtengo los datos del recurso para comprobar si es convention bureau
			Recurso recurso = this.recursoService.getDatosRecurso(multimedia.getCodRecurso());
			// en caso de ser convention bureau
			if (Constants.N3_SUBTIPO_CONVENTION_BUREAU.equals(recurso.getSubtipo())) {
				tipo = Constants.N3_CONVENTION_BUREAU;
			}
		}

		// Comprobamos si podemos introducir multimedias para ese recurso
		if (!Constants.EDITABLES_MAP.get(tipo).isMultimedia()) {
			logger.warn("Se ha intentado introducir un multimedia para un tipo no soportado: '{}'",
					multimedia.getTipoRecurso());

			throw new ValidacionException(this.messageSource.getMessage("error.multimediaNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeMultimediaNoEditable", null, LocaleContextHolder.getLocale()));
		}

		multimedia.getImagen().setBytes(file.getBytes());
		multimedia.getImagen().setTipo(file.getContentType());

		multimedia.getImagen().setBytes(Utilidades.recortarImagen(multimedia.getImagen()));
		multimedia.getImagen().setLength(multimedia.getImagen().getBytes().length);

		multimedia.getImagen().setNombre(Utilidades.escapeFileName(multimedia.getImagen().getNombre()));

		return this.recursoService.guardarMultimedia(multimedia);
	}

	/**
	 * Elimina un elemento multimedia.
	 * 
	 * @param multimedia datos del elemento multimedia
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usuario/recursos/eliminarMultimedia", method = RequestMethod.POST)
	public @ResponseBody()
	String eliminarMultimedia(@RequestBody() Multimedia multimedia) throws XmlMappingException,
			DatatypeConfigurationException, IOException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), multimedia.getCodRecurso())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'",
					multimedia.getCodRecurso());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		String tipo = multimedia.getTipoRecurso();
		// para los tipos N3 - Convention bureau (0009) si que está habilitado el multimedia ya que tiene ficha base con
		// galería
		if (Constants.EMPRESAS_RELACIONADAS.equals(tipo)) {
			// obtengo los datos del recurso para comprobar si es convention bureau
			Recurso recurso = this.recursoService.getDatosRecurso(multimedia.getCodRecurso());
			// en caso de ser convention bureau
			if (Constants.N3_SUBTIPO_CONVENTION_BUREAU.equals(recurso.getSubtipo())) {
				tipo = Constants.N3_CONVENTION_BUREAU;
			}
		}

		// Comprobamos si podemos introducir multimedias para ese recurso
		if (!Constants.EDITABLES_MAP.get(tipo).isMultimedia()) {
			logger.warn("Se ha intentado introducir un multimedia para un tipo no soportado: '{}'",
					multimedia.getTipoRecurso());

			throw new ValidacionException(this.messageSource.getMessage("error.multimediaNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeMultimediaNoEditable", null, LocaleContextHolder.getLocale()));
		}

		return this.recursoService.eliminarMultimedia(multimedia);
	}

	/**
	 * Método que busca un archivo en la tabla TMP_IMAGEN, alternativamente mediante código de multimedia (pantalla
	 * multimedia) o código de recurso (pantalla ofertas)
	 * 
	 * @param nombre nombre del archivo multimedia
	 * @param codRecurso código del recurso
	 * @return codigo del recurso
	 */
	@RequestMapping(value = "/usuario/recursos/existeTmpFile", method = RequestMethod.GET)
	public @ResponseBody()
	Long exiteTmpFile(@RequestParam(value = "nombre", required = false) String nombre,
			@RequestParam(value = "codRecurso", required = false) Long codRecurso) {

		return this.recursoService.exiteTmpFile(nombre, codRecurso);

	}

	/**
	 * Devuelve el nombre y array de bytes de una imagen
	 * 
	 * @param codMultimedia código del registro multimedia
	 * @param codRecurso código del recurso
	 * @param response objeto response
	 * @return datos de la imagen
	 */
	@RequestMapping(value = "/usuario/recursos/descargarImagen", method = RequestMethod.GET)
	public @ResponseBody()
	ModelAndView getFicheroDescargaById(@RequestParam(value = "codMultimedia", required = false) Long codMultimedia,
			@RequestParam(value = "codRecurso", required = false) Long codRecurso, HttpServletResponse response) {

		try {

			Imagen imagen = this.recursoService.getImagenById(codMultimedia, codRecurso);
			String fileExtension = Utilidades.getFileExtension(imagen.getNombre());
			response.setContentType("image/" + fileExtension);
			response.setContentLength(imagen.getBytes().length);
			// response.setHeader("Content-Disposition", "attachment; filename=\"" + imagen.getNombre() + "\"");
			response.setHeader("Content-Disposition", "inline; filename=\"" + imagen.getNombre() + "\"");
			FileCopyUtils.copy(imagen.getBytes(), response.getOutputStream());
			response.getOutputStream().flush();
			return null;
		} catch (Exception e) {
			logger.error("Error al descargar el documento", e);
			throw new ValidacionException(this.messageSource.getMessage("error.descargarImagen", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.descargarImagen", null,
					LocaleContextHolder.getLocale()));
		}
	}

	// ******** GESTION RECURSOS FIN **********

	/**
	 * Obtiene la lista de notificaciones
	 * 
	 * @param filtro Criterios de búsqueda
	 * 
	 * @return Lista de nofiticadiones
	 */
	@RequestMapping(value = "/gestion/gestionNotificaciones/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaNotificaciones(@ModelAttribute() FiltroNotificacion filtro) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.notificacionService.findNotificacionByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.notificacionService.findNotificacionByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Notificacion>());
		}

		return resultado;
	}

	/**
	 * Actualiza el estado a "V" (Vista) de una lista de notificaciones
	 * 
	 * @param codigos lista de codigos de notificaciones a actualizar
	 * @return resultado de la operación
	 * 
	 */
	@RequestMapping(value = "/gestion/gestionNotificacion/actualizarEstadoVista", method = RequestMethod.POST)
	public @ResponseBody()
	String actualizarEstadoNotificacionVista(@RequestBody() List<Long> codigos) {

		return this.notificacionService.actualizarEstadoNotificacionVista(codigos);
	}

	/**
	 * Obtiene la lista de sugerencias
	 * 
	 * @param filtro Criterios de búsqueda
	 * 
	 * @return Lista de sugerencias
	 */
	@RequestMapping(value = "/gestion/gestionSugerencias/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaSugerencias(@ModelAttribute() FiltroSugerencia filtro) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.sugerenciaService.findSugerenciaByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.sugerenciaService.findSugerenciaByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Sugerencia>());
		}

		return resultado;
	}

	/**
	 * Actualiza el estado a "V" (Vista) de una lista de sugerencias
	 * 
	 * @param codigos lista de codigos de sugerencias a actualizar
	 * @return resultado de la operación
	 * 
	 */
	@RequestMapping(value = "/gestion/gestionSugerencias/actualizarEstadoVista", method = RequestMethod.POST)
	public @ResponseBody()
	String actualizarEstadoSugerenciaVista(@RequestBody() List<Long> codigos) {

		return this.sugerenciaService.actualizarEstadoSugerenciaVista(codigos);
	}

	/**
	 * Obtiene la lista de los precios de un recurso
	 * 
	 * @param codigo Código del recurso
	 * @return Lista de precios
	 */
	@RequestMapping(value = "/usuario/recursos/getListaPrecios", method = RequestMethod.GET)
	public @ResponseBody()
	List<Precio> getListaPrecios(@RequestParam(value = "codigo") String codigo) {
		List<Precio> resultado = new ArrayList<Precio>();

		try {
			long codigoRecurso = Long.parseLong(codigo);

			resultado = this.recursoService.findPreciosByCorec(codigoRecurso);
		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codigo);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoEncontrado",
					null, LocaleContextHolder.getLocale()));
		}

		return resultado;
	}

	/**
	 * Guarda la lista de precios de un recurso
	 * 
	 * @param recurso datos del recurso con el listado de precos
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	@RequestMapping(value = "/usuario/recursos/guardarPrecios", method = RequestMethod.POST)
	public @ResponseBody()
	String eliminarMultimedia(@RequestBody() @Valid() Recurso recurso) throws XmlMappingException,
			DatatypeConfigurationException, IOException {

		// Comprobamos si el usuario tiene acceso a ese recurso
		if (!Utilidades.tieneAcceso(Utilidades.getDetallesUsuario(), recurso.getCodigo())) {
			logger.warn("Se ha intentado acceder a un recurso no vinculado al usuario: '{}'", recurso.getCodigo());

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoAccesible", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoAccesible",
					null, LocaleContextHolder.getLocale()));
		}

		// Comprobamos si podemos introducir precios para ese recurso
		if (!Utilidades.getEditable(recurso).isPrecios()) {
			logger.warn("Se ha intentado introducir un precio para un tipo no soportado: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.precioNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajePrecioNoEditable",
					null, LocaleContextHolder.getLocale()));
		}

		return this.recursoService.guardarPreciosRecurso(recurso);
	}

	/**
	 * Obtiene la lista de municipios
	 * 
	 * @param filtroMunicipio filtro aplicado
	 * @return Lista de municipios
	 */
	@RequestMapping(value = "/usuario/recursos/getListaMunicipios", method = RequestMethod.POST)
	public @ResponseBody()
	List<Municipio> getListaMunicipios(@RequestBody() Municipio filtroMunicipio) {
		return this.fcaService.findMunicipios(filtroMunicipio);
	}

	/**
	 * Obtiene la lista de localidades
	 * 
	 * @param filtroLocalidad filtro aplicado
	 * @return Lista de localidades
	 */
	@RequestMapping(value = "/usuario/recursos/getListaLocalidades", method = RequestMethod.POST)
	public @ResponseBody()
	List<Localidad> getListaLocalidades(@RequestBody() Localidad filtroLocalidad) {
		return this.fcaService.findLocalidades(filtroLocalidad);
	}

	/**
	 * Obtiene la lista de condiciones de una oferta
	 * 
	 * @param codRecurso Código del recurso
	 * @return Lista de condiciones
	 */
	@RequestMapping(value = "/usuario/recursos/getListaCondiciones", method = RequestMethod.GET)
	public @ResponseBody()
	List<Condicion> getListaCondiciones(@RequestParam(value = "codRecurso") String codRecurso) {
		List<Condicion> resultado = new ArrayList<Condicion>();

		try {
			long codigoRecurso = Long.parseLong(codRecurso);
			resultado = this.recursoService.findCondicionesOfertaByCorec(codigoRecurso);

		} catch (NumberFormatException nfe) {
			logger.warn("El codigo de recurso seleccionado: '{}' no es válido", codRecurso);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoEncontrado",
					null, LocaleContextHolder.getLocale()));
		}

		return resultado;
	}

	/**
	 * Obtiene la lista de municipios de una provincia
	 * 
	 * @param filtroMunicipio filtro aplicado
	 * @return Lista de municipios
	 */
	@RequestMapping(value = "/usuario/recursos/getListaMunicipiosByProvincia", method = RequestMethod.POST)
	public @ResponseBody()
	List<Municipio> getListaMunicipiosByProvincia(@RequestBody() Municipio filtroMunicipio) {
		return this.fcaService.findMunicipiosByProvincia(filtroMunicipio);
	}

	/**
	 * Obtiene la lista de localidades de un municipio
	 * 
	 * @param filtroLocalidad filtro aplicado
	 * @return Lista de localidades
	 */
	@RequestMapping(value = "/usuario/recursos/getListaLocalidadesByMunicipio", method = RequestMethod.POST)
	public @ResponseBody()
	List<Localidad> getListaLocalidadesByMunicipio(@RequestBody() Localidad filtroLocalidad) {
		return this.fcaService.findLocalidadesByMunicipio(filtroLocalidad);
	}

}
