package com.ejie.aa80a.configuration;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.web.DefaultRedirectStrategy;
import org.springframework.security.web.RedirectStrategy;
import org.springframework.security.web.authentication.SimpleUrlAuthenticationSuccessHandler;
import org.springframework.stereotype.Component;

import com.ejie.aa80a.util.Utilidades;

/**
 * Clase que se encarga de redirigir desde la página de login a la página inicial dependiendo del perfil del usuario que
 * ha entrado
 * 
 *  
 */
@Component()
public class CustomSuccessHandler extends SimpleUrlAuthenticationSuccessHandler {

	private static final Logger logger = LoggerFactory.getLogger(CustomSuccessHandler.class);

	/** The redirect strategy. */
	private RedirectStrategy redirectStrategy = new DefaultRedirectStrategy();

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.springframework.security.web.authentication.AbstractAuthenticationTargetUrlRequestHandler#handle(javax.servlet
	 * .http.HttpServletRequest, javax.servlet.http.HttpServletResponse,
	 * org.springframework.security.core.Authentication)
	 */
	@Override()
	protected void handle(HttpServletRequest request, HttpServletResponse response, Authentication authentication)
			throws IOException {
		String targetUrl = determineTargetUrl(authentication);

		if (response.isCommitted()) {
			logger.error("No se puede realizar la redirección");
			return;
		}

		this.redirectStrategy.sendRedirect(request, response, targetUrl);
	}

	/**
	 * Determina la url de destino dependiendo del perfil del usuario logado
	 * 
	 * @param authentication Objeto con la información de autenticación
	 * 
	 * @return url de destino
	 */
	protected String determineTargetUrl(Authentication authentication) {
		String url = "";

		Collection<? extends GrantedAuthority> authorities = authentication.getAuthorities();

		List<String> roles = new ArrayList<String>();

		for (GrantedAuthority a : authorities) {
			roles.add(a.getAuthority());
		}

		if (Utilidades.esAdministrador(roles)) {
			url = "/inicioCoordinador";
		} else if (Utilidades.esOficina(roles)) {
			url = "/inicioOficina";
		} else if (Utilidades.esUsuario(roles)) {
			url = "/inicioUsuario";
		} else {
			url = "/accessDenied";
		}

		return url;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.springframework.security.web.authentication.AbstractAuthenticationTargetUrlRequestHandler#setRedirectStrategy
	 * (org.springframework.security.web.RedirectStrategy)
	 */
	@Override()
	public void setRedirectStrategy(RedirectStrategy redirectStrategy) {
		this.redirectStrategy = redirectStrategy;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.springframework.security.web.authentication.AbstractAuthenticationTargetUrlRequestHandler#getRedirectStrategy
	 * ()
	 */
	@Override()
	protected RedirectStrategy getRedirectStrategy() {
		return this.redirectStrategy;
	}

}