package com.ejie.aa80a.util;

import java.awt.AlphaComposite;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.GregorianCalendar;
import java.util.List;

import javax.imageio.ImageIO;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;

import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;

import com.ejie.aa80a.model.Editables;
import com.ejie.aa80a.model.Imagen;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.UsuarioDetalles;

/**
 * Clase con métodos de utilidad común.
 * 
 *  
 * 
 */
public class Utilidades {

	private static final int SIZE_PASSWORD = 10;

	private final static int MAX_LENGTH_NOMBRE_CONTENIDO = 10;

	/**
	 * Constructor privado. Clase de Utilidades.
	 */
	private Utilidades() {
	}

	/**
	 * Obtiene la consulta incluyendo la paginación.
	 * 
	 * @param query La consulta a la que añadir la paginación.
	 * @param numRegistro El número de registro del primer registro a devolver.
	 * @param numRegistrosPagina El número de registros por página.
	 * @param columnaOrden Columna por la que se ordena, opcional.
	 * @param orden El tipo de orden si se ha informado columna por la que ordenar, asc: ascendiente, desc:
	 *            descendiente.
	 * @return StringBuilder Devuelve la consulta que incluye la paginación.
	 */
	public static StringBuilder getPaginationQuery(StringBuilder query, Integer numRegistro,
			Integer numRegistrosPagina, String columnaOrden, String orden) {
		StringBuilder paginationQuery = new StringBuilder();
		if (columnaOrden != null) {
			paginationQuery.append(" ORDER BY ");
			paginationQuery.append(columnaOrden);
			paginationQuery.append(" ");
			paginationQuery.append(orden);
			query.append(paginationQuery);
		}
		paginationQuery = new StringBuilder();

		if (numRegistrosPagina != null && numRegistro != null) {
			paginationQuery
					.append("SELECT b.* FROM (SELECT rownum rnum, a.*  FROM (" + query + ")a) b where rnum between ")
					.append((numRegistro + 1)).append(" and ").append((numRegistro + numRegistrosPagina));
		}/*
		 * else if (numRegistro != null) { TODO Revisar si este caso va a hacer falta
		 * paginationQuery.append("SELECT b.* FROM (SELECT rownum rnum, a.*  FROM (" + query).append(
		 * ")a) b where rnum > 0 and rnum < " + (numRegistro + 1)); }
		 */else {
			return query;
		}
		return paginationQuery;
	}

	/**
	 * Genera un password aleatorio alfanumérico y de SIZE_PASSWORD caracteres.
	 * 
	 * @return password aleatorio
	 */
	public static String getRandomPassword() {
		return RandomStringUtils.randomAlphanumeric(SIZE_PASSWORD);
	}

	/**
	 * Obtiene el content type en función del tipo de recurso.
	 * 
	 * @param tipoRecurso Tipo de recurso
	 * 
	 * @return content type
	 */
	public static String getContentType(String tipoRecurso) {
		if (tipoRecurso == null) {
			return null;
		}

		if (tipoRecurso.length() > 0) {
			String contentTypeKey = tipoRecurso.substring(0, 1).toUpperCase();
			return Constants.CONTENT_TYPE_MAP.get(contentTypeKey);
		} else {
			return null;
		}
	}

	/**
	 * Obtiene el nombre del contenido en el Portal a partir del código de recurso y del tipo.
	 * 
	 * @param codigoRecurso Código de recurso
	 * @param tipoRecurso Tipo de recurso
	 * 
	 * @return Nombre del contenido en el Gestor de Contenidos
	 */
	public static String getNombreContenido(long codigoRecurso, String tipoRecurso) {

		StringBuilder nombreContenido = new StringBuilder(Long.toString(codigoRecurso));
		while (nombreContenido.length() < MAX_LENGTH_NOMBRE_CONTENIDO) {
			nombreContenido.insert(0, "0");
		}
		nombreContenido.append("_");
		nombreContenido.append(tipoRecurso.toLowerCase());
		nombreContenido.append("_rec_turismo");
		return nombreContenido.toString();
	}

	/**
	 * Obtiene la URL del contenido.
	 * 
	 * @param codigoRecurso Código de recurso
	 * @param tipoRecurso Tipo de recurso
	 * @param idioma Idioma [es | eu | en | fr | de ]
	 * 
	 * @return URL relativa (sin dominio) del contenido
	 */
	public static String getURLContenido(long codigoRecurso, String tipoRecurso, String idioma) {

		StringBuilder pathContenido = new StringBuilder();
		String typologyRecurso = getContentType(tipoRecurso);

		pathContenido.append("/contenidos/");
		pathContenido.append(typologyRecurso);
		pathContenido.append("/");
		pathContenido.append(getNombreContenido(codigoRecurso, tipoRecurso));
		pathContenido.append("/");
		pathContenido.append(idioma);
		pathContenido.append("_");
		pathContenido.append(codigoRecurso);
		pathContenido.append("/");

		return pathContenido.toString();
	}

	/**
	 * Obtiene la URL relativa al dominio de la ficha de un recurso.
	 * 
	 * @param codigoRecurso Código de recurso
	 * @param tipoRecurso Tipo de recurso
	 * @param idioma Idioma [es | eu | en | fr | de ]
	 * 
	 * @return URL relativa (sin dominio) de la ficha
	 */
	public static String getURLFichaPortal(long codigoRecurso, String tipoRecurso, String idioma) {

		StringBuilder pathFicha = new StringBuilder();

		// Preguntamos el tipo de recurso, para mostrar la ficha en su página correspondiente
		if (Constants.EXPERIENCIAS.equals(tipoRecurso)) {
			pathFicha.append("/" + Constants.PAGINA_FICHA_EXPERIENCIAS + "/");
		} else {
			pathFicha.append("/" + Constants.PAGINA_FICHA + "/");
		}
		pathFicha.append(idioma);
		pathFicha.append(getURLContenido(codigoRecurso, tipoRecurso, idioma));

		if (Constants.AGENDA.equals(tipoRecurso) || Constants.OFERTAS.equals(tipoRecurso)) {
			pathFicha.append(codigoRecurso).append(".html");
		} else {
			pathFicha.append(codigoRecurso).append(Constants.HTML_FICHA);
		}

		return pathFicha.toString();
	}

	/**
	 * Obtiene la url amigable de la ficha del portal del recurso. En caso de no existir, obtiene la url relativa
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * @param host raiz de la ruta del portal de turismo
	 * 
	 * @return url de la ficha
	 */
	public static String getURLAmigableFichaPortal(Recurso recurso, String host) {
		String urlPortal = "";

		if (recurso.getUrlPortal() != null) { // Existe URL amigable
			urlPortal = host
					+ "/"
					+ recurso.getUrlPortal()
					+ (Constants.EXPERIENCIAS.equals(recurso.getTipo()) ? Constants.PAGINA_FICHA_EXPERIENCIAS
							: Constants.PAGINA_FICHA) + "/" + LocaleContextHolder.getLocale().getLanguage() + "/";
		} else { // Se genera una URL no amigable en función del código de recurso y del tipo
			urlPortal = host
					+ getURLFichaPortal(recurso.getCodigo(), recurso.getTipo(), LocaleContextHolder.getLocale()
							.getLanguage());
		}

		return urlPortal;
	}

	/**
	 * Comprueba si el usuario tiene el rol de USUARIO
	 * 
	 * @param roles lista de roles del usuario logado
	 * 
	 * @return true, si tiene el rol de USUARIO
	 */
	public static boolean esUsuario(List<String> roles) {
		if (roles.contains("ROLE_USUARIO")) {
			return true;
		}
		return false;
	}

	/**
	 * Comprueba si el usuario tiene el rol de OFICINA.
	 * 
	 * @param roles lista de roles del usuario logado
	 * @return true, si tiene el rol de OFICINA
	 */
	public static boolean esOficina(List<String> roles) {
		if (roles.contains("ROLE_OFICINA")) {
			return true;
		}
		return false;
	}

	/**
	 * Comprueba si el usuario tiene el rol de ADMINISTRADOR
	 * 
	 * @param roles lista de roles del usuario logado
	 * 
	 * @return true, si tiene el rol de ADMINISTRADOR
	 */
	public static boolean esAdministrador(List<String> roles) {
		if (roles.contains("ROLE_ADMINISTRADOR")) {
			return true;
		}
		return false;
	}

	/**
	 * Comprueba si el usuario tiene acceso a un recurso.
	 * 
	 * Si el usuario tiene los roles de ADMINISTRADOR u OFICINA tiene acceso Si el usuario tiene el rol de USUARIO solo
	 * puede acceder a su recurso asociado
	 * 
	 * @param usuario Datos del usuario logado
	 * @param codigoRecurso Código de recurso
	 * 
	 * @return true, si tiene acceso
	 */
	public static boolean tieneAcceso(UsuarioDetalles usuario, long codigoRecurso) {
		Collection<? extends GrantedAuthority> authorities = usuario.getAuthorities();

		List<String> roles = new ArrayList<String>();

		for (GrantedAuthority a : authorities) {
			roles.add(a.getAuthority());
		}

		if (Utilidades.esUsuario(roles) && usuario.getCodigoRecurso() != codigoRecurso) {
			return false;
		}

		return true;
	}

	/**
	 * Obtiene los datos del usuario autenticado del contexto.
	 * 
	 * @return datos del usuario autenticado
	 */
	public static UsuarioDetalles getDetallesUsuario() {
		UsuarioDetalles usuarioDetalles = null;
		Object principal = SecurityContextHolder.getContext().getAuthentication().getPrincipal();

		if (principal instanceof UsuarioDetalles) {
			usuarioDetalles = (UsuarioDetalles) principal;
		}

		return usuarioDetalles;
	}

	/**
	 * Obtiene la fecha y hora actual en formato XMLGregorianCalendar
	 * 
	 * @return Fecha y hora actual en formato XMLGregorianCalendar
	 * 
	 * @throws DatatypeConfigurationException Excepción
	 */
	public static XMLGregorianCalendar getXMLGregorianCalendarNow() throws DatatypeConfigurationException {
		GregorianCalendar gregorianCalendar = new GregorianCalendar();
		return DatatypeFactory.newInstance().newXMLGregorianCalendar(gregorianCalendar);
	}

	/**
	 * Recorta una imagen y devuelve su array de bytes
	 * 
	 * @param file array de bytes de la imagen
	 * @param cropX coordenada x del recorte
	 * @param cropY coordenada y del recorte
	 * @param cropWidth ancho del recorte
	 * @param cropHeight alto del recorte
	 * @param imageWidth ancho de la imagen en el momento del recorte
	 * @param imageHeight alto de la imagen en el momento del recorte
	 * @return array de bytes de la imagen
	 * @throws IOException excepción
	 */
	// public static byte[] recortarImagen(byte[] file, int cropX, int cropY, int cropWidth, int cropHeight,
	// int imageWidth, int imageHeight) throws IOException {
	//
	// InputStream in = new ByteArrayInputStream(file);
	// BufferedImage originalImage = ImageIO.read(in);
	//
	// int type = originalImage.getType() == 0 ? BufferedImage.TYPE_INT_ARGB : originalImage.getType();
	//
	// BufferedImage resizeImage = resizeImageWithHint(originalImage, type, imageWidth, imageHeight);
	// BufferedImage cropImage = resizeImage.getSubimage(cropX, cropY, cropWidth, cropHeight);
	//
	// ByteArrayOutputStream baos = new ByteArrayOutputStream();
	// ImageIO.write(cropImage, "jpg", baos);
	// baos.flush();
	// byte[] imageInByte = baos.toByteArray();
	// baos.close();
	//
	// return imageInByte;
	// }

	/**
	 * Recorta una imagen y devuelve su array de bytes
	 * 
	 * @param imagen objeto con los datos de la imagen
	 * @return array de bytes de la imagen
	 * @throws IOException excepción
	 */
	public static byte[] recortarImagen(Imagen imagen) throws IOException {

		InputStream in = new ByteArrayInputStream(imagen.getBytes());
		BufferedImage originalImage = ImageIO.read(in);

		int type = originalImage.getType() == 0 ? BufferedImage.TYPE_INT_ARGB : originalImage.getType();

		BufferedImage resizeImage = resizeImageWithHint(originalImage, type, imagen.getImageWidth(),
				imagen.getImageHeight());
		BufferedImage cropImage = resizeImage.getSubimage(imagen.getCropX(), imagen.getCropY(), imagen.getCropWidth(),
				imagen.getCropHeight());

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		ImageIO.write(cropImage, getFileExtension(imagen.getNombre()), baos);
		baos.flush();
		byte[] imageInByte = baos.toByteArray();
		baos.close();

		return imageInByte;
	}

	/**
	 * Redimensiona una imagen
	 * 
	 * @param originalImage imagen original
	 * @param type tipo de imagen
	 * @param width ancho a aplicar en la imagen
	 * @param height alto a aplicar en la imagen
	 * @return imagen redimensionada
	 */
	private static BufferedImage resizeImageWithHint(BufferedImage originalImage, int type, int width, int height) {
		BufferedImage resizedImage = new BufferedImage(width, height, type);
		Graphics2D g = resizedImage.createGraphics();
		g.drawImage(originalImage, 0, 0, width, height, null);
		g.dispose();
		g.setComposite(AlphaComposite.Src);

		g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BILINEAR);
		g.setRenderingHint(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);
		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);

		return resizedImage;
	}

	/**
	 * devuelve la extensión a partir del nombre de un fichero
	 * 
	 * @param fileName nombre del fichero
	 * @return extensión del fichero
	 */
	public static String getFileExtension(String fileName) {
		String extension = "";

		int i = fileName.lastIndexOf('.');
		if (i > 0) {
			extension = fileName.substring(i + 1);
		}
		return extension;

	}

	/**
	 * Elimina del nombre de un fichero todos aquellos caracteres que no sean letras o numeros
	 * 
	 * @param fileName nombre del fichero
	 * @return nombre del fichero escapado
	 */
	public static String escapeFileName(String fileName) {
		String extension = getFileExtension(fileName);
		int i = fileName.lastIndexOf('.');
		String name = fileName.substring(0, i);
		name = name.replaceAll("[^A-Za-z0-9_]", "");

		return name + "." + extension.toLowerCase();

	}

	/**
	 * Devuelve el objeto Editables con los permisos de edición de un recurso
	 * 
	 * @param recurso datos del recurso
	 * @return permisos de edición
	 */
	public static Editables getEditable(Recurso recurso) {
		if (Constants.EMPRESAS_RELACIONADAS.equals(recurso.getTipo())
				&& (Constants.N3_AGENCIA_VIAJES_MAYORISTA.equals(recurso.getSubtipo())
						|| Constants.N3_AGENCIA_VIAJES_MINORISTA.equals(recurso.getSubtipo()) || Constants.N3_AGENCIA_VIAJES_MINORISTA_MAYORISTA
							.equals(recurso.getSubtipo()))) {
			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			return Constants.EDITABLES_MAP.get(Constants.N3_AGENCIAS_VIAJES);

		} else if (Constants.EMPRESAS_RELACIONADAS.equals(recurso.getTipo())
				&& Constants.N3_SUBTIPO_CONVENTION_BUREAU.equals(recurso.getSubtipo())) {

			return Constants.EDITABLES_MAP.get(Constants.N3_CONVENTION_BUREAU);

		} else {
			// Mostrar o no campos editables, opciones de menú y botonera. Depende del tipo / subtipo de recurso
			return Constants.EDITABLES_MAP.get(recurso.getTipo());
		}
	}
}