package com.ejie.aa80a.service;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.dao.UsuarioDao;
import com.ejie.aa80a.model.FiltroUsuario;
import com.ejie.aa80a.model.TipoPerfilUsuario;
import com.ejie.aa80a.model.Usuario;
import com.ejie.aa80a.model.UsuarioDetalles;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;
import com.ejie.aa80a.util.exception.NegocioException;

/**
 * Clase que implementa los servicios de acceso a los datos del usuario
 * 
 *  
 */
@Service(value = "usuarioService")
@Transactional()
public class UsuarioServiceImpl implements UsuarioService {

	private static final Logger logger = LoggerFactory.getLogger(UsuarioServiceImpl.class);

	@Autowired()
	private UsuarioDao dao;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private MailService mailService;

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#findByUsuario(java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Usuario findByUsuario(String login) {
		return this.dao.findByUsuario(login);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#recordarPassword(java.lang.String)
	 */
	@Override()
	public String recordarPassword(String email) {
		// Comprobamos que el correo está en la base de datos
		if (this.dao.existeEmail(email, null)) {
			List<Usuario> usuarios = this.dao.findByEmail(email);

			logger.debug("Se ha introducido el email asociado a los usuarios: {}", usuarios);

			StringBuilder body = new StringBuilder();
			BCryptPasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

			// Si hay más de un usuario con el mismo email se reseteará la contraseña de todos ellos y
			// se enviará un único email informativo con los nuevos datos de acceso
			for (Usuario usuario : usuarios) {
				// Se genera un nuevo password
				String password = Utilidades.getRandomPassword();

				// TODO: Eliminar al subir a Producción
				// logger.debug("Nueva password para el usuario: {} / {}", usuario.getLogin(), password);
				// //////////////////////////////////////

				// Se actualiza la password en base de datos
				this.dao.changePassword(usuario.getLogin(), passwordEncoder.encode(password));

				body.append(
						this.messageSource.getMessage("login.email.body",
								new Object[] { password, usuario.getLogin() }, LocaleContextHolder.getLocale()))
						.append("\n\n");
			}

			// Se envía el mail
			String asunto = this.messageSource.getMessage("login.email.asunto", null, LocaleContextHolder.getLocale());

			this.mailService.sendMail(email, asunto, body.toString());

		} else {
			logger.warn("Se ha introducido un email que no está asociado a ningún usuario: {}", email);

			throw new NegocioException("error.emailNoExiste");
		}

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#guardarUsuario(com.ejie.aa80a.model.Usuario)
	 */
	@Override()
	public String guardarUsuario(Usuario usuario) {

		// El campo de login debe ser único
		// Se realiza la comprobación.
		if (this.dao.existeUsuario(usuario.getLogin(), usuario.getIdUsuario())) {
			logger.warn("Se ha introducido un login que ya está asociado a algún usuario: {}", usuario.getLogin());

			throw new NegocioException("error.loginDuplicado");
		}

		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.dao.actualizarUsuario(usuario, datosLogin.getUsername());

		// si se ha modificado el campo usuario
		if (!usuario.getLogin().equals(usuario.getLoginAnterior())) {
			// devuelvo un 1 para indicar que vuelva a realizar el login
			return "1";
		}

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#crearUsuario(com.ejie.aa80a.model.Usuario)
	 */
	@Override()
	public String crearUsuario(Usuario usuario) {

		// El campo de login debe ser único
		// Se realiza la comprobación.
		if (this.dao.existeUsuario(usuario.getLogin(), null)) {
			logger.warn("Se ha introducido un login que ya está asociado a algún usuario: {}", usuario.getLogin());

			throw new NegocioException("error.loginDuplicado");
		}

		// Mensaje de correo personalizado según el tipo de recurso asociado
		String mensajeKey = "usuario.alta.email.bodyAlojamiento";

		// TODO: Revisar
		// Asignamos el Perfil dependiendo del tipo de recurso asociado
		// A1: USUARIO
		// E1: OFICINA
		// Sin recurso seleccionado: COORDINADOR
		if (usuario.getRecurso() != null && usuario.getRecurso().getCodigo() != null) {
			if (Constants.ALOJAMIENTO.equals(usuario.getRecurso().getTipo())) {
				usuario.setPerfil(TipoPerfilUsuario.valueOf("USUARIO"));

				mensajeKey = "usuario.alta.email.bodyAlojamiento";
			} else if (Constants.OFICINA_TURISMO.equals(usuario.getRecurso().getTipo())) {
				usuario.setPerfil(TipoPerfilUsuario.valueOf("OFICINA"));

				mensajeKey = "usuario.alta.email.bodyOtros";
			}
		} else {
			usuario.setPerfil(TipoPerfilUsuario.valueOf("ADMINISTRADOR"));

			mensajeKey = "usuario.alta.email.bodyOtros";
		}

		// Se genera un nuevo password
		String password = Utilidades.getRandomPassword();

		// TODO: Eliminar al subir a Producción
		// logger.debug("Password para el nuevo usuario: {} / {}", usuario.getLogin(), password);
		// //////////////////////////////////////

		// Se asigna la password encriptada para almacenar en base de datos
		BCryptPasswordEncoder passwordEncoder = new BCryptPasswordEncoder();
		usuario.setPassword(passwordEncoder.encode(password));

		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.dao.insertarUsuario(usuario, datosLogin.getUsername());

		// Se envía un mail al nuevo usuario con el usuario y la password
		String asunto = this.messageSource.getMessage("usuario.alta.email.subject", null,
				LocaleContextHolder.getLocale());
		String body = this.messageSource.getMessage(mensajeKey, new Object[] { usuario.getLogin(), password },
				LocaleContextHolder.getLocale());

		this.mailService.sendMail(usuario.getEmail(), asunto, body);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#cambiarPassword(java.lang.String, java.lang.String)
	 */
	@Override()
	public String cambiarPassword(String password, String nuevaPassword) {
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		BCryptPasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

		// Comprobamos que la password introducida se corresponde con la almacenada en BD
		if (!passwordEncoder.matches(password, this.dao.obtenerPassword(datosLogin.getUsername()))) {
			logger.warn("Se ha introducido una password incorrecta");

			throw new NegocioException("error.passwordIncorrecta");
		}

		// Se actualiza la password en base de datos

		this.dao.changePassword(datosLogin.getUsername(), passwordEncoder.encode(nuevaPassword));

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#findByCriteria(com.ejie.aa80a.model.FiltroUsuario)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Usuario> findByCriteria(FiltroUsuario filtro) {
		return this.dao.findByCriteria(filtro);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#findByCountCriteria(com.ejie.aa80a.model.FiltroUsuario)
	 */
	@Transactional(readOnly = true)
	@Override()
	public int findByCountCriteria(FiltroUsuario filtro) {
		return this.dao.findByCountCriteria(filtro);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.UsuarioService#eliminarUsuario(java.lang.Integer)
	 */
	@Override()
	public String eliminarUsuario(Integer idUsuario) {
		this.dao.eliminarUsuario(idUsuario);

		return "OK";
	}

}
