package com.ejie.aa80a.service;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.transform.stream.StreamResult;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.oxm.Marshaller;
import org.springframework.oxm.XmlMappingException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.dao.HistoricoDao;
import com.ejie.aa80a.dao.RecursoDao;
import com.ejie.aa80a.dao.SugerenciaDao;
import com.ejie.aa80a.jms.InvoiceQueueSender;
import com.ejie.aa80a.model.Combo;
import com.ejie.aa80a.model.Condicion;
import com.ejie.aa80a.model.FiltroRecurso;
import com.ejie.aa80a.model.Imagen;
import com.ejie.aa80a.model.Localizacion;
import com.ejie.aa80a.model.Multimedia;
import com.ejie.aa80a.model.Oferta;
import com.ejie.aa80a.model.Precio;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Servicio;
import com.ejie.aa80a.model.Usuario;
import com.ejie.aa80a.model.UsuarioDetalles;
import com.ejie.aa80a.schema.gaurkotu.ActualizacionGaurkotu;
import com.ejie.aa80a.schema.notification.NotificationBody;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;
import com.ejie.aa80a.util.exception.ValidacionException;

/**
 * Clase que implementa los servicios de acceso a los datos del recurso.
 * 
 *  
 */
@Service(value = "recursoService")
@Transactional()
public class RecursoServiceImpl implements RecursoService {

	private static final Logger logger = LoggerFactory.getLogger(RecursoServiceImpl.class);

	private static final int EDITABLE = 1;
	private static final int NO_EDITABLE = 0;

	@Autowired()
	private RecursoDao recursoDao;

	@Autowired()
	private HistoricoDao historicoDao;

	@Autowired()
	private SugerenciaDao sugerenciaDao;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private Properties appConfiguration;

	@Autowired()
	private InvoiceQueueSender invoiceQueueSender;

	@Autowired()
	@Qualifier(value = "jaxbMarshallerActualizacionGaurkotuBean")
	private Marshaller marshallerActualizacionGaurkotu;

	@Autowired()
	@Qualifier(value = "jaxbMarshallerNotificationBean")
	private Marshaller marshallerNotification;

	@Autowired()
	private MailService mailService;

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findByUsuario(com.ejie.aa80a.model.Usuario)
	 */
	@Override()
	public Map<String, String> findByUsuario(Usuario usuario) {
		HashMap<String, String> nombreRecurso = new HashMap<String, String>();

		if (usuario.getRecurso() != null) {
			nombreRecurso.put("es", this.recursoDao.findNombreByCorecTipoIdioma(usuario.getRecurso().getCodigo(),
					usuario.getRecurso().getTipo(), "es"));
			nombreRecurso.put("eu", this.recursoDao.findNombreByCorecTipoIdioma(usuario.getRecurso().getCodigo(),
					usuario.getRecurso().getTipo(), "eu"));
		}

		return nombreRecurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findByCriteria(com.ejie.aa80a.model.FiltroRecurso, boolean)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Recurso> findByCriteria(FiltroRecurso filtro, boolean editables) {
		return this.recursoDao.findByCriteria(filtro, editables);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findByCountCriteria(com.ejie.aa80a.model.FiltroRecurso, boolean)
	 */
	@Transactional(readOnly = true)
	@Override()
	public int findByCountCriteria(FiltroRecurso filtro, boolean editables) {
		return this.recursoDao.findByCountCriteria(filtro, editables);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getDatosRecurso(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Recurso getDatosRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoByCorecIdioma(codigoRecurso, LocaleContextHolder.getLocale()
				.getLanguage());

		// Se rellenan los campos idiomáticos de nombreTipo, nombreSubtipo y nombreCategoría
		recurso.setNombreTipo(this.messageSource.getMessage("tipoRecurso." + recurso.getTipo(), null,
				LocaleContextHolder.getLocale()));

		if (recurso.getSubtipo() != null) {
			recurso.setNombreSubtipo(this.messageSource.getMessage("subtipoRecurso." + recurso.getTipo() + "."
					+ recurso.getSubtipo(), null, LocaleContextHolder.getLocale()));
		}

		if (recurso.getCategoria() != null) {

			/*
			 * Interpretar la categoría S como sin categoría para los recursos de subtipo camping con modalidad distinta
			 * de C (Camping).
			 * 
			 * si se trata de un alojamiento tipo Camping pero la modalidad es autocaravanas o área natural de acampada
			 */
			if (Constants.ALOJAMIENTO.equalsIgnoreCase(recurso.getTipo())
					&& Constants.ALOJAMIENTO_SUBTIPO_CAMPING.equalsIgnoreCase(recurso.getSubtipo())
					&& !Constants.MODALIDAD_CAMPING_CAMPING.equalsIgnoreCase(recurso.getModalidad())) {

				recurso.setNombreCategoria(this.messageSource.getMessage("categoria.B.S", null,
						LocaleContextHolder.getLocale())); // Sin categoría

			} else {
				recurso.setNombreCategoria(this.messageSource.getMessage("categoria." + recurso.getSubtipo() + "."
						+ recurso.getCategoria(), null, LocaleContextHolder.getLocale()));
			}
		}

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		return recurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarDatosRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public String guardarDatosRecurso(Recurso recurso) throws DatatypeConfigurationException, XmlMappingException,
			IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(recurso.getCodigo());

		// Guardamos los datos del recurso en la tabla correspondiente
		this.recursoDao.actualizarDatosRecurso(recurso);

		// Actualizamos la tabla RECURSO con la fecha y el usuario (GAURKOTU) que ha actualizado datos
		this.recursoDao.actualizarRecurso(recurso);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(recurso.getCodigo(), Constants.ACTUALIZACION_DATOS,
				datosLogin.getUsername());

		String mensaje = "";
		// si se ha subido una nueva imagen principal (para los casos de ficha reducida)
		if (recurso.getImagen() != null && recurso.getImagen().getBytes() != null) {
			// Enviar a AA80T un mensaje para subir la foto principal (tambien regenera y republica)
			mensaje = crearMensajeActualizacion(Constants.ACCION_SUBIR_MULTIMEDIA, recurso.getCodigo());
		} else {
			// Enviar a AA80T un mensaje de regeneración y republicación del recurso
			mensaje = crearMensajeActualizacion(Constants.ACCION_REGENERAR_PUBLICAR, recurso.getCodigo());
		}

		this.invoiceQueueSender.sendMesage(mensaje);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getServiciosRecurso(java.lang.Long)
	 */
	@Override()
	public Recurso getServiciosRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos básicos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoBasicosByCorecIdioma(codigoRecurso, LocaleContextHolder
				.getLocale().getLanguage());

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		// Comprobamos si los servicios del recurso son editables
		if (!Utilidades.getEditable(recurso).isServicios()) {
			logger.warn("Se ha intentado editar servicios no editables: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.serviciosNoEditables", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeServiciosNoEditables", null, LocaleContextHolder.getLocale()));
		}

		// Obtenemos las características del recurso
		List<Servicio> caracteristicas = this.recursoDao.findServiciosByRecursoIdiomaEditable(recurso,
				LocaleContextHolder.getLocale().getLanguage(), NO_EDITABLE);

		// Obtenemos los servicios del recurso
		List<Servicio> servicios = this.recursoDao.findServiciosByRecursoIdiomaEditable(recurso, LocaleContextHolder
				.getLocale().getLanguage(), EDITABLE);

		recurso.setCaracteristicas(caracteristicas);
		recurso.setServicios(servicios);

		return recurso;
	}

	// /**
	// * Obtiene la url a la ficha del portal del recurso.
	// *
	// * @param recurso {@link Recurso} objeto con los datos del recurso
	// *
	// * @return url
	// */
	// private String obtenerURLFichaPortal(Recurso recurso) {
	// String urlPortal = "";
	//
	// if (recurso.getUrlPortal() != null) { // Existe URL amigable
	// urlPortal = this.appConfiguration.getProperty("portal.host")
	// + "/"
	// + recurso.getUrlPortal()
	// + (Constants.EXPERIENCIAS.equals(recurso.getTipo()) ? Constants.PAGINA_FICHA_EXPERIENCIAS
	// : Constants.PAGINA_FICHA) + "/" + LocaleContextHolder.getLocale().getLanguage() + "/";
	// } else { // Se genera una URL no amigable en función del código de recurso y del tipo
	// urlPortal = this.appConfiguration.getProperty("portal.host")
	// + Utilidades.getURLFichaPortal(recurso.getCodigo(), recurso.getTipo(), LocaleContextHolder
	// .getLocale().getLanguage());
	// }
	//
	// return urlPortal;
	// }

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarServiciosRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public String guardarServiciosRecurso(Recurso recurso) throws XmlMappingException, DatatypeConfigurationException,
			IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(recurso.getCodigo());

		// Guardamos los servicios del recurso
		this.recursoDao.actualizarServiciosRecurso(recurso);

		// Actualizamos la tabla RECURSO con la fecha y el usuario (GAURKOTU) que ha actualizado datos
		this.recursoDao.actualizarRecurso(recurso);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(recurso.getCodigo(), Constants.ACTUALIZACION_SERVICIOS,
				datosLogin.getUsername());

		// Enviar a AA80T un mensaje de regeneración y republicación del recurso
		String mensaje = crearMensajeActualizacion(Constants.ACCION_REGENERAR_PUBLICAR, recurso.getCodigo());
		this.invoiceQueueSender.sendMesage(mensaje);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getSugerenciasRecurso(java.lang.Long)
	 */
	@Override()
	public Recurso getSugerenciasRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos básicos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoBasicosByCorecIdioma(codigoRecurso, LocaleContextHolder
				.getLocale().getLanguage());

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		return recurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarSugerenciasRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public String guardarSugerenciasRecurso(Recurso recurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(recurso.getCodigo());

		// Guardamos la sugerencia en BBDD
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.sugerenciaDao.insertarSugerenciaRecurso(recurso, datosLogin.getUsername());

		// Insertamos un registro en el histórico de modificaciones
		this.historicoDao.insertarHistoricoRecurso(recurso.getCodigo(), Constants.ACTUALIZACION_SUGERENCIAS,
				datosLogin.getUsername());

		// Enviamos un correo a la persona de contacto del Departamento

		// Preparamos el mail para enviar a la persona de contacto del departamento
		List<String> args = new ArrayList<String>();
		args.add(recurso.getNombre());

		String subject = this.messageSource.getMessage("sugerencia.guardar.email.subject", args.toArray(),
				LocaleContextHolder.getLocale());

		args = new ArrayList<String>();
		args.add(datosLogin.getUsername());

		args.add(recurso.getNombre());
		args.add(recurso.getSugerencia());

		String body = this.messageSource.getMessage("sugerencia.guardar.email.body", args.toArray(),
				LocaleContextHolder.getLocale());

		String mailTo = this.appConfiguration.getProperty("email.to");

		this.mailService.sendMail(mailTo, subject, body);

		return "OK";
	}

	/**
	 * Comprueba la existencia del recurso a partir del código de recurso. Si el recurso no existe se lanza una
	 * excepción.
	 * 
	 * @param codigoRecurso Código de recurso
	 */
	private void comprobarExistenciaRecurso(Long codigoRecurso) {
		if (!this.recursoDao.existeRecurso(codigoRecurso)) {
			logger.warn("El codigo de recurso: '{}' no es válido", codigoRecurso);

			throw new ValidacionException(this.messageSource.getMessage("error.recursoNoEncontrado", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajeRecursoNoEncontrado",
					null, LocaleContextHolder.getLocale()));
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getLocalizacionRecurso(java.lang.Long)
	 */
	@Override()
	public Recurso getLocalizacionRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoByCorecIdioma(codigoRecurso, LocaleContextHolder.getLocale()
				.getLanguage());

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		// Comprobamos si la localización del recurso es editable
		if (!Utilidades.getEditable(recurso).isLocalizacion()) {
			logger.warn("Se ha intentado editar una localización no editable: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.localizacionNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeLocalizacionNoEditable", null, LocaleContextHolder.getLocale()));
		}

		// Obtenemos los datos de georeferenciación del recurso
		Localizacion localizacion = this.recursoDao.findLocalizacionByRecurso(recurso);

		recurso.setLocalizacion(localizacion);

		return recurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarLocalizacionRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public String guardarLocalizacionRecurso(Recurso recurso) throws XmlMappingException,
			DatatypeConfigurationException, IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(recurso.getCodigo());

		// Guardamos la localización en BBDD
		this.recursoDao.actualizarLocalizacionRecurso(recurso);

		// Actualizamos la tabla RECURSO con la fecha y el usuario (GAURKOTU) que ha actualizado datos
		this.recursoDao.actualizarRecurso(recurso);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(recurso.getCodigo(), Constants.ACTUALIZACION_LOCALIZACION,
				datosLogin.getUsername());

		// Enviar a AA80T un mensaje de regeneración y republicación del recurso
		String mensaje = crearMensajeActualizacion(Constants.ACCION_REGENERAR_PUBLICAR, recurso.getCodigo());
		this.invoiceQueueSender.sendMesage(mensaje);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getOfertasRecurso(long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Recurso getOfertasRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos básicos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoBasicosByCorecIdioma(codigoRecurso, LocaleContextHolder
				.getLocale().getLanguage());

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		return recurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findOfertasByCorec(long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Oferta> findOfertasByCorec(Long codigoRecurso) {
		return this.recursoDao.findOfertasByCorec(codigoRecurso);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarOfertaRecurso(com.ejie.aa80a.model.Oferta)
	 */
	@Override()
	public String guardarOfertaRecurso(Oferta oferta) throws XmlMappingException, DatatypeConfigurationException,
			IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(oferta.getCodigoPadre());
		Boolean modoAlta = true;
		// Comprobamos si se trata de una oferta nueva o una modificación
		if (oferta.getCodigo() != null) {
			// Modificación

			modoAlta = false;
			// Comprobamos la existencia de la oferta
			this.comprobarExistenciaRecurso(oferta.getCodigo());

			// Actualizamos la oferta en BBDD
			this.recursoDao.actualizarOfertaRecurso(oferta);

		} else {
			// Alta

			// Insertamos la oferta en BBDD
			this.recursoDao.crearOfertaRecurso(oferta);
		}

		// guardamos las condiciones de la oferta
		this.guardarCondicionesOferta(oferta);

		// Enviar a AA80T un mensaje para que se cree el contenido asociado a la oferta en el Gestor de
		// Contenidos si no lo está ya, suba la imagen al Gestor de Contenidos si se ha introducido
		// y regenere y republique el recurso si es posible
		String mensaje = crearMensajeActualizacion(Constants.ACCION_CREAR_OFERTA, oferta.getCodigo());
		this.invoiceQueueSender.sendMesage(mensaje);

		// Enviar a AA80T un mensaje de regeneración y republicación del recurso padre para que se vea la oferta en su
		// ficha si está publicada. No necesario.
		// mensaje = crearMensajeActualizacion(Constants.ACCION_REGENERAR_PUBLICAR, oferta.getCodigoPadre());
		// this.invoiceQueueSender.sendMesage(mensaje);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(oferta.getCodigoPadre(), Constants.ACTUALIZACION_OFERTAS,
				datosLogin.getUsername());

		// Enviamos un correo a la persona de contacto del Departamento

		// Preparamos el mail para enviar a la persona de contacto del departamento

		// obtengo los datos del recurso para indicarlos en el mail
		Recurso recurso = this.recursoDao.findDatosRecursoByCorecIdioma(oferta.getCodigoPadre(), Constants.CASTELLANO);

		List<String> args = new ArrayList<String>();
		args.add(recurso.getNombre());

		String subject = "";
		String keySubject = "oferta.alta.email.subject";
		String keyBody = "oferta.alta.email.body";

		// Comprobamos si se trata de una modificación
		if (!modoAlta) {
			// en ese caso, cambiamos las key de los textos a cargar en el mail
			keySubject = "oferta.modificar.email.subject";
			keyBody = "oferta.modificar.email.body";

		}
		subject = this.messageSource.getMessage(keySubject, args.toArray(), LocaleContextHolder.getLocale());

		args = new ArrayList<String>();
		args.add(datosLogin.getUsername());

		args.add(recurso.getNombre());
		args.add(oferta.getNombreCastellano());

		String body = this.messageSource.getMessage(keyBody, args.toArray(), LocaleContextHolder.getLocale());

		String mailTo = this.appConfiguration.getProperty("email.to");

		this.mailService.sendMail(mailTo, subject, body);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getOfertaRecurso(java.lang.Long)
	 */
	@Override()
	public Oferta getOfertaRecurso(Long codigoOferta) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoOferta);

		return this.recursoDao.findOfertaByCorec(codigoOferta);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#eliminarOfertaRecurso(java.lang.Long)
	 */
	@Override()
	public String eliminarOfertaRecurso(Long codigoOferta) throws XmlMappingException, DatatypeConfigurationException,
			IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoOferta);

		// obtengo los datos de la oferta para indicarlos en el mail
		Oferta oferta = this.recursoDao.findOfertaByCorec(codigoOferta);

		this.recursoDao.eliminarOfertaRecurso(codigoOferta);

		// Enviar a AA80T un mensaje para que se despublique el contenido asociado a la oferta en el Gestor de
		// Contenidos
		String mensaje = crearMensajeActualizacion(Constants.ACCION_ELIMINAR_OFERTA, codigoOferta);
		this.invoiceQueueSender.sendMesage(mensaje);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(codigoOferta, Constants.ACTUALIZACION_OFERTAS,
				datosLogin.getUsername()); // TODO: Se debería insertar el código del padre

		// Enviamos un correo a la persona de contacto del Departamento

		// obtengo los datos del recurso para indicarlos en el mail
		Recurso recurso = this.recursoDao.findDatosRecursoByCorecIdioma(oferta.getCodigoPadre(), Constants.CASTELLANO);

		// Preparamos el mail para enviar a la persona de contacto del departamento
		List<String> args = new ArrayList<String>();
		args.add(recurso.getNombre());

		String subject = this.messageSource.getMessage("oferta.eliminar.email.subject", args.toArray(),
				LocaleContextHolder.getLocale());

		args = new ArrayList<String>();
		args.add(datosLogin.getUsername());

		args.add(recurso.getNombre());
		args.add(oferta.getNombreCastellano());

		String body = this.messageSource.getMessage("oferta.eliminar.email.body", args.toArray(),
				LocaleContextHolder.getLocale());

		String mailTo = this.appConfiguration.getProperty("email.to");

		this.mailService.sendMail(mailTo, subject, body);

		return "OK";
	}

	@Override()
	public Recurso getPreciosRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoByCorecIdioma(codigoRecurso, LocaleContextHolder.getLocale()
				.getLanguage());

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		// Comprobamos si los precios del recurso son editables
		if (!Utilidades.getEditable(recurso).isPrecios()) {
			logger.warn("Se ha intentado editar un precio no editable: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.PreciosNoEditables", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage("error.mensajePreciosNoEditables",
					null, LocaleContextHolder.getLocale()));
		}

		return recurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getMultimediaRecurso(java.lang.Long)
	 */
	@Override()
	public Recurso getMultimediaRecurso(Long codigoRecurso) {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(codigoRecurso);

		// Obtenemos los datos del recurso
		Recurso recurso = this.recursoDao.findDatosRecursoByCorecIdioma(codigoRecurso, LocaleContextHolder.getLocale()
				.getLanguage());

		recurso.setEsTop(this.recursoDao.getIsRecursoTop(codigoRecurso));

		// Se completa la URL a la ficha del recursos en el Portal de Turismo
		recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
				this.appConfiguration.getProperty("portal.host")));

		// Comprobamos si los multimedia del recurso son editables
		if (!Utilidades.getEditable(recurso).isMultimedia()) {
			logger.warn("Se ha intentado editar un multimedia no editable: '{}'", recurso.getTipo());

			throw new ValidacionException(this.messageSource.getMessage("error.multimediaNoEditable", null,
					LocaleContextHolder.getLocale()), this.messageSource.getMessage(
					"error.mensajeMultimediaNoEditable", null, LocaleContextHolder.getLocale()));
		}

		return recurso;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findMultimediasByCorec(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Multimedia> findMultimediasByCorec(Long codigoRecurso) {
		return this.recursoDao.findMultimediasByCorec(codigoRecurso);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarMultimedia(com.ejie.aa80a.model.Multimedia)
	 */
	@Override()
	public String guardarMultimedia(Multimedia multimedia) throws XmlMappingException, DatatypeConfigurationException,
			IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(multimedia.getCodRecurso());

		this.recursoDao.guardarMultimedia(multimedia);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(multimedia.getCodRecurso(), Constants.ACTUALIZACION_MULTIMEDIA,
				datosLogin.getUsername());

		// Enviar a AA80T un mensaje para subir el multimedia al Gestor de Contenidos
		String mensaje = crearMensajeActualizacion(Constants.ACCION_SUBIR_MULTIMEDIA, multimedia.getCodRecurso());
		this.invoiceQueueSender.sendMesage(mensaje);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#eliminarMultimedia(com.ejie.aa80a.model.Multimedia)
	 */
	@Override()
	public String eliminarMultimedia(Multimedia multimedia) throws XmlMappingException, DatatypeConfigurationException,
			IOException {
		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(multimedia.getCodRecurso());

		this.recursoDao.eliminarMultimediaRecurso(multimedia);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(multimedia.getCodRecurso(), Constants.ACTUALIZACION_MULTIMEDIA,
				datosLogin.getUsername());

		// Enviar a AA80T un mensaje de regeneración y republicación del recurso
		String mensaje = crearMensajeActualizacion(Constants.ACCION_REGENERAR_PUBLICAR, multimedia.getCodRecurso());
		this.invoiceQueueSender.sendMesage(mensaje);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#exiteTmpFile(java.lang.String, java.lang.Long)
	 */
	@Override()
	public Long exiteTmpFile(String nombre, Long codRecurso) {
		return this.recursoDao.exiteTmpFile(nombre, codRecurso);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getImagenById(java.lang.Long, java.lang.Long)
	 */
	@Override()
	public Imagen getImagenById(Long codMultimedia, Long codRecurso) {
		return this.recursoDao.getImagenById(codMultimedia, codRecurso);
	}

	/**
	 * Crea el mensaje XML de actualización que se enviará a la cola para que AA80T realice las operaciones solicitadas.
	 * 
	 * @param accion Acción a incluir en el mensaje
	 * @param codigoRecurso Código del recurso a actualizar
	 * 
	 * @return Mensaje en forma de String
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws XmlMappingException Excepción de mapeo de XML
	 * @throws IOException Excepción de entrada/salida
	 */
	private String crearMensajeActualizacion(String accion, Long codigoRecurso) throws DatatypeConfigurationException,
			XmlMappingException, IOException {
		ActualizacionGaurkotu actualizacionGaurkotu = new ActualizacionGaurkotu();
		actualizacionGaurkotu.setAccion(accion);
		actualizacionGaurkotu.setFechaHora(Utilidades.getXMLGregorianCalendarNow());
		actualizacionGaurkotu.setCodigo(codigoRecurso);

		StringWriter writerActualizacionGaurkotu = new StringWriter();
		this.marshallerActualizacionGaurkotu.marshal(actualizacionGaurkotu, new StreamResult(
				writerActualizacionGaurkotu));

		NotificationBody notificacion = new NotificationBody();
		notificacion.setXmlValue(writerActualizacionGaurkotu.toString());

		StringWriter writerNotificationBody = new StringWriter();
		this.marshallerNotification.marshal(notificacion, new StreamResult(writerNotificationBody));

		return writerNotificationBody.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#getComboTiposPrecio(java.lang.String)
	 */
	@Override()
	public List<Combo> getComboTiposPrecio(String subTipo) {
		return this.recursoDao.getComboTiposPrecio(subTipo);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findPreciosByCorec(java.lang.Long)
	 */
	@Override()
	public List<Precio> findPreciosByCorec(Long codigoRecurso) {
		return this.recursoDao.findPreciosByCorec(codigoRecurso);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#guardarPreciosRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public String guardarPreciosRecurso(Recurso recurso) throws XmlMappingException, DatatypeConfigurationException,
			IOException {

		// Comprobamos la existencia del recurso
		this.comprobarExistenciaRecurso(recurso.getCodigo());

		this.recursoDao.actualizarPreciosRecurso(recurso);

		// Actualizamos la tabla RECURSO con la fecha y el usuario (GAURKOTU) que ha actualizado datos
		this.recursoDao.actualizarRecurso(recurso);

		// Insertamos un registro en el histórico de modificaciones
		UsuarioDetalles datosLogin = Utilidades.getDetallesUsuario();
		this.historicoDao.insertarHistoricoRecurso(recurso.getCodigo(), Constants.ACTUALIZACION_PRECIOS,
				datosLogin.getUsername());

		// Enviar a AA80T un mensaje de regeneración y republicación del recurso
		String mensaje = this.crearMensajeActualizacion(Constants.ACCION_REGENERAR_PUBLICAR, recurso.getCodigo());
		this.invoiceQueueSender.sendMesage(mensaje);

		return "OK";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.service.RecursoService#findCondicionesOfertaByCorec(java.lang.Long)
	 */
	@Override()
	public List<Condicion> findCondicionesOfertaByCorec(Long codigoRecurso) {
		return this.recursoDao.findCondicionesOfertaByCorec(codigoRecurso);
	}

	/**
	 * Guarda las condiciones de una oferta
	 * 
	 * @param oferta datos de la oferta con la lista de condiciones
	 */
	private void guardarCondicionesOferta(Oferta oferta) {
		// Primero eliminamos las condiciones de la oferta
		this.recursoDao.eliminarCondicionesOferta(oferta.getCodigo());

		final String[] VERSIONES_IDIOMATICAS = { "es", "eu", "en", "fr", "de" };

		for (String vi : VERSIONES_IDIOMATICAS) {

			for (Condicion condicion : oferta.getCondiciones()) {

				condicion.setCodRecurso(oferta.getCodigo());

				if (vi.contentEquals(Constants.CASTELLANO)) {
					// CASTELLANO
					condicion.setIdioma(Constants.CASTELLANO);
					condicion.setTexto(condicion.getTextoEs());
					this.recursoDao.altaCondicionIdioma(condicion);
				} else if (vi.contentEquals(Constants.EUSKERA)) {
					// EUSKERA
					condicion.setIdioma(Constants.EUSKERA);
					condicion.setTexto(condicion.getTextoEu());
					this.recursoDao.altaCondicionIdioma(condicion);
				} else if (vi.contentEquals(Constants.INGLES)) {
					// INGLES
					condicion.setIdioma(Constants.INGLES);
					condicion.setTexto(condicion.getTextoEn());
					this.recursoDao.altaCondicionIdioma(condicion);
				} else if (vi.contentEquals(Constants.FRANCES)) {
					// FRANCES
					condicion.setIdioma(Constants.FRANCES);
					condicion.setTexto(condicion.getTextoFr());
					this.recursoDao.altaCondicionIdioma(condicion);
				} else if (vi.contentEquals(Constants.ALEMAN)) {
					// ALEMAN
					condicion.setIdioma(Constants.ALEMAN);
					condicion.setTexto(condicion.getTextoDe());
					this.recursoDao.altaCondicionIdioma(condicion);
				}
			}
		}

	}

}
