package com.ejie.aa80a.service;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import javax.xml.datatype.DatatypeConfigurationException;

import org.springframework.oxm.XmlMappingException;

import com.ejie.aa80a.model.Combo;
import com.ejie.aa80a.model.Condicion;
import com.ejie.aa80a.model.FiltroRecurso;
import com.ejie.aa80a.model.Imagen;
import com.ejie.aa80a.model.Multimedia;
import com.ejie.aa80a.model.Oferta;
import com.ejie.aa80a.model.Precio;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Usuario;

/**
 * Interfaz de los servicios de acceso a los datos del recurso
 * 
 *  
 */
public interface RecursoService {

	/**
	 * Obtiene los datos del recurso a partir de los datos del usuario
	 * 
	 * @param usuario Objeto con los datos del usuario
	 * 
	 * @return mapa con el nombre del recurso en castellano y euskera (claves 'es' y 'eu')
	 */
	Map<String, String> findByUsuario(Usuario usuario);

	/**
	 * Obtiene los recursos dependiendo del filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * @param editables Indica el conjunto de tipos de recursos por el que buscar true: busca sobre los tipos de recurso
	 *            editables por Garkotu, false: busca sobre los tipos de recurso con acceso a Garkotu
	 * 
	 * @return lista con los recursos.
	 */
	List<Recurso> findByCriteria(FiltroRecurso filtro, boolean editables);

	/**
	 * Obtiene el número de los recursos que cumplen el filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * @param editables Indica el conjunto de tipos de recursos por el que buscar true: busca sobre los tipos de recurso
	 *            editables por Garkotu, false: busca sobre los tipos de recurso con acceso a Garkotu
	 * 
	 * @return int número de recursos que cumplen el filtro aplicado.
	 */
	int findByCountCriteria(FiltroRecurso filtro, boolean editables);

	/**
	 * Obtiene los datos del recurso.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getDatosRecurso(Long codigoRecurso);

	/**
	 * Guarda los datos del recurso modificados en Base de datos. Guarda información de la actualización en la tabla de
	 * históricos. Envía a AA80T un mensaje de regeneración y republicación del recurso
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String guardarDatosRecurso(Recurso recurso) throws DatatypeConfigurationException, XmlMappingException, IOException;

	/**
	 * Obtiene los datos básicos del recurso y sus servicios asociados.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getServiciosRecurso(Long codigoRecurso);

	/**
	 * Guarda los servicios del recurso modificados en Base de datos. Guarda información de la actualización en la tabla
	 * de históricos. Envía a AA80T un mensaje de regeneración y republicación del recurso
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String guardarServiciosRecurso(Recurso recurso) throws XmlMappingException, DatatypeConfigurationException,
			IOException;

	/**
	 * Obtiene los datos del recurso y sus datos de goreferenciación.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getLocalizacionRecurso(Long codigoRecurso);

	/**
	 * Guarda la localización del recurso en Base de datos. Guarda información de la actualización en la tabla de
	 * históricos. Envía a AA80T un mensaje de regeneración y republicación del recurso
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String guardarLocalizacionRecurso(Recurso recurso) throws XmlMappingException, DatatypeConfigurationException,
			IOException;

	/**
	 * Obtiene los datos básicos del recurso.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getSugerenciasRecurso(Long codigoRecurso);

	/**
	 * Guarda las sugerencias introducidas para un recurso en Base de datos. Guarda información de la actualización en
	 * la tabla de históricos. Envía a un correo a la persona de contacto
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 * @return Resultado de la operación
	 */
	String guardarSugerenciasRecurso(Recurso recurso);

	/**
	 * Obtiene los datos básicos del recurso.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getOfertasRecurso(Long codigoRecurso);

	/**
	 * Obtiene las ofertas asociadas a un recurso.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return lista con las ofertas.
	 */
	List<Oferta> findOfertasByCorec(Long codigoRecurso);

	/**
	 * Guarda la oferta introducida para un recurso en Base de datos. Guarda información de la actualización en la tabla
	 * de históricos. Envía a un correo a la persona de contacto
	 * 
	 * @param oferta {@link Oferta} objeto con los datos de la oferta
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String guardarOfertaRecurso(Oferta oferta) throws XmlMappingException, DatatypeConfigurationException, IOException;

	/**
	 * Obtiene los datos de una oferta.
	 * 
	 * @param codigoOferta Código de la oferta
	 * 
	 * @return {@link Oferta} objeto con los datos de la oferta
	 */
	Oferta getOfertaRecurso(Long codigoOferta);

	/**
	 * Elimina una oferta (borrado físico en AA80T). Guarda información de la actualización en la tabla de históricos.
	 * Envía a un correo a la persona de contacto
	 * 
	 * @param codigoOferta Código de la oferta
	 * 
	 * @return Resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String eliminarOfertaRecurso(Long codigoOferta) throws XmlMappingException, DatatypeConfigurationException,
			IOException;

	/**
	 * Obtiene los datos del recurso para la pantalla de multimedia.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getMultimediaRecurso(Long codigoRecurso);

	/**
	 * Obtiene la lista de imágenes multimedia de un recurso
	 * 
	 * @param codigoRecurso Código del recurso
	 * @return Lista con los multimedia
	 */
	List<Multimedia> findMultimediasByCorec(Long codigoRecurso);

	/**
	 * Guarda un registro en la tabla MULTIMEDIA y su imagen asociada en la tabla temporal
	 * 
	 * @param multimedia datos del multimedia
	 * 
	 * @return resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String guardarMultimedia(Multimedia multimedia) throws XmlMappingException, DatatypeConfigurationException,
			IOException;

	/**
	 * Elimina un elemento multimedia
	 * 
	 * @param multimedia datos del elemento multimedia a eliminar
	 * 
	 * @return resultado de la operación
	 * 
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String eliminarMultimedia(Multimedia multimedia) throws XmlMappingException, DatatypeConfigurationException,
			IOException;

	/**
	 * Método que busca un archivo en la tabla TMP_IMAGEN, alternativamente mediante código de multimedia (pantalla
	 * multimedia) o código de recurso (pantalla ofertas)
	 * 
	 * @param nombre nombre del archivo multimedia
	 * @param codRecurso código del recurso
	 * @return codigo del recurso
	 */
	Long exiteTmpFile(String nombre, Long codRecurso);

	/**
	 * Devuelve el nombre y array de bytes de una imagen
	 * 
	 * @param codMultimedia código del registro multimedia
	 * @param codRecurso código del recurso
	 * @return datos de la imagen
	 */
	Imagen getImagenById(Long codMultimedia, Long codRecurso);

	/**
	 * Obtiene los datos del recurso para la pantalla de precios.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso
	 */
	Recurso getPreciosRecurso(Long codigoRecurso);

	/**
	 * Devuelve la lista de tipos de precio para cargar un combo
	 * 
	 * @param subTipo subtipo del recurso del que hay que cargar la lista de tipos de precio
	 * @return lista de tipos de precios
	 */
	List<Combo> getComboTiposPrecio(String subTipo);

	/**
	 * Actualiza la lista de precios de un recurso
	 * 
	 * @param recurso datos del recurso con la lista de precios
	 * @return resultado de la operación
	 * @throws DatatypeConfigurationException Excepción de configuración de tipos
	 * @throws IOException Excepción de entrada/salida
	 * @throws XmlMappingException Excepción de mapeo de XML
	 */
	String guardarPreciosRecurso(Recurso recurso) throws XmlMappingException, DatatypeConfigurationException,
			IOException;

	/**
	 * Devuelve la lista de precios de un recurso
	 * 
	 * @param codigoRecurso código del recurso
	 * @return lista de precios
	 */
	List<Precio> findPreciosByCorec(Long codigoRecurso);

	/**
	 * Devuelve la lista de condiciones asociadas a un recurso (para ofertas)
	 * 
	 * @param codigoRecurso código del recurso
	 * @return lista de condiciones
	 */
	List<Condicion> findCondicionesOfertaByCorec(Long codigoRecurso);

}