package com.ejie.aa80a.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.model.Usuario;
import com.ejie.aa80a.model.UsuarioDetalles;

/**
 * Clase encargada de la autenticación del usuario en la aplicación
 * 
 *  
 */
@Service(value = "customUserDetailsService")
public class CustomUserDetailsService implements UserDetailsService {

	private static final Logger logger = LoggerFactory.getLogger(CustomUserDetailsService.class);

	@Autowired()
	private UsuarioService usuarioService;

	@Autowired()
	private RecursoService recursoService;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.springframework.security.core.userdetails.UserDetailsService# loadUserByUsername(java.lang.String)
	 */
	@Override()
	@Transactional(readOnly = true)
	public UserDetails loadUserByUsername(String login) throws UsernameNotFoundException {

		Usuario usuario = this.usuarioService.findByUsuario(login);
		logger.debug("Usuario: {}", usuario);

		if (usuario == null) {
			logger.warn("Usuario no encontrado");
			throw new UsernameNotFoundException("Usuario no encontrado");
		}

		UsuarioDetalles usuarioDetalles = new UsuarioDetalles(usuario.getLogin(), usuario.getPassword(),
				getGrantedAuthorities(usuario),
				(usuario.getRecurso() != null ? usuario.getRecurso().getCodigo() : null), usuario.getNombre(),
				usuario.getApellidos());

		// Si el usuario está asociado a un recurso obtenemos el nombre en castellano y en euskera del recurso
		if (usuario.getRecurso() != null) {
			Map<String, String> nombreRecurso = this.recursoService.findByUsuario(usuario);
			usuarioDetalles.setNombreRecursoCastellano(nombreRecurso.get("es"));
			usuarioDetalles.setNombreRecursoEuskera(nombreRecurso.get("eu"));
		} else {
			// Es un coordinador
			usuarioDetalles.setNombreRecursoCastellano(this.messageSource.getMessage("comun.coordinador", null,
					new Locale("es")));
			usuarioDetalles.setNombreRecursoEuskera(this.messageSource.getMessage("comun.coordinador", null,
					new Locale("eu")));
		}

		return usuarioDetalles;
	}

	/**
	 * Obtiene las autorizaciones de acceso del usuario
	 * 
	 * @param usuario Usuario
	 * 
	 * @return lista de autorizaciones
	 */
	private List<GrantedAuthority> getGrantedAuthorities(Usuario usuario) {
		List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();

		if (usuario.getPerfil() != null) {
			logger.debug("PerfilUsuario: {}", usuario.getPerfil());
			authorities.add(new SimpleGrantedAuthority("ROLE_" + usuario.getPerfil().getTipoPerfilUsuario()));
		}

		logger.debug("authorities: {}", authorities);
		return authorities;
	}

}
