package com.ejie.aa80a.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.model.FiltroUsuario;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.TipoPerfilUsuario;
import com.ejie.aa80a.model.Usuario;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;

/**
 * Clase que accede a la base de datos para obtener datos de usuario. Tabla: USUARIOS
 * 
 *  
 */
@Repository()
@Transactional()
public class UsuarioDaoImpl implements UsuarioDao {

	private static final String SELECT_USUARIOS = "SELECT U.IDUSUARIO, U.USUARIO, U.PASSWORD, U.COREC AS CODIGO, U.DNI, U.NOMBRE AS NOMBRE_USU, U.APELLIDOS, U.CARGO, U.EMAIL, U.TELEFONO, U.PERFIL, R.COTIPREC, R.COREGNB29, DG.NOMBRE AS NOMBRE_REC";
	private static final String FROM_SELECT_USUARIOS = " FROM USUARIOS U LEFT JOIN RECURSO R ON U.COREC = R.COREC LEFT JOIN DATOS_GENERALES DG ON DG.COREC = R.COREC AND DG.COIDIOMA = ? ";

	private JdbcTemplate jdbcTemplate;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla USUARIOS con los objetos de la clase Usuario
	 */
	private static final class UsuarioRowMapper implements RowMapper<Usuario> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Usuario mapRow(ResultSet rs, int rowNum) throws SQLException {
			Usuario usuario = new Usuario();
			usuario.setIdUsuario(rs.getInt("IDUSUARIO"));
			usuario.setLogin(rs.getString("USUARIO"));
			usuario.setPassword(rs.getString("PASSWORD"));
			usuario.setDni(rs.getString("DNI"));
			usuario.setNombre(rs.getString("NOMBRE_USU"));
			usuario.setApellidos(rs.getString("APELLIDOS"));
			usuario.setCargo(rs.getString("CARGO"));
			usuario.setEmail(rs.getString("EMAIL"));
			usuario.setTelefono(rs.getString("TELEFONO"));
			usuario.setPerfil(TipoPerfilUsuario.valueOf(rs.getString("PERFIL")));

			if (rs.getLong("CODIGO") != 0) {
				Recurso recurso = new Recurso();
				recurso.setCodigo(rs.getLong("CODIGO"));
				recurso.setNombre(rs.getString("NOMBRE_REC"));
				recurso.setSignatura(rs.getString("COREGNB29"));
				recurso.setTipo(rs.getString("COTIPREC"));
				usuario.setRecurso(recurso);
			}

			return usuario;
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#findByUsuario(java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Usuario findByUsuario(String login) {
		Usuario usuario = this.jdbcTemplate.queryForObject(SELECT_USUARIOS + FROM_SELECT_USUARIOS
				+ " WHERE U.USUARIO = ?", new UsuarioRowMapper(), Constants.CASTELLANO, login);

		return usuario;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#findByEmail(java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Usuario> findByEmail(String email) {
		List<Usuario> usuarios = this.jdbcTemplate.query(SELECT_USUARIOS + FROM_SELECT_USUARIOS + " WHERE U.EMAIL = ?",
				new UsuarioRowMapper(), Constants.CASTELLANO, email);

		return usuarios;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#changePassword(java.lang.String, java.lang.String)
	 */
	@Override()
	public void changePassword(String usuario, String nuevaPassword) {
		this.jdbcTemplate.update(
				"UPDATE USUARIOS SET PASSWORD = ?, USUMODIF = ?, FECMODIF = SYSDATE WHERE USUARIO = ?", nuevaPassword,
				usuario, usuario);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#existeEmail(java.lang.String, java.lang.Integer)
	 */
	@Transactional(readOnly = true)
	@Override()
	public boolean existeEmail(String email, Integer idUsuarioActual) {
		Integer count = 0;

		if (idUsuarioActual == null) {
			count = this.jdbcTemplate.queryForObject("SELECT COUNT(USUARIO) FROM USUARIOS WHERE EMAIL = ?",
					Integer.class, email);
		} else {
			// Se trata de una modificación por lo que hay que descartar el propio usuario modificado
			count = this.jdbcTemplate.queryForObject(
					"SELECT COUNT(USUARIO) FROM USUARIOS WHERE EMAIL = ? AND IDUSUARIO != ?", Integer.class, email,
					idUsuarioActual);
		}

		if (count > 0) {
			return true;
		}

		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#existeUsuario(java.lang.String, java.lang.Integer)
	 */
	@Transactional(readOnly = true)
	@Override()
	public boolean existeUsuario(String login, Integer idUsuarioActual) {
		Integer count = 0;

		if (idUsuarioActual == null) {
			count = this.jdbcTemplate.queryForObject("SELECT COUNT(USUARIO) FROM USUARIOS WHERE USUARIO = ?",
					Integer.class, login);
		} else {
			// Se trata de una modificación por lo que hay que descartar el propio usuario modificado
			count = this.jdbcTemplate.queryForObject(
					"SELECT COUNT(USUARIO) FROM USUARIOS WHERE USUARIO = ? AND IDUSUARIO != ?", Integer.class, login,
					idUsuarioActual);
		}

		if (count > 0) {
			return true;
		}

		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#actualizarUsuario(com.ejie.aa80a.model.Usuario)
	 */
	@Override()
	public void actualizarUsuario(Usuario usuario, String username) {
		this.jdbcTemplate
				.update("UPDATE USUARIOS SET USUARIO = ?, DNI = ?, NOMBRE = ?, APELLIDOS = ?, CARGO = ?, TELEFONO = ?, EMAIL = ?, USUMODIF = ?, FECMODIF = SYSDATE WHERE IDUSUARIO = ?",
						usuario.getLogin(), usuario.getDni(), usuario.getNombre(), usuario.getApellidos(),
						usuario.getCargo(), usuario.getTelefono(), usuario.getEmail(), username, usuario.getIdUsuario());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#insertarUsuario(com.ejie.aa80a.model.Usuario, java.lang.String)
	 */
	@Override()
	public void insertarUsuario(Usuario usuario, String username) {
		Object[] insertObj = { usuario.getLogin(), usuario.getPassword(),
				(usuario.getRecurso() != null ? usuario.getRecurso().getCodigo() : null), usuario.getDni(),
				usuario.getNombre(), usuario.getApellidos(), usuario.getCargo(), usuario.getTelefono(),
				usuario.getEmail(), usuario.getPerfil().getTipoPerfilUsuario(), username };

		this.jdbcTemplate
				.update("INSERT INTO USUARIOS (IDUSUARIO, USUARIO, PASSWORD, COREC, DNI, NOMBRE, APELLIDOS, CARGO, TELEFONO, EMAIL, PERFIL, USUMODIF, FECMODIF) VALUES (AA80A01Q00.NEXTVAL,?,?,?,?,?,?,?,?,?,?,?,SYSDATE)",
						insertObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#obtenerPassword(java.lang.String)
	 */
	@Override()
	public String obtenerPassword(String usuario) {
		String password = this.jdbcTemplate.queryForObject("SELECT PASSWORD FROM USUARIOS WHERE USUARIO = ?",
				String.class, usuario);

		return password;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#findByCriteria(com.ejie.aa80a.model.FiltroUsuario)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Usuario> findByCriteria(FiltroUsuario filtro) {
		StringBuilder query = new StringBuilder(SELECT_USUARIOS + FROM_SELECT_USUARIOS);

		StringBuilder whereSql = new StringBuilder();

		List<Object> argumentos = new ArrayList<Object>();

		argumentos.add(Constants.CASTELLANO);

		// Cuidado, si cambia algo del filtro, hay que cambiar en el método findByCountCriteria que devuelve el contador
		// de la consulta para la paginación.
		if (filtro.getDni() != null && !"".equals(filtro.getDni())) {
			whereSql.append(" WHERE UPPER(U.DNI) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getDni() + "%");
		}
		if (filtro.getNombre() != null && !"".equals(filtro.getNombre())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(U.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombre() + "%");
		}
		if (filtro.getApellidos() != null && !"".equals(filtro.getApellidos())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(U.APELLIDOS) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getApellidos() + "%");
		}
		if (filtro.getLogin() != null && !"".equals(filtro.getLogin())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(U.USUARIO) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getLogin() + "%");
		}
		if (filtro.getCodigoRecurso() != null && !"".equals(filtro.getCodigoRecurso())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" U.COREC = ? ");
			argumentos.add(filtro.getCodigoRecurso());
		}
		if (filtro.getNombreRecurso() != null && !"".equals(filtro.getNombreRecurso())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(DG.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombreRecurso() + "%");
		}
		if (filtro.getSignatura() != null && !"".equals(filtro.getSignatura())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" R.COREGNB29 = ? ");
			argumentos.add(filtro.getSignatura());
		}

		query.append(whereSql).append(" ORDER BY DG.NOMBRE");

		// Se obtiene la consulta final incluyendo la paginación.
		StringBuilder queryFinal = new StringBuilder(Utilidades.getPaginationQuery(query, filtro.getOffset(),
				filtro.getLimit(), null, null).toString());

		List<Usuario> usuarios = this.jdbcTemplate.query(queryFinal.toString(), new UsuarioRowMapper(),
				argumentos.toArray());

		for (Usuario usuario : usuarios) {
			if (usuario.getRecurso() != null) {
				usuario.setNombreTipoRecurso(this.messageSource.getMessage("tipoRecurso."
						+ usuario.getRecurso().getTipo(), null, LocaleContextHolder.getLocale()));
			} else {
				usuario.setNombreTipoRecurso(this.messageSource.getMessage("comun.coordinador", null,
						LocaleContextHolder.getLocale()));
			}
		}

		return usuarios;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#findByCountCriteria(com.ejie.aa80a.model.FiltroUsuario)
	 */
	@Transactional(readOnly = true)
	@Override()
	public int findByCountCriteria(FiltroUsuario filtro) {
		StringBuilder query = new StringBuilder("SELECT COUNT(U.IDUSUARIO) " + FROM_SELECT_USUARIOS);

		StringBuilder whereSql = new StringBuilder();

		List<Object> argumentos = new ArrayList<Object>();

		argumentos.add(Constants.CASTELLANO);

		// Cuidado, si cambia algo del filtro en el método findByCriteria hay que cambiarlo también aquí.
		if (filtro.getDni() != null && !"".equals(filtro.getDni())) {
			whereSql.append(" WHERE UPPER(U.DNI) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getDni() + "%");
		}
		if (filtro.getNombre() != null && !"".equals(filtro.getNombre())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(U.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombre() + "%");
		}
		if (filtro.getApellidos() != null && !"".equals(filtro.getApellidos())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(U.APELLIDOS) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getApellidos() + "%");
		}
		if (filtro.getLogin() != null && !"".equals(filtro.getLogin())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(U.USUARIO) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getLogin() + "%");
		}
		if (filtro.getCodigoRecurso() != null && !"".equals(filtro.getCodigoRecurso())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" U.COREC = ? ");
			argumentos.add(filtro.getCodigoRecurso());
		}
		if (filtro.getNombreRecurso() != null && !"".equals(filtro.getNombreRecurso())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" UPPER(DG.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombreRecurso() + "%");
		}
		if (filtro.getSignatura() != null && !"".equals(filtro.getSignatura())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" R.COREGNB29 = ? ");
			argumentos.add(filtro.getSignatura());
		}

		return this.jdbcTemplate.queryForObject(query.append(whereSql).toString(), argumentos.toArray(), Integer.class);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.UsuarioDao#eliminarUsuario(java.lang.Integer)
	 */
	@Override()
	public void eliminarUsuario(Integer idUsuario) {
		this.jdbcTemplate.update("DELETE FROM USUARIOS WHERE IDUSUARIO = ?", idUsuario);

	}

}
