package com.ejie.aa80a.dao;

import java.util.List;

import com.ejie.aa80a.model.FiltroUsuario;
import com.ejie.aa80a.model.Usuario;

/**
 * Interfaz de acceso a la base de datos para obtener datos de usuario. Tabla: USUARIOS
 * 
 *  
 */
public interface UsuarioDao {

	/**
	 * Busca por login de usuario.
	 * 
	 * @param login Login del usuario
	 * 
	 * @return Objeto con los datos del usuario
	 */
	Usuario findByUsuario(String login);

	/**
	 * Busca por email de usuario.
	 * 
	 * @param email Email del usuario
	 * 
	 * @return Objeto con los datos de los usuarios asociados a ese email
	 */
	List<Usuario> findByEmail(String email);

	/**
	 * Comprueba si el email corresponde con algún usuario.
	 * 
	 * @param email Email del usuario
	 * @param idUsuarioActual Id de usuario actual, para modificaciones
	 * 
	 * @return true si el email pertenece a algún usuario
	 */
	boolean existeEmail(String email, Integer idUsuarioActual);

	/**
	 * Comprueba si el login corresponde con algún usuario.
	 * 
	 * @param login Login del usuario
	 * @param idUsuarioActual Id de usuario actual, para modificaciones
	 * 
	 * @return true si el login pertenece a algún usuario
	 */
	boolean existeUsuario(String login, Integer idUsuarioActual);

	/**
	 * Cambia la password de un usuario.
	 * 
	 * @param usuario Login del usuario
	 * @param nuevaPassword Nueva password
	 */
	void changePassword(String usuario, String nuevaPassword);

	/**
	 * Actualiza los datos del usuario.
	 * 
	 * @param usuario Objeto con los datos del usuario
	 * @param username Login actual
	 */
	void actualizarUsuario(Usuario usuario, String username);

	/**
	 * Inserta un nuevo usuario.
	 * 
	 * @param usuario Objeto con los datos del usuario
	 * @param username Login actual
	 */
	void insertarUsuario(Usuario usuario, String username);

	/**
	 * Obtiene la password (codificada) de un usuario.
	 * 
	 * @param usuario Login del usuario
	 * 
	 * @return password actual (codificada) del usuario
	 */
	String obtenerPassword(String usuario);

	/**
	 * Obtiene los usuarios que cumplen el filtro proporcionado.
	 * 
	 * @param filtro {@link Usuario} objeto con el filtro a cumplir.
	 * 
	 * @return List<Usuario> Devuelve el listado de usuarios que cumplen con el filtro.
	 */
	List<Usuario> findByCriteria(FiltroUsuario filtro);

	/**
	 * Obtiene el número de usuarios que cumplen el filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return int número de usuarios que cumplen el filtro aplicado.
	 */
	int findByCountCriteria(FiltroUsuario filtro);

	/**
	 * Elimina un usuario.
	 * 
	 * @param idUsuario Id del usuario a eliminar
	 */
	void eliminarUsuario(Integer idUsuario);

}
