package com.ejie.aa80a.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.model.FiltroSugerencia;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Sugerencia;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;

/**
 * Clase que accede a la base de datos para tratar datos de sugerencias. Tablas: SUGERENCIAS
 * 
 *  
 */
@Repository()
@Transactional()
public class SugerenciaDaoImpl implements SugerenciaDao {

	private JdbcTemplate jdbcTemplate;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private Properties appConfiguration;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla SUGERENCIAS con los objetos de la clase Sugerencia
	 */
	private static final class SugerenciaRowMapper implements RowMapper<Sugerencia> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Sugerencia mapRow(ResultSet rs, int rowNum) throws SQLException {
			Sugerencia sugerencia = new Sugerencia();
			sugerencia.setCodigo(rs.getLong("IDSUGERENCIA"));
			sugerencia.setEstado(rs.getString("ESTADO"));
			sugerencia.setFecha(rs.getDate("FECMODIF"));
			sugerencia.setUsuario(rs.getString("USUMODIF"));
			sugerencia.setTexto(rs.getString("TEXTO"));

			Recurso recurso = new Recurso();
			recurso.setCodigo(rs.getLong("COREC"));
			recurso.setNombre(rs.getString("NOMBRE"));
			recurso.setTipo(rs.getString("COTIPREC"));
			recurso.setUrlPortal(rs.getString("URLAMIGABLE")); // url amigable de la ficha en el portal de turismo

			sugerencia.setRecurso(recurso);
			return sugerencia;
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.SugerenciaDao#insertarSugerenciaRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public void insertarSugerenciaRecurso(Recurso recurso, String username) {
		this.jdbcTemplate
				.update("INSERT INTO SUGERENCIAS (IDSUGERENCIA, COREC, ESTADO, TEXTO, USUMODIF, FECMODIF) VALUES (AA80A03Q00.NEXTVAL, ?, ?, ?, ?, SYSDATE)",
						recurso.getCodigo(), Constants.SUGERENCIA_ENVIADA, recurso.getSugerencia(), username);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.SugerenciaDao#findSugerenciaByCriteria(com.ejie.aa80a.model.FiltroSugerencia)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Sugerencia> findSugerenciaByCriteria(FiltroSugerencia filtro) {
		StringBuilder query = new StringBuilder();

		query.append("SELECT S.IDSUGERENCIA, S.COREC, S.ESTADO, S.USUMODIF, S.FECMODIF , DG.NOMBRE, R.COTIPREC, POS.URLAMIGABLE, S.TEXTO ");

		// Where clause & Params
		Map<String, ?> mapaWhere = getWhereMapSugerenciaByCriteria(filtro);
		StringBuilder where = new StringBuilder();
		where.append(mapaWhere.get("query"));
		query.append(where);

		query.append(" ORDER BY S.ESTADO ASC, S.FECMODIF DESC");

		List<?> argumentos = (List<?>) mapaWhere.get("params");

		// Se obtiene la consulta final incluyendo la paginación.
		StringBuilder queryFinal = new StringBuilder(Utilidades.getPaginationQuery(query, filtro.getOffset(),
				filtro.getLimit(), null, null).toString());

		List<Sugerencia> sugerencias = this.jdbcTemplate.query(queryFinal.toString(), new SugerenciaRowMapper(),
				argumentos.toArray());

		for (Sugerencia sugerencia : sugerencias) {

			sugerencia.getRecurso().setNombreTipo(
					this.messageSource.getMessage("tipoRecurso." + sugerencia.getRecurso().getTipo(), null,
							LocaleContextHolder.getLocale()));

			sugerencia.setDescEstado(this.messageSource.getMessage("notificacion.estado." + sugerencia.getEstado(),
					null, LocaleContextHolder.getLocale()));

			sugerencia.getRecurso().setUrlPortal(
					Utilidades.getURLAmigableFichaPortal(sugerencia.getRecurso(),
							this.appConfiguration.getProperty("portal.host")));
		}

		return sugerencias;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.SugerenciaDao#findSugerenciaByCountCriteria(com.ejie.aa80a.model.FiltroSugerencia)
	 */
	@Transactional(readOnly = true)
	@Override()
	public int findSugerenciaByCountCriteria(FiltroSugerencia filtro) {
		StringBuilder query = new StringBuilder("SELECT COUNT(S.IDSUGERENCIA) ");

		// Where clause & Params
		Map<String, ?> mapaWhere = getWhereMapSugerenciaByCriteria(filtro);
		StringBuilder where = new StringBuilder();
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> argumentos = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForObject(query.toString(), argumentos.toArray(), Integer.class);
	}

	/**
	 * Devuelve un mapa con la query a ejecutar y los parámetros de búsqueda
	 * 
	 * @param filtro datos del filtro
	 * @return query a ejecutar
	 */
	private Map<String, ?> getWhereMapSugerenciaByCriteria(FiltroSugerencia filtro) {
		StringBuilder query = new StringBuilder();
		List<Object> argumentos = new ArrayList<Object>();

		query.append(" FROM SUGERENCIAS S ");
		query.append(" INNER JOIN RECURSO R ON R.COREC = S.COREC ");
		query.append(" INNER JOIN POSICIONAMIENTO POS ON POS.COREC = S.COREC AND POS.COIDIOMA = ? ");
		query.append(" INNER JOIN DATOS_GENERALES DG ON DG.COREC = R.COREC AND DG.COIDIOMA = ?");

		argumentos.add(LocaleContextHolder.getLocale().getLanguage());
		argumentos.add(LocaleContextHolder.getLocale().getLanguage());

		query.append(" WHERE 1=1 ");

		if (filtro.getNombreRecurso() != null && !"".equals(filtro.getNombreRecurso())) {
			query.append(" AND UPPER(DG.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombreRecurso() + "%");
		}

		if (filtro.getTipoRecurso() != null && !"".equals(filtro.getTipoRecurso())) {
			query.append(" AND R.COTIPREC = ? ");
			argumentos.add(filtro.getTipoRecurso());
		}

		if (filtro.getEstado() != null && !"".equals(filtro.getEstado())) {
			query.append(" AND S.ESTADO = ?");
			argumentos.add(filtro.getEstado());
		}

		if (filtro.getFechaDesde() != null) {
			query.append(" AND TRUNC(S.FECMODIF) >= ?");
			argumentos.add(filtro.getFechaDesde());
		}
		if (filtro.getFechaHasta() != null) {
			query.append(" AND TRUNC(S.FECMODIF) <= ?");
			argumentos.add(filtro.getFechaHasta());
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", query);
		mapWhere.put("params", argumentos);

		return mapWhere;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.SugerenciaDao#actualizarEstadoSugerencia(java.lang.Long, java.lang.String)
	 */
	@Override()
	public void actualizarEstadoSugerencia(Long codigo, String estado) {
		// Hacemos un update
		this.jdbcTemplate.update("UPDATE SUGERENCIAS SET ESTADO = ? WHERE IDSUGERENCIA = ?", estado, codigo);
	}

}
