package com.ejie.aa80a.dao;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.support.SqlLobValue;
import org.springframework.jdbc.support.lob.DefaultLobHandler;
import org.springframework.jdbc.support.lob.LobHandler;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.model.Combo;
import com.ejie.aa80a.model.Condicion;
import com.ejie.aa80a.model.FiltroRecurso;
import com.ejie.aa80a.model.Imagen;
import com.ejie.aa80a.model.Localizacion;
import com.ejie.aa80a.model.Multimedia;
import com.ejie.aa80a.model.Oferta;
import com.ejie.aa80a.model.Precio;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Servicio;
import com.ejie.aa80a.util.Constants;
import com.ejie.aa80a.util.Utilidades;

/**
 * Clase que accede a la base de datos para obtener datos de recurso. Tablas: TIPO_RECURSO, OFICINASTURISMO, ALOJAMIENTO
 * 
 *  
 */
@Repository()
@Transactional()
public class RecursoDaoImpl implements RecursoDao {

	private static final Logger logger = LoggerFactory.getLogger(RecursoDaoImpl.class);

	private static final String RECURSOS_CON_ACCESO = "'A1','E1'";
	private static final String RECURSOS_EDITABLES = "'A1','B1','C1','C2','C3','C6','D1','D2','D3','D4','D5','D6','E1','G1','G2','H1','H10','H4','H5','H6',"
			+ "'I2','I31','I32','J1','J2','J4','K1','K3','K5','K7','K8','L1','L2','L3','M1','M2','N1','N2','N3','N5'";

	// Constantes de estados de AA80T
	private static final String SITUACION_RECURSO_EN_EDICION = "EE";

	private static final String SITUACION_VI_PENDIENTE_TRADUCCION = "PT";
	private static final String SITUACION_VI_EN_EDICION = "EE";

	private static final String TIPO_OFERTA_ALOJAMIENTO = "0100";
	private static final String TIPO_OFERTA_RESTAURANTE = "0101";
	private static final String TIPO_OFERTA_OFICINA_TURISMO = "0108";

	// Versiones idiomáticas (VI) contempladas en AA80T: es - Castellano, eu - Euskera, en - Inglés, fr - Francés, de -
	// Alemán
	private static final String[] VERSIONES_IDIOMATICAS = { "es", "eu", "en", "fr", "de" };

	private static final int MAX_VERSIONES_IDIOMATICAS = 5;

	private JdbcTemplate jdbcTemplate;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private Properties appConfiguration;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla RECURSO con los objetos de la clase Recurso
	 */
	private static final class RecursoRowMapper implements RowMapper<Recurso> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Recurso mapRow(ResultSet rs, int rowNum) throws SQLException {
			Recurso recurso = new Recurso();
			recurso.setCodigo(rs.getLong("COREC"));
			recurso.setNombre(rs.getString("NOMBRE"));
			recurso.setSignatura(rs.getString("COREGNB29"));
			recurso.setTipo(rs.getString("COTIPREC"));
			recurso.setUrlPortal(rs.getString("URLAMIGABLE")); // url amigable de la ficha en el portal de turismo
			return recurso;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la consulta con los objetos de la clase Recurso
	 */
	private static final class DatosRecursoRowMapper implements RowMapper<Recurso> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Recurso mapRow(ResultSet rs, int rowNum) throws SQLException {
			Recurso recurso = new Recurso();

			// Datos comunes
			recurso.setCodigo(rs.getLong("COREC"));
			recurso.setNombre(rs.getString("NOMBRE"));
			recurso.setSignatura(rs.getString("COREGNB29"));
			recurso.setTipo(rs.getString("COTIPREC"));
			recurso.setProvincia(rs.getString("PROVINCIA"));
			recurso.setMunicipio(rs.getString("MUNICIPIO"));
			recurso.setLocalidad(rs.getString("LOCALIDAD"));
			recurso.setUrlPortal(rs.getString("URLAMIGABLE"));
			recurso.setDireccion(rs.getString("DIRECCION"));
			recurso.setCodigoPostal(rs.getString("CODIGOPOSTAL"));
			recurso.setEmail(rs.getString("MAIL"));
			recurso.setTelefono1(rs.getString("TELEFONO1"));
			recurso.setTelefono2(rs.getString("TELEFONO2"));
			recurso.setWeb(rs.getString("URL"));
			recurso.setSubtipo(rs.getString("CODSUBTIPO"));

			// Datos particulares
			if (Constants.AGENDA.equals(recurso.getTipo()) || Constants.OFERTAS.equals(recurso.getTipo())) {
				recurso.setFechaInicio(rs.getDate("FECHAINICIO"));
				recurso.setFechaFin(rs.getDate("FECHAFIN"));
			} else if (Constants.ALOJAMIENTO.equals(recurso.getTipo())) {
				recurso.setCategoria(rs.getString("CATEGORIA"));
				recurso.setModalidad(rs.getString("MODALIDAD"));
			}

			if (Constants.TIPO_RECURSOS_FICHA_REDUCIDAD.contains(recurso.getTipo())) {
				Imagen imagen = new Imagen();
				imagen.setNombre(rs.getString("ARCHIVOMULT"));
				recurso.setImagen(imagen);
			}

			recurso.setRutaRecurso(Utilidades.getURLContenido(recurso.getCodigo(), recurso.getTipo(),
					Constants.CASTELLANO)); // ruta del recurso de castellano utilizada para visualizar la foto

			return recurso;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la consulta con los objetos de la clase Recurso
	 */
	private static final class DatosRecursoBasicosRowMapper implements RowMapper<Recurso> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Recurso mapRow(ResultSet rs, int rowNum) throws SQLException {
			Recurso recurso = new Recurso();

			// Datos comunes
			recurso.setCodigo(rs.getLong("COREC"));
			recurso.setNombre(rs.getString("NOMBRE"));
			recurso.setTipo(rs.getString("COTIPREC"));
			recurso.setUrlPortal(rs.getString("URLAMIGABLE"));
			recurso.setSubtipo(rs.getString("CODSUBTIPO"));

			return recurso;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la consulta con los objetos de la clase Servicio
	 */
	private static final class ServiciosRowMapper implements RowMapper<Servicio> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Servicio mapRow(ResultSet rs, int rowNum) throws SQLException {
			Servicio servicio = new Servicio();

			servicio.setCodigo(rs.getLong("COSERVICIO"));
			servicio.setNombre(rs.getString("NOSERVICIO"));
			servicio.setSeleccionado(rs.getString("SELECCIONADO"));

			return servicio;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla GEOREFERENCIACION con los objetos de la clase
	 * Localizacion
	 */
	private static final class LocalizacionRowMapper implements RowMapper<Localizacion> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Localizacion mapRow(ResultSet rs, int rowNum) throws SQLException {
			Localizacion localizacion = new Localizacion();
			localizacion.setGmLng(rs.getDouble("GMLNG"));
			localizacion.setGmLat(rs.getDouble("GMLAT"));
			localizacion.setGmZoom(rs.getInt("GMZOOM"));

			return localizacion;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla OFERTAS con los objetos de la clase Oferta
	 */
	private static final class ListaOfertasResultSetExtractor implements ResultSetExtractor<List<Oferta>> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.ResultSetExtractor#extractData(java.sql.ResultSet)
		 */
		@Override()
		public List<Oferta> extractData(ResultSet rs) throws SQLException, DataAccessException {
			List<Oferta> listaOfertas = new ArrayList<Oferta>();

			// Vienen dos filas por oferta, una correspondiente a la VI de castellano y otra a la de euskera
			while (rs.next()) {
				Oferta oferta = new Oferta();
				oferta.setCodigo(rs.getLong("COREC"));

				if (listaOfertas.contains(oferta)) {
					// Si existe se trata de la VI de euskera, ya que por la ordenación viene la segunda
					oferta = listaOfertas.get(listaOfertas.indexOf(oferta));
					oferta.setNombreEuskera(rs.getString("NOMBRE"));
				} else {
					// Si no existe se trata de la VI de castellano, ya que por la ordenación viene la primera
					oferta.setNombreCastellano(rs.getString("NOMBRE"));
					oferta.setFechaDesde(rs.getDate("FECHAINICIO"));
					oferta.setFechaHasta(rs.getDate("FECHAFIN"));
					oferta.setPrecio(rs.getDouble("PRECIO"));
					listaOfertas.add(oferta);
				}
			}

			return listaOfertas;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla OFERTAS con los objetos de la clase Oferta
	 */
	private static final class OfertaResultSetExtractor implements ResultSetExtractor<Oferta> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.ResultSetExtractor#extractData(java.sql.ResultSet)
		 */
		@Override()
		public Oferta extractData(ResultSet rs) throws SQLException, DataAccessException {
			Oferta oferta = new Oferta();

			// Vienen tantas filas como VI tenga la oferta
			while (rs.next()) {

				if (Constants.CASTELLANO.equals(rs.getString("COIDIOMA"))) {
					oferta.setCodigo(rs.getLong("COREC"));
					oferta.setNombreCastellano(rs.getString("NOMBRE"));
					oferta.setDescripcionCastellano(rs.getString("DESCRIPCION"));
					oferta.setFechaDesde(rs.getDate("FECHAINICIO"));
					oferta.setFechaHasta(rs.getDate("FECHAFIN"));
					oferta.setPrecio(rs.getDouble("PRECIO"));
					oferta.setWeb(rs.getString("URL"));
					oferta.setEmail(rs.getString("MAIL"));
					oferta.setCodigoPadre(rs.getLong("COPARENTREC"));
					// dirección
					oferta.setDireccion(rs.getString("DIRECCION"));
					oferta.setTelefono(rs.getString("TELEFONO1"));
					// provincia
					oferta.setCodigoProvincia(rs.getInt("COPROV"));
					oferta.setNombreProvinciaEs(rs.getString("DS_PROV"));
					oferta.setNombreProvinciaEu(rs.getString("DSE_PROV"));

					// municipio
					oferta.setCodigoMunicipio(rs.getString("COMUN"));
					oferta.setNombreMunicipioEs(rs.getString("DS_MUNI"));

					// localidad
					oferta.setCodigoLocalidad(rs.getLong("COLOC"));
					oferta.setNombreLocalidadEs(rs.getString("DS_LOC"));

					if (rs.getString("ARCHIVOMULT") != null) {
						Imagen imagen = new Imagen();
						imagen.setNombre(rs.getString("ARCHIVOMULT"));

						oferta.setImagen(imagen);
					}
				} else if (Constants.EUSKERA.equals(rs.getString("COIDIOMA"))) {
					oferta.setNombreEuskera(rs.getString("NOMBRE"));
					oferta.setDescripcionEuskera(rs.getString("DESCRIPCION"));

				} else if (Constants.INGLES.equals(rs.getString("COIDIOMA"))) {
					oferta.setNombreIngles(rs.getString("NOMBRE"));
					oferta.setDescripcionIngles(rs.getString("DESCRIPCION"));

				} else if (Constants.FRANCES.equals(rs.getString("COIDIOMA"))) {
					oferta.setNombreFrances(rs.getString("NOMBRE"));
					oferta.setDescripcionFrances(rs.getString("DESCRIPCION"));

				} else if (Constants.ALEMAN.equals(rs.getString("COIDIOMA"))) {
					oferta.setNombreAleman(rs.getString("NOMBRE"));
					oferta.setDescripcionAleman(rs.getString("DESCRIPCION"));

				}
			}

			oferta.setRutaRecurso(Utilidades.getURLContenido(oferta.getCodigo(), Constants.OFERTAS,
					Constants.CASTELLANO)); // ruta del recurso de castellano utilizada para visualizar la foto

			return oferta;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla MULTIMEDIA con los objetos de la clase Multimedia
	 */
	private static final class MultimediaRowMapper implements RowMapper<Multimedia> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Multimedia mapRow(ResultSet rs, int rowNum) throws SQLException {
			Multimedia multimedia = new Multimedia();
			multimedia.setCodigo(rs.getLong("COMULT"));
			multimedia.setCodRecurso(rs.getLong("COREC"));
			multimedia.setTipoRecurso(rs.getString("COTIPREC"));
			multimedia.setCodUsuario(rs.getString("COUSER"));

			multimedia.setImagen(new Imagen());
			multimedia.getImagen().setNombre(rs.getString("ARCHIVOMULT"));
			multimedia.getImagen().setDescripcion(rs.getString("DEMULT"));
			return multimedia;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla RECURSO con los objetos de la clase Recurso
	 */
	private static final class ImagenRowMapper implements RowMapper<Imagen> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Imagen mapRow(ResultSet rs, int rowNum) throws SQLException {
			Imagen imagen = new Imagen();
			imagen.setNombre(rs.getString("NOMBRE"));
			imagen.setBytes(rs.getBytes("IMAGEN"));
			return imagen;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla MULTIMEDIA con los objetos de la clase Multimedia
	 */
	private static final class TipoPrecioRowMapper implements RowMapper<Combo> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Combo mapRow(ResultSet rs, int rowNum) throws SQLException {
			return new Combo(rs.getString("COPRECIO"), rs.getString("NOMBRE"));
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla MULTIMEDIA con los objetos de la clase Multimedia
	 */
	private static final class PrecioRowMapper implements RowMapper<Precio> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Precio mapRow(ResultSet rs, int rowNum) throws SQLException {
			Precio precio = new Precio();

			precio.setId(rs.getString("ID"));
			precio.setCodigoPrecio(rs.getLong("COPRECIO"));
			precio.setCapacidad(rs.getInt("CAPACIDAD"));
			precio.setPrecioMinimo(rs.getDouble("PRECIOMINIMO"));
			precio.setPrecioMaximo(rs.getDouble("PRECIOMAXIMO"));
			precio.setParcelaAdulto(rs.getInt("OTROADULTO"));
			precio.setParcelaInfantil(rs.getInt("OTROINFANTIL"));
			precio.setParcelaTienda(rs.getInt("OTROTIENDA"));
			precio.setParcelaAutomovil(rs.getInt("OTROAUTOMOVIL"));
			precio.setParcelaConexionElectrica(rs.getInt("OTROCONEXELECTRICA"));
			precio.setTipoPrecio(rs.getString("NOMPRECIO"));
			return precio;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla CONDICIONES con los objetos de la clase Condiciones
	 */
	private static final class CondicionRowMapper implements RowMapper<Condicion> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Condicion mapRow(ResultSet rs, int rowNum) throws SQLException {
			Condicion condicion = new Condicion();
			condicion.setOrden(rs.getInt("ORDEN"));
			condicion.setTextoEs(rs.getString("ES"));
			condicion.setTextoEu(rs.getString("EU"));
			condicion.setTextoEn(rs.getString("EN"));
			condicion.setTextoFr(rs.getString("FR"));
			condicion.setTextoDe(rs.getString("DE"));
			return condicion;
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findNombreByCorecTipoIdioma(java.lang.Long, java.lang.String,
	 * java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public String findNombreByCorecTipoIdioma(Long codigoRecurso, String tipoRecurso, String idioma) {
		StringBuilder query = new StringBuilder("SELECT NOMBRE FROM DATOS_GENERALES WHERE COREC = ? AND COIDIOMA = ?");

		return this.jdbcTemplate.queryForObject(query.toString(), String.class, codigoRecurso, idioma);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findByCriteria(com.ejie.aa80a.model.FiltroRecurso, boolean)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Recurso> findByCriteria(FiltroRecurso filtro, boolean editables) {

		StringBuilder query = new StringBuilder();

		query.append("SELECT DISTINCT R.COREC, R.COTIPREC, R.COREGNB29, POS.URLAMIGABLE, DG.NOMBRE ");

		// Where clause & Params
		Map<String, ?> mapaWhere = getWhereMapRecursoByCriteria(filtro, editables);
		StringBuilder where = new StringBuilder();
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> argumentos = (List<?>) mapaWhere.get("params");

		query.append("ORDER BY DG.NOMBRE");

		// Se obtiene la consulta final incluyendo la paginación.
		StringBuilder queryFinal = new StringBuilder(Utilidades.getPaginationQuery(query, filtro.getOffset(),
				filtro.getLimit(), null, null).toString());

		List<Recurso> recursos = this.jdbcTemplate.query(queryFinal.toString(), new RecursoRowMapper(),
				argumentos.toArray());

		for (Recurso recurso : recursos) {
			recurso.setNombreTipo(this.messageSource.getMessage("tipoRecurso." + recurso.getTipo(), null,
					LocaleContextHolder.getLocale()));

			recurso.setUrlPortal(Utilidades.getURLAmigableFichaPortal(recurso,
					this.appConfiguration.getProperty("portal.host")));
		}

		return recursos;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findByCountCriteria(com.ejie.aa80a.model.FiltroRecurso, boolean)
	 */
	@Transactional(readOnly = true)
	@Override()
	public int findByCountCriteria(FiltroRecurso filtro, boolean editables) {
		StringBuilder query = new StringBuilder();
		query.append("SELECT COUNT(DISTINCT R.COREC)");
		// Where clause & Params
		Map<String, ?> mapaWhere = getWhereMapRecursoByCriteria(filtro, editables);
		StringBuilder where = new StringBuilder();
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> argumentos = (List<?>) mapaWhere.get("params");
		return this.jdbcTemplate.queryForObject(query.toString(), argumentos.toArray(), Integer.class);

	}

	/**
	 * Devuelve un mapa con la query a ejecutar y los parámetros de búsqueda
	 * 
	 * @param filtro datos del filtro
	 * @param editables Indica el conjunto de tipos de recursos por el que buscar true: busca sobre los tipos de recurso
	 *            editables por Garkotu, false: busca sobre los tipos de recurso con acceso a Garkotu
	 * @return query a ejecutar
	 */
	private Map<String, ?> getWhereMapRecursoByCriteria(FiltroRecurso filtro, boolean editables) {
		StringBuilder query = new StringBuilder();
		List<Object> argumentos = new ArrayList<Object>();
		query.append(" FROM RECURSO R ");
		query.append(" INNER JOIN  DATOS_GENERALES DG ON R.COREC = DG.COREC AND DG.COIDIOMA = ? ");
		query.append(" LEFT JOIN POSICIONAMIENTO POS ON R.COREC = POS.COREC AND DG.COIDIOMA = POS.COIDIOMA");
		argumentos.add(LocaleContextHolder.getLocale().getLanguage());

		if (filtro.getCodProvincia() != null || filtro.getCodMunicipio() != null || filtro.getCodLocalidad() != null
				|| (filtro.getNomLocalidad() != null && !"".equals(filtro.getNomLocalidad()))
				|| (filtro.getNomMunicipio() != null && !"".equals(filtro.getNomMunicipio()))) {
			query.append(" LEFT JOIN REC_GEOCATALOG GEO ON R.COREC = GEO.COREC ");
			query.append(" LEFT JOIN T17_PROVINCIA ON T17_PROVINCIA.ID = GEO.COPROV ");
			query.append(" LEFT JOIN T17_MUNICIPIO ON T17_MUNICIPIO.ID = GEO.COMUN AND T17_MUNICIPIO.PROVINCIA_ID = GEO.COPROV ");
			query.append(" LEFT JOIN T17_LOCALIDAD ON T17_LOCALIDAD.ID = GEO.COLOC AND T17_LOCALIDAD.PROVINCIA_ID = GEO.COPROV AND T17_LOCALIDAD.MUNICIPIO_ID = GEO.COMUN ");

		}

		query.append(" WHERE 1=1 ");

		if (filtro.getCodProvincia() != null || filtro.getCodMunicipio() != null || filtro.getCodLocalidad() != null
				|| (filtro.getNomLocalidad() != null && !"".equals(filtro.getNomLocalidad()))
				|| (filtro.getNomMunicipio() != null && !"".equals(filtro.getNomMunicipio()))) {
			if (filtro.getCodProvincia() != null) {
				query.append(" AND GEO.COPROV = ? ");
				argumentos.add(filtro.getCodProvincia());
			}

			if (filtro.getCodMunicipio() != null) {
				query.append(" AND GEO.COMUN = ? ");
				argumentos.add(filtro.getCodMunicipio());
			} else if (!"".equals(filtro.getNomMunicipio())) {
				query.append(" AND UPPER(T17_MUNICIPIO.DS_O) LIKE UPPER(?) ");
				argumentos.add("%" + filtro.getNomMunicipio() + "%");
			}

			if (filtro.getCodLocalidad() != null) {
				query.append(" AND GEO.COLOC = ? ");
				argumentos.add(filtro.getCodLocalidad());
			} else if (!"".equals(filtro.getNomLocalidad())) {
				query.append(" AND UPPER(T17_LOCALIDAD.DS_O) LIKE UPPER(?) ");
				argumentos.add("%" + filtro.getNomLocalidad() + "%");
			}

		}

		// Solo incluimos los que están publicados
		query.append(" AND R.FEPUBLICACION IS NOT NULL ");

		// No incluimos los eventos (agenda, ofertas) que no estén vigentes
		query.append(" AND (R.FEPUBLICACIONPROGRAMADA IS NULL OR (R.FEPUBLICACIONPROGRAMADA < SYSDATE AND R.FEDESPUBLICACIONPROGRAMADA > SYSDATE)) ");

		if (filtro.getNombre() != null && !"".equals(filtro.getNombre())) {
			query.append(" AND UPPER(DG.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombre() + "%");
		}
		if (filtro.getTipo() != null && !"".equals(filtro.getTipo())) {
			query.append(" AND R.COTIPREC = ? ");
			argumentos.add(filtro.getTipo());
		} else {
			if (editables) {
				query.append(" AND R.COTIPREC IN (");
				query.append(RECURSOS_EDITABLES);
				query.append(") ");
			} else {
				query.append(" AND R.COTIPREC IN (");
				query.append(RECURSOS_CON_ACCESO);
				query.append(") ");
			}
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", query);
		mapWhere.put("params", argumentos);

		return mapWhere;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#existeRecurso(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public boolean existeRecurso(Long codigoRecurso) {
		Integer count = this.jdbcTemplate.queryForObject("SELECT COUNT(COREC) FROM RECURSO WHERE COREC = ?",
				Integer.class, codigoRecurso);

		if (count > 0) {
			return true;
		}

		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findDatosRecursoByCorecIdioma(java.lang.Long, java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Recurso findDatosRecursoByCorecIdioma(Long codigoRecurso, String idioma) {

		String tipoRecurso = this.jdbcTemplate.queryForObject("SELECT COTIPREC FROM RECURSO WHERE COREC = ?",
				String.class, codigoRecurso);

		// Obtenemos el nombre de la tabla sobre la que consultar en función del tipo de recurso
		String nombreTabla = this.jdbcTemplate.queryForObject("SELECT NOBDTABLE FROM TIPO_RECURSO WHERE COTIPREC = ?",
				String.class, tipoRecurso);

		StringBuilder query = new StringBuilder(
				"SELECT R.COREC, R.COTIPREC, DG.NOMBRE, R.COREGNB29, AA80A.GEO_PROVINCIA(R.COREC) AS PROVINCIA, ");
		query.append(" AA80A.GEO_MUNICIPIO(R.COREC) AS MUNICIPIO, AA80A.GEO_LOCALIDAD(R.COREC) AS LOCALIDAD, POS.URLAMIGABLE, ");
		query.append(" DG.DIRECCION, DG.CODIGOPOSTAL, DG.MAIL, DG.TELEFONO1, DG.TELEFONO2, DG.URL, DG.CODSUBTIPO ");

		if (Constants.TIPO_RECURSOS_FICHA_REDUCIDAD.contains(tipoRecurso)) {
			query.append(", MULT.ARCHIVOMULT ");
		}

		if (Constants.TABLA_M1.equals(nombreTabla)) {
			query.append(" , AGE.FECHAINICIO, AGE.FECHAFIN ");
		} else if (Constants.TABLA_M2.equals(nombreTabla)) {
			query.append(" , OFER.FECHAINICIO, OFER.FECHAFIN ");
		} else if (Constants.TABLA_A1.equals(nombreTabla)) {
			query.append(" , ALO.CATEGORIA, ALO.MODALIDAD ");
		}

		query.append(" FROM RECURSO R INNER JOIN DATOS_GENERALES DG ON R.COREC = DG.COREC AND DG.COIDIOMA = ?");
		query.append(" LEFT JOIN POSICIONAMIENTO POS ON R.COREC = POS.COREC AND DG.COIDIOMA = POS.COIDIOMA ");
		if (Constants.TABLA_A1.equals(nombreTabla)) {
			query.append(" LEFT JOIN ALOJAMIENTO ALO ON R.COREC = ALO.COREC AND DG.COIDIOMA = ALO.COIDIOMA ");
		} else if (Constants.TABLA_M1.equals(nombreTabla)) {
			query.append(" LEFT JOIN AGENDA AGE ON R.COREC = AGE.COREC AND DG.COIDIOMA = AGE.COIDIOMA ");
		} else if (Constants.TABLA_M2.equals(nombreTabla)) {
			query.append(" LEFT JOIN OFERTAS OFER ON R.COREC = OFER.COREC AND DG.COIDIOMA = OFER.COIDIOMA ");
		}

		if (Constants.TIPO_RECURSOS_FICHA_REDUCIDAD.contains(tipoRecurso)) {
			query.append(" LEFT JOIN MULTIMEDIA MULT ON R.COREC = MULT.COREC AND DG.COIDIOMA = MULT.COIDIOMA AND MULT.TIPOIMAGEN = ?");
		}

		query.append(" WHERE R.COREC = ?");

		if (Constants.TIPO_RECURSOS_FICHA_REDUCIDAD.contains(tipoRecurso)) {
			return this.jdbcTemplate.queryForObject(query.toString(), new DatosRecursoRowMapper(), idioma,
					Constants.TIPO_IMAGEN_MINIATURA_FICHA_REDUCIDA, codigoRecurso);
		}

		return this.jdbcTemplate.queryForObject(query.toString(), new DatosRecursoRowMapper(), idioma, codigoRecurso);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#actualizarDatosRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public void actualizarDatosRecurso(Recurso recurso) {
		// Obtenemos el nombre de la tabla sobre la que consultar en función del tipo de recurso
		String nombreTabla = this.jdbcTemplate.queryForObject("SELECT NOBDTABLE FROM TIPO_RECURSO WHERE COTIPREC = ?",
				String.class, recurso.getTipo());

		// Actualizamos los datos en la tabla correspondiente
		ArrayList<Object> largs = new ArrayList<Object>();
		ArrayList<Integer> largTypes = new ArrayList<Integer>();

		StringBuilder update = new StringBuilder("UPDATE DATOS_GENERALES DG");
		update.append(" SET ");

		// TODO: Se puede unificar permitiendo la modificación de todos los campos ya que existen
		// para todos los tipos
		if (Constants.TABLA_G2.equals(nombreTabla) || Constants.TABLA_D2.equals(nombreTabla)) {
			update.append(" URL = ? ");
			largs.add(recurso.getWeb());
			largTypes.add(Types.VARCHAR);
		} else if (Constants.TABLA_H6.equals(nombreTabla)) {
			update.append(" TELEFONO1 = ?, TELEFONO2 = ? ");
			largs.add(recurso.getTelefono1());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getTelefono2());
			largTypes.add(Types.VARCHAR);
		} else if (Constants.TABLA_I31.equals(nombreTabla)) {
			update.append(" MAIL = ?, TELEFONO1 = ?, TELEFONO2 = ?, URL = ? ");
			largs.add(recurso.getEmail());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getTelefono1());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getTelefono2());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getWeb());
			largTypes.add(Types.VARCHAR);
		} else if (Constants.TABLA_L2.equals(nombreTabla)) {
			update.append(" TELEFONO1 = ?, TELEFONO2 = ?, URL = ? ");
			largs.add(recurso.getTelefono1());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getTelefono2());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getWeb());
			largTypes.add(Types.VARCHAR);
		} else {
			update.append(" DG.MAIL = ?,  DG.TELEFONO1 = ?, DG.TELEFONO2 = ?, DG.URL = ? ");
			largs.add(recurso.getEmail());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getTelefono1());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getTelefono2());
			largTypes.add(Types.VARCHAR);
			largs.add(recurso.getWeb());
			largTypes.add(Types.VARCHAR);
		}

		update.append(" WHERE COREC = ? ");
		largs.add(recurso.getCodigo());
		largTypes.add(Types.NUMERIC);

		int[] argTypes = new int[largTypes.size()];
		for (int i = 0; i < argTypes.length; i++) {
			argTypes[i] = largTypes.get(i).intValue();
		}

		this.jdbcTemplate.update(update.toString(), largs.toArray(), argTypes);

		if (Constants.TABLA_M1.equals(nombreTabla)) {
			this.jdbcTemplate.update("UPDATE AGENDA SET FECHAINICIO = ?, FECHAFIN = ? WHERE COREC = ?",
					recurso.getFechaInicio(), recurso.getFechaFin(), recurso.getCodigo());
		} else if (Constants.TABLA_M2.equals(nombreTabla)) {
			this.jdbcTemplate.update("UPDATE OFERTAS SET FECHAINICIO = ?, FECHAFIN = ? WHERE COREC = ? ",
					recurso.getFechaInicio(), recurso.getFechaFin(), recurso.getCodigo());
		}

		// Para M1 y M2 posible modificación de la fecha de despublicación programada de la tabla RECURSO
		if (Constants.TABLA_M1.equals(nombreTabla) || Constants.TABLA_M2.equals(nombreTabla)) {
			this.jdbcTemplate.update("UPDATE RECURSO SET FEDESPUBLICACIONPROGRAMADA = ? WHERE COREC = ?",
					recurso.getFechaFin(), recurso.getCodigo());
		}

		// Si han modificado la imagen de la miniatura, en el caso de recursos con ficha reducida, actualizo el campo
		// con el nombre de la foto
		if (recurso.getImagen() != null && recurso.getImagen().getBytes() != null) {

			Integer count = this.jdbcTemplate.queryForObject("SELECT COUNT(COREC) FROM MULTIMEDIA "
					+ " WHERE COREC = ? AND TIPOIMAGEN = ? ", Integer.class, recurso.getCodigo(),
					Constants.TIPO_IMAGEN_MINIATURA_FICHA_REDUCIDA);

			if (count == 0) {

				Integer numOrden = this.jdbcTemplate.queryForObject(
						"SELECT NVL(MAX(ORDEN),0)+1 FROM MULTIMEDIA WHERE COREC = ?", Integer.class,
						recurso.getCodigo());

				for (String vi : VERSIONES_IDIOMATICAS) {

					Long codigoMult = this.jdbcTemplate.queryForObject("SELECT AA80T.AA80T17Q00.NEXTVAL FROM dual",
							Long.class);

					this.jdbcTemplate
							.update("INSERT INTO MULTIMEDIA (COMULT, COREC, COIDIOMA, COTIPMULT, ARCHIVOMULT, DEMULT, COUSER, FEREG, ORDEN, TIPOIMAGEN, FORMATO, PROPORCION, PUBLICABLE) VALUES (?, ?, ?, ?, ?, (SELECT NOMBRE FROM DATOS_GENERALES WHERE COREC = ? AND COIDIOMA = ?), ?, SYSDATE, ?, ?, ?, ?, 1)",
									codigoMult, recurso.getCodigo(), vi, Constants.MULTIMEDIA_IMAGEN, recurso
											.getImagen().getNombre(), recurso.getCodigo(), vi,
									Constants.USUARIO_GAURKOTU, numOrden,
									Constants.TIPO_IMAGEN_MINIATURA_FICHA_REDUCIDA, Constants.FORMATO_APAISADO,
									Constants.PROPORCION_IMAGEN_8);
				}

			} else {
				// Se modifica el campo orden para romper la posible agrupación de imágenes que se realiza en AA80T al
				// subir la misma imagen para distintos tipos
				Integer numOrden = this.jdbcTemplate.queryForObject(
						"SELECT NVL(MAX(ORDEN),0)+1 FROM MULTIMEDIA WHERE COREC = ?", Integer.class,
						recurso.getCodigo());

				for (String vi : VERSIONES_IDIOMATICAS) {
					this.jdbcTemplate
							.update("UPDATE MULTIMEDIA SET COUSER = ?, FEREG = SYSDATE, ORDEN = ?, ARCHIVOMULT = ?, DEMULT = (SELECT NOMBRE FROM DATOS_GENERALES WHERE COREC = ? AND COIDIOMA = ?) WHERE COREC = ? AND TIPOIMAGEN = ? AND COIDIOMA = ?",
									Constants.USUARIO_GAURKOTU, numOrden, recurso.getImagen().getNombre(),
									recurso.getCodigo(), vi, recurso.getCodigo(),
									Constants.TIPO_IMAGEN_MINIATURA_FICHA_REDUCIDA, vi);
				}
			}
		}

		// Se guarda la imagen, si existe, de forma temporal en la tabla TMP_IMAGEN
		if (recurso.getImagen() != null && recurso.getImagen().getBytes() != null) {

			// antes de subir una nueva imagen, borro de la tabla TMP_IMAGEN las foto que tenga asociada (en caso de
			// existir)
			this.jdbcTemplate.update("DELETE FROM TMP_IMAGEN WHERE COREC = ?", recurso.getCodigo());

			InputStream imageIs = new ByteArrayInputStream(recurso.getImagen().getBytes());
			LobHandler lobHandler = new DefaultLobHandler();

			this.jdbcTemplate.update("INSERT INTO TMP_IMAGEN (COREC, NOMBRE, CONTENTTYPE, IMAGEN) VALUES (?, ?, ?, ?)",
					new Object[] { recurso.getCodigo(), recurso.getImagen().getNombre(), recurso.getImagen().getTipo(),
							new SqlLobValue(imageIs, recurso.getImagen().getLength(), lobHandler) }, new int[] {
							Types.NUMERIC, Types.VARCHAR, Types.VARCHAR, Types.BLOB });
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findDatosRecursoServiciosByCorecIdioma(java.lang.Long, java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Recurso findDatosRecursoBasicosByCorecIdioma(Long codigoRecurso, String idioma) {
		StringBuilder query = new StringBuilder(
				"SELECT R.COREC, R.COTIPREC, DG.NOMBRE, POS.URLAMIGABLE, DG.CODSUBTIPO ");
		query.append(" FROM RECURSO R INNER JOIN DATOS_GENERALES DG ON R.COREC = DG.COREC AND DG.COIDIOMA = ?");
		query.append(" INNER JOIN POSICIONAMIENTO POS ON R.COREC = POS.COREC AND DG.COIDIOMA = POS.COIDIOMA ");
		query.append(" WHERE R.COREC = ?");

		return this.jdbcTemplate.queryForObject(query.toString(), new DatosRecursoBasicosRowMapper(), idioma,
				codigoRecurso);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findServiciosByRecursoIdiomaEditable(com.ejie.aa80a.model.Recurso,
	 * java.lang.String, int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Servicio> findServiciosByRecursoIdiomaEditable(Recurso recurso, String idioma, int editable) {
		StringBuilder query = new StringBuilder(
				"SELECT ST.COSERVICIO, SN.NOSERVICIO, NVL2(RS.COREC, 1, 0) AS SELECCIONADO");
		query.append(" FROM SRV_TREC ST INNER JOIN SRV_NAME SN ON ST.COSERVICIO = SN.COSERVICIO ");
		query.append(" LEFT JOIN REC_SRV RS ON ST.COSERVICIO = RS.COSERVICIO AND RS.COREC = ? ");
		query.append(" WHERE ST.COTIPREC = ? AND ST.SUBTIPOSREC LIKE ? AND ST.TIPOSERVICIO = ? AND SN.COIDIOMA = ? ");
		query.append(" ORDER BY SN.NOSERVICIO");

		return this.jdbcTemplate.query(query.toString(), new ServiciosRowMapper(), recurso.getCodigo(),
				recurso.getTipo(), "%" + recurso.getSubtipo() + "%", editable, idioma);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#actualizarServiciosRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public void actualizarServiciosRecurso(Recurso recurso) {
		// Borramos de la tabla REC_SRV todos aquellos servicios que son modificables desde Gaurkotu.
		StringBuilder delete = new StringBuilder("DELETE FROM REC_SRV WHERE COREC = ? AND ");
		delete.append(" COSERVICIO IN (SELECT COSERVICIO FROM SRV_TREC WHERE COTIPREC = ? AND SUBTIPOSREC LIKE ? AND TIPOSERVICIO = '1')");

		this.jdbcTemplate.update(delete.toString(), recurso.getCodigo(), recurso.getTipo(), "%" + recurso.getSubtipo()
				+ "%");

		// Insertamos todos los servicios que vienen en el objeto recurso, que corresponden con los marcados en Gaurkotu
		if (recurso.getServicios() != null) {
			for (Servicio servicio : recurso.getServicios()) {
				this.jdbcTemplate.update("INSERT INTO REC_SRV (COREC, COSERVICIO) VALUES (?, ?)", recurso.getCodigo(),
						servicio.getCodigo());
			}
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findLocalizacionByRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Localizacion findLocalizacionByRecurso(Recurso recurso) {
		// Comprobamos si existen datos de georeferenciación para el recurso (opcionales)
		Integer count = this.jdbcTemplate.queryForObject("SELECT COUNT(COREC) FROM GEOREFERENCIACION "
				+ " WHERE COREC = ?", Integer.class, recurso.getCodigo());

		if (count == 0) {
			logger.warn("El recurso no tiene datos de georeferenciación: '{}'", recurso.getCodigo());

			return null;
		}

		StringBuilder query = new StringBuilder("SELECT GMLNG, GMLAT, GMZOOM ");
		query.append(" FROM GEOREFERENCIACION ");
		query.append(" WHERE COREC = ?");

		return this.jdbcTemplate.queryForObject(query.toString(), new LocalizacionRowMapper(), recurso.getCodigo());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#actualizarLocalizacionRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public void actualizarLocalizacionRecurso(Recurso recurso) {
		// Comprobamos si existen datos de georeferenciación para el recurso (opcionales)
		Integer count = this.jdbcTemplate.queryForObject("SELECT COUNT(COREC) FROM GEOREFERENCIACION "
				+ " WHERE COREC = ?", Integer.class, recurso.getCodigo());

		if (count == 0) {
			// Hacemos un insert
			this.jdbcTemplate
					.update("INSERT INTO GEOREFERENCIACION (COREC, GMLAT, GMLNG, GMZOOM, GMMAPCLAT, GMMAPCLNG) VALUES (?, ?, ?, ?, ?, ?)",
							recurso.getCodigo(), recurso.getLocalizacion().getGmLat(), recurso.getLocalizacion()
									.getGmLng(), recurso.getLocalizacion().getGmZoom(), recurso.getLocalizacion()
									.getGmLat(), recurso.getLocalizacion().getGmLng());

		} else {
			// Hacemos un update
			this.jdbcTemplate
					.update("UPDATE GEOREFERENCIACION SET GMLAT = ?, GMLNG = ?, GMZOOM = ?, GMMAPCLAT = ?, GMMAPCLNG = ? WHERE COREC = ?",
							recurso.getLocalizacion().getGmLat(), recurso.getLocalizacion().getGmLng(), recurso
									.getLocalizacion().getGmZoom(), recurso.getLocalizacion().getGmLat(), recurso
									.getLocalizacion().getGmLng(), recurso.getCodigo());

		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findOfertasByCorec(long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Oferta> findOfertasByCorec(Long codigoRecurso) {
		StringBuilder query = new StringBuilder(
				"SELECT DISTINCT DG.NOMBRE, R.COREC, DG.COIDIOMA, O.PRECIO, O.FECHAINICIO, O.FECHAFIN ");
		query.append(" FROM RECURSO R INNER JOIN DATOS_GENERALES DG ON R.COREC = DG.COREC AND DG.COIDIOMA IN ('es', 'eu')");
		query.append(" LEFT JOIN OFERTAS O ON O.COREC = R.COREC AND O.COIDIOMA IN ('es', 'eu')");
		query.append(" WHERE O.COPARENTREC = ? ");
		query.append(" ORDER BY R.COREC, DG.COIDIOMA ");

		List<Oferta> ofertas = this.jdbcTemplate.query(query.toString(), new ListaOfertasResultSetExtractor(),
				codigoRecurso);

		return ofertas;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#crearOfertaRecurso(com.ejie.aa80a.model.Oferta)
	 */
	@Override()
	public void crearOfertaRecurso(Oferta oferta) {
		// OJO: Código altamente dependiente del modelo de datos de AA80T

		// Obtenemos un nuevo código de recurso de la secuencia asociada
		oferta.setCodigo(this.jdbcTemplate.queryForObject("SELECT AA80T.AA80T03Q00.NEXTVAL FROM dual", Long.class));

		// Se crea un nuevo registro en la tabla RECURSO
		this.jdbcTemplate
				.update("INSERT INTO RECURSO (COREC, COTIPREC, COESTADOREC, FECREACION, COUSERCREACION, FEPUBLICACIONPROGRAMADA, FEDESPUBLICACIONPROGRAMADA) VALUES (?, ?, ?, SYSDATE, ?, SYSDATE, ?)",
						oferta.getCodigo(), Constants.OFERTAS, SITUACION_RECURSO_EN_EDICION,
						Constants.USUARIO_GAURKOTU, oferta.getFechaHasta());

		Integer numOrden = this.jdbcTemplate.queryForObject(
				"SELECT NVL(MAX(ORDEN),0)+1 FROM MULTIMEDIA WHERE COREC = ?", Integer.class, oferta.getCodigo());

		oferta.getImagen().setOrden(numOrden);

		// Se crean tantos registros en las tablas VERSION_IDIOMATICA, DATOS_GENERALES, POSICIONAMIENTO, TOP_EUSKADI,
		// MULTIMEDIA y OFERTAS como versiones idiomáticas se contemplan en AA80T (actualmente 5)
		for (String vi : VERSIONES_IDIOMATICAS) {
			this.crearVersionIdiomaticaOferta(vi, oferta);
		}

		this.jdbcTemplate
				.update("INSERT INTO HISTORICO (COREC, COIDIOMA, COACCION, COESTADO, COESTADOREC, FEREGISTRO, COUSER) VALUES (?, ?, ?, ?, ?, SYSDATE, ?)",
						oferta.getCodigo(), null, Constants.ACCION_CREACION_RECURSO, null,
						Constants.ESTADO_RECURSO_EN_EDICION, Constants.USUARIO_GAURKOTU);

		// Se guarda la imagen, si existe, de forma temporal en la tabla TMP_IMAGEN
		if (oferta.getImagen() != null && oferta.getImagen().getBytes() != null) {
			InputStream imageIs = new ByteArrayInputStream(oferta.getImagen().getBytes());
			LobHandler lobHandler = new DefaultLobHandler();

			this.jdbcTemplate.update("INSERT INTO TMP_IMAGEN (COREC, NOMBRE, CONTENTTYPE, IMAGEN) VALUES (?, ?, ?, ?)",
					new Object[] { oferta.getCodigo(), oferta.getImagen().getNombre(), oferta.getImagen().getTipo(),
							new SqlLobValue(imageIs, oferta.getImagen().getLength(), lobHandler) }, new int[] {
							Types.NUMERIC, Types.VARCHAR, Types.VARCHAR, Types.BLOB });
		}

		// Guardamos los datos de la localización en la tabla REC_GEOCATALOG
		this.guardarGeoCatalogOferta(oferta);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#actualizarOfertaRecurso(com.ejie.aa80a.model.Oferta)
	 */
	@Override()
	public void actualizarOfertaRecurso(Oferta oferta) {
		// OJO: Código altamente dependiente del modelo de datos de AA80T

		// Obtenemos las VI que existen para esa oferta. Al menos deben existir 2 (es y eu).
		List<String> vis = this.jdbcTemplate.queryForList("SELECT COIDIOMA FROM OFERTAS WHERE COREC = ?", String.class,
				oferta.getCodigo());

		String nombre = "";
		String descripcion = "";

		for (String vi : vis) {
			if (Constants.CASTELLANO.equals(vi)) {
				nombre = oferta.getNombreCastellano();
				descripcion = oferta.getDescripcionCastellano();
			} else if (Constants.EUSKERA.equals(vi)) {
				nombre = oferta.getNombreEuskera();
				descripcion = oferta.getDescripcionEuskera();
			} else if (Constants.INGLES.equals(vi)) {
				nombre = oferta.getNombreIngles();
				descripcion = oferta.getDescripcionIngles();
			} else if (Constants.FRANCES.equals(vi)) {
				nombre = oferta.getNombreFrances();
				descripcion = oferta.getDescripcionFrances();
			} else if (Constants.ALEMAN.equals(vi)) {
				nombre = oferta.getNombreAleman();
				descripcion = oferta.getDescripcionAleman();
			}

			// Actualización de las VIs existentes
			this.jdbcTemplate.update(
					"UPDATE OFERTAS SET FECHAINICIO = ?, FECHAFIN = ?, PRECIO = ? WHERE COREC = ? AND COIDIOMA = ?",
					oferta.getFechaDesde(), oferta.getFechaHasta(), oferta.getPrecio(), oferta.getCodigo(), vi);
			this.jdbcTemplate
					.update("UPDATE DATOS_GENERALES SET NOMBRE = ?, DESCRIPCION = ?, MAIL = ?, URL = ? WHERE COREC = ? AND COIDIOMA = ?",
							nombre, descripcion, oferta.getEmail(), oferta.getWeb(), oferta.getCodigo(), vi);
			this.jdbcTemplate
					.update("UPDATE MULTIMEDIA SET ARCHIVOMULT = ?, DEMULT = ? WHERE COREC = ? AND COIDIOMA = ? AND TIPOIMAGEN = ?",
							oferta.getImagen().getNombre(), nombre, oferta.getCodigo(), vi,
							Constants.TIPO_IMAGEN_GALERIA);
		}

		// Posible actualización de la fecha de despublicación en la tabla RECURSO y actualización de la fecha de
		// modificación
		this.jdbcTemplate
				.update("UPDATE RECURSO SET FEDESPUBLICACIONPROGRAMADA = ?, FEACTUALIZACION = SYSDATE, COUSERACTUALIZACION = ? WHERE COREC = ?",
						oferta.getFechaHasta(), Constants.USUARIO_GAURKOTU, oferta.getCodigo());

		// Comprobamos si hay que crear alguna VI
		if (vis.size() < MAX_VERSIONES_IDIOMATICAS) {

			logger.warn("Creadas VIs para el recurso: '{}'", oferta.getCodigo());

			Integer numOrden = this.jdbcTemplate.queryForObject(
					"SELECT ORDEN WHERE COREC = ?, COIDOIMA = ?, COTIPMULT = ?, TIPOIMAGEN = ?", Integer.class,
					oferta.getCodigo(), Constants.CASTELLANO, Constants.MULTIMEDIA_IMAGEN,
					Constants.TIPO_IMAGEN_PRINCIPAL);

			oferta.getImagen().setOrden(numOrden);

			if (!vis.contains(Constants.INGLES)
					&& ((oferta.getNombreIngles() != null && !"".equals(oferta.getNombreIngles())) || (oferta
							.getDescripcionIngles() != null) && !"".equals(oferta.getDescripcionIngles()))) {
				// hay que crear la vi de inglés
				crearVersionIdiomaticaOferta(Constants.INGLES, oferta);
			}

			if (!vis.contains(Constants.FRANCES)
					&& ((oferta.getNombreFrances() != null && !"".equals(oferta.getNombreFrances())) || (oferta
							.getDescripcionFrances() != null) && !"".equals(oferta.getDescripcionFrances()))) {
				// hay que crear la vi de francés
				crearVersionIdiomaticaOferta(Constants.FRANCES, oferta);
			}

			if (!vis.contains(Constants.ALEMAN)
					&& ((oferta.getNombreAleman() != null && !"".equals(oferta.getNombreAleman())) || (oferta
							.getDescripcionAleman() != null) && !"".equals(oferta.getDescripcionAleman()))) {
				// hay que crear la vi de alemán
				crearVersionIdiomaticaOferta(Constants.ALEMAN, oferta);

			}

			// TODO: Evitar que falle AA80T al crear VIs sin el correspondiente documento en el Gestor de Contenidos
			// Se puede asumir que en Producción siempre van a existir las 5 VIs.
		}

		// Se guarda la imagen, si existe, de forma temporal en la tabla TMP_IMAGEN
		if (oferta.getImagen() != null && oferta.getImagen().getBytes() != null) {

			// antes de subir una nueva imagen, borro de la tabla TMP_IMAGEN las foto que tenga asociada (en caso de
			// existir)
			this.jdbcTemplate.update("DELETE FROM TMP_IMAGEN WHERE COREC = ?", oferta.getCodigo());

			InputStream imageIs = new ByteArrayInputStream(oferta.getImagen().getBytes());
			LobHandler lobHandler = new DefaultLobHandler();

			this.jdbcTemplate.update("INSERT INTO TMP_IMAGEN (COREC, NOMBRE, CONTENTTYPE, IMAGEN) VALUES (?, ?, ?, ?)",
					new Object[] { oferta.getCodigo(), oferta.getImagen().getNombre(), oferta.getImagen().getTipo(),
							new SqlLobValue(imageIs, oferta.getImagen().getLength(), lobHandler) }, new int[] {
							Types.NUMERIC, Types.VARCHAR, Types.VARCHAR, Types.BLOB });
		}

		// guardamos los datos de la localización en la tabla REC_GEOCATALOG
		this.guardarGeoCatalogOferta(oferta);

	}

	/**
	 * Guarda los datos de la geocatalogación de una oferta en la tabla REC_GEOCATALOG
	 * 
	 * @param oferta datos de la oferta
	 */
	private void guardarGeoCatalogOferta(Oferta oferta) {
		// guardamos los datos de la localización en la tabla REC_GEOCATALOG
		this.jdbcTemplate.update("DELETE FROM REC_GEOCATALOG WHERE COREC = ?", oferta.getCodigo());

		// si ha seleccionado al menos la provincia, guardo los datos en la tabla de geocatalogación
		if (oferta.getCodigoProvincia() != null) {
			StringBuilder query = new StringBuilder();
			query.append("INSERT INTO REC_GEOCATALOG ");
			query.append(" ( ");
			query.append(" 		COREC, COGEOCATALOG, COPROV, COMUN, COLOC, ORDEN");
			query.append(" ) VALUES ( ");
			query.append("?, ?, ?, ?, ?, 1");
			query.append(" ");
			query.append(" ) ");

			// genero el código para el id de la fila
			StringBuilder cogeocatalog = new StringBuilder();
			cogeocatalog.append(String.format("%02d", oferta.getCodigoProvincia()));
			if (oferta.getCodigoMunicipio() != null) {
				cogeocatalog.append(String.format("%03d", Integer.parseInt(oferta.getCodigoMunicipio())));

			} else {
				cogeocatalog.append("000");
			}
			if (oferta.getCodigoLocalidad() != null) {
				cogeocatalog.append(String.format("%015d", oferta.getCodigoLocalidad()));
			} else {
				cogeocatalog.append("000000000000000");
			}

			this.jdbcTemplate.update(query.toString(), oferta.getCodigo(), cogeocatalog,
					String.format("%02d", oferta.getCodigoProvincia()), oferta.getCodigoMunicipio(),
					oferta.getCodigoLocalidad());
		}
	}

	/**
	 * Crea en la BBDD de AA80T una version idiomática de una oferta.
	 * 
	 * @param idioma Idioma de la versión idiomática [es|eu|en|fr|de]
	 * @param oferta {@link Oferta} objeto con los datos de la oferta
	 */
	private void crearVersionIdiomaticaOferta(String idioma, Oferta oferta) {
		String nombre = "";
		String descripcion = "";
		String situacion = "";

		// Obtener el tipo de recurso del padre a partir de su código (COPARENTREC -> tabla OFERTAS) ya que
		// ahora no se guarda en la tabla

		String tipoOferta = "";

		if (Constants.ALOJAMIENTO.equals(oferta.getTipoPadre())) {
			tipoOferta = TIPO_OFERTA_ALOJAMIENTO;
		} else if (Constants.RESTAURACION.equals(oferta.getTipoPadre())) {
			tipoOferta = TIPO_OFERTA_RESTAURANTE;
		} else if (Constants.OFICINA_TURISMO.equals(oferta.getTipoPadre())) {
			tipoOferta = TIPO_OFERTA_OFICINA_TURISMO;
		}

		if (Constants.CASTELLANO.equals(idioma)) {
			nombre = oferta.getNombreCastellano();
			descripcion = oferta.getDescripcionCastellano();
			// La VI de castellano (es) se pone en estado de "En edición" (EE) el resto a
			// "Pendiente de traducción" (PT)
			situacion = SITUACION_VI_EN_EDICION;
		} else if (Constants.EUSKERA.equals(idioma)) {
			nombre = oferta.getNombreEuskera();
			descripcion = oferta.getDescripcionEuskera();
			situacion = SITUACION_VI_PENDIENTE_TRADUCCION;
		} else if (Constants.INGLES.equals(idioma)) {
			nombre = oferta.getNombreIngles();
			descripcion = oferta.getDescripcionIngles();
			situacion = SITUACION_VI_PENDIENTE_TRADUCCION;
		} else if (Constants.FRANCES.equals(idioma)) {
			nombre = oferta.getNombreFrances();
			descripcion = oferta.getDescripcionFrances();
			situacion = SITUACION_VI_PENDIENTE_TRADUCCION;
		} else if (Constants.ALEMAN.equals(idioma)) {
			nombre = oferta.getNombreAleman();
			descripcion = oferta.getDescripcionAleman();
			situacion = SITUACION_VI_PENDIENTE_TRADUCCION;
		}

		// Tabla VERSION_IDIOMATICA
		this.jdbcTemplate.update(
				"INSERT INTO VERSION_IDIOMATICA (COREC, COIDIOMA, COESTADO, FEMODIFICACION) VALUES (?, ?, ?, SYSDATE)",
				oferta.getCodigo(), idioma, situacion);

		// Tabla OFERTAS
		this.jdbcTemplate
				.update("INSERT INTO OFERTAS (COREC, COIDIOMA, FECHAINICIO, FECHAFIN, PRECIO, COPARENTREC) VALUES (?, ?, ?, ?, ?, ?)",
						oferta.getCodigo(), idioma, oferta.getFechaDesde(), oferta.getFechaHasta(), oferta.getPrecio(),
						oferta.getCodigoPadre());

		// Tabla DATOS_GENERALES
		this.jdbcTemplate
				.update("INSERT INTO DATOS_GENERALES (COREC, COIDIOMA, COTIPREC, NOMBRE, NOMBREPT, CODSUBTIPO, DESCRIPCION, DESCRIPCIONPT, DIRECCION, MAIL, TELEFONO1, DESCTELEFONO1PT, DESCTELEFONO2PT, DESCTELEFONO3PT, URL, ENTRADILLAPT, OBSERVACIONESPT) VALUES (?, ?, ?, ?, 0, ?, ?, 0, ?, ?, ?, 0, 0, 0, ?, 0, 0)",
						oferta.getCodigo(), idioma, Constants.OFERTAS, nombre, tipoOferta, descripcion,
						oferta.getDireccion(), oferta.getEmail(), oferta.getTelefono(), oferta.getWeb());

		// Tabla MULTIMEDIA
		Long codigoMult = this.jdbcTemplate.queryForObject("SELECT AA80T.AA80T17Q00.NEXTVAL FROM dual", Long.class);

		this.jdbcTemplate
				.update("INSERT INTO MULTIMEDIA (COMULT, COREC, COIDIOMA, COTIPMULT, ARCHIVOMULT, DEMULT, ORDEN, TIPOIMAGEN, FORMATO, PROPORCION, PUBLICABLE) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1)",
						codigoMult, oferta.getCodigo(), idioma, Constants.MULTIMEDIA_IMAGEN, oferta.getImagen()
								.getNombre(), nombre, oferta.getImagen().getOrden(), Constants.TIPO_IMAGEN_GALERIA,
						Constants.FORMATO_APAISADO, Constants.PROPORCION_IMAGEN_8);

		// Tabla POSICIONAMIENTO
		this.jdbcTemplate.update("INSERT INTO POSICIONAMIENTO (COREC, COIDIOMA) VALUES (?, ?)", oferta.getCodigo(),
				idioma);

		// Tabla TOP_EUSKADI
		this.jdbcTemplate.update("INSERT INTO TOP_EUSKADI (COREC, COIDIOMA) VALUES (?, ?)", oferta.getCodigo(), idioma);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findOfertaByCorec(java.lang.Long)
	 */
	@Override()
	public Oferta findOfertaByCorec(Long codigoOferta) {
		StringBuilder query = new StringBuilder(
				"SELECT R.COREC, DG.COIDIOMA, DG.NOMBRE, DG.DESCRIPCION, O.PRECIO, DG.MAIL, MULT.ARCHIVOMULT, O.COPARENTREC, DG.URL, O.FECHAINICIO, O.FECHAFIN, DG.DIRECCION, DG.TELEFONO1,");
		query.append(" C.COPROV, T17_PROVINCIA.DS_O AS DS_PROV, T17_PROVINCIA.DS_E AS DSE_PROV, C.COMUN, T17_MUNICIPIO.DS_O AS DS_MUNI, C.COLOC, T17_LOCALIDAD.DS_O AS DS_LOC");
		query.append(" FROM RECURSO R INNER JOIN DATOS_GENERALES DG ON R.COREC = DG.COREC AND DG.COIDIOMA IN ('es', 'eu', 'en', 'fr', 'de')");
		query.append(" LEFT JOIN OFERTAS O ON O.COREC = R.COREC AND O.COIDIOMA = ?");
		query.append(" LEFT JOIN MULTIMEDIA MULT ON MULT.COREC = R.COREC AND DG.COIDIOMA = MULT.COIDIOMA  AND MULT.TIPOIMAGEN = ?");
		query.append(" LEFT JOIN REC_GEOCATALOG C ON R.COREC = C.COREC");
		query.append(" LEFT JOIN T17_PROVINCIA ON T17_PROVINCIA.ID = C.COPROV");
		query.append(" LEFT JOIN T17_MUNICIPIO ON T17_MUNICIPIO.ID = C.COMUN AND T17_MUNICIPIO.PROVINCIA_ID = C.COPROV");
		query.append(" LEFT JOIN T17_LOCALIDAD ON T17_LOCALIDAD.ID = C.COLOC AND T17_LOCALIDAD.PROVINCIA_ID = C.COPROV AND T17_LOCALIDAD.MUNICIPIO_ID = C.COMUN");
		query.append(" WHERE R.COREC = ?");

		return this.jdbcTemplate.query(query.toString(), new OfertaResultSetExtractor(), Constants.CASTELLANO,
				Constants.TIPO_IMAGEN_GALERIA, codigoOferta);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#eliminarOfertaRecurso(java.lang.Long)
	 */
	@Override()
	public void eliminarOfertaRecurso(Long codigoOferta) {
		// Borrado de las tablas dependientes:
		// eliminar NOTIFICACIONES
		this.jdbcTemplate.update("DELETE FROM NOTIFICACIONES WHERE COREC = ? ", codigoOferta);

		// eliminar ORDENES_TRABAJO
		this.jdbcTemplate
				.update("DELETE FROM ORDENES_TRABAJO WHERE CODORDEN IN (SELECT CODORDEN FROM RECURSOS_ORDENES_TRABAJO WHERE COREC = ? )",
						codigoOferta);

		// Borrado físico del recurso en la tabla RECURSO
		this.jdbcTemplate.update("DELETE FROM RECURSO WHERE COREC = ?", codigoOferta);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findMultimediasByCorec(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Multimedia> findMultimediasByCorec(Long codigoRecurso) {
		StringBuilder query = new StringBuilder(
				"SELECT M.COMULT, R.COREC, R.COTIPREC, M.ARCHIVOMULT, M.DEMULT, M.COUSER ");
		query.append(" FROM RECURSO R INNER JOIN MULTIMEDIA M ON R.COREC = M.COREC ");
		query.append(" WHERE M.COREC = ? ");

		// Solo obtenemos la VI de castellano
		query.append(" AND M.COIDIOMA = 'es' ");

		query.append(" AND M.COTIPMULT = ? ");
		query.append(" ORDER BY M.ORDEN ");

		List<Multimedia> multimedias = this.jdbcTemplate.query(query.toString(), new MultimediaRowMapper(),
				codigoRecurso, Constants.MULTIMEDIA_IMAGEN);

		return multimedias;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#actualizarRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public void actualizarRecurso(Recurso recurso) {
		this.jdbcTemplate.update(
				"UPDATE RECURSO SET FEACTUALIZACION = SYSDATE, COUSERACTUALIZACION = ? WHERE COREC = ?",
				Constants.USUARIO_GAURKOTU, recurso.getCodigo());

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#guardarMultimedia(com.ejie.aa80a.model.Multimedia)
	 */
	@Override()
	public void guardarMultimedia(Multimedia multimedia) {
		// OJO: Código altamente dependiente del modelo de datos de AA80T

		// Obtenemos el número de orden correspondiente al multimedia
		// Los archivos multimedia añadidos desde Gaurkotu se ponen los últimos en la galería
		Integer numOrden = this.jdbcTemplate
				.queryForObject("SELECT NVL(MAX(ORDEN),0)+1 FROM MULTIMEDIA WHERE COREC = ?", Integer.class,
						multimedia.getCodRecurso());

		// Se crean tantos registros en la tabla MULTIMEDIA
		// versiones idiomáticas se contemplan en AA80T (actualmente 5)
		for (String vi : VERSIONES_IDIOMATICAS) {

			// Obtenemos un nuevo código de multimedia de la secuencia asociada
			multimedia.setCodigo(this.jdbcTemplate.queryForObject("SELECT AA80T.AA80T17Q00.NEXTVAL FROM dual",
					Long.class));

			this.jdbcTemplate
					.update("INSERT INTO MULTIMEDIA (COMULT,COREC,COIDIOMA,COTIPMULT,ARCHIVOMULT,DEMULT,COUSER,FEREG,ORDEN,FORMATO,PROPORCION,TIPOIMAGEN) VALUES(?,?,?,?,?,(SELECT NOMBRE FROM DATOS_GENERALES WHERE COREC = ? AND COIDIOMA = ?),?,SYSDATE,?,?,?,?)",
							multimedia.getCodigo(), multimedia.getCodRecurso(), vi, Constants.MULTIMEDIA_IMAGEN,
							multimedia.getImagen().getNombre(), multimedia.getCodRecurso(), vi,
							Constants.USUARIO_GAURKOTU, numOrden, Constants.FORMATO_APAISADO,
							Constants.PROPORCION_IMAGEN_8, Constants.TIPO_IMAGEN_GALERIA);
		}

		// Se guarda la imagen, si existe, de forma temporal en la tabla TMP_IMAGEN
		if (multimedia.getImagen() != null && multimedia.getImagen().getBytes() != null) {
			InputStream imageIs = new ByteArrayInputStream(multimedia.getImagen().getBytes());
			LobHandler lobHandler = new DefaultLobHandler();

			this.jdbcTemplate.update("INSERT INTO TMP_IMAGEN (COREC, NOMBRE, CONTENTTYPE, IMAGEN) VALUES (?, ?, ?, ?)",
					new Object[] { multimedia.getCodRecurso(), multimedia.getImagen().getNombre(),
							multimedia.getImagen().getTipo(),
							new SqlLobValue(imageIs, multimedia.getImagen().getLength(), lobHandler) }, new int[] {
							Types.NUMERIC, Types.VARCHAR, Types.VARCHAR, Types.BLOB });
		}

		// actualización de la fecha de modificación de la tabla recurso
		this.jdbcTemplate.update(
				"UPDATE RECURSO SET FEACTUALIZACION = SYSDATE, COUSERACTUALIZACION = ? WHERE COREC = ?",
				Constants.USUARIO_GAURKOTU, multimedia.getCodRecurso());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#eliminarMultimediaRecurso(com.ejie.aa80a.model.Multimedia)
	 */
	@Override()
	public void eliminarMultimediaRecurso(Multimedia multimedia) {
		// OJO: Código altamente dependiente del modelo de datos de AA80T
		// Elimino el elemento multimedia en todos los idiomas
		Integer numOrden = this.jdbcTemplate.queryForObject(
				"SELECT ORDEN FROM MULTIMEDIA WHERE COREC = ? AND COMULT = ?", Integer.class,
				multimedia.getCodRecurso(), multimedia.getCodigo());

		this.jdbcTemplate.update("DELETE FROM MULTIMEDIA WHERE COREC = ? AND ORDEN = ? AND COTIPMULT = ?",
				multimedia.getCodRecurso(), numOrden, Constants.MULTIMEDIA_IMAGEN);

		// Actualización de la fecha de modificación de la tabla recurso
		this.jdbcTemplate.update(
				"UPDATE RECURSO SET FEACTUALIZACION = SYSDATE, COUSERACTUALIZACION = ? WHERE COREC = ?",
				Constants.USUARIO_GAURKOTU, multimedia.getCodRecurso());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#exiteTmpFile(java.lang.String, java.lang.Long)
	 */
	@Override()
	public Long exiteTmpFile(String nombre, Long codRecurso) {
		StringBuilder query = new StringBuilder("SELECT COUNT(*) FROM TMP_IMAGEN ");

		if (nombre != null) {// si estoy buscando el archivo por nombre de archivo multimedia (en el caso de multimedia)
			query.append(" WHERE COREC = ? AND NOMBRE = ?");
			return this.jdbcTemplate.queryForObject(query.toString(), Long.class, codRecurso, nombre);

		} else {// si estoy buscando el archivo por código de recurso (en el caso de ofertas)
			query.append(" WHERE COREC = ?");
			return this.jdbcTemplate.queryForObject(query.toString(), Long.class, codRecurso);

		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#getImagenById(java.lang.Long, java.lang.Long)
	 */
	@Override()
	public Imagen getImagenById(Long codMultimedia, Long codRecurso) {

		String sqlWhere = "";
		Long codigo;
		if (codMultimedia != null) {// si estoy buscando el archivo por código de multimedia (en el caso de multimedia)
			codigo = codMultimedia;
			sqlWhere = "WHERE COMULT = ?";
		} else {// si estoy buscando el archivo por código de recurso (en el caso de ofertas)
			codigo = codRecurso;
			sqlWhere = "WHERE COREC = ?";
		}

		StringBuilder query = new StringBuilder("SELECT NOMBRE, IMAGEN FROM TMP_IMAGEN ").append(sqlWhere);
		return this.jdbcTemplate.queryForObject(query.toString(), new ImagenRowMapper(), codigo);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#getComboTiposPrecio(java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Combo> getComboTiposPrecio(String subTipo) {
		StringBuilder query = new StringBuilder();
		query.append(" SELECT COPRECIO, ");

		if (Constants.CASTELLANO.equals(LocaleContextHolder.getLocale().getLanguage())) {
			query.append(" NOPRECIO_ES NOMBRE ");
		} else {
			query.append(" NOPRECIO_EU NOMBRE ");
		}

		query.append(" FROM PRECIO ");

		query.append(" WHERE COTIPO LIKE ? ");
		query.append(" ORDER BY ORDEN ");

		List<Combo> tipoPrecios = this.jdbcTemplate.query(query.toString(), new TipoPrecioRowMapper(), "%" + subTipo
				+ "%");

		return tipoPrecios;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findPreciosByCorec(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Precio> findPreciosByCorec(Long codigoRecurso) {
		StringBuilder query = new StringBuilder();
		query.append(" SELECT RP.ID, RP.COREC, RP.COPRECIO, RP.CAPACIDAD, RP.PRECIOMINIMO, ");
		query.append(" RP.PRECIOMAXIMO, RP.OTROADULTO, RP.OTROINFANTIL, RP.OTROTIENDA, RP.OTROAUTOMOVIL, RP.OTROCONEXELECTRICA, ");
		if (Constants.CASTELLANO.equals(LocaleContextHolder.getLocale().getLanguage())) {
			query.append(" P.NOPRECIO_ES NOMPRECIO ");
		} else {
			query.append(" P.NOPRECIO_EU NOMPRECIO ");
		}

		query.append(" FROM REC_PRECIO RP ");
		query.append(" INNER JOIN PRECIO P ON P.COPRECIO = RP.COPRECIO ");
		query.append(" WHERE RP.COREC = ? ");
		query.append(" ORDER BY P.ORDEN ");

		List<Precio> precios = this.jdbcTemplate.query(query.toString(), new PrecioRowMapper(), codigoRecurso);

		return precios;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#actualizarPreciosRecurso(com.ejie.aa80a.model.Recurso)
	 */
	@Override()
	public void actualizarPreciosRecurso(Recurso recurso) {
		// Borramos de la tabla REC_PRECIO todos los precios del recurso
		StringBuilder delete = new StringBuilder("DELETE FROM REC_PRECIO WHERE COREC = ? ");

		this.jdbcTemplate.update(delete.toString(), recurso.getCodigo());

		StringBuilder insert = new StringBuilder();
		insert.append("INSERT INTO REC_PRECIO ");
		insert.append("(");
		insert.append(" ID, COREC, COPRECIO, CAPACIDAD, PRECIOMINIMO, PRECIOMAXIMO, OTROADULTO, OTROINFANTIL, OTROTIENDA, OTROAUTOMOVIL, OTROCONEXELECTRICA ");
		insert.append(") VALUES ( ");
		insert.append(" ?,?,?,?,?,?,?,?,?,?,? ");
		insert.append(" ) ");

		// Insertamos todos los precios que hay en pantalla
		for (Precio precio : recurso.getPrecios()) {
			List<Object> params = new ArrayList<Object>();
			params.add(this.jdbcTemplate.queryForObject("SELECT AA80T.AA80TY3Q00.NEXTVAL FROM dual", Long.class));
			params.add(recurso.getCodigo());
			params.add(precio.getCodigoPrecio());
			params.add(precio.getCapacidad());
			params.add(precio.getPrecioMinimo());
			params.add(precio.getPrecioMaximo());
			params.add(precio.getParcelaAdulto());
			params.add(precio.getParcelaInfantil());
			params.add(precio.getParcelaTienda());
			params.add(precio.getParcelaAutomovil());
			params.add(precio.getParcelaConexionElectrica());

			this.jdbcTemplate.update(insert.toString(), params.toArray());
		}

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#getIsRecursoTop(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Boolean getIsRecursoTop(Long codigoRecurso) {
		// Obtenemos el campo TOP que determina si se trata de un recurso TOP
		Integer num = this.jdbcTemplate.queryForObject("SELECT TOP FROM RECURSO WHERE COREC = ?", Integer.class,
				codigoRecurso);

		return (num > 0);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#findCondicionesOfertaByCorec(java.lang.Long)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Condicion> findCondicionesOfertaByCorec(Long codigoRecurso) {
		StringBuilder query = new StringBuilder();
		List<Object> params = new ArrayList<Object>();

		query.append(" SELECT ORDEN, MAX(ES) ES, MAX(EU) EU, MAX(EN) EN, MAX(FR) FR, MAX(DE) DE FROM ");
		query.append(" ( ");

		query.append("  	SELECT ORDEN, TEXTOCONDICION AS ES, NULL AS EU, NULL AS EN, NULL AS FR, NULL AS DE FROM CONDICIONES WHERE COIDIOMA = ? AND COREC = ? ");

		params.add(Constants.CASTELLANO);
		params.add(codigoRecurso);

		query.append("  	UNION ");
		query.append("  	SELECT ORDEN, NULL AS ES, TEXTOCONDICION AS EU, NULL AS EN, NULL AS FR, NULL AS DE FROM CONDICIONES WHERE COIDIOMA = ? AND COREC = ? ");

		params.add(Constants.EUSKERA);
		params.add(codigoRecurso);

		query.append("  	UNION ");
		query.append("  	SELECT ORDEN, NULL AS ES, NULL AS EU, TEXTOCONDICION AS EN, NULL AS FR, NULL AS DE FROM CONDICIONES WHERE COIDIOMA = ? AND COREC = ? ");

		params.add(Constants.INGLES);
		params.add(codigoRecurso);

		query.append("  	UNION ");
		query.append("  	SELECT ORDEN, NULL AS ES, NULL AS EU, NULL AS EN, TEXTOCONDICION AS FR, NULL AS DE FROM CONDICIONES WHERE COIDIOMA = ? AND COREC = ? ");

		params.add(Constants.FRANCES);
		params.add(codigoRecurso);

		query.append("  	UNION ");
		query.append("  	SELECT ORDEN, NULL AS ES, NULL AS EU, NULL AS EN, NULL AS FR, TEXTOCONDICION AS DE FROM CONDICIONES WHERE COIDIOMA = ? AND COREC = ? ");

		params.add(Constants.ALEMAN);
		params.add(codigoRecurso);

		query.append(" ) ");
		query.append(" GROUP BY ORDEN ");
		query.append(" ORDER BY ORDEN ");

		return this.jdbcTemplate.query(query.toString(), new CondicionRowMapper(), params.toArray());

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#altaCondicionIdioma(com.ejie.aa80a.model.Condicion)
	 */
	@Override()
	public void altaCondicionIdioma(Condicion condicion) {
		// Obtenemos un nuevo código de condición de la secuencia asociada
		condicion
				.setCodigo(this.jdbcTemplate.queryForObject("SELECT AA80T.AA80TM3Q00.NEXTVAL FROM dual", String.class));
		this.jdbcTemplate
				.update("INSERT INTO CONDICIONES (COCONDICION, COREC, COIDIOMA, TEXTOCONDICION, ORDEN, COUSER, FREG) VALUES(?,?,?,?,?,?,SYSDATE)",
						condicion.getCodigo(), condicion.getCodRecurso(), condicion.getIdioma(), condicion.getTexto(),
						condicion.getOrden(), Constants.USUARIO_GAURKOTU);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.RecursoDao#eliminarCondicionesOferta(java.lang.Long)
	 */
	@Override()
	public void eliminarCondicionesOferta(Long codigoRecurso) {
		// OJO: Código altamente dependiente del modelo de datos de AA80T

		// Elimino las condiciones de una oferta
		this.jdbcTemplate.update("DELETE FROM CONDICIONES WHERE COREC = ?", codigoRecurso);

	}

}
