package com.ejie.aa80a.dao;

import java.util.List;

import com.ejie.aa80a.model.Combo;
import com.ejie.aa80a.model.Condicion;
import com.ejie.aa80a.model.FiltroRecurso;
import com.ejie.aa80a.model.Imagen;
import com.ejie.aa80a.model.Localizacion;
import com.ejie.aa80a.model.Multimedia;
import com.ejie.aa80a.model.Oferta;
import com.ejie.aa80a.model.Precio;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.model.Servicio;

/**
 * Interfaz de acceso a la base de datos para obtener datos de recurso. Tablas: RECURSO, TIPO_RECURSO, OFICINASTURISMO,
 * ALOJAMIENTO,...
 * 
 *  
 */
public interface RecursoDao {

	/**
	 * Busca el nombre un recurso en función del código, del tipo y del idioma
	 * 
	 * @param codigoRecurso Código del recurso
	 * @param tipoRecurso Tipo del recurso
	 * @param idioma Idioma
	 * 
	 * @return Nombre del recurso en función de los criterios de búsqueda
	 */
	String findNombreByCorecTipoIdioma(Long codigoRecurso, String tipoRecurso, String idioma);

	/**
	 * Obtiene los recursos que cumplen el filtro proporcionado.
	 * 
	 * @param filtro {@link FiltroRecurso} objeto con el filtro a cumplir.
	 * @param editables Indica el conjunto de tipos de recursos por el que buscar true: busca sobre los tipos de recurso
	 *            editables por Garkotu, false: busca sobre los tipos de recurso con acceso a Garkotu
	 * 
	 * @return List<Recurso> Devuelve el listado de recursos que cumplen con el filtro.
	 */
	List<Recurso> findByCriteria(FiltroRecurso filtro, boolean editables);

	/**
	 * Obtiene el número de recursos que cumplen el filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * @param editables Indica el conjunto de tipos de recursos por el que buscar true: busca sobre los tipos de recurso
	 *            editables por Garkotu, false: busca sobre los tipos de recurso con acceso a Garkotu
	 * 
	 * @return int número de recursos que cumplen el filtro aplicado.
	 */
	int findByCountCriteria(FiltroRecurso filtro, boolean editables);

	/**
	 * Comprueba si existe el recurso.
	 * 
	 * @param codigoRecurso Código de recurso
	 * 
	 * @return true, si el recurso existe
	 */
	boolean existeRecurso(Long codigoRecurso);

	/**
	 * Obtiene los datos del recurso en función del código de recurso y del idioma.
	 * 
	 * @param codigoRecurso Código de recurso
	 * @param idioma Idioma en el que se obtendrán los datos
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso.
	 */
	Recurso findDatosRecursoByCorecIdioma(Long codigoRecurso, String idioma);

	/**
	 * Guarda los datos del recurso modificados en Base de datos.
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 */
	void actualizarDatosRecurso(Recurso recurso);

	/**
	 * Obtiene los datos del recurso básicos en función del código de recurso y del idioma.
	 * 
	 * @param codigoRecurso Código de recurso
	 * @param idioma Idioma en el que se obtendrán los datos
	 * 
	 * @return {@link Recurso} objeto con los datos del recurso.
	 */
	Recurso findDatosRecursoBasicosByCorecIdioma(Long codigoRecurso, String idioma);

	/**
	 * Obtiene los servicios del recurso en función del idioma.
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * @param idioma Idioma en el que se obtendrán los datos
	 * @param editable Indica si los servicios son editables (1 - servicios) o no editables (0 - características)
	 * 
	 * @return Lista de objetos {@link Servicio} con los servicios del recurso.
	 */
	List<Servicio> findServiciosByRecursoIdiomaEditable(Recurso recurso, String idioma, int editable);

	/**
	 * Guarda los servicios del recurso modificados en Base de datos.
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 */
	void actualizarServiciosRecurso(Recurso recurso);

	/**
	 * Obtiene los datos de georeferenciación del recurso.
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 * @return Lista de objetos {@link Localizacion} con los datos de georeferenciación del recurso.
	 */
	Localizacion findLocalizacionByRecurso(Recurso recurso);

	/**
	 * Guarda la localización del recurso en Base de datos.
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 */
	void actualizarLocalizacionRecurso(Recurso recurso);

	/**
	 * Obtiene las ofertas asociadas a un recurso.
	 * 
	 * @param codigoRecurso Código del recurso
	 * 
	 * @return lista con las ofertas.
	 */
	List<Oferta> findOfertasByCorec(Long codigoRecurso);

	/**
	 * Crea una oferta asociada a un recurso en Base de datos.
	 * 
	 * @param oferta {@link Oferta} objeto con los datos de la oferta
	 * 
	 */
	void crearOfertaRecurso(Oferta oferta);

	/**
	 * Actualiza una oferta asociada a un recurso en Base de datos.
	 * 
	 * @param oferta {@link Oferta} objeto con los datos de la oferta
	 * 
	 */
	void actualizarOfertaRecurso(Oferta oferta);

	/**
	 * Obtiene los datos de una oferta.
	 * 
	 * @param codigoOferta Código de la oferta
	 * 
	 * @return {@link Oferta} objeto con los datos de la oferta
	 */
	Oferta findOfertaByCorec(Long codigoOferta);

	/**
	 * Elimina una oferta asociada a un recurso. Baja física en AA80T.
	 * 
	 * @param codigoOferta Código de la oferta
	 * 
	 */
	void eliminarOfertaRecurso(Long codigoOferta);

	/**
	 * Obtiene la lista de imágenes multimedia de un recurso
	 * 
	 * @param codigoRecurso Código del recurso
	 * @return lista con imagenes multimedia
	 */
	List<Multimedia> findMultimediasByCorec(Long codigoRecurso);

	/**
	 * Actualiza en la tabla RECURSO los campos de fecha y usuario de actualización
	 * 
	 * @param recurso {@link Recurso} objeto con los datos del recurso
	 * 
	 */
	void actualizarRecurso(Recurso recurso);

	/**
	 * Guarda un registro en la tabla MULTIMEDIA y su imagen asociada en la tabla temporal
	 * 
	 * @param multimedia datos del multimedia
	 */
	void guardarMultimedia(Multimedia multimedia);

	/**
	 * Elimina un elemento multimedia
	 * 
	 * @param multimedia datos del elemento multimedia a eliminar
	 */
	void eliminarMultimediaRecurso(Multimedia multimedia);

	/**
	 * Método que busca un archivo en la tabla TMP_IMAGEN, alternativamente mediante código de multimedia (pantalla
	 * multimedia) o código de recurso (pantalla ofertas)
	 * 
	 * @param nombre nombre del archivo multimedia
	 * @param codRecurso código del recurso
	 * @return codigo del recurso
	 */
	Long exiteTmpFile(String nombre, Long codRecurso);

	/**
	 * Devuelve el nombre y array de bytes de una imagen
	 * 
	 * @param codMultimedia código del registro multimedia
	 * @param codRecurso código del recurso
	 * @return datos de la imagen
	 */
	Imagen getImagenById(Long codMultimedia, Long codRecurso);

	/**
	 * Devuelve la lista de tipos de precio para cargar un combo
	 * 
	 * @param subTipo subtipo del recurso del que hay que cargar la lista de tipos de precio
	 * @return lista de tipos de precios
	 */
	List<Combo> getComboTiposPrecio(String subTipo);

	/**
	 * Actualiza la lista de precios de un recurso
	 * 
	 * @param recurso datos del recurso con la lista de precios
	 */
	void actualizarPreciosRecurso(Recurso recurso);

	/**
	 * Devuelve la lista de precios de un recurso
	 * 
	 * @param codigoRecurso código del recurso
	 * @return lista de precios
	 */
	List<Precio> findPreciosByCorec(Long codigoRecurso);

	/**
	 * Devuelve 'true' cuando se trata de un recurso TOP
	 * 
	 * @param codigoRecurso código del recurso
	 * @return valor del campo "ACTIVO"
	 */
	Boolean getIsRecursoTop(Long codigoRecurso);

	/**
	 * Devuelve la lista de condiciones asociadas a un recurso (para ofertas)
	 * 
	 * @param codigoRecurso código del recurso
	 * @return lista de condiciones
	 */
	List<Condicion> findCondicionesOfertaByCorec(Long codigoRecurso);

	/**
	 * Guarda una condición en un idioma concreto
	 * 
	 * @param condicion datos de la condición
	 */
	void altaCondicionIdioma(Condicion condicion);

	/**
	 * Elimina la lista de condiciones de una oferta
	 * 
	 * @param codigoRecurso codigo del recurso de la oferta
	 */
	void eliminarCondicionesOferta(Long codigoRecurso);

}
