package com.ejie.aa80a.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.model.FiltroNotificacion;
import com.ejie.aa80a.model.Notificacion;
import com.ejie.aa80a.model.Recurso;
import com.ejie.aa80a.util.Utilidades;

/**
 * Clase que accede a la base de datos para tratar datos de notificaciones. Tablas: HISTORICO_GAURKOTU
 * 
 *  
 * 
 */
@Repository()
@Transactional()
public class NotificacionDaoImpl implements NotificacionDao {

	private JdbcTemplate jdbcTemplate;

	@Autowired()
	@Qualifier(value = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private Properties appConfiguration;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla HISTORICO_GAURKOTU con los objetos de la clase
	 * Notificacion
	 */
	private static final class NotificacionRowMapper implements RowMapper<Notificacion> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Notificacion mapRow(ResultSet rs, int rowNum) throws SQLException {
			Notificacion notificacion = new Notificacion();
			notificacion.setCodigo(rs.getLong("IDHISTORICO"));
			notificacion.setEstado(rs.getString("ESTADO"));
			notificacion.setFecha(rs.getDate("FECMODIF"));
			notificacion.setUsuario(rs.getString("USUMODIF"));
			notificacion.setTipoActualizacion(rs.getString("TIPO"));

			Recurso recurso = new Recurso();
			recurso.setCodigo(rs.getLong("COREC"));
			recurso.setNombre(rs.getString("NOMBRE"));
			recurso.setTipo(rs.getString("COTIPREC"));
			recurso.setUrlPortal(rs.getString("URLAMIGABLE")); // url amigable de la ficha en el portal de turismo

			notificacion.setRecurso(recurso);
			return notificacion;
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.NotificacionDao#findNotificacionByCriteria(com.ejie.aa80a.model.FiltroNotificacion)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Notificacion> findNotificacionByCriteria(FiltroNotificacion filtro) {
		StringBuilder query = new StringBuilder();

		query.append("SELECT H.IDHISTORICO, H.COREC COREC, H.TIPO, H.ESTADO, H.USUMODIF, H.FECMODIF , DG.NOMBRE, R.COTIPREC, POS.URLAMIGABLE ");

		// Where clause & Params
		Map<String, ?> mapaWhere = getWhereMapNotificacionByCriteria(filtro);
		StringBuilder where = new StringBuilder();
		where.append(mapaWhere.get("query"));
		query.append(where);

		query.append(" ORDER BY H.ESTADO ASC, H.FECMODIF DESC");

		List<?> argumentos = (List<?>) mapaWhere.get("params");

		// Se obtiene la consulta final incluyendo la paginación.
		StringBuilder queryFinal = new StringBuilder(Utilidades.getPaginationQuery(query, filtro.getOffset(),
				filtro.getLimit(), null, null).toString());

		List<Notificacion> notificaciones = this.jdbcTemplate.query(queryFinal.toString(), new NotificacionRowMapper(),
				argumentos.toArray());

		for (Notificacion notificacion : notificaciones) {

			notificacion.getRecurso().setNombreTipo(
					this.messageSource.getMessage("tipoRecurso." + notificacion.getRecurso().getTipo(), null,
							LocaleContextHolder.getLocale()));

			notificacion.setDescEstado(this.messageSource.getMessage("notificacion.estado." + notificacion.getEstado(),
					null, LocaleContextHolder.getLocale()));

			notificacion.setDescTipoActualizacion(this.messageSource.getMessage("notificacion.actualizacion."
					+ notificacion.getTipoActualizacion(), null, LocaleContextHolder.getLocale()));

			notificacion.getRecurso().setUrlPortal(
					Utilidades.getURLAmigableFichaPortal(notificacion.getRecurso(),
							this.appConfiguration.getProperty("portal.host")));
		}

		return notificaciones;

	}

	@Transactional(readOnly = true)
	@Override()
	public int findNotificacionByCountCriteria(FiltroNotificacion filtro) {
		StringBuilder query = new StringBuilder("SELECT COUNT(H.IDHISTORICO) ");

		// Where clause & Params
		Map<String, ?> mapaWhere = getWhereMapNotificacionByCriteria(filtro);
		StringBuilder where = new StringBuilder();
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> argumentos = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForObject(query.toString(), argumentos.toArray(), Integer.class);
	}

	/**
	 * Devuelve un mapa con la query a ejecutar y los parámetros de búsqueda
	 * 
	 * @param filtro datos del filtro
	 * @return query a ejecutar
	 */
	private Map<String, ?> getWhereMapNotificacionByCriteria(FiltroNotificacion filtro) {
		StringBuilder query = new StringBuilder();
		List<Object> argumentos = new ArrayList<Object>();

		query.append(" FROM HISTORICO_GAURKOTU H ");
		query.append(" INNER JOIN RECURSO R ON R.COREC = H.COREC ");
		query.append(" INNER JOIN DATOS_GENERALES DG ON R.COREC = DG.COREC AND DG.COIDIOMA = ?");
		query.append(" INNER JOIN POSICIONAMIENTO POS ON POS.COREC = H.COREC AND POS.COIDIOMA = ? ");

		argumentos.add(LocaleContextHolder.getLocale().getLanguage());
		argumentos.add(LocaleContextHolder.getLocale().getLanguage());

		query.append(" WHERE 1=1 ");

		if (filtro.getNombreRecurso() != null && !"".equals(filtro.getNombreRecurso())) {
			query.append(" AND UPPER(DG.NOMBRE) LIKE UPPER(?) ");
			argumentos.add("%" + filtro.getNombreRecurso() + "%");
		}

		if (filtro.getTipoRecurso() != null && !"".equals(filtro.getTipoRecurso())) {
			query.append(" AND R.COTIPREC = ? ");
			argumentos.add(filtro.getTipoRecurso());
		}

		if (filtro.getTipoActualizacion() != null && !"".equals(filtro.getTipoActualizacion())) {
			query.append(" AND H.TIPO = ?");
			argumentos.add(filtro.getTipoActualizacion());
		}

		if (filtro.getEstado() != null && !"".equals(filtro.getEstado())) {
			query.append(" AND H.ESTADO = ?");
			argumentos.add(filtro.getEstado());
		}

		if (filtro.getFechaDesde() != null) {
			query.append(" AND TRUNC(H.FECMODIF) >= ?");
			argumentos.add(filtro.getFechaDesde());
		}
		if (filtro.getFechaHasta() != null) {
			query.append(" AND TRUNC(H.FECMODIF) <= ?");
			argumentos.add(filtro.getFechaHasta());
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", query);
		mapWhere.put("params", argumentos);

		return mapWhere;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.NotificacionDao#actualizarEstadoNotificacion(com.ejie.aa80a.model.Notificacion)
	 */
	@Override()
	public void actualizarEstadoNotificacion(Long codigo, String estado) {
		// Hacemos un update
		this.jdbcTemplate.update("UPDATE HISTORICO_GAURKOTU SET ESTADO = ? WHERE IDHISTORICO = ?", estado, codigo);
	}

}
