package com.ejie.aa80a.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.aa80a.model.Localidad;
import com.ejie.aa80a.model.Municipio;
import com.ejie.aa80a.util.Constants;

/**
 * Clase que accede a la base de datos para obtener datos de las tablas FCA de Municipios y Localidades
 * 
 *  
 */
@Repository()
@Transactional()
public class FcaDaoImpl implements FcaDao {

	private JdbcTemplate jdbcTemplate;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla MUNICIPIOS con los objetos de la clase Municipio
	 */
	private static final class MunicipioRowMapper implements RowMapper<Municipio> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Municipio mapRow(ResultSet rs, int rowNum) throws SQLException {
			Municipio municipio = new Municipio();
			municipio.setCodMunicipio(rs.getString("ID_MUNI"));
			municipio.setCodProvincia(rs.getLong("ID_PROV"));
			municipio.setNombre(rs.getString("DS_MUNI"));
			municipio.setNombreCastellano(rs.getString("DS_MUNI"));

			return municipio;
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla LOCALIDADES con los objetos de la clase Localidad
	 */
	private static final class LocalidadRowMapper implements RowMapper<Localidad> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Localidad mapRow(ResultSet rs, int rowNum) throws SQLException {
			Localidad localidad = new Localidad();
			localidad.setCodProvincia(rs.getLong("ID_PROV"));
			localidad.setCodMunicipio(rs.getString("ID_MUNI"));
			localidad.setCodLocalidad(rs.getLong("ID_LOC"));
			localidad.setNombre(rs.getString("DS_LOC"));
			localidad.setNombreCastellano(rs.getString("DS_LOC"));

			return localidad;

		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.FcaDao#findMunicipios(java.lang.String)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Municipio> findMunicipios(Municipio filtroMunicipio) {
		StringBuilder query = new StringBuilder();
		query.append("SELECT T17_PROVINCIA.ID AS ID_PROV, T17_MUNICIPIO.ID AS ID_MUNI, T17_MUNICIPIO.DS_O AS DS_MUNI");
		query.append(" FROM T17_MUNICIPIO INNER JOIN T17_PROVINCIA ON T17_MUNICIPIO.PROVINCIA_ID = T17_PROVINCIA.ID");
		query.append(" WHERE T17_PROVINCIA.ID IN (?,?,?) ");
		query.append(" AND UPPER(T17_MUNICIPIO.DS_O) LIKE UPPER(?)");
		query.append(" ORDER BY T17_MUNICIPIO.DS_O ");

		return this.jdbcTemplate.query(query.toString(), new MunicipioRowMapper(), Constants.PROVINCIA_ALAVA,
				Constants.PROVINCIA_BIZKAIA, Constants.PROVINCIA_GIPUZKOA, "%" + filtroMunicipio.getNombre() + "%");

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.FcaDao#findMunicipiosByProvincia(com.ejie.aa80a.model.Municipio)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Municipio> findMunicipiosByProvincia(Municipio filtroMunicipio) {
		StringBuilder query = new StringBuilder();
		query.append("SELECT T17_PROVINCIA.ID AS ID_PROV, T17_MUNICIPIO.ID AS ID_MUNI, T17_MUNICIPIO.DS_O AS DS_MUNI");
		query.append(" FROM T17_MUNICIPIO INNER JOIN T17_PROVINCIA ON T17_MUNICIPIO.PROVINCIA_ID = T17_PROVINCIA.ID");
		query.append(" WHERE T17_PROVINCIA.ID = ? ");
		query.append(" ORDER BY T17_MUNICIPIO.DS_O ");

		return this.jdbcTemplate.query(query.toString(), new MunicipioRowMapper(), filtroMunicipio.getCodProvincia());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.FcaDao#findLocalidades(com.ejie.aa80a.model.Localidad)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Localidad> findLocalidades(Localidad filtroLocalidad) {
		StringBuilder query = new StringBuilder();
		List<Object> argumentos = new ArrayList<Object>();

		query.append("SELECT T17_PROVINCIA.ID AS ID_PROV, T17_MUNICIPIO.ID AS ID_MUNI, T17_LOCALIDAD.ID AS ID_LOC, T17_LOCALIDAD.DS_O AS DS_LOC");
		query.append(" FROM T17_PROVINCIA INNER JOIN T17_MUNICIPIO ON T17_MUNICIPIO.PROVINCIA_ID = T17_PROVINCIA.ID ");
		query.append(" INNER JOIN T17_LOCALIDAD ON T17_LOCALIDAD.PROVINCIA_ID = T17_PROVINCIA.ID AND T17_LOCALIDAD.MUNICIPIO_ID = T17_MUNICIPIO.ID");

		if (filtroLocalidad.getCodProvincia() != null) {
			query.append(" WHERE T17_PROVINCIA.ID = ? ");
			argumentos.add(filtroLocalidad.getCodProvincia());
		} else {
			query.append(" WHERE T17_PROVINCIA.ID IN (?,?,?) ");
			argumentos.add(Constants.PROVINCIA_ALAVA);
			argumentos.add(Constants.PROVINCIA_BIZKAIA);
			argumentos.add(Constants.PROVINCIA_GIPUZKOA);
		}

		if (filtroLocalidad.getCodMunicipio() != null) {
			query.append(" AND T17_MUNICIPIO.ID = ? ");
			argumentos.add(filtroLocalidad.getCodMunicipio());
		}
		if (filtroLocalidad.getNombre() != null && filtroLocalidad.getNombre() != "") {
			query.append(" AND UPPER(T17_LOCALIDAD.DS_O) LIKE UPPER(?)");
			argumentos.add("%" + filtroLocalidad.getNombre() + "%");
		}

		query.append(" ORDER BY T17_LOCALIDAD.DS_O ");

		return this.jdbcTemplate.query(query.toString(), new LocalidadRowMapper(), argumentos.toArray());

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.aa80a.dao.FcaDao#findLocalidadesByMunicipio(com.ejie.aa80a.model.Localidad)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Localidad> findLocalidadesByMunicipio(Localidad filtroLocalidad) {
		StringBuilder query = new StringBuilder();
		List<Object> argumentos = new ArrayList<Object>();

		query.append("SELECT T17_PROVINCIA.ID AS ID_PROV, T17_MUNICIPIO.ID AS ID_MUNI, T17_LOCALIDAD.ID AS ID_LOC, T17_LOCALIDAD.DS_O AS DS_LOC");
		query.append(" FROM T17_PROVINCIA INNER JOIN T17_MUNICIPIO ON T17_MUNICIPIO.PROVINCIA_ID = T17_PROVINCIA.ID ");
		query.append(" INNER JOIN T17_LOCALIDAD ON T17_LOCALIDAD.PROVINCIA_ID = T17_PROVINCIA.ID AND T17_LOCALIDAD.MUNICIPIO_ID = T17_MUNICIPIO.ID");
		query.append(" WHERE T17_PROVINCIA.ID = ? ");
		query.append(" AND T17_MUNICIPIO.ID = ? ");
		query.append(" ORDER BY DS_LOC ");

		argumentos.add(filtroLocalidad.getCodProvincia());
		argumentos.add(filtroLocalidad.getCodMunicipio());

		return this.jdbcTemplate.query(query.toString(), new LocalidadRowMapper(), argumentos.toArray());

	}
}
