/*
 * Copyright 2011 E.J.I.E., S.A.
 *
 * Licencia con arreglo a la EUPL, Versión 1.1 exclusivamente (la «Licencia»);
 * Solo podrá usarse esta obra si se respeta la Licencia.
 * Puede obtenerse una copia de la Licencia en
 *
 * http://ec.europa.eu/idabc/eupl.html
 *
 * Salvo cuando lo exija la legislación aplicable o se acuerde por escrito,
 * el programa distribuido con arreglo a la Licencia se distribuye «TAL CUAL»,
 * SIN GARANTÍAS NI CONDICIONES DE NINGÚN TIPO, ni expresas ni implícitas.
 * Véase la Licencia en el idioma concreto que rige los permisos y limitaciones
 * que establece la Licencia.
 */
package com.ejie.aa80a.aop;

import org.aspectj.lang.ProceedingJoinPoint;

import com.ejie.aa80a.log.IncidenceLoggingAdvice;
import com.ejie.aa80a.log.LoggingAdvice;

/**
 * 
 * Advice principal que se encarga de gestionar toda la información interceptada
 * por los PointCuts. Actualmente solo se usa para el aspecto de Logging.
 * 
 *  
 * 
 */
public class MainAdvice {

	private LoggingAdvice filterLoggingAdvice;
	private LoggingAdvice serviceLoggingAdvice;
	private LoggingAdvice daoLoggingAdvice;
	private IncidenceLoggingAdvice incidenceLoggingAdvice;

	/**
	 * MainAdvice que se ejecuta alrededor de llamadas a un filtro, de tal
	 * manera que loguea la petición y la respuesta de éste.
	 * 
	 * @param call Llamada interceptada.
	 * 
	 * @return Retorno de la llamada interceptada.
	 * 
	 * @throws Throwable Throwable Excepcion que se deja pasar, en caso de que
	 *             el Target la genere.
	 */
	public Object filterCall(ProceedingJoinPoint call) throws Throwable {
		this.filterLoggingAdvice.preLogging(call);
		Object ret = call.proceed();
		this.filterLoggingAdvice.postLogging(call, ret);
		return ret;
	}

	/**
	 * MainAdvice que se ejecuta alrededor de llamadas a servicios, de tal
	 * manera que loguea la petición y la respuesta de estas.
	 * 
	 * @param call Llamada interceptada.
	 * 
	 * @return Retorno de la llamada interceptada.
	 * 
	 * @throws Throwable Excepcion que se deja pasar, en caso de que el Target
	 *             la genere.
	 */
	public Object serviceLogCall(ProceedingJoinPoint call) throws Throwable {
		this.serviceLoggingAdvice.preLogging(call);
		Object ret = call.proceed();
		this.serviceLoggingAdvice.postLogging(call, ret);
		return ret;
	}

	/**
	 * MainAdvice que loguea las llamadas a los DAOs, registrando los valores de
	 * entrada y salida.
	 * 
	 * @param call Llamada interceptada.
	 * 
	 * @return Retorno de la llamada interceptada.
	 * 
	 * @throws Throwable Throwable Excepción que se deja pasar, en caso de que
	 *             el Target la genere.
	 */
	public Object daoLogCall(ProceedingJoinPoint call) throws Throwable {
		this.daoLoggingAdvice.preLogging(call);
		Object ret = call.proceed();
		this.daoLoggingAdvice.postLogging(call, ret);
		return ret;
	}

	/**
	 * Advice que se encarga de loguear las excepciones no controladas.
	 * 
	 * @param target Clase que lanza la excepción no controlada.
	 * @param exception Excepción capturada.
	 */
	public void logIncidence(Object target, Exception exception) {
		this.incidenceLoggingAdvice.logIncidence(target, exception);
	}

	/**
	 * Establece el campo filterLoggingAdvice.
	 * 
	 * @param filterLoggingAdvice filterLoggingAdvice
	 */
	public void setFilterLoggingAdvice(LoggingAdvice filterLoggingAdvice) {
		this.filterLoggingAdvice = filterLoggingAdvice;
	}

	/**
	 * Obtiene el campo serviceLoggingAdvice.
	 * 
	 * @return serviceLoggingAdvice
	 */
	public LoggingAdvice getServiceLoggingAdvice() {
		return this.serviceLoggingAdvice;
	}

	/**
	 * Establece el campo serviceLoggingAdvice.
	 * 
	 * @param serviceLoggingAdvice serviceLoggingAdvice
	 */
	public void setServiceLoggingAdvice(LoggingAdvice serviceLoggingAdvice) {
		this.serviceLoggingAdvice = serviceLoggingAdvice;
	}

	/**
	 * Obtiene el campo daoLoggingAdvice.
	 * 
	 * @return daoLoggingAdvice
	 */
	public LoggingAdvice getDaoLoggingAdvice() {
		return this.daoLoggingAdvice;
	}

	/**
	 * Establece el campo daoLoggingAdvice.
	 * 
	 * @param daoLoggingAdvice daoLoggingAdvice
	 */
	public void setDaoLoggingAdvice(LoggingAdvice daoLoggingAdvice) {
		this.daoLoggingAdvice = daoLoggingAdvice;
	}

	/**
	 * Obtiene el campo incidenceLoggingAdvice.
	 * 
	 * @return incidenceLoggingAdvice
	 */
	public IncidenceLoggingAdvice getIncidenceLoggingAdvice() {
		return this.incidenceLoggingAdvice;
	}

	/**
	 * Establece el campo incidenceLoggingAdvice.
	 * 
	 * @param incidenceLoggingAdvice incidenceLoggingAdvice
	 */
	public void setIncidenceLoggingAdvice(IncidenceLoggingAdvice incidenceLoggingAdvice) {
		this.incidenceLoggingAdvice = incidenceLoggingAdvice;
	}
}