package com.ejie.y40a.utils;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Locale;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.i18n.LocaleContextHolder;

import com.ejie.y40a.constantes.Y40aConstantes;
import com.ejie.y40a.utils.fecha.DateFormatSymbolsProvider_eu;
import com.ejie.y40a.utils.mail.Y40aMailMessage;
import com.ejie.y40a.utils.mail.Y40aMailSender;
import com.sun.org.apache.xerces.internal.impl.xpath.regex.ParseException;

public class Y40aUtils {
	private static Y40aUtils INSTANCE = null;

	private static final Logger logger = LoggerFactory
			.getLogger(Y40aUtils.class);

	// Private constructor suppresses
	private Y40aUtils() {

	}

	// creador sincronizado para protegerse de posibles problemas multi-hilo
	// otra prueba para evitar instanciación múltiple
	private synchronized static void createInstance() {
		if (INSTANCE == null) {
			INSTANCE = new Y40aUtils();
		}
	}

	public static Y40aUtils getInstance() {
		if (INSTANCE == null)
			createInstance();
		return INSTANCE;
	}

	/**
	 * Muestra la etiqueta parseada para una estructura json - arbol
	 * 
	 * @param lblEs
	 * @param lblEu
	 * @return
	 */
	public static String printTreeLabel(String lblEs, String lblEu) {
		return isLanguage(lblEs, lblEu);
	}

	/**
	 * Muestra la etiqueta en el idioma correspondiente
	 * 
	 * @param lblEs
	 * @param lblEu
	 * @return
	 */
	private static String isLanguage(String lblEs, String lblEu) {
		// Idioma
		Locale locale = LocaleContextHolder.getLocale();

		if (Y40aConstantes.EUSKERA.equals(locale.getLanguage())) {
			return filtroDB(lblEu);
		} else {
			return filtroDB(lblEs);
		}

	}

	public static String filtroDB(String input) {

		logger.info(input);
		if (input == null) {
			return "";
		} else {
			input = input.replaceAll("\"", "");
		}

		return input;
	}

	/**
	 * Devuelve true si el string vienen informado y false si el string es nulo
	 * o vacio
	 * 
	 * @param stringIn
	 *            Dato a evaluar
	 * @return boolean true o false
	 */
	public static boolean isFilled(String stringIn) {
		if (stringIn == null || stringIn.equals("")) {
			return false;
		}
		return true;
	}

	/**
	 * Transforma un java.sql.Date a la mascara de formato de fecha recibida
	 * como parámetro.
	 * 
	 * @param fecha
	 *            String
	 * @param formato
	 *            String con el formato a parsear
	 * @return String fecha formateada
	 * @exception ParseException
	 *                Se produce la excepción cuando la fecha está mal
	 *                formateada.
	 */
	public static String formatearDateToString(java.util.Date fecha,
			String formato) throws ParseException {
		DateFormat sdf = null;

		if (fecha == null) {
			return null;
		} else {
			sdf = new SimpleDateFormat(formato);
		}

		return sdf.format(fecha);
	}

	/**
	 * Parsea un string de fecha con el formato correspondiente al idioma del
	 * locale y devuelve la fecha.
	 * 
	 * @param dateString
	 * @param locale
	 * @return
	 * @throws Exception
	 */
	public static Date parseDate(String dateString, Locale locale)
			throws Exception {
		if (dateString != null && !dateString.equals("")) {
			DateFormat dateFormat = null;
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				dateFormat = new SimpleDateFormat("dd/MM/yyyy");
			} else if (Y40aConstantes.EUSKERA.equals(locale.getLanguage())) {
				dateFormat = new SimpleDateFormat("yyyy/MM/dd");
			} else {
				throw new Exception(
						"Language date parse support not implemented for this language.");
			}
			return dateFormat.parse(dateString);
		}
		return null;
	}

	/**
	 * Parsea un string de fecha con el formato correspondiente al idioma del
	 * locale y devuelve la fecha. Recibe un tipo date
	 * 
	 * @param date
	 *            (Date)
	 * @param locale
	 * @return
	 * @throws Exception
	 */
	public static Date parseDate(Date date, Locale locale) throws Exception {
		if (date != null) {
			SimpleDateFormat formatter = null;
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				formatter = new SimpleDateFormat("dd/MM/yyyy");
			} else if (Y40aConstantes.EUSKERA.equals(locale.getLanguage())) {
				formatter = new SimpleDateFormat("yyyy/MM/dd");
			} else {
				throw new Exception(
						"Language date parse support not implemented for this language.");
			}
			return formatter.parse(formatter.format(date));

		}
		return null;
	}

	/**
	 * Normaliza datos de entrada de formularios a "" - excepto combos
	 * 
	 * @param input
	 *            String
	 * @return String
	 */
	public static String normalizarVacio(String input) {

		String rt = Y40aConstantes.VACIO;
		if (input != null) {
			if (!"null".equals(input)) {
				rt = input;
			}
		}

		return rt;
	}

	/**
	 * Convierte un Long en String
	 * 
	 * @param input
	 *            Long
	 * @return String
	 */
	public static String convertLong2String(Long input) {

		if (input != null) {
			return input.toString();
		} else {
			return Y40aConstantes.VACIO;
		}
	}

	/**
	 * 
	 * @param fecha
	 * @param lang
	 * @return
	 */
	public static String parseDateToString(Date fecha, String lang) {
		String fechaFormateada = "";
		DecimalFormat df = new DecimalFormat("00");

		if (fecha != null) {
			Calendar calendar = Calendar.getInstance();
			calendar.setTime(fecha);

			int dia = calendar.get(Calendar.DAY_OF_MONTH); // dia del mes
			int mes = calendar.get(Calendar.MONTH) + 1; // mes, de 0 a 11
			int anio = calendar.get(Calendar.YEAR); // año

			if (Y40aConstantes.EUSKERA.equals(lang)) {
				fechaFormateada = anio + "/" + df.format(mes) + "/"
						+ df.format(dia);
			} else {
				fechaFormateada = df.format(dia) + "/" + df.format(mes) + "/"
						+ anio;
			}
		}
		return fechaFormateada;
	}

	// Metodo que devuelve un list con todos los campos seleccionados de una
	// consulta
	public static List<String> cargarDatosBDVO(ResultSet rsSelect) {
		List<String> listRetorno = null;
		try {
			ResultSetMetaData rMetaData = rsSelect.getMetaData();
			int columnasRegistro = rMetaData.getColumnCount();
			String datosColumna = "";
			for (int intColumna = 1; intColumna <= columnasRegistro; intColumna++) {
				if (listRetorno == null) {
					listRetorno = new ArrayList<String>();
				}
				datosColumna = rMetaData.getColumnName(intColumna);
				listRetorno.add(datosColumna);
			}

		} catch (Exception e) {
		}
		return listRetorno;
	}

	/**
	 * Transforma un java.sql.Date a la mascara de formato de fecha recibida
	 * como parÃ¡metro.
	 * 
	 * @param fecha
	 *            String
	 * @param formato
	 *            String con el formato a parsear
	 * @param idioma
	 *            String con el idioma
	 * @return String fecha formateada
	 * @exception ParseException
	 *                Se produce la excepciÃ³n cuando la fecha estÃ¡ mal
	 *                formateada.
	 */
	public static String formatearDateToString(java.util.Date fecha,
			String formato, String idioma) throws ParseException {

		SimpleDateFormat sdf = null;

		if (fecha == null) {
			return null;
		} else {
			sdf = new SimpleDateFormat(formato, new Locale(idioma));
		}

		return sdf.format(fecha);
	}

	/**
	 * Transforma un java.sql.Date a la mascara de formato de fecha recibida
	 * como parÃ¡metro.
	 * 
	 * @param fecha
	 *            String
	 * @param formato
	 *            String con el formato a parsear
	 * @param idioma
	 *            String con el idioma
	 * @return String fecha formateada
	 * @exception ParseException
	 *                Se produce la excepciÃ³n cuando la fecha estÃ¡ mal
	 *                formateada.
	 */
	public static String formatearDateToStringEuskera(java.util.Date fecha,
			String formato, String idioma) throws ParseException {

		SimpleDateFormat sdf = null;
		DateFormatSymbolsProvider_eu dateFormatSymbolsProvider_eu = new DateFormatSymbolsProvider_eu();

		if (fecha == null) {
			return null;
		} else {
			sdf = new SimpleDateFormat(formato,
					dateFormatSymbolsProvider_eu
							.getInstance(new Locale(idioma)));
		}

		return sdf.format(fecha);
	}

	/**
	 * Se envia un mail
	 * 
	 * @param smtp
	 *            Servidor smtp
	 * @param asunto
	 *            Asunto del mensaje
	 * @param body
	 *            Cuerpo del mensaje
	 * @param from
	 *            Email quien envia el mensaje
	 * @param to
	 *            Email a quien va dirigido el mensaje
	 * @param attachments
	 *            Ficheros adjuntos en el mensaje
	 * @return boolean True si el mensaje se envio correctamente y False en caso
	 *         contrario
	 */
	public static boolean sendMail(String smtp, String asunto, String body,
			String from, String to, Collection attachments) {

		try {

			Y40aMailMessage eMailMessage = new Y40aMailMessage();
			Y40aMailSender sendMessage = new Y40aMailSender();
			eMailMessage.setFrom(from);
			eMailMessage.setEmailServer(smtp);
			eMailMessage.setSubject(asunto);
			eMailMessage.setNotDeliveryAddress("");
			eMailMessage.setMsgBodyText(body);

			// para introducir los destinatarios
			// Cambiado para que vayan con copia oculta
			String[] destinatarios = to.toString().split(";");
			for (int i = 0; i < destinatarios.length; i++) {
				if (!"".equals(destinatarios[i])) {
					eMailMessage.addBCC(destinatarios[i]);
				}
			}

			eMailMessage.setAttachments((ArrayList) attachments);

			sendMessage.send(eMailMessage);
		} catch (Exception e) {
			return false;
		}
		return true;
	}

}
