package com.ejie.y40a.service;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y40a.constantes.Y40aConstantes;
import com.ejie.y40a.dao.CategoriaDao;
import com.ejie.y40a.dao.ContactoDao;
import com.ejie.y40a.dao.EntidadDao;
import com.ejie.y40a.dao.FormacionDao;
import com.ejie.y40a.dao.ParametrosDao;
import com.ejie.y40a.dao.SolicitudDao;
import com.ejie.y40a.model.Categoria;
import com.ejie.y40a.model.CodesEntities;
import com.ejie.y40a.model.CodesEntity;
import com.ejie.y40a.model.ConsultaDinamicaBusqueda;
import com.ejie.y40a.model.ConsultaDinamicaResultado;
import com.ejie.y40a.model.Contacto;
import com.ejie.y40a.model.Entidad;
import com.ejie.y40a.model.EntidadDetalle;
import com.ejie.y40a.model.Etiqueta;
import com.ejie.y40a.model.Formacion;
import com.ejie.y40a.model.Parametros;
import com.ejie.y40a.model.PlanificacionFormacionBusqueda;
import com.ejie.y40a.model.PlanificacionSolicitanteBusqueda;
import com.ejie.y40a.model.PlanificacionSolicitanteBusquedaCartas;
import com.ejie.y40a.model.Solicitud;
import com.ejie.y40a.model.SolicitudDetalle;
import com.ejie.y40a.utils.Y40aUIDGenerator;
import com.ejie.y40a.utils.Y40aUtils;
import com.ejie.y40a.utils.exception.Y40aUDAException;
import com.ejie.y40a.utils.jaxb.Y40aJAXBFactory;

/**
 * * SolicitudServiceImpl  
 * 
 *  
 */

@Service(value = "solicitudService")
public class SolicitudServiceImpl implements SolicitudService {

	private static final Logger logger = LoggerFactory
			.getLogger(SolicitudServiceImpl.class);
	@Autowired
	private SolicitudDao solicitudDao;
	@Autowired
	private EntidadDao entidadDao;
	@Autowired
	private ContactoDao contactoDao;
	@Autowired
	private CategoriaDao categoriaDao;
	@Autowired
	private FormacionDao formacionDao;
	@Autowired
	private ParametrosDao parametrosDao;

	/**
	 * Inserts a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Solicitud add(Solicitud solicitud) {
		return this.solicitudDao.add(solicitud);
	}

	/**
	 * Inserts a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Solicitud addSolicitud(Solicitud solicitud) {
		try {
			// Validacion de numero maximo de asistentes
			Formacion formacion = new Formacion();
			formacion.setIdFor(solicitud.getFormacion().getIdFor());
			formacion = this.formacionDao.find(formacion);

			if (formacion.getCategoria().getNumAsistentes() > 0
					&& solicitud.getParticipantes() > formacion.getCategoria()
							.getNumAsistentes()) {
				throw new Y40aUDAException(
						"error.solicitud.numParticipantesException", true,
						formacion.getCategoria().getNumAsistentes().toString(),
						new Exception());
			}

			// Validacion: No se debe permitir 2 solicitudes para la misma
			// formación/categoría para el mismo grupo e idioma.
			Solicitud solicitudAux = new Solicitud();
			solicitudAux.setEntidad(solicitud.getEntidad());
			solicitudAux.setGrupo(solicitud.getGrupo());
			solicitudAux.setFormacion(formacion);
			solicitudAux.setIdioma(solicitud.getIdioma());

			Long numFormacionesGrupoIdioma = this.solicitudDao
					.findNumFormacionGrupoIdioma(solicitudAux);

			if (numFormacionesGrupoIdioma > 0) {
				throw new Y40aUDAException(
						"error.solicitud.solicitudyaexistente", true,
						new Exception());
			}

			// Generar la clave
			solicitud.setIdSol(Y40aUIDGenerator.getInstance().generateId(
					Y40aConstantes.PK_SIZE));

			// Evita problemas con las fechas desde hasta
			Locale locale = LocaleContextHolder.getLocale();
			Date today = new Date();
			Date todayWithZeroTime = Y40aUtils.parseDate(today, locale);

			solicitud.setFechaSol(todayWithZeroTime);

			// Add Solicitud
			this.solicitudDao.add(solicitud);

			// Validacion de numero maximo de formaciones por grupo.
			solicitudAux = new Solicitud();
			solicitudAux.setEntidad(solicitud.getEntidad());
			solicitudAux.setGrupo(solicitud.getGrupo());
			solicitudAux.setFormacion(formacion);

			Long numFormacionesGrupo = this.solicitudDao
					.findNumFormacionGrupo(solicitudAux);

			if (formacion.getCategoria().getNumFormaciones() > 0
					&& numFormacionesGrupo > formacion.getCategoria()
							.getNumFormaciones()) {
				throw new Y40aUDAException(
						"error.solicitud.numFormacionesException",
						true,
						formacion.getCategoria().getNumFormaciones().toString(),
						new Exception());
			}

			// add etiquetas
			this.solicitudDao.addSolicitudEtiqueta(solicitud);

			return solicitud;
		} catch (Y40aUDAException e) {
			throw e;
		} catch (Exception e) {
			throw new Y40aUDAException("error.solicitud.addSolicitud", true, e);
		}
	}

	/**
	 * Updates a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Solicitud updateSolicitud(Solicitud solicitud) {
		// Validacion de numero maximo de asistentes
		Formacion formacion = new Formacion();
		formacion.setIdFor(solicitud.getFormacion().getIdFor());
		formacion = this.formacionDao.find(formacion);

		if (formacion.getCategoria().getNumAsistentes() > 0
				&& solicitud.getParticipantes() > formacion.getCategoria()
						.getNumAsistentes()) {
			throw new Y40aUDAException(
					"error.solicitud.numParticipantesException", true,
					formacion.getCategoria().getNumAsistentes().toString(),
					new Exception());
		}

		this.solicitudDao.update(solicitud);

		// Validacion: No se debe permitir 2 solicitudes para la misma
		// formación/categoría para el mismo grupo e idioma.
		Solicitud solicitudAux = new Solicitud();
		solicitudAux.setEntidad(solicitud.getEntidad());
		solicitudAux.setGrupo(solicitud.getGrupo());
		solicitudAux.setFormacion(formacion);
		solicitudAux.setIdioma(solicitud.getIdioma());

		Long numFormacionesGrupoIdioma = this.solicitudDao
				.findNumFormacionGrupoIdioma(solicitudAux);

		if (numFormacionesGrupoIdioma > 1) {
			throw new Y40aUDAException("error.solicitud.solicitudyaexistente",
					true, new Exception());
		}

		// Validacion de numero maximo de formaciones por grupo.
		solicitudAux = new Solicitud();
		solicitudAux.setEntidad(solicitud.getEntidad());
		solicitudAux.setGrupo(solicitud.getGrupo());
		solicitudAux.setFormacion(formacion);

		Long numFormacionesGrupo = this.solicitudDao
				.findNumFormacionGrupo(solicitudAux);

		if (formacion.getCategoria().getNumFormaciones() > 0
				&& numFormacionesGrupo > formacion.getCategoria()
						.getNumFormaciones()) {
			throw new Y40aUDAException(
					"error.solicitud.numFormacionesException", true, formacion
							.getCategoria().getNumFormaciones().toString(),
					new Exception());
		}

		// Etiquetas
		this.solicitudDao.removeAllSolicitudEtiqueta(solicitud);
		this.solicitudDao.addSolicitudEtiqueta(solicitud);

		return solicitud;
	}

	/**
	 * Updates a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Solicitud update(Solicitud solicitud) {
		return this.solicitudDao.update(solicitud);
	}

	/**
	 * Finds a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Solicitud
	 */
	public Solicitud find(Solicitud solicitud) {
		return (Solicitud) this.solicitudDao.find(solicitud);
	}

	/**
	 * Finds a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return SolicitudDetalle
	 */
	public SolicitudDetalle findSolicitud(Solicitud solicitud) {
		Solicitud solicitudAux = this.solicitudDao.find(solicitud);

		Solicitud solicitudAuxEtiquetas = this.solicitudDao.find(solicitud);

		solicitudAuxEtiquetas = this.solicitudDao.findSolicitudEtiqueta(
				solicitud, null, null);
		List<Etiqueta> lstEtiquetasFormacion = solicitudAuxEtiquetas
				.getEtiquetas();
		List<String> lstIdsEtiquetas = new ArrayList<String>();
		List<String> lstDescsEtiquetas = new ArrayList<String>();
		StringBuffer strIdsEtiquetas = new StringBuffer();
		StringBuffer strDescsEtiquetas = new StringBuffer();

		strDescsEtiquetas.append("[");

		boolean firstElement = true;

		// se recorre para obtener los ids de etiquetas
		for (Etiqueta etiqueta : lstEtiquetasFormacion) {
			lstIdsEtiquetas.add(etiqueta.getIdEtiqueta());
			lstDescsEtiquetas.add(etiqueta.getNombre());
			if (firstElement) {
				strIdsEtiquetas.append(etiqueta.getIdEtiqueta());
				strDescsEtiquetas.append(etiqueta.getNombre());
				firstElement = false;
			} else {
				strIdsEtiquetas.append(",").append(etiqueta.getIdEtiqueta());
				strDescsEtiquetas.append(",").append(etiqueta.getNombre());
			}
		}
		strDescsEtiquetas.append("]");

		Categoria categoriaAux = new Categoria();
		categoriaAux.setIdCat(solicitudAux.getFormacion().getCategoria()
				.getIdCat());

		String jerarquiaCategoria = obtenerJerarquiaCategoria(categoriaAux);
		StringBuffer jerarquia = new StringBuffer();
		Locale locale = LocaleContextHolder.getLocale();

		categoriaAux = this.categoriaDao.find(solicitudAux.getFormacion()
				.getCategoria());

		if (Y40aUtils.isFilled(jerarquiaCategoria)) {
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				jerarquia.append(jerarquiaCategoria).append(">")
						.append(categoriaAux.getNombreEs());
			} else {
				jerarquia.append(jerarquiaCategoria).append(">")
						.append(categoriaAux.getNombreEu());
			}
		} else {
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				jerarquia.append(categoriaAux.getNombreEs());
			} else {
				jerarquia.append(categoriaAux.getNombreEu());
			}
		}

		if (Y40aUtils.isFilled(jerarquia.toString())) {
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				jerarquia.append(">").append(
						solicitudAux.getFormacion().getNombreEs());
			} else {
				jerarquia.append(">").append(
						solicitudAux.getFormacion().getNombreEu());
			}
		} else {
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				jerarquia.append(solicitudAux.getFormacion().getNombreEs());
			} else {
				jerarquia.append(solicitudAux.getFormacion().getNombreEu());
			}
		}

		SolicitudDetalle solicitudDetalle = new SolicitudDetalle(
				solicitudAux.getIdSol(), solicitudAux.getFechaSol(),
				solicitudAux.getFormacion().getIdFor(), solicitudAux
						.getFormacion().getNombreEs(), solicitudAux
						.getFormacion().getNombreEu(), jerarquia.toString(),
				solicitudAux.getEntidad().getEntidadId(), solicitudAux
						.getEntidad().getNombreEs(), solicitudAux.getEntidad()
						.getNombreEu(),
				solicitudAux.getEntidad().getNoraProvinciaId(),
				solicitudAux.getGrupo(),
				solicitudAux.getParticipantes(),
				solicitudAux.getProvincia(),
				solicitudAux.getDiaImparticion(),
				// FIXME solicitudAux.getHorarioImparticion(),
				solicitudAux.getHoraInicio(), solicitudAux.getHoraFin(),
				solicitudAux.getAsistentes(), solicitudAux.getNumMujeres(),
				solicitudAux.getNumHombres(), solicitudAux.getCancelado(),
				solicitudAux.getIdioma(), strIdsEtiquetas.toString(),
				strDescsEtiquetas.toString());

		solicitudDetalle.setObservaciones(solicitudAux.getObservaciones());
		solicitudDetalle.setNombapecontacto(solicitudAux.getNombapecontacto());
		solicitudDetalle.setEmailcontacto(solicitudAux.getEmailcontacto());
		solicitudDetalle.setTelcontacto(solicitudAux.getTelcontacto());
		solicitudDetalle.setObservasolicitante(solicitudAux
				.getObservasolicitante());

		// Recogemos si hay que visulizar los campos de contacto
		solicitudDetalle.setVisibleContactoCat(categoriaAux
				.getVisiblecontacto());

		solicitudDetalle.setBloqueada(solicitudAux.getBloqueada());
		solicitudDetalle.setIdMotivo(solicitudAux.getIdMotivo());

		return solicitudDetalle;
	}

	/**
	 * Finds a List of rows in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Solicitud> findAll(Solicitud solicitud, Pagination pagination) {
		return (List<Solicitud>) this.solicitudDao.findAll(solicitud,
				pagination);
	}

	/**
	 * Counts rows in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Long
	 */
	public Long findAllCount(Solicitud solicitud) {
		return this.solicitudDao.findAllCount(solicitud);
	}

	/**
	 * Finds a List of rows in the Solicitud table.
	 * 
	 * @param planificacionSolicitanteBusqueda
	 *            PlanificacionSolicitanteBusqueda
	 * @param perfiles
	 *            Vector<String>
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<PlanificacionSolicitanteBusqueda> findAllPlanificacionSolicitante(
			PlanificacionSolicitanteBusqueda planificacionSolicitanteBusqueda,
			Vector<String> perfiles, Pagination pagination) {

		if (pagination != null) {

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreEs")) {
				pagination.setSort("entidadNombreEs");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreEu")) {
				pagination.setSort("entidadNombreEu");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreMunicipio")) {
				pagination.setSort("entidadNombreMunicipio");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("fechaSolicitud")) {
				pagination.setSort("FECHA_SOLICITUD");
			}

		}

		String categoriasHijas = "";
		if (planificacionSolicitanteBusqueda.getIdCat() != null
				&& !planificacionSolicitanteBusqueda.getIdCat().equals("")) {
			categoriasHijas = obtenerCategoriasHijas(
					planificacionSolicitanteBusqueda.getIdCat(), perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			planificacionSolicitanteBusqueda.setCategorias(categorias);
		}

		List<PlanificacionSolicitanteBusqueda> listaPlanificacionSolicitanteBusqueda = this.solicitudDao
				.findAllPlanificacionSolicitante(
						planificacionSolicitanteBusqueda, perfiles, pagination);

		Locale locale = LocaleContextHolder.getLocale();

		for (PlanificacionSolicitanteBusqueda planifiSolicitanteBusqueda : listaPlanificacionSolicitanteBusqueda) {
			StringBuffer intervaloImparticion = new StringBuffer();
			if (planifiSolicitanteBusqueda.getFechaImparticionDesde() != null) {
				intervaloImparticion.append(Y40aUtils.normalizarVacio(Y40aUtils
						.parseDateToString(planifiSolicitanteBusqueda
								.getFechaImparticionDesde(), locale
								.getLanguage())));
				intervaloImparticion.append("-");
				intervaloImparticion.append(Y40aUtils.normalizarVacio(Y40aUtils
						.parseDateToString(planifiSolicitanteBusqueda
								.getFechaImparticionHasta(), locale
								.getLanguage())));
			}

			planifiSolicitanteBusqueda
					.setIntervaloImparticion(intervaloImparticion.toString());
		}

		return listaPlanificacionSolicitanteBusqueda;
	}

	/**
	 * 
	 * Obtiene la categoria y todas sus categorias hijas.
	 * 
	 * @param idCategoria
	 *            String
	 * @param perfiles
	 *            Vector<String>
	 * @return String
	 */
	private String obtenerCategoriasHijas(String idCategoria,
			Vector<String> perfiles) {

		StringBuffer categoriasHijas = new StringBuffer();
		categoriasHijas.append(idCategoria);

		Categoria categoriaAux = new Categoria();
		Categoria categoriaPadre = new Categoria();
		categoriaPadre.setIdCat(idCategoria);
		categoriaAux.setCategoria(categoriaPadre);
		categoriaAux.setVisibleWeb(null);

		Locale locale = LocaleContextHolder.getLocale();

		Pagination pagination = new Pagination();
		if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
			pagination.setSort("NOMBREES");
			pagination.setAscDsc("ASC");
		} else {
			pagination.setSort("NOMBREEU");
			pagination.setAscDsc("ASC");
		}

		List<Categoria> lstCategoriaChildren = this.categoriaDao
				.findAllCategoriaSeguridad(categoriaAux, perfiles, null,
						pagination);

		String idcategoria = "";
		if (lstCategoriaChildren.size() > 0) {

			for (Categoria categoria : lstCategoriaChildren) {

				idcategoria = obtenerCategoriasHijas(categoria.getIdCat(),
						perfiles);

				categoriasHijas.append(",").append(idcategoria);
			}
		}

		return categoriasHijas.toString();
	}

	/**
	 * Counts rows in the Solicitud table.
	 * 
	 * @param planificacionSolicitanteBusqueda
	 *            PlanificacionSolicitanteBusqueda
	 * @param perfiles
	 *            Vector<String>
	 * @return Long
	 */
	public Long findAllPlanificacionSolicitanteCount(
			PlanificacionSolicitanteBusqueda planificacionSolicitanteBusqueda,
			Vector<String> perfiles) {

		String categoriasHijas = "";
		if (planificacionSolicitanteBusqueda.getIdCat() != null
				&& !planificacionSolicitanteBusqueda.getIdCat().equals("")) {
			categoriasHijas = obtenerCategoriasHijas(
					planificacionSolicitanteBusqueda.getIdCat(), perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			planificacionSolicitanteBusqueda.setCategorias(categorias);
		}

		return this.solicitudDao.findAllPlanificacionSolicitanteCount(
				planificacionSolicitanteBusqueda, perfiles);
	}

	/**
	 * Finds a List of rows in the Solicitud table.
	 * 
	 * @param planificacionFormacionBusqueda
	 *            PlanificacionFormacionBusqueda
	 * @param perfiles
	 *            Vector<String>
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<PlanificacionFormacionBusqueda> findAllPlanificacionFormacion(
			PlanificacionFormacionBusqueda planificacionFormacionBusqueda,
			Vector<String> perfiles, Pagination pagination) {

		if (pagination != null) {

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreCortoFormacionEs")) {
				pagination.setSort("FORMACIONNOMBRECORTOES");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreCortoFormacionEu")) {
				pagination.setSort("FORMACIONNOMBRECORTOEU");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreFormacionEs")) {
				pagination.setSort("NOMBREES");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreEntidadEs")) {
				pagination.setSort("ENTIDADNOMBREES");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreFormacionEu")) {
				pagination.setSort("NOMBREEU");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreEntidadEu")) {
				pagination.setSort("ENTIDADNOMBREEU");
			}
		}

		String categoriasHijas = "";
		if (planificacionFormacionBusqueda.getIdCat() != null
				&& !planificacionFormacionBusqueda.getIdCat().equals("")) {
			categoriasHijas = obtenerCategoriasHijas(
					planificacionFormacionBusqueda.getIdCat(), perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			planificacionFormacionBusqueda.setCategorias(categorias);
		}

		return (List<PlanificacionFormacionBusqueda>) this.solicitudDao
				.findAllPlanificacionFormacion(planificacionFormacionBusqueda,
						perfiles, pagination);
	}

	/**
	 * Counts rows in the Solicitud table.
	 * 
	 * @param planificacionFormacionBusqueda
	 *            PlanificacionFormacionBusqueda
	 * @param perfiles
	 *            Vector<String>
	 * @return Long
	 */
	public Long findAllPlanificacionFormacionCount(
			PlanificacionFormacionBusqueda planificacionFormacionBusqueda,
			Vector<String> perfiles) {

		String categoriasHijas = "";
		if (planificacionFormacionBusqueda.getIdCat() != null
				&& !planificacionFormacionBusqueda.getIdCat().equals("")) {
			categoriasHijas = obtenerCategoriasHijas(
					planificacionFormacionBusqueda.getIdCat(), perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			planificacionFormacionBusqueda.setCategorias(categorias);
		}

		return this.solicitudDao.findAllPlanificacionFormacionCount(
				planificacionFormacionBusqueda, perfiles);
	}

	/**
	 * Finds a single row in the Solicitante table.
	 * 
	 * @param entidadDetalle
	 *            EntidadDetalle
	 * @return EntidadDetalle
	 */
	public EntidadDetalle findEntidadSolicitante(EntidadDetalle entidadDetalle) {

		Entidad entidad = new Entidad();
		entidad.setEntidadId(entidadDetalle.getEntidadId());
		entidad = this.entidadDao.find(entidad);

		List<Contacto> listContacto = this.contactoDao.findAllContactoEntidad(
				entidad, null);

		Contacto contacto = new Contacto();

		if (listContacto != null && listContacto.size() > 0) {
			contacto = listContacto.get(0);
		}

		entidadDetalle = new EntidadDetalle(entidad.getEntidadId(),
				entidad.getNombreEs(), entidad.getNombreEu(),
				entidad.getActiva(), entidad.getIskontsumogest(),
				entidad.getIskontsumonet(), entidad.getIsofertaform(),
				entidad.getIsconsultasweb(), entidad.getCodOrg(),
				entidad.getObservaciones(), entidad.getTelefono(),
				entidad.getFax(), entidad.getEmail(), entidad.getUrl(),
				entidad.getUsuario(), entidad.getPassword(),
				entidad.getIsnacional(), entidad.getNoraPaisId(),
				entidad.getNombrePais(), entidad.getNoraProvinciaId(),
				entidad.getNombreProvincia(), entidad.getNoraMunicipioId(),
				entidad.getNombreMunicipio(), entidad.getNoraLocalidadId(),
				entidad.getNombreLocalidad(), entidad.getNoraCalleId(),
				entidad.getNombreCalle(), entidad.getNoraCpId(),
				entidad.getNombreCp(), entidad.getNoraPortalId(),
				entidad.getNombrePortal(), entidad.getUsuultmod(),
				entidad.getFecultmod(), entidad.getCompleDirec(),
				entidad.getOrinbdco(), contacto.getNombre(),
				contacto.getEmail(), entidad.getObservacionesEntidad()
						.getObservaciones());

		entidadDetalle.setNumHistObserv(entidad.getNumHistObserv());
		return entidadDetalle;
	}

	/**
	 * Finds rows in the Solicitud table using like.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Solicitud> findAllLike(Solicitud solicitud,
			Pagination pagination, Boolean startsWith) {
		return (List<Solicitud>) this.solicitudDao.findAllLike(solicitud,
				pagination, startsWith);
	}

	/**
	 * Deletes a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void remove(Solicitud solicitud) {
		this.solicitudDao.remove(solicitud);
	}

	/**
	 * Deletes a single row in the Solicitud table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeSolicitud(Solicitud solicitud) {
		try {
			// Etiquetas
			this.solicitudDao.removeAllSolicitudEtiqueta(solicitud);

			this.solicitudDao.remove(solicitud);
		} catch (DataIntegrityViolationException e) {
			throw new Y40aUDAException(
					"error.solicitud.DataIntegrityViolationException", true, e);
		}
	}

	/**
	 * Deletes multiple rows in the Solicitud table.
	 * 
	 * @param solicitudList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeMultiple(ArrayList<Solicitud> solicitudList) {
		for (Solicitud solicitudAux : solicitudList) {
			this.solicitudDao.remove(solicitudAux);
		}
	}

	/**
	 * Inserts a single row in the SolicitudEtiqueta table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @return Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Solicitud addSolicitudEtiqueta(Solicitud solicitud) {
		return this.solicitudDao.addSolicitudEtiqueta(solicitud);
	}

	/**
	 * Deletes a single row in the SolicitudEtiqueta table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeSolicitudEtiqueta(Solicitud solicitud) {
		this.solicitudDao.removeSolicitudEtiqueta(solicitud);
	}

	/**
	 * Find a single row in the findSolicitudEtiqueta Many To Many relationship.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @param etiqueta
	 *            Etiqueta
	 * @param pagination
	 *            Pagination
	 * @return Solicitud
	 */
	public Solicitud findSolicitudEtiqueta(Solicitud solicitud,
			Etiqueta etiqueta, Pagination pagination) {
		return this.solicitudDao.findSolicitudEtiqueta(solicitud, etiqueta,
				pagination);
	}

	/**
	 * Counts rows in the SolicitudEtiqueta table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * @param etiqueta
	 *            Etiqueta
	 * @return List
	 */
	public Long findSolicitudEtiquetaCount(Solicitud solicitud,
			Etiqueta etiqueta) {
		return this.solicitudDao
				.findSolicitudEtiquetaCount(solicitud, etiqueta);
	}

	/**
	 * Getter method for SolicitudDao
	 * 
	 * @return SolicitudDao
	 */
	public SolicitudDao getSolicitudDao() {
		return this.solicitudDao;
	}

	/**
	 * Setter method for SolicitudDao.
	 * 
	 * @param solicitudDao
	 *            SolicitudDao
	 * @return
	 */
	public void setSolicitudDao(SolicitudDao solicitudDao) {
		logger.info("Setting Dependency " + solicitudDao);
		this.solicitudDao = solicitudDao;
	}

	/**
	 * Finds a List of rows in the Solicitud table.
	 * 
	 * @param consultaDinamicaBusqueda
	 *            ConsultaDinamicaBusqueda
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<ConsultaDinamicaResultado> findAllConsultaDinamica(
			ConsultaDinamicaBusqueda consultaDinamicaBusqueda,
			Pagination pagination) {
		List<ConsultaDinamicaResultado> listConsultaDinamicaResultado = this.solicitudDao
				.findAllConsultaDinamica(consultaDinamicaBusqueda, pagination);

		for (ConsultaDinamicaResultado consultaDinamicaResultadoAux : listConsultaDinamicaResultado) {
			Categoria categoriaAux = new Categoria();
			categoriaAux.setIdCat(consultaDinamicaResultadoAux.getIdCat());
			// Consulta de la jerarquia de categorias
			String jerarquiaCategoria = this
					.obtenerJerarquiaCategoria(categoriaAux);
			StringBuffer jerarquia = new StringBuffer();
			Locale locale = LocaleContextHolder.getLocale();

			if (Y40aUtils.isFilled(jerarquiaCategoria)) {
				if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
					jerarquia
							.append(jerarquiaCategoria)
							.append(">")
							.append(consultaDinamicaResultadoAux
									.getNombreCategoriaEs());
				} else {
					jerarquia
							.append(jerarquiaCategoria)
							.append(">")
							.append(consultaDinamicaResultadoAux
									.getNombreCategoriaEu());
				}
			} else {
				if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
					jerarquia.append(jerarquiaCategoria)
							.append(consultaDinamicaResultadoAux
									.getNombreCategoriaEs());
				} else {
					jerarquia.append(jerarquiaCategoria)
							.append(consultaDinamicaResultadoAux
									.getNombreCategoriaEu());
				}
			}

			consultaDinamicaResultadoAux.setJerarquiaCategoria(jerarquia
					.toString());

			// Consulta para obtener las etiquetas de formacion
			Formacion formacion = new Formacion();
			formacion.setIdFor(consultaDinamicaResultadoAux.getIdFor());

			Etiqueta etiqueta = new Etiqueta();

			formacion = this.formacionDao.findFormacionEtiqueta(formacion,
					etiqueta, null);

			StringBuffer etiquetas = new StringBuffer();

			// for para crear el string con todas las etiquetas separadas por ;
			boolean firstElement = true;
			for (Etiqueta etiquetaAux : formacion.getEtiquetas()) {
				if (firstElement) {
					etiquetas.append(etiquetaAux.getNombre());
					firstElement = false;
				} else {
					etiquetas.append(";").append(etiquetaAux.getNombre());
				}
			}

			consultaDinamicaResultadoAux.setEtiquetasFormacion(etiquetas
					.toString());

			// Consulta para obtener las etiquetas de solicitud
			Solicitud solicitud = new Solicitud();
			solicitud.setIdSol(consultaDinamicaResultadoAux.getIdSol());

			solicitud = this.solicitudDao.findSolicitudEtiqueta(solicitud,
					etiqueta, null);

			etiquetas = new StringBuffer();

			// for para crear el string con todas las etiquetas separadas por ;
			firstElement = true;
			for (Etiqueta etiquetaAux : solicitud.getEtiquetas()) {
				if (firstElement) {
					etiquetas.append(etiquetaAux.getNombre());
					firstElement = false;
				} else {
					etiquetas.append(";").append(etiquetaAux.getNombre());
				}
			}

			consultaDinamicaResultadoAux.setEtiquetasSolicitud(etiquetas
					.toString());
		}

		return listConsultaDinamicaResultado;
	}

	/**
	 * Recupera la jerarquia de una Categoria.
	 * 
	 * @param categoria
	 *            Categoria
	 * @return String
	 */
	private String obtenerJerarquiaCategoria(Categoria categoria) {
		StringBuffer jerarquiaCategoria = new StringBuffer();

		Locale locale = LocaleContextHolder.getLocale();

		boolean firstElement = true;
		boolean tienePadre = true;

		while (tienePadre) {

			Categoria categoriaAux = this.categoriaDao
					.findCategoriaPadre(categoria);

			if (Y40aUtils.isFilled(categoriaAux.getIdCat())) {
				if (firstElement) {
					if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
						jerarquiaCategoria.append(categoriaAux.getNombreEs());
					} else {
						jerarquiaCategoria.append(categoriaAux.getNombreEu());
					}

					firstElement = false;
				} else {
					String jerarquia = jerarquiaCategoria.toString();
					jerarquiaCategoria = new StringBuffer();

					if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
						jerarquiaCategoria.append(categoriaAux.getNombreEs())
								.append(">").append(jerarquia);
					} else {
						jerarquiaCategoria.append(categoriaAux.getNombreEu())
								.append(">").append(jerarquia);
					}
				}
				categoria.setIdCat(categoriaAux.getIdCat());
			} else {
				tienePadre = false;
			}
		}

		return jerarquiaCategoria.toString();
	}

	/**
	 * Counts rows in the Solicitud table.
	 * 
	 * @param consultaDinamicaBusqueda
	 *            ConsultaDinamicaBusqueda
	 * @return Long
	 */
	public Long findAllConsultaDinamicaCount(
			ConsultaDinamicaBusqueda consultaDinamicaBusqueda) {
		return this.solicitudDao
				.findAllConsultaDinamicaCount(consultaDinamicaBusqueda);
	}

	/**
	 * Devuelve el xml para cargarlo en la aplicacion de organismos.
	 * 
	 * @param consultaDinamicaBusqueda
	 *            ConsultaDinamicaBusqueda
	 * @return String
	 */
	public String findAllConsultaDinamicaXML(
			ConsultaDinamicaBusqueda consultaDinamicaBusqueda) {
		try {
			List<ConsultaDinamicaResultado> listConsultaDinamicaResultado = this.solicitudDao
					.findAllConsultaDinamicaXML(consultaDinamicaBusqueda, null);

			CodesEntities ces = new CodesEntities();

			for (ConsultaDinamicaResultado consultaDinamicaResultadoAux : listConsultaDinamicaResultado) {
				CodesEntity e1 = new CodesEntity();
				e1.setCode(consultaDinamicaResultadoAux.getEntidadId());

				ces.getEntidades().add(e1);
			}

			String xml = Y40aJAXBFactory.marshal(ces);

			return xml;
		} catch (Exception e) {
			throw new Y40aUDAException("error.consultadinamica.xml", true, e);
		}
	}

	/**
	 * Inserts a single row in the Denuncia table.
	 * 
	 * @param solicitud
	 *            Solicitud
	 * 
	 * @return Solicitud
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Solicitud copiarSolicitud(Solicitud solicitud) {
		try {

			// Buscar la solicitud que se quiere duplicar
			Solicitud solCopia = this.find(solicitud);

			// Generar id para la copia
			solCopia.setIdSol(Y40aUIDGenerator.getInstance().generateId(
					Y40aConstantes.PK_SIZE));

			// Añadir la copia
			// this.solicitudDao.add(solCopia);

			// obtener etiquetas del original
			Solicitud solCopiaEtiquetas = this.solicitudDao
					.findSolicitudEtiqueta(solicitud, null, null);

			if (solCopiaEtiquetas != null
					&& solCopiaEtiquetas.getEtiquetas() != null
					&& solCopiaEtiquetas.getEtiquetas().size() > 0) {
				// actualizar id al de la copia
				solCopiaEtiquetas.setIdSol(solCopia.getIdSol());

				// añadir etiquetas a la copia
				this.solicitudDao.addSolicitudEtiqueta(solCopiaEtiquetas);
			}

			return solCopia;

		} catch (Exception e) {
			throw new Y40aUDAException("error.generico", true, e);
		}
	}

	/**
	 * Finds a List of rows in the Solicitud table.
	 * 
	 * @param planificacionSolicitanteBusquedaCartas
	 *            PlanificacionSolicitanteBusquedaCartas
	 * @param perfiles
	 *            Vector<String>
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<PlanificacionSolicitanteBusquedaCartas> findAllPlanificacionSolicitanteCartas(
			PlanificacionSolicitanteBusquedaCartas planificacionSolicitanteBusquedaCartas,
			Vector<String> perfiles, Pagination pagination) {

		if (pagination != null) {

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreEs")) {
				pagination.setSort("entidadNombreEs");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreEu")) {
				pagination.setSort("entidadNombreEu");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("nombreMunicipio")) {
				pagination.setSort("entidadNombreMunicipio");
			}

			if (pagination.getSort() != null
					&& pagination.getSort().equals("fechaSolicitud")) {
				pagination.setSort("FECHA_SOLICITUD");
			}

		}
		String idCatPadre = "";
		String categoriasHijas = "";
		if (planificacionSolicitanteBusquedaCartas.getIdCat() != null
				&& !planificacionSolicitanteBusquedaCartas.getIdCat()
						.equals("")) {
			idCatPadre = planificacionSolicitanteBusquedaCartas.getIdCat();
			categoriasHijas = obtenerCategoriasHijas(
					planificacionSolicitanteBusquedaCartas.getIdCat(), perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			planificacionSolicitanteBusquedaCartas.setCategorias(categorias);
		}

		List<PlanificacionSolicitanteBusquedaCartas> listaPlanificacionSolicitanteBusquedaCartas = this.solicitudDao
				.findAllPlanificacionSolicitanteCartas(idCatPadre,
						planificacionSolicitanteBusquedaCartas, perfiles,
						pagination);

		// Locale locale = LocaleContextHolder.getLocale();

		for (PlanificacionSolicitanteBusquedaCartas planifiSolicitanteBusquedaCartas : listaPlanificacionSolicitanteBusquedaCartas) {
			StringBuffer intervaloImparticionEs = new StringBuffer();
			StringBuffer intervaloImparticionEu = new StringBuffer();
			if (planifiSolicitanteBusquedaCartas.getFechaImparticionDesde() != null) {
				intervaloImparticionEs.append(Y40aUtils
						.normalizarVacio(Y40aUtils.parseDateToString(
								planifiSolicitanteBusquedaCartas
										.getFechaImparticionDesde(),
								Y40aConstantes.CASTELLANO)));
				intervaloImparticionEs.append("-");
				intervaloImparticionEs.append(Y40aUtils
						.normalizarVacio(Y40aUtils.parseDateToString(
								planifiSolicitanteBusquedaCartas
										.getFechaImparticionHasta(),
								Y40aConstantes.CASTELLANO)));

				intervaloImparticionEu.append(Y40aUtils
						.normalizarVacio(Y40aUtils.parseDateToString(
								planifiSolicitanteBusquedaCartas
										.getFechaImparticionDesde(),
								Y40aConstantes.EUSKERA)));
				intervaloImparticionEu.append("-");
				intervaloImparticionEu.append(Y40aUtils
						.normalizarVacio(Y40aUtils.parseDateToString(
								planifiSolicitanteBusquedaCartas
										.getFechaImparticionHasta(),
								Y40aConstantes.EUSKERA)));

			}

			planifiSolicitanteBusquedaCartas
					.setIntervaloImparticionEs(intervaloImparticionEs
							.toString());
			planifiSolicitanteBusquedaCartas
					.setIntervaloImparticionEu(intervaloImparticionEu
							.toString());
		}

		return listaPlanificacionSolicitanteBusquedaCartas;
	}

	/**
	 * Blocks multiple rows in the Solicitud table.
	 * 
	 * @param solicitudList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void blockMultiple(List<Solicitud> solicitudList) {
		for (Solicitud solicitudAux : solicitudList) {
			this.solicitudDao.block(solicitudAux);
		}
	}

	/**
	 * Unblocks multiple rows in the Solicitud table.
	 * 
	 * @param solicitudList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void unblockMultiple(List<Solicitud> solicitudList) {
		for (Solicitud solicitudAux : solicitudList) {
			this.solicitudDao.unblock(solicitudAux);
		}
	}

	/**
	 * Blocks multiple rows in the Solicitud table.
	 * 
	 * @param solicitudList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void bloquearSolicitudesAutomaticamente() {
		try {
			// recuperar valores de bloqueo automatico y fecha bloqueo
			Parametros parametro = new Parametros();
			parametro
					.setDescripcionEs(Y40aConstantes.PARAMETRO_BLOQUEO_AUTOMATICO_DESC);
			parametro = this.parametrosDao.findByDescES(parametro);
			String valorBloqueo = parametro.getValor();

			logger.info("Bloqueo activo (1 si, 0 no): " + valorBloqueo);

			parametro
					.setDescripcionEs(Y40aConstantes.PARAMETRO_FECHA_BLOQUEO_DESC);
			parametro = this.parametrosDao.findByDescES(parametro);
			String valorFechaBloqueo = parametro.getValor();

			parametro
					.setDescripcionEs(Y40aConstantes.PARAMETRO_DIAS_BLOQUEO_SOLICITUDES);
			parametro = this.parametrosDao.findByDescES(parametro);
			String valorDiasBloqueo = parametro.getValor();

			logger.info("Días bloqueo (not null, >0 ?): " + valorDiasBloqueo);

			Date today = new Date();
			String strToday = Y40aUtils.parseDateToString(today,
					Y40aConstantes.CASTELLANO);

			logger.info("Fecha actual: " + strToday);
			logger.info("FechaBloqueo: " + valorFechaBloqueo);

			logger.info("BLOQUEO SI en caso de que fecha bloqueo<fechaactual y bloqueo automatico = 1");

			// si fecha bloqueo<fechaactual y bloqueo automatico = 1

			Locale locale = new Locale(Y40aConstantes.CASTELLANO);

			Date date1 = Y40aUtils.parseDate(strToday, locale);
			Date date2 = Y40aUtils.parseDate(valorFechaBloqueo, locale);

			if ((date2 == null || (date2.compareTo(date1) < 0))
					&& (valorBloqueo
							.equals(Y40aConstantes.PARAMETRO_BLOQUEO_AUTOMATICO_ACTIVO))) {

				this.solicitudDao
						.bloquearSolicitudesAutomaticamente(valorDiasBloqueo);

				// actualizar fecha registro
				Parametros parametroAux = new Parametros();
				parametroAux
						.setDescripcionEs(Y40aConstantes.PARAMETRO_FECHA_BLOQUEO_DESC);
				parametroAux.setValor(strToday);

				this.parametrosDao.updateValorByDescES(parametroAux);

			}
		} catch (Exception e) {
			throw new Y40aUDAException("error.generico", true, e);
		}

	}
}
