package com.ejie.y40a.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y40a.constantes.Y40aConstantes;
import com.ejie.y40a.dao.CategoriaDao;
import com.ejie.y40a.dao.FormacionDao;
import com.ejie.y40a.model.Categoria;
import com.ejie.y40a.model.Etiqueta;
import com.ejie.y40a.model.Formacion;
import com.ejie.y40a.model.FormacionBusqueda;
import com.ejie.y40a.utils.Y40aUIDGenerator;
import com.ejie.y40a.utils.Y40aUtils;
import com.ejie.y40a.utils.exception.Y40aUDAException;

/**
 * * FormacionServiceImpl  
 * 
 *  
 */

@Service(value = "formacionService")
public class FormacionServiceImpl implements FormacionService {

	private static final Logger logger = LoggerFactory
			.getLogger(FormacionServiceImpl.class);
	@Autowired
	private FormacionDao formacionDao;
	@Autowired
	private CategoriaDao categoriaDao;

	/**
	 * Inserts a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Formacion add(Formacion formacion) {
		return this.formacionDao.add(formacion);
	}

	/**
	 * Inserts a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Formacion addFormacion(Formacion formacion) {

		// Comprobar que el nombre de la Formacion en castellano sea unico
		// dentro de la categoria
		FormacionBusqueda formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setNombreEs(formacion.getNombreEs());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		Long numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 0) {
			throw new Y40aUDAException("error.formacion.nombreesrepetido",
					true, new Exception());
		}

		// Comprobar que el nombre de la Formacion en euskera sea unico dentro
		// de la categoria
		formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setNombreEu(formacion.getNombreEu());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 0) {
			throw new Y40aUDAException("error.formacion.nombreeurepetido",
					true, new Exception());
		}

		// Comprobar que el nombre de la Formacion corto sea unico dentro de la
		// categoria
		formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setNombreCortoEs(formacion.getNombreCortoEs());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 0) {
			throw new Y40aUDAException("error.formacion.nombrecortoesrepetido",
					true, new Exception());
		}

		formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setNombreCortoEu(formacion.getNombreCortoEu());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 0) {
			throw new Y40aUDAException("error.formacion.nombrecortoeurepetido",
					true, new Exception());
		}

		formacion.setIdFor(Y40aConstantes.PREFIJO_FORMACION
				+ Y40aUIDGenerator.getInstance().generateId(
						Y40aConstantes.PK_SIZE - 2));

		this.formacionDao.add(formacion);

		this.formacionDao.addFormacionEtiqueta(formacion);

		return formacion;
	}

	/**
	 * Updates a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Formacion update(Formacion formacion) {
		return this.formacionDao.update(formacion);
	}

	/**
	 * Updates a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Formacion updateFormacion(Formacion formacion) {

		this.formacionDao.update(formacion);

		// Comprobar que el nombre de la Formacion en castellano sea unico
		// dentro de la categoria
		FormacionBusqueda formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setIdFor(formacion.getIdFor());
		formacionBusquedaAux.setNombreEs(formacion.getNombreEs());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		Long numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 1) {
			throw new Y40aUDAException("error.formacion.nombreesrepetido",
					true, new Exception());
		}

		// Comprobar que el nombre de la Formacion en euskera sea unico dentro
		// de la categoria
		formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setIdFor(formacion.getIdFor());
		formacionBusquedaAux.setNombreEu(formacion.getNombreEu());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 1) {
			throw new Y40aUDAException("error.formacion.nombreeurepetido",
					true, new Exception());
		}

		// Comprobar que el nombre de la Formacion corto sea unico dentro de la
		// categoria
		formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setNombreCortoEs(formacion.getNombreCortoEs());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 1) {
			throw new Y40aUDAException("error.formacion.nombrecortoesrepetido",
					true, new Exception());
		}

		formacionBusquedaAux = new FormacionBusqueda();
		formacionBusquedaAux.setNombreCortoEu(formacion.getNombreCortoEu());

		if (formacion.getCategoria() != null) {
			formacionBusquedaAux.setIdCat(formacion.getCategoria().getIdCat());
		}

		numFormaciones = this.formacionDao
				.findAllNameFormacionCount(formacionBusquedaAux);

		if (numFormaciones > 1) {
			throw new Y40aUDAException("error.formacion.nombrecortoeurepetido",
					true, new Exception());
		}

		// Etiquetas
		this.formacionDao.removeAllFormacionEtiqueta(formacion);
		this.formacionDao.addFormacionEtiqueta(formacion);

		return formacion;
	}

	/**
	 * Updates a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Formacion updateVisibleWeb(Formacion formacion) {
		return this.formacionDao.updateVisibleWeb(formacion);
	}

	/**
	 * Finds a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	public Formacion find(Formacion formacion) {
		return (Formacion) this.formacionDao.find(formacion);
	}

	/**
	 * Finds a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	public Formacion findFormacion(Formacion formacion) {
		formacion = this.formacionDao.find(formacion);

		Categoria categoriaAux = new Categoria();
		categoriaAux.setIdCat(formacion.getCategoria().getIdCat());

		String jerarquiaCategoria = obtenerJerarquiaCategoria(categoriaAux);

		StringBuffer jerarquia = new StringBuffer();
		Locale locale = LocaleContextHolder.getLocale();
		if (Y40aUtils.isFilled(jerarquiaCategoria)) {
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				jerarquia.append(jerarquiaCategoria).append(">")
						.append(formacion.getCategoria().getNombreEs());
			} else {
				jerarquia.append(jerarquiaCategoria).append(">")
						.append(formacion.getCategoria().getNombreEu());
			}
		} else {
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				jerarquia.append(formacion.getCategoria().getNombreEs());
			} else {
				jerarquia.append(formacion.getCategoria().getNombreEu());
			}
		}

		formacion.getCategoria().setJerarquiaCategoria(jerarquia.toString());

		if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
			jerarquia.append(jerarquiaCategoria).append(">")
					.append(formacion.getNombreEs());
		} else {
			jerarquia.append(jerarquiaCategoria).append(">")
					.append(formacion.getNombreEu());
		}

		formacion.setJerarquiaFormacion(jerarquia.toString());

		if (!Y40aUtils.isFilled(formacion.getDescripcionEs())) {
			formacion.setDescripcionEs("");
		}

		if (!Y40aUtils.isFilled(formacion.getDescripcionEu())) {
			formacion.setDescripcionEu("");
		}

		return formacion;
	}

	/**
	 * Recupera la jerarquia de una Categoria.
	 * 
	 * @param categoria
	 *            Categoria
	 * @return String
	 */
	private String obtenerJerarquiaCategoria(Categoria categoria) {
		StringBuffer jerarquiaCategoria = new StringBuffer();

		Locale locale = LocaleContextHolder.getLocale();

		boolean firstElement = true;
		boolean tienePadre = true;

		while (tienePadre) {

			Categoria categoriaAux = this.categoriaDao
					.findCategoriaPadre(categoria);

			if (Y40aUtils.isFilled(categoriaAux.getIdCat())) {
				if (firstElement) {
					if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
						jerarquiaCategoria.append(categoriaAux.getNombreEs());
					} else {
						jerarquiaCategoria.append(categoriaAux.getNombreEu());
					}

					firstElement = false;
				} else {
					String jerarquia = jerarquiaCategoria.toString();
					jerarquiaCategoria = new StringBuffer();

					if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
						jerarquiaCategoria.append(categoriaAux.getNombreEs())
								.append(">").append(jerarquia);
					} else {
						jerarquiaCategoria.append(categoriaAux.getNombreEu())
								.append(">").append(jerarquia);
					}
				}
				categoria.setIdCat(categoriaAux.getIdCat());
			} else {
				tienePadre = false;
			}
		}

		return jerarquiaCategoria.toString();
	}

	/**
	 * Finds a List of rows in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @param pagination
	 *            Pagination
	 * @return List
	 */
	public List<Formacion> findAll(Formacion formacion, Pagination pagination) {
		return (List<Formacion>) this.formacionDao.findAll(formacion,
				pagination);
	}

	/**
	 * Counts rows in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Long
	 */
	public Long findAllCount(Formacion formacion) {
		return this.formacionDao.findAllCount(formacion);
	}

	/**
	 * Finds rows in the Formacion table using like.
	 * 
	 * @param formacion
	 *            Formacion
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<Formacion> findAllLike(Formacion formacion,
			Pagination pagination, Boolean startsWith) {
		return (List<Formacion>) this.formacionDao.findAllLike(formacion,
				pagination, startsWith);
	}

	/**
	 * Finds rows in the Formacion table using like.
	 * 
	 * @param formacion
	 *            Formacion
	 * @param perfiles
	 *            Vector<String>
	 * @param pagination
	 *            Pagination
	 * @param startsWith
	 *            Boolean
	 * @return List
	 */
	public List<FormacionBusqueda> findAllLikeFormacion(
			FormacionBusqueda formacionBusqueda, Vector<String> perfiles,
			Pagination pagination, Boolean startsWith) {

		String categoriasHijas = "";
		if (formacionBusqueda.getIdCat() != null
				&& !formacionBusqueda.getIdCat().equals("")) {
			categoriasHijas = obtenerCategoriasHijas(formacionBusqueda,
					perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			formacionBusqueda.setCategorias(categorias);
		}

		List<FormacionBusqueda> listFormacionBusqueda = this.formacionDao
				.findAllLikeFormacion(formacionBusqueda, perfiles, pagination,
						startsWith);

		for (FormacionBusqueda formacionBusquedaAux : listFormacionBusqueda) {
			Categoria categoriaAux = new Categoria();
			categoriaAux.setIdCat(formacionBusquedaAux.getIdCat());

			String jerarquiaCategoria = obtenerJerarquiaCategoria(categoriaAux);

			StringBuffer jerarquia = new StringBuffer();
			Locale locale = LocaleContextHolder.getLocale();
			if (Y40aUtils.isFilled(jerarquiaCategoria)) {
				if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
					jerarquia
							.append(jerarquiaCategoria)
							.append(">")
							.append(formacionBusquedaAux.getCategoriaNombreEs());
				} else {
					jerarquia
							.append(jerarquiaCategoria)
							.append(">")
							.append(formacionBusquedaAux.getCategoriaNombreEu());
				}
			} else {
				if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
					jerarquia.append(formacionBusquedaAux
							.getCategoriaNombreEs());
				} else {
					jerarquia.append(formacionBusquedaAux
							.getCategoriaNombreEu());
				}
			}

			formacionBusquedaAux.setJerarquiaCategoria(jerarquia.toString());

		}

		return listFormacionBusqueda;
	}

	/**
	 * 
	 * Obtiene la categoria y todas sus categorias hijas.
	 * 
	 * @param formacionBusqueda
	 *            FormacionBusqueda
	 * @param perfiles
	 *            Vector<String>
	 * @return String
	 */
	private String obtenerCategoriasHijas(FormacionBusqueda formacionBusqueda,
			Vector<String> perfiles) {

		StringBuffer categoriasHijas = new StringBuffer();
		categoriasHijas.append(formacionBusqueda.getIdCat());

		Categoria categoriaAux = new Categoria();
		Categoria categoriaPadre = new Categoria();
		categoriaPadre.setIdCat(formacionBusqueda.getIdCat());
		categoriaAux.setCategoria(categoriaPadre);
		categoriaAux.setVisibleWeb(null);

		Locale locale = LocaleContextHolder.getLocale();

		Pagination pagination = new Pagination();
		if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
			pagination.setSort("NOMBREES");
			pagination.setAscDsc("ASC");
		} else {
			pagination.setSort("NOMBREEU");
			pagination.setAscDsc("ASC");
		}

		List<Categoria> lstCategoriaChildren = this.categoriaDao
				.findAllCategoriaSeguridad(categoriaAux, perfiles, null,
						pagination);

		String idcategoria = "";
		if (lstCategoriaChildren.size() > 0) {

			for (Categoria categoria : lstCategoriaChildren) {

				FormacionBusqueda formacionBusquedaAux = new FormacionBusqueda();
				formacionBusquedaAux.setIdCat(categoria.getIdCat());

				idcategoria = obtenerCategoriasHijas(formacionBusquedaAux,
						perfiles);

				categoriasHijas.append(",").append(idcategoria);
			}
		}

		return categoriasHijas.toString();
	}

	/**
	 * Counts rows in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @param perfiles
	 *            Vector<String>
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	public Long findAllLikeFormacionCount(FormacionBusqueda formacionBusqueda,
			Vector<String> perfiles, Boolean startsWith) {
		String categoriasHijas = "";
		if (formacionBusqueda.getIdCat() != null
				&& !formacionBusqueda.getIdCat().equals("")) {
			categoriasHijas = obtenerCategoriasHijas(formacionBusqueda,
					perfiles);

			List<Categoria> categorias = new ArrayList<Categoria>();
			// Generar el listado de categorias
			if (null != categoriasHijas && !"".equals(categoriasHijas)) {
				Categoria categoria = null;
				for (String id : categoriasHijas.split(",")) {
					categoria = new Categoria();// NOPMD 2012/02/29 (IDA)
					categoria.setIdCat(id);
					categorias.add(categoria);
				}
			}
			formacionBusqueda.setCategorias(categorias);
		}
		return this.formacionDao.findAllLikeFormacionCount(formacionBusqueda,
				perfiles, startsWith);
	}

	/**
	 * Deletes a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void remove(Formacion formacion) {
		this.formacionDao.remove(formacion);
	}

	/**
	 * Deletes a single row in the Formacion table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeFormacion(Formacion formacion) {
		try {
			// Etiquetas
			this.formacionDao.removeAllFormacionEtiqueta(formacion);

			this.formacionDao.remove(formacion);
		} catch (DataIntegrityViolationException e) {
			throw new Y40aUDAException(
					"error.formacion.DataIntegrityViolationException", true, e);
		}
	}

	/**
	 * Deletes multiple rows in the Formacion table.
	 * 
	 * @param formacionList
	 *            ArrayList
	 * @return
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeMultiple(ArrayList<Formacion> formacionList) {
		for (Formacion formacionAux : formacionList) {
			this.formacionDao.remove(formacionAux);
		}
	}

	/**
	 * Inserts a single row in the FormacionEtiqueta table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @return Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public Formacion addFormacionEtiqueta(Formacion formacion) {
		return this.formacionDao.addFormacionEtiqueta(formacion);
	}

	/**
	 * Deletes a single row in the FormacionEtiqueta table.
	 * 
	 * @param formacion
	 *            Formacion
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void removeFormacionEtiqueta(Formacion formacion) {
		this.formacionDao.removeFormacionEtiqueta(formacion);
	}

	/**
	 * Find a single row in the findFormacionEtiqueta Many To Many relationship.
	 * 
	 * @param formacion
	 *            Formacion
	 * @param etiqueta
	 *            Etiqueta
	 * @param pagination
	 *            Pagination
	 * @return Formacion
	 */
	public Formacion findFormacionEtiqueta(Formacion formacion,
			Etiqueta etiqueta, Pagination pagination) {
		return this.formacionDao.findFormacionEtiqueta(formacion, etiqueta,
				pagination);
	}

	/**
	 * Counts rows in the FormacionEtiqueta table.
	 * 
	 * @param formacion
	 *            Formacion
	 * @param etiqueta
	 *            Etiqueta
	 * @return List
	 */
	public Long findFormacionEtiquetaCount(Formacion formacion,
			Etiqueta etiqueta) {
		return this.formacionDao
				.findFormacionEtiquetaCount(formacion, etiqueta);
	}

	/**
	 * Getter method for FormacionDao
	 * 
	 * @return FormacionDao
	 */
	public FormacionDao getFormacionDao() {
		return this.formacionDao;
	}

	/**
	 * Setter method for FormacionDao.
	 * 
	 * @param formacionDao
	 *            FormacionDao
	 * @return
	 */
	public void setFormacionDao(FormacionDao formacionDao) {
		logger.info("Setting Dependency " + formacionDao);
		this.formacionDao = formacionDao;
	}
}
