package com.ejie.y40a.service;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.x38.dto.Pagination;
import com.ejie.y40a.constantes.Y40aConstantes;
import com.ejie.y40a.dao.CategoriaDao;
import com.ejie.y40a.dao.FormacionDao;
import com.ejie.y40a.model.Categoria;
import com.ejie.y40a.model.Formacion;
import com.ejie.y40a.utils.exception.Y40aUDAException;

/**
 * * CategoriaServiceImpl  
 * 
 *  
 */

@Service(value = "campaniaService")
public class CampaniaServiceImpl implements CampaniaService {

	private static final Logger logger = LoggerFactory
			.getLogger(CampaniaServiceImpl.class);

	@Autowired
	private CategoriaService categoriaService;

	@Autowired
	private FormacionService formacionService;

	@Autowired
	private CategoriaDao categoriaDao;

	@Autowired
	private FormacionDao formacionDao;

	/**
	 * Busca todas las categorias/campañas y sus hijas y actualiza la
	 * informacion encotrada con la asignada por pantalla a la nueva campania
	 * que se quiere copiar.
	 * 
	 * @param categoria
	 *            Categoria desde la que se busca
	 * @return List<Categoria> donde se deja todas las categorias encontradas
	 */
	public List<Categoria> dameCategorias(Categoria categoria) {

		List<Categoria> arrCat = new ArrayList<Categoria>();

		List<Categoria> lstCategorias = new ArrayList<Categoria>();
		this.categoriaService.contarCategorias(categoria, lstCategorias);
		for (Categoria catAux : lstCategorias) {
			Categoria catPadre = this.categoriaDao.find(catAux);
			catPadre.setNombreEs(catAux.getNombreEs());
			catPadre.setNombreEu(catAux.getNombreEu());
			arrCat.add(catPadre);
		}// end for

		return arrCat;
	}// end dameCategorias

	/**
	 * Partiendo de una campania ya existente, copia todas sus categorias hijas
	 * y formaciones asociadas, creando una campania nueva.
	 * 
	 * @param categoria
	 *            Categoria objecto con la informacion de la campania
	 * @return List<Categoria> con la estructura de la nueva campania copiada y
	 *         todas sus categorias asociadas
	 */
	@Transactional(rollbackFor = Throwable.class)
	public void copiaCampaina(Categoria categoria) {

		try {
			Categoria cc = this.categoriaDao.find(categoria);
			cc.setNombreEs(categoria.getNombreEs());
			cc.setNombreEu(categoria.getNombreEu());
			cc.setFechaIniSol(categoria.getFechaIniSol());
			cc.setFechaFinSol(categoria.getFechaFinSol());
			// copia campania
			cc = this.categoriaService.addCategoria(cc);
			// String idCatNew = cc.getIdCat();
			String idCatOld = categoria.getIdCat();
			// guarda los grupos asociados a la categoria vieja para la
			// categoria nueva
			this.guardaGruposCategoria(cc, idCatOld);
			// guarda los perfiles asociados a la categoria vieja para la
			// categoria nueva
			this.guardaPerfilesCategoria(cc, idCatOld);
			// copia formaciones
			this.guardaFormacionesCategoria(cc, idCatOld);

			// de forma recursiva buscaremos todos los descendientes de la
			// campania y se copiaran
			this.copiaCampaniaHijos(cc, idCatOld);
		} catch (Y40aUDAException e) {
			throw e;
		} catch (Exception e) {
			logger.error("error al copiar las categorias: " + e.getMessage());
			throw new Y40aUDAException("error.campania.copia.categorias", true,
					new Exception());
		}
	}// end copiaCampaina

	/**
	 * Metodo recursivo que nos copia las categorias y formaciones, grupos... de
	 * una categoria padre/campania.
	 * 
	 * @param catNew
	 *            Categoria con la informacion de la categoria a copiar
	 * @param idOld
	 *            String con el identificativo de la categoria a copiar
	 */
	private void copiaCampaniaHijos(Categoria catNew, String idOld) {

		Categoria categoriaAux = new Categoria();
		Categoria categoriaPadre = new Categoria();
		categoriaPadre.setIdCat(idOld);
		categoriaAux.setCategoria(categoriaPadre);

		try {
			// busca info. de todas las hijas de la categoria seleccionada en
			// pantalla
			List<Categoria> lstCategoriaChildren = this.categoriaDao.findAll(
					categoriaAux, null);
			if (lstCategoriaChildren.size() > 0) {
				for (Categoria categoriaHijo : lstCategoriaChildren) {
					Categoria cc = this.categoriaDao.find(categoriaHijo);
					// actualiza el idPadre con el idCat nuevo
					cc.getCategoria().setIdCat(catNew.getIdCat());

					cc.setFechaIniSol((catNew.getFechaIniSol()));
					cc.setFechaFinSol((catNew.getFechaFinSol()));

					String idCatOld = categoriaHijo.getIdCat();
					cc = this.categoriaService.addCategoria(cc);
					// guarda los grupos asociados a la categoria vieja para la
					// categoria nueva
					this.guardaGruposCategoria(cc, idCatOld);
					// guarda los perfiles asociados a la categoria vieja para
					// la
					// categoria nueva
					this.guardaPerfilesCategoria(cc, idCatOld);
					// copia formaciones
					this.guardaFormacionesCategoria(cc, idCatOld);
					this.copiaCampaniaHijos(cc, idCatOld);

				}// end for
			}// end if
		} catch (Y40aUDAException e) {
			logger.error("error", e);
		} catch (Exception e) {
			logger.error("error", e);
		}
	}// end copiaCampaniaHijos

	/**
	 * Guarda los grupos asociados a la categoria vieja para la categoria nueva.
	 * 
	 * @param catAux
	 *            Categoria objeto con la informacion de la categoria nueva
	 * @param idCatOld
	 *            String con el identificador de la categoria de la que se
	 *            quiere copiar
	 */
	private void guardaGruposCategoria(Categoria categoria, String idCatOld) {

		try {
			Categoria catAux = new Categoria();
			catAux.setIdCat(idCatOld);
			// busca el id de los grupos asociados a la categoria vieja
			List<Object> lstGrupo = this.categoriaDao
					.buscaCategoriaGrupo(catAux);
			for (Object idGrupo : lstGrupo) {
				this.categoriaDao.copiaCategoriaGrupo(categoria.getIdCat(),
						idGrupo.toString());
			}// end for
		} catch (Exception e) {
			logger.error("Error al guardar los grupos: " + e.getMessage());
			throw new Y40aUDAException("error.campania.copia.grupos", true,
					new Exception());
		}
	}// end guardaGruposCategoria

	/**
	 * Guarda los perfiles asociados a la categoria vieja para la categoria
	 * nueva.
	 * 
	 * @param catAux
	 *            Categoria objeto con la informacion de la categoria nueva
	 * @param idCatOld
	 *            String con el identificador de la categoria de la que se
	 *            quiere copiar
	 */
	private void guardaPerfilesCategoria(Categoria categoria, String idCatOld) {

		try {
			Categoria catAux = new Categoria();
			catAux.setIdCat(idCatOld);
			// busca el id de los perfiles asociados a la categoria vieja
			List<Object> lstPerfil = this.categoriaDao
					.buscaCategoriaPerfil(catAux);
			for (Object idGrupo : lstPerfil) {
				this.categoriaDao.copiaCategoriaPerfil(categoria.getIdCat(),
						idGrupo.toString());
			}// end for
		} catch (Exception e) {
			logger.error("Error al guardar los perfiles: " + e.getMessage());
			throw new Y40aUDAException("error.campania.copia.perfiles", true,
					new Exception());
		}
	}// end guardaPerfilesCategoria

	/**
	 * Guarda las formaciones asociados a la categoria vieja para la categoria
	 * nueva
	 * 
	 * @param newLstCategorias
	 *            List<Categoria> lista con todas las categorias que se acaban
	 *            de copiar
	 * @param lstCategorias
	 *            List<Categoria> lista con todas las categorias de las que se
	 *            quiere copiar sus formaciones
	 */
	public void guardaFormacionesCategoria(Categoria categoria, String idOldCat) {

		try {
			// Busca las formaciones de la categoria
			Locale locale = LocaleContextHolder.getLocale();
			Pagination pagination = new Pagination();
			if (Y40aConstantes.CASTELLANO.equals(locale.getLanguage())) {
				pagination.setSort("NOMBREES");
				pagination.setAscDsc("ASC");
			} else {
				pagination.setSort("NOMBREEU");
				pagination.setAscDsc("ASC");
			}
			Formacion formacionAux = new Formacion();
			Formacion fAuxiliar = new Formacion();
			Categoria catAuxiliar = new Categoria();
			catAuxiliar.setIdCat(idOldCat);
			formacionAux.setCategoria(catAuxiliar);
			// busca las formaciones para la categoria
			List<Formacion> lstFormacionChildren = this.formacionDao.findAll(
					formacionAux, pagination);
			// para cada formacion
			for (Formacion forma : lstFormacionChildren) {
				// busca las etiquetas asociadas a la formacion
				List<Object> lstEtiqueta = this.formacionDao
						.buscaFormacionEtiqueta(forma);
				// busca la categoria en la nueva lista
				// forma el objeto para realizar la copia
				fAuxiliar = new Formacion(null, forma.getNombreEs(),
						forma.getNombreEu(), forma.getNombreCortoEs(),
						forma.getNombreCortoEu(), forma.getDescripcionEs(),
						forma.getDescripcionEu(), forma.getVisibleWeb(),
						forma.getNueva(), categoria);
				fAuxiliar = this.formacionService.addFormacion(fAuxiliar);

				// por cada etiqueta, copia las etiquetas a la formacion
				for (Object etiqueta : lstEtiqueta) {
					this.formacionDao.copiaFormacionEtiqueta(
							fAuxiliar.getIdFor(), etiqueta.toString());
				}// end for(etiqueta)
			}// end for(forma)
		} catch (Exception e) {
			logger.error("Error al guardar las formaciones para la copia: "
					+ e.getMessage());
			throw new Y40aUDAException("error.campania.copia.formaciones",
					true, new Exception());
		}

	}// end guardaFormacionesCategoria

}
