package com.ejie.u74a.control;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;

import javax.annotation.Resource;
import javax.validation.Valid;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;

import com.ejie.u74a.model.Articulo;
import com.ejie.u74a.model.Combo;
import com.ejie.u74a.model.CriteriosBusqueda;
import com.ejie.u74a.model.FichaSolicitud;
import com.ejie.u74a.model.ListaPaginadaServidor;
import com.ejie.u74a.model.Material;
import com.ejie.u74a.model.Peticion;
import com.ejie.u74a.model.RegistroES;
import com.ejie.u74a.model.TemaMaterial;
import com.ejie.u74a.model.Ubicacion;
import com.ejie.u74a.model.UbicacionInforme;
import com.ejie.u74a.service.ArticuloService;
import com.ejie.u74a.service.ComboService;
import com.ejie.u74a.service.GestionStockService;
import com.ejie.u74a.service.MaterialService;
import com.ejie.u74a.service.PeticionService;
import com.ejie.u74a.service.SolicitudService;
import com.ejie.u74a.service.TemaMaterialService;
import com.ejie.u74a.service.UbicacionService;
import com.ejie.u74a.util.Constants;
import com.ejie.u74a.util.Utilidades;
import com.ejie.u74a.util.exception.ParameterValidationException;

/**
 * Controlador para las peticiones AJAX.
 * 
 *  
 */
@Controller()
public class AjaxController {

	private static final Logger logger = LoggerFactory.getLogger(AjaxController.class);

	@Autowired()
	private TemaMaterialService temaMaterialService;

	@Autowired()
	private MaterialService materialService;

	@Autowired()
	private ArticuloService articuloService;

	@Autowired()
	private SolicitudService solicitudService;

	@Autowired()
	private UbicacionService ubicacionService;

	@Autowired()
	private PeticionService peticionService;

	@Autowired()
	private GestionStockService gestionStockService;

	@Autowired()
	private ComboService comboService;

	@Resource(name = "appMessageSource")
	private ReloadableResourceBundleMessageSource messageSource;

	/**
	 * Obtiene los temas asociados a un tipo de material vía ajax.
	 * 
	 * @param codigoTipoMaterial Código de tipo de material
	 * @param locale Locale actual
	 * 
	 * @return temas asociados en formato JSON
	 */
	@RequestMapping(value = "/usuario/getTemaMaterial", method = RequestMethod.GET)
	public @ResponseBody()
	List<Combo> getTemaMaterialAjax(@RequestParam(value = "codigoTipoMaterial") String codigoTipoMaterial, Locale locale) {
		List<Combo> temaMaterialSelect = new ArrayList<Combo>();

		try {
			int codigo = Integer.parseInt(codigoTipoMaterial);

			List<TemaMaterial> temasMaterial = this.temaMaterialService.findByTipoMaterial(codigo);

			for (TemaMaterial temaMaterial : temasMaterial) {
				temaMaterialSelect.add(new Combo(String.valueOf(temaMaterial.getCodigoTemaMaterial()), temaMaterial
						.getNombre(locale)));
			}
		} catch (NumberFormatException nfe) {
			// Si se selecciona la opción vacía se devuelve el mapa vacío
			logger.warn("El codigo de tipo de material seleccionado: '{}' no es un número entero", codigoTipoMaterial);
		}

		return temaMaterialSelect;

	}

	// ******** CATALOGO GENERAL INICIO **********

	/**
	 * Obtiene la lista de materiales del catálogo general
	 * 
	 * @param filtro CriteriosBusqueda
	 * @param locale Locale actual
	 * 
	 * @return List<catalogoGeneral>
	 */
	@RequestMapping(value = "/gestion/catalogoGeneral/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaCatalogoGeneral(@ModelAttribute() CriteriosBusqueda filtro, Locale locale) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.articuloService.findByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.articuloService.findByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Articulo>());
		}

		return resultado;

	}

	// ******** CATALOGO GENERAL FIN **********

	// ******** SOLICITUD FOLLETOS INICIO **********

	/**
	 * Obtiene la lista de articulos para la solicitud de folletos.
	 * 
	 * @param filtro CriteriosBusqueda
	 * @param locale Locale actual
	 * 
	 * @return List<Articulo>
	 */
	@RequestMapping(value = "/usuario/solicitudFolletos/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaSolicitudFolletos(@ModelAttribute() CriteriosBusqueda filtro, Locale locale) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		filtro.setDescatalogado("N"); // filtramos para que no aparezcan los descatalogados

		int numRegistros = this.articuloService.findByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.articuloService.findByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Articulo>());
		}

		return resultado;
	}

	/**
	 * Obtiene los datos a mostrar en el detalle de la solicitud.
	 * 
	 * @param codigosArticulosLst Lista de códigos de articulos seleccionados.
	 * 
	 * @return Objeto {@link FichaSolicitud} con los datos a mostrar en el detalle de la solicitud.
	 */
	@RequestMapping(value = "/usuario/solicitudFolletos/detalleSolicitud", method = RequestMethod.POST)
	public @ResponseBody()
	FichaSolicitud detalleSolicitud(@RequestBody() List<Integer> codigosArticulosLst) { // codigosMaterialesLst

		return this.solicitudService.detalleSolicitud(codigosArticulosLst); // codigosMaterialesLst
	}

	/**
	 * Obtiene los datos de una ubicacion para el código proporcionado.
	 * 
	 * @param codigoUbicacion El código de la ubicación.
	 * 
	 * @return Objeto {@link Ubicacion} con los datos de la ubicación.
	 */
	@RequestMapping(value = "/usuario/solicitudFolletos/getDatosUbicacion", method = RequestMethod.GET)
	public @ResponseBody()
	Ubicacion getDatosUbicacion(@RequestParam(value = "codigoUbicacion") int codigoUbicacion) {

		return this.ubicacionService.findByCodigo(codigoUbicacion);
	}

	/**
	 * Guarda la lista de artículos solicitados.
	 * 
	 * @param fichaSolicitud {@link FichaSolicitud} Objeto con los datos de la solicitud
	 * 
	 * @return String mensaje con el error si ha fallado al guardar la solicitud, blanco en caso contrario.
	 */
	@RequestMapping(value = "/usuario/solicitudFolletos/guardarSolicitud", method = RequestMethod.POST)
	public @ResponseBody()
	String guardarSolicitud(@RequestBody() @Valid() FichaSolicitud fichaSolicitud) {

		return this.solicitudService.guardarSolicitud(fichaSolicitud);
	}

	// ******** SOLICITUD FOLLETOS FIN **********

	// ******** FICHA MATERIAL INICIO **********

	/**
	 * Obtiene la lista de articulos asociados a los materiales seleccionados, la oficina del usuario.
	 * 
	 * @param codMaterial the cod material
	 * @return List<catalogoGeneral>
	 */
	@RequestMapping(value = "/usuario/fichaMaterial/verFichaMaterial", method = RequestMethod.POST)
	public @ResponseBody()
	Material getFichaMaterial(@RequestBody() int codMaterial) {

		return this.materialService.getDetalleMaterial(codMaterial, true, true);
	}

	/**
	 * Obtiene la lista de articulos asociados a los materiales seleccionados.
	 * 
	 * @param codArticulo El código del artículo.
	 * @return Articulo
	 */
	@RequestMapping(value = "/usuario/fichaMaterial/verFichaArticulo", method = RequestMethod.POST)
	public @ResponseBody()
	Articulo verFichaArticulo(@RequestBody() int codArticulo) {
		return this.articuloService.findByCodigo(codArticulo);
	}

	/**
	 * Guarda el tema asociado con el material, si el tema es principal en la tabla Material, si no en MaterialTema.
	 * 
	 * @param temaMaterial datos del tema
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/fichaMaterial/guardarTema", method = RequestMethod.POST)
	public @ResponseBody()
	TemaMaterial guardarTema(@RequestBody() TemaMaterial temaMaterial) {
		// return this.materialService.insertTemaMaterial(codMaterial,
		// codTemaMaterial, principal);
		return this.materialService.insertTemaMaterial(temaMaterial);
	}

	/**
	 * Elimina el tema asociado con el material, si el tema es principal en la tabla Material, si no en MaterialTema.
	 * 
	 * @param temaMaterial datos del tema
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/fichaMaterial/borrarTema", method = RequestMethod.POST)
	public @ResponseBody()
	TemaMaterial borrarTema(@RequestBody() TemaMaterial temaMaterial) {
		return this.materialService.borrarTemaMaterial(temaMaterial);
	}

	/**
	 * Si el código de material viene vacío crea un nuevo material, si no, lo actualiza.
	 * 
	 * @param material {@link Material} objeto con la información.
	 * @return String mensaje con código del material si ha ido bien si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/fichaMaterial/guardarMaterial", method = RequestMethod.POST)
	public @ResponseBody()
	Material guardarMaterial(@RequestBody() Material material) {

		if (material == null) {
			throw new ParameterValidationException("El material no puede estar nulooo");
		}

		// Dependiendo de si viene con codigoMaterial o no, será un alta o una
		// modificación.

		// TODO si no llega descatalogado es porque no se ha enviado el check y
		// es N, mandarlo siempre?
		if (material.getDescatalogado() == null) {
			material.setDescatalogado("N");
		}

		// en caso de se modo envío por ejemplares
		if (material.getArticuloPromocional() != null
				&& material.getArticuloPromocional().getModoPeticion().equals(Constants.MODO_ENVIO_EJEMPLARES)) {
			material.getArticuloPromocional().setNumEjemplares(null); // vacio el campo de num ejemplares caja
		}

		if (material.getCodigoMaterial() > 0) {
			return this.materialService.actualizarMaterial(material);
		} else {
			return this.materialService.insertarMaterial(material);
		}

	}// guardarMaterial

	/**
	 * Si el código de artículo viene vacío crea un nuevo artículo, si no, lo actualiza.
	 * 
	 * @param articulo {@link Articulo} objeto con la información.
	 * @return String mensaje con código del material si ha ido bien si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/fichaMaterial/guardarArticulo", method = RequestMethod.POST)
	public @ResponseBody()
	Articulo guardarArticulo(@RequestBody() @Valid() Articulo articulo) {

		if (articulo == null) {
			throw new ParameterValidationException(this.messageSource.getMessage("articulo.errorNulo", null,
					LocaleContextHolder.getLocale()));
		}

		// Dependiendo de si viene con codArticulo o no, será un alta o una
		// modificación.

		// TODO si no llega descatalogado es porque no se ha enviado el check y
		// es N, mandarlo siempre?
		if (articulo.getDescatalogado() == null) {
			articulo.setDescatalogado("N");
		}

		// en caso de se modo envío por ejemplares
		if (articulo.getModoPeticion().equals(Constants.MODO_ENVIO_EJEMPLARES)) {
			articulo.setNumEjemplares(null); // vacio el campo de num ejemplares caja
		}

		if (articulo.getCodArticulo() != null) {
			return this.articuloService.actualizarArticulo(articulo);
		} else {
			return this.articuloService.insertarArticulo(articulo);
		}
	}// guardarArticulo

	/**
	 * Borra el artículo y sus idiomas para el código de artículo proporcionado.
	 * 
	 * @param codigoArticulo El código del artículo a eliminar.
	 * @param validarPdtes Indica si hay que validar las peticiones pendientes a S, en otro caso no valida.
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/fichaMaterial/borrarArticuloIdiomas", method = RequestMethod.GET)
	public @ResponseBody()
	String borrarArticuloIdiomas(@RequestParam(value = "codigoArticulo") int codigoArticulo,
			@RequestParam(value = "validarPdtes") String validarPdtes) {

		return this.articuloService.borrarArticuloIdiomas(codigoArticulo, validarPdtes);
	}// borrarArticuloIdiomas

	/**
	 * Borra todos los artículo, sus idiomas y peticiones para el código de material proporcionado, graba el nuevo tipo
	 * de idioma.
	 * 
	 * @param codigoMaterial El código del material.
	 * @param monolingue S para indicar que es monolingue, en cualquier otro caso se informará como N.
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/fichaMaterial/cambioTipoIdiomaDeMaterial", method = RequestMethod.GET)
	public @ResponseBody()
	String cambioTipoIdiomaDeMaterial(@RequestParam(value = "codigoMaterial") int codigoMaterial,
			@RequestParam(value = "monolingue") String monolingue) {

		return this.materialService.cambioTipoIdiomaDeMaterial(codigoMaterial, monolingue);
	}// cambioTipoIdiomaDeMaterial

	// ******** FICHA MATERIAL FIN **********

	// ******** SOLICITUES RECIBIDAS INICIO **********

	/**
	 * Obtiene la lista de solicitudes recibidas.
	 * 
	 * @param filtro CriteriosBusqueda
	 * @param locale Locale actual
	 * 
	 * @return List<Peticion>
	 */
	@RequestMapping(value = "/gestion/solicitudesRecibidas/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaSolicitudRecibidas(@ModelAttribute() CriteriosBusqueda filtro, Locale locale) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.peticionService.findByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.peticionService.findByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Articulo>());
		}

		return resultado;
	}

	/**
	 * Obtiene la lista de solicitudes recibidas seleccionadas para aceptar/rechazar
	 * 
	 * @param params Parámetros de la request
	 * @return List<Peticion>
	 */
	@RequestMapping(value = "/gestion/solicitudesRecibidas/getListaSeleccionadas", method = RequestMethod.POST)
	public @ResponseBody()
	List<Peticion> getListaSolicitudRecibidas(@RequestBody() LinkedHashMap<String, Object> params) {

		// this.peticionService.findByCriteria(filtro);
		@SuppressWarnings("unchecked")
		List<Integer> codigosPeticionLst = (List<Integer>) params.get("codigos");

		// si tengo ids para cargar, hago la búsqueda de sus datos
		if (!codigosPeticionLst.isEmpty()) {
			CriteriosBusqueda filtro = new CriteriosBusqueda();
			filtro.setCodigos(codigosPeticionLst);
			return this.peticionService.findByCriteria(filtro);
		} else {// si no hay ningún id, devuelvo una lista vacia
			return new ArrayList<Peticion>();
		}

	}

	/**
	 * Cambia el estado a una petición.
	 * 
	 * @param peticiones Lista de peticiones a aceptar/rechazar
	 * 
	 * @return String con resultado Ok si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/solicitudesRecibidas/cambiarEstadoPeticion", method = RequestMethod.POST)
	public @ResponseBody()
	// String cambiarEstadoPeticion(@RequestBody() LinkedHashMap<Object, Object> params) { //
	String cambiarEstadoPeticion(@RequestBody() List<Peticion> peticiones) { //

		// String estado = (String) params.get("estado");
		// List<Peticion> peticiones = (List<Peticion>) params.get("peticiones");
		// Peticion peticion = peticiones.get(0);
		// return peticion.getDescripcionIdioma();
		return this.peticionService.cambiarEstadoPeticion(peticiones);

	}

	/**
	 * Genera un informe para enviar a almacén con un conjunto de peticiones aceptadas.
	 * 
	 * @param params Parámetros de la request
	 * 
	 * @return Objeto con toda la información necesaria para la generación del informe.
	 */
	@RequestMapping(value = "/gestion/solicitudesRecibidas/generarInforme", method = RequestMethod.POST)
	public @ResponseBody()
	List<UbicacionInforme> generarInforme(@RequestBody() LinkedHashMap<String, Object> params) { //

		/*
		 * @RequestParam(value = "codigosPeticion", required = false) List<Integer> codigosPeticionLst,
		 * 
		 * @RequestParam(value = "estado", required = false) String estado
		 */

		List<Integer> codigosPeticionLst = (List<Integer>) params.get("codigosPeticion");
		return this.peticionService.generarInforme(codigosPeticionLst);

	}

	// ******** SOLICITUES RECIBIDAS FIN **********

	// ******** GESTION STOCK INICIO **********

	/**
	 * Obtiene la lista de articulos para la gestion de stocks.
	 * 
	 * @param filtro CriteriosBusqueda
	 * @param locale Locale actual
	 * 
	 * @return List<Articulo>
	 */
	@RequestMapping(value = "/gestion/gestionStock/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaGestionStock(@ModelAttribute() CriteriosBusqueda filtro, Locale locale) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		filtro.setDescatalogado("N");
		int numRegistros = this.gestionStockService.findByCountCriteria(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.gestionStockService.findByCriteria(filtro));
		} else {
			resultado.setRows(new ArrayList<Articulo>());
		}

		return resultado;
	}

	/**
	 * Obtiene la información del artículo para la gestión de stocks
	 * 
	 * @param codigoArticulo El código del artículo.
	 * @return Articulo
	 */
	@RequestMapping(value = "/gestion/gestionStock/getArticulo", method = RequestMethod.POST)
	public @ResponseBody()
	Articulo getArticuloGestionStock(@RequestBody() int codigoArticulo) {
		return this.articuloService.findByCodigoExtra(codigoArticulo);
	}

	/**
	 * Obtiene la lista de registros de entrada salidas para un artículo para la gestión de stocks.
	 * 
	 * @param filtro objeto con los parámetros codigoArticulo, offset, limit.
	 * @return objeto {@link ListaPaginadaServidor} con la lista de registros {@link List<RegistrosES>} y la información
	 *         para la paginación.
	 */
	@RequestMapping(value = "/gestion/gestionStock/getListaES", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaGestionStock(@ModelAttribute() CriteriosBusqueda filtro) {

		/*
		 * int codigoArticulo = (Integer) params.get("codigoArticulo"); int offset = (Integer) params.get("offset"); int
		 * limit = (Integer) params.get("limit");
		 */
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		if (filtro.getCodArticulo() != null) {
			int numRegistros = this.gestionStockService.findRegistrosESByCodigoCount(filtro);
			resultado.setTotal(numRegistros);
			if (numRegistros > 0) {
				resultado.setRows(this.gestionStockService.findRegistrosESByCodigo(filtro));
			} else {
				resultado.setRows(new ArrayList<RegistroES>());
			}
		} else {
			resultado.setRows(new ArrayList<RegistroES>());
			resultado.setTotal(0);
		}

		return resultado;
	}

	/**
	 * Actualiza la información del stock e inventario para el artículo..
	 * 
	 * @param articulo {@link Articulo} objeto con la información.
	 * @return String mensaje con el resultado Ok si ha ido bien.
	 */
	@RequestMapping(value = "/gestion/gestionStock/actualizarStock", method = RequestMethod.POST)
	public @ResponseBody()
	String actualizarStock(@RequestBody() Articulo articulo) {

		if (articulo == null) {
			throw new ParameterValidationException(this.messageSource.getMessage("articulo.errorNulo", null,
					LocaleContextHolder.getLocale()));
		}

		return this.articuloService.actualizarInfoStockArticulo(articulo);
	}// actualizarStock

	// ******** GESTION STOCK FIN **********

	/**
	 * Obtiene la lista de solicitudes realizadas por el usuario conectado.
	 * 
	 * @param filtro CriteriosBusqueda
	 * @param locale Locale actual
	 * 
	 * @return List<Articulo>
	 */
	@RequestMapping(value = "/usuario/misSolicitudes/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	ListaPaginadaServidor getListaMisSolicitud(@ModelAttribute() CriteriosBusqueda filtro, Locale locale) {
		ListaPaginadaServidor resultado = new ListaPaginadaServidor();

		int numRegistros = this.peticionService.findByCountCriteriaMisSolicitudes(filtro);
		resultado.setTotal(numRegistros);
		if (numRegistros > 0) {
			resultado.setRows(this.peticionService.findByCriteriaMisSolicitudes(filtro));
		} else {
			resultado.setRows(new ArrayList<Articulo>());
		}

		return resultado;
	}

	/**
	 * Obtiene los datos de una solicitud de un folleto
	 * 
	 * @param codigoPeticion código de la petición
	 * @return detalle de la petición
	 */
	@RequestMapping(value = "/usuario/misSolicitudes/getDatosSolicitud", method = RequestMethod.GET)
	public @ResponseBody()
	Peticion getDatosSolicitud(@RequestParam(value = "codigoPeticion") int codigoPeticion) {
		return this.peticionService.findDetalleByCodigo(codigoPeticion);
	}

	/**
	 * 
	 * @return Listado de puntos de recepción
	 */
	@RequestMapping(value = "/usuario/puntosRecepcion/getLista", method = RequestMethod.POST)
	public @ResponseBody()
	List<Ubicacion> getListaPuntosRecepcion() {
		return this.ubicacionService.findByCodigoOficina(Utilidades.getDetallesUsuario().getCodigoOficina());

	}

	/**
	 * Obtiene la lista de municipios para cargar un combo
	 * 
	 * @param codigoProvincia código de la provincia
	 * @return lista de municipios
	 */
	@RequestMapping(value = "/usuario/municipios/getCombo", method = RequestMethod.GET)
	public @ResponseBody()
	List<Combo> getComboMunicipiosByProvincia(@RequestParam(value = "codigoProvincia") int codigoProvincia) {
		return this.comboService.findMunicipiosByProvincia(codigoProvincia);

	}

	/**
	 * Obtiene la lista de municipios para cargar un combo
	 * 
	 * @param codigoProvincia código de la provincia
	 * @param codigoMunicipio código del municipio
	 * @return lista de municipios
	 */
	@RequestMapping(value = "/usuario/localidades/getCombo", method = RequestMethod.GET)
	public @ResponseBody()
	List<Combo> getComboLocalidadesByMunicipio(@RequestParam(value = "codigoProvincia") int codigoProvincia,
			@RequestParam(value = "codigoMunicipio") int codigoMunicipio) {
		return this.comboService.findLocalidadesByMunicipio(codigoProvincia, codigoMunicipio);

	}

	/**
	 * Guarda los datos de un punto de recepción
	 * 
	 * @param ubicacion datos del punto de recepción
	 * @return resultado de la operación
	 */
	@RequestMapping(value = "/usuario/puntosRecepcion/guardar", method = RequestMethod.POST)
	public @ResponseBody()
	int guardarPuntoRecepcion(@RequestBody() Ubicacion ubicacion) {

		return this.ubicacionService.saveUbicacion(ubicacion);
	}

	/**
	 * Borra un punto de recepción
	 * 
	 * @param codigoUbicacion código de la ubicación
	 * @return resultado de la operación
	 */
	@RequestMapping(value = "/usuario/puntosRecepcion/borrar", method = RequestMethod.GET)
	public @ResponseBody()
	int deletePuntoRecepcion(@RequestParam(value = "codigoUbicacion") int codigoUbicacion) {
		return this.ubicacionService.deleteUbicacion(codigoUbicacion);

	}

	/**
	 * Obtiene la lista de provincias, excepto las de euskadi
	 * 
	 * @return lista de provincias (excepto euskadi)
	 */
	@RequestMapping(value = "/usuario/provincias/getCombo", method = RequestMethod.GET)
	public @ResponseBody()
	List<Combo> getComboProvincias() {
		return this.comboService.findProvincias();

	}
}
