/*
 * Copyright 2011 E.J.I.E., S.A.
 *
 * Licencia con arreglo a la EUPL, Versión 1.1 exclusivamente (la «Licencia»);
 * Solo podrá usarse esta obra si se respeta la Licencia.
 * Puede obtenerse una copia de la Licencia en
 *
 * http://ec.europa.eu/idabc/eupl.html
 *
 * Salvo cuando lo exija la legislación aplicable o se acuerde por escrito,
 * el programa distribuido con arreglo a la Licencia se distribuye «TAL CUAL»,
 * SIN GARANTÍAS NI CONDICIONES DE NINGÚN TIPO, ni expresas ni implícitas.
 * Véase la Licencia en el idioma concreto que rige los permisos y limitaciones
 * que establece la Licencia.
 */
package com.ejie.u74a.util;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

import com.ejie.u74a.log.LogConstants;

/**
 * 
 *  
 * 
 */
public class DateTimeManager {

	private static final int INCREMENTO_HTTP_EXPIRED_DATA = 3;

	/**
	 * Constructor privado. Clase de utilidad.
	 */
	private DateTimeManager() {
	}

	/**
	 * Obtiene un formateador de fecha dependiente del locale para los idiomas
	 * castellano, francés, inglés y euskera
	 * 
	 * @param locale locale
	 * 
	 * @return formateador de fecha
	 */
	public static SimpleDateFormat getDateTimeFormat(Locale locale) {
		if (locale.getLanguage().equals(Constants.EUSKARA)) {
			return Constants.YYYYMMDD_DATE_FORMAT;
		} else if (locale.getLanguage().equals(Constants.FRANCAIS) || locale.getLanguage().equals(Constants.CASTELLANO)
				|| locale.getLanguage().equals(Constants.ENGLISH)) {
			return Constants.DDMMYYYY_DATE_FORMAT;
		} else {
			return Constants.DDMMYYYY_DATE_FORMAT;
		}
	}

	/**
	 * Obtiene un formateador de fecha-hora dependiente del locale para los
	 * idiomas castellano, francés, inglés y euskera
	 * 
	 * @param locale locale
	 * 
	 * @return formateador de fecha-hora
	 */
	public static SimpleDateFormat getTimestampFormat(Locale locale) {
		if (locale.getLanguage().equals(Constants.EUSKARA)) {
			return Constants.YYYYMMDD_HHMMSS_DATE_FORMAT;
		} else if (locale.getLanguage().equals(Constants.FRANCAIS) || locale.getLanguage().equals(Constants.CASTELLANO)
				|| locale.getLanguage().equals(Constants.ENGLISH)) {
			return Constants.DDMMYYYY_HHMMSS_DATE_FORMAT;
		} else {
			return Constants.DDMMYYYY_HHMMSS_DATE_FORMAT;
		}
	}

	/**
	 * Obtiene un formateador de fecha-hora dependiente del locale
	 * 
	 * @param locale locale
	 * 
	 * @return formateador de hora
	 */
	public static SimpleDateFormat getTimeFormat(Locale locale) {
		return Constants.HHMMSS_TIME_FORMAT;
	}

	/**
	 * Obtiene la fecha y hora actual formateadas para logs
	 * 
	 * @return fecha y hora actual
	 */
	public static String getDateTime() {
		DateFormat dateFormat = new SimpleDateFormat(LogConstants.DATETIMEFORMAT);
		Date date = new Date();
		return dateFormat.format(date);
	}

	/**
	 * Obtiene la fecha de expiración http formateada a partir de la fecha
	 * actual incrementada en 3 días
	 * 
	 * @return fecha de expiración http formateada
	 */
	public static String getHttpExpiredDate() {
		final Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.DAY_OF_YEAR, DateTimeManager.INCREMENTO_HTTP_EXPIRED_DATA);

		final DateFormat httpDateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US);
		httpDateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));

		return httpDateFormat.format(calendar.getTime());
	}

	/**
	 * Obtiene la fecha de expiración http formateada a partir de una fecha dada
	 * 
	 * @param date fecha
	 * 
	 * @return fecha de expiración http formateada
	 */
	public static String getHttpExpiredDate(Date date) {

		final DateFormat httpDateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US);
		httpDateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));

		return httpDateFormat.format(date);
	}
}