package com.ejie.u74a.service;

import java.util.List;

import com.ejie.u74a.model.CriteriosBusqueda;
import com.ejie.u74a.model.Material;
import com.ejie.u74a.model.TemaMaterial;

/**
 * Interfaz de los servicios de acceso a los datos de materiales.
 * 
 *  
 */
public interface MaterialService {

	/**
	 * Obtiene los materiales dependiendo del filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return lista con los materiales
	 */
	List<Material> findByCriteria(CriteriosBusqueda filtro);

	/**
	 * Obtiene el número de los materiales que cumplen del filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return int número de los materiales que cumplen el filtro aplicado.
	 */
	int findByCountCriteria(CriteriosBusqueda filtro);

	/**
	 * Obtiene los materiales dependiendo para el código proporcionado.
	 * 
	 * @param codigoMaterial El código del material.
	 * @param incluirTemasSecundarios true: Indica si en la consulta se deben obtener también los temas secundarios del
	 *            material si los hubiera, false: para no incluirlos.
	 * @param incluirInfoArticulo true:Indica si en la consulta se deben obtener también los articulos e idiomas del
	 *            material si los hubiera, false: para no incluirlos.
	 * @return Objeto {@link Material} con los datos del material.
	 */
	Material getDetalleMaterial(int codigoMaterial, boolean incluirTemasSecundarios, boolean incluirInfoArticulo);

	/**
	 * Guarda el tema asociado con el material, si el tema es principal en la tabla Material, si no en MaterialTema.
	 * 
	 * @param temaMaterial objeto TemaMaterial
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	TemaMaterial insertTemaMaterial(TemaMaterial temaMaterial);

	/**
	 * Borra el tema asociado con el material, si el tema es principal en la tabla Material, si no en MaterialTema.
	 * 
	 * @param temaMaterial objeto TemaMaterial
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	TemaMaterial borrarTemaMaterial(TemaMaterial temaMaterial);

	/**
	 * Actualizar el material con los datos proporcionados.
	 * 
	 * @param material objeto {@link Material} con la información.
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	Material actualizarMaterial(Material material);

	/**
	 * Inserta un nuevo material.
	 * 
	 * @param material objeto {@link Material} con la información.
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	Material insertarMaterial(Material material);

	/**
	 * Borra todos los artículo, sus idiomas y peticiones para el código de material proporcionado, graba el nuevo tipo
	 * de idioma.
	 * 
	 * @param codigoMaterial El código del material.
	 * @param monolingue S para indicar que es monolingue, en cualquier otro caso se informará como N.
	 * @return String mensaje con resultado OK si ha ido bien.
	 */
	String cambioTipoIdiomaDeMaterial(int codigoMaterial, String monolingue);

}