package com.ejie.u74a.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.u74a.model.PerfilUsuario;
import com.ejie.u74a.model.TipoPerfilUsuario;
import com.ejie.u74a.model.Usuario;
import com.ejie.u74a.util.Constants;

/**
 * Clase que accede a la base de datos para obtener datos de usuario. Tablas: PERSONAS, PERFILES y PERFILPERSONA
 * 
 *  
 */
@Repository()
@Transactional()
public class UsuarioDaoImpl implements UsuarioDao {

	// Administrador de Oficina
	private static final int ADMINISTRADOR_OFICINA = 2;

	private JdbcTemplate jdbcTemplate;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla PERSONAS con los objetos de la clase Usuario
	 */
	private static final class UsuarioRowMapper implements RowMapper<Usuario> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Usuario mapRow(ResultSet rs, int rowNum) throws SQLException {
			int codigoUsuario = rs.getInt("T03_COUSU");
			String codigoOficina = rs.getString("T03_COOFI");
			String nombre = rs.getString("T03_NOUSU");
			String cargo = rs.getString("T03_NOCAR");
			String telefono = rs.getString("T03_NUTLF");
			String fax = rs.getString("T03_NUFAX");
			String email = rs.getString("T03_COEML");
			String login = rs.getString("T03_LOGIN");
			String password = rs.getString("T03_PASS");

			return new Usuario(codigoUsuario, codigoOficina, nombre, cargo, telefono, fax, email, login, password);
		}
	}

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla PERFILPERSONA con los objetos de la clase PerfilUsuario
	 */
	private static final class PerfilUsuarioRowMapper implements RowMapper<PerfilUsuario> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public PerfilUsuario mapRow(ResultSet rs, int rowNum) throws SQLException {
			int codigoPerfil = rs.getInt("T05_COPER");

			String tipoPerfil = TipoPerfilUsuario.get(codigoPerfil).getTipoPerfilUsuario();

			return new PerfilUsuario(codigoPerfil, tipoPerfil);
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UsuarioDao#findByCodigoUsuario(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Usuario findByCodigoUsuario(int codigoUsuario) {
		Usuario usuario = this.jdbcTemplate
				.queryForObject(
						"SELECT T03_COUSU, T03_COOFI, T03_NOUSU, T03_NOCAR, T03_NUTLF, T03_NUFAX, T03_COEML, T03_LOGIN, T03_PASS FROM PERSONAS WHERE T03_COUSU = ?",
						new UsuarioRowMapper(), codigoUsuario);
		List<PerfilUsuario> perfilesUsuario = this.jdbcTemplate.query(
				"SELECT T05_COPER FROM PERFILPERSONA WHERE T05_COUSU = ?", new PerfilUsuarioRowMapper(),
				usuario.getCodigoUsuario());

		usuario.setPerfilesUsuario(perfilesUsuario);

		// Si el usuario pertenece a la oficina del Departamento:
		// 1C028 - GOBIERNO VASCO - DIRECCION DE PROMOCION TURISTICA
		// y es Administrador de Oficina tendrá el perfil de GESTOR
		if (Constants.CODIGO_OFICINA_GV.equals(usuario.getCodigoOficina())) {
			if (usuario.getPerfilesUsuario().contains(
					new PerfilUsuario(ADMINISTRADOR_OFICINA, TipoPerfilUsuario.USUARIO.getTipoPerfilUsuario()))) {
				for (PerfilUsuario perfil : usuario.getPerfilesUsuario()) {
					if (perfil.getCodigoPerfil() == ADMINISTRADOR_OFICINA) {
						perfil.setTipoPerfil(TipoPerfilUsuario.GESTOR.getTipoPerfilUsuario());
					}
				}
			}

		}

		return usuario;
	}

}
