package com.ejie.u74a.dao;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Locale;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.u74a.model.Ubicacion;
import com.ejie.u74a.util.Constants;

/**
 * Clase que accede a la base de datos para obtener datos de ubicaciones. Tabla: UBICACION
 * 
 *  
 */
@Repository()
@Transactional()
public class UbicacionDaoImpl implements UbicacionDao {

	private JdbcTemplate jdbcTemplate;

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla UBICACION con los objetos de la clase Ubicacion
	 */
	private static final class UbicacionRowMapper implements RowMapper<Ubicacion> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Ubicacion mapRow(ResultSet rs, int rowNum) throws SQLException {

			Ubicacion ubicacion = new Ubicacion();

			ubicacion.setCodigoUbicacion(rs.getInt("T07_COUBICA"));
			ubicacion.setCodigoOficina(rs.getString("T07_COOR")); // TODO revisar Este campo igual sobra
			ubicacion.setNombreUbicacion(rs.getString("T07_NOUBICA"));
			ubicacion.setNombreDireccion(rs.getString("T07_NODIR"));
			ubicacion.setCodigoPostal(rs.getString("T07_COCP"));
			ubicacion.setObservaciones(rs.getString("T07_TEOBS"));
			ubicacion.setCodigoProvincia(rs.getInt("T07_CODPROVINC"));
			ubicacion.setCodigoMunicipio(rs.getInt("T07_CODMUNICIP"));
			ubicacion.setCodigoLocalidad(rs.getInt("T07_COLO"));

			ubicacion.setNombreContacto(rs.getString("T07_NOCON"));
			ubicacion.setTelefono(rs.getString("T07_NUTLF"));
			ubicacion.setFax(rs.getString("T07_NUFAX"));
			ubicacion.setEmail(rs.getString("T07_COEML"));

			// Datos externos
			ubicacion.setNombreProvinciaCastellano(rs.getString("NOMPROVINC"));
			ubicacion.setNombreProvinciaEuskera(rs.getString("NOMPROVINCE"));
			ubicacion.setNombreLocalidad(rs.getString("NOMLOCALID"));
			ubicacion.setNombreMunicipio(rs.getString("NOMMUNICIP"));

			if (new Locale("eu").equals(LocaleContextHolder.getLocale())) {
				ubicacion.setNombreProvincia(ubicacion.getNombreProvinciaEuskera());
			} else {
				ubicacion.setNombreProvincia(ubicacion.getNombreProvinciaCastellano());
			}

			ubicacion.setExtranjero(rs.getInt("T07_EXTRANJERA"));
			ubicacion.setDireccionExtranjero(rs.getString("T07_NODIREXTRANJERA"));

			return ubicacion;
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#findByCodigo(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Ubicacion findByCodigo(int codigoUbicacion) {
		StringBuilder query = new StringBuilder(
				"SELECT T07_COUBICA,T07_COOR,T07_NOUBICA,T07_NODIR,T07_COCP,T07_TEOBS,T07_CODPROVINC,T07_CODMUNICIP,T07_COLO,")
				.append(" T07_NOCON, T07_NUTLF, T07_NUFAX, T07_COEML, ")
				.append(" P.NOMPROVINC, P.NOMPROVINCE, M.NOMMUNICIP, L.NOMLOCALID, ")
				.append(" T07_EXTRANJERA, T07_NODIREXTRANJERA ")
				.append(" FROM UBICACIONES U ")
				.append("LEFT JOIN PROVINCIA P ON P.CODPROVINC=U.T07_CODPROVINC ")
				.append("LEFT JOIN MUNICIPIO M ON M.CODPROVINC=U.T07_CODPROVINC and M.CODMUNICIP=U.T07_CODMUNICIP ")
				.append("LEFT JOIN LOCALIDAD L ON L.CODPROVINC=U.T07_CODPROVINC and L.CODMUNICIP=U.T07_CODMUNICIP AND L.CODLOCALID=U.T07_COLO  WHERE T07_COUBICA = ?");

		return this.jdbcTemplate.query(query.toString(), new UbicacionRowMapper(), codigoUbicacion).get(0);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#findByCodigoOficina(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Ubicacion> findByCodigoOficina(String codigoOficina) {
		StringBuilder query = new StringBuilder();
		query.append("SELECT U.T07_COUBICA, U.T07_COOR, U.T07_NOUBICA, U.T07_NODIR, U.T07_COCP, U.T07_TEOBS, U.T07_CODPROVINC, U.T07_CODMUNICIP, U.T07_COLO, ");
		query.append(" U.T07_NOCON, U.T07_NUTLF, U.T07_NUFAX, U.T07_COEML, ");
		query.append(" P.NOMPROVINC, P.NOMPROVINCE, M.NOMMUNICIP, L.NOMLOCALID, ");
		query.append(" T07_EXTRANJERA, T07_NODIREXTRANJERA ");
		query.append(" FROM UBICACIONES U");
		query.append(" LEFT JOIN PROVINCIA P ON P.CODPROVINC=U.T07_CODPROVINC ");
		query.append(" LEFT JOIN MUNICIPIO M ON M.CODPROVINC=U.T07_CODPROVINC and M.CODMUNICIP=U.T07_CODMUNICIP ");
		query.append(" LEFT JOIN LOCALIDAD L ON L.CODPROVINC=U.T07_CODPROVINC and L.CODMUNICIP=U.T07_CODMUNICIP AND L.CODLOCALID=U.T07_COLO ");
		query.append(" WHERE U.T07_COOR = ? AND T07_BAJA = ?");

		query.append(" ORDER BY U.T07_NOUBICA ");

		return this.jdbcTemplate.query(query.toString(), new UbicacionRowMapper(), codigoOficina,
				Constants.REGISTRO_ACTIVO);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#updateUbicacion(com.ejie.u74a.model.Ubicacion)
	 */
	@Transactional()
	@Override()
	public int updateUbicacion(Ubicacion ubicacion) {
		StringBuilder query = new StringBuilder();
		query.append(" UPDATE UBICACIONES SET ");
		query.append(" T07_NOUBICA = ?, ");
		query.append(" T07_NOCON = ?, ");
		query.append(" T07_NUTLF = ?, ");
		query.append(" T07_NUFAX = ?, ");
		query.append(" T07_COEML = ?, ");
		query.append(" T07_COLO = ?,");
		query.append(" T07_NODIR = ?, ");
		query.append(" T07_COCP = ?, ");
		query.append(" T07_TEOBS = ?, ");
		query.append(" T07_CODPROVINC = ?, ");
		query.append(" T07_CODMUNICIP = ?, ");
		query.append(" T07_EXTRANJERA = ?, ");
		query.append(" T07_NODIREXTRANJERA = ? ");
		query.append(" WHERE T07_COUBICA = ? ");

		return this.jdbcTemplate.update(query.toString(), ubicacion.getNombreUbicacion(),
				ubicacion.getNombreContacto(), ubicacion.getTelefono(), ubicacion.getFax(), ubicacion.getEmail(),
				ubicacion.getCodigoLocalidad(), ubicacion.getNombreDireccion(), ubicacion.getCodigoPostal(),
				ubicacion.getObservaciones(), ubicacion.getCodigoProvincia(), ubicacion.getCodigoMunicipio(),
				ubicacion.getExtranjero() == null ? 0 : ubicacion.getExtranjero(), ubicacion.getDireccionExtranjero(),
				ubicacion.getCodigoUbicacion());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#insertUbicacion(com.ejie.u74a.model.Ubicacion)
	 */
	@Transactional()
	@Override()
	public int insertUbicacion(Ubicacion ubicacion) {
		StringBuilder query = new StringBuilder();
		query.append(" INSERT INTO UBICACIONES ( ");

		query.append(" T07_COUBICA,T07_COOR,T07_NOUBICA, ");

		query.append(" T07_NOCON,T07_NUTLF,T07_NUFAX, ");
		query.append(" T07_COEML,T07_COLO,T07_NODIR, ");
		query.append(" T07_COCP,T07_TEOBS,T07_CODPROVINC,T07_CODMUNICIP, ");
		query.append(" T07_EXTRANJERA, T07_NODIREXTRANJERA ) ");

		query.append(" VALUES (SEQ_Q5307T00.NEXTVAL,?,?,?,?,?,?,?,?,?,?,?,?,?,?) ");

		return this.jdbcTemplate.update(query.toString(), ubicacion.getCodigoOficina(), ubicacion.getNombreUbicacion(),
				ubicacion.getNombreContacto(), ubicacion.getTelefono(), ubicacion.getFax(), ubicacion.getEmail(),
				ubicacion.getCodigoLocalidad(), ubicacion.getNombreDireccion(), ubicacion.getCodigoPostal(),
				ubicacion.getObservaciones(), ubicacion.getCodigoProvincia(), ubicacion.getCodigoMunicipio(),
				ubicacion.getExtranjero() == null ? 0 : ubicacion.getExtranjero(), ubicacion.getDireccionExtranjero());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#deleteLogicaUbicacion(java.lang.Integer)
	 */
	@Transactional()
	@Override()
	public int deleteLogicaUbicacion(Integer codUbicacion) {
		StringBuilder query = new StringBuilder();
		query.append(" UPDATE UBICACIONES SET ");
		query.append(" T07_BAJA = ? ");
		query.append(" WHERE T07_COUBICA = ? ");

		return this.jdbcTemplate.update(query.toString(), Constants.REGISTRO_BORRADO, codUbicacion);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#deleteFisicaUbicacion(java.lang.Integer)
	 */
	@Transactional()
	@Override()
	public int deleteFisicaUbicacion(Integer codUbicacion) {
		StringBuilder query = new StringBuilder();
		query.append(" DELETE UBICACIONES ");
		query.append(" WHERE T07_COUBICA = ? ");

		return this.jdbcTemplate.update(query.toString(), codUbicacion);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.UbicacionDao#findExisteUbicacionPeticion(java.lang.Integer)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Boolean findExisteUbicacionPeticion(Integer codUbicacion) {

		StringBuilder query = new StringBuilder("SELECT COUNT(*) FROM PETICION WHERE COUBICA = ?");

		Object[] args = { codUbicacion };
		int num = this.jdbcTemplate.queryForObject(query.toString(), args, Integer.class);
		if (num > 0) {
			return true;
		}
		return false;

	}
}
