package com.ejie.u74a.dao;

import java.math.BigDecimal;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.u74a.model.Articulo;
import com.ejie.u74a.model.CriteriosBusqueda;
import com.ejie.u74a.model.Idioma;
import com.ejie.u74a.model.Material;
import com.ejie.u74a.model.TemaMaterial;
import com.ejie.u74a.model.TipoMaterial;
import com.ejie.u74a.util.Constants;

/**
 * Clase que accede a la base de datos para obtener datos de materiales. Tabla: MATERIAL
 * 
 *  
 */
@Repository()
@Transactional()
public class MaterialDaoImpl implements MaterialDao {

	private JdbcTemplate jdbcTemplate;

	// @Resource(name = "appMessageSource")
	// private ReloadableResourceBundleMessageSource messageSource;

	@Autowired()
	private ArticuloDao articuloDao;

	private StringBuilder fromFindByCriteria = new StringBuilder(
			" FROM MATERIAL M LEFT JOIN TEMAMATERIAL TM ON  M.COTEMAMAT = TM.COTEMAMAT LEFT JOIN TIPOMATERIAL TIM ON M.COTIPOMAT = TIM.COTIPOMAT ");

	/**
	 * Clase que especifica el mapeo de las columnas de la tabla MATERIAL con los objetos de la clase Material
	 */
	private static final class MaterialRowMapper implements RowMapper<Material> {

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.springframework.jdbc.core.RowMapper#mapRow(java.sql.ResultSet, int)
		 */
		@Override()
		public Material mapRow(ResultSet rs, int rowNum) throws SQLException {
			Material material = new Material();

			material.setCodigoMaterial(rs.getInt("COMATERIAL"));
			material.setCodigoTipoMaterial(rs.getInt("COTIPOMAT"));
			material.setCodigoOficinaOrigen(rs.getString("COOFIORIGEN"));
			material.setDescatalogado(rs.getString("DESCATALOGADO"));
			material.setFechaDescatalogado(rs.getDate("FEDESCATALOGADO"));
			material.setTituloCastellano(rs.getString("TITULO"));
			material.setTituloEuskera(rs.getString("TITULO_EU"));
			material.setMonolingue(rs.getString("MONOLINGUE"));
			material.setAnioEdicion(rs.getString("ANIOEDICION"));
			material.setCodigoTema(rs.getInt("COTEMAMAT"));
			material.setFormato(rs.getString("FORMATO"));
			material.setFechaModificacion(rs.getDate("FECHAMOD"));
			material.setCodigoUsuarioModificacion(rs.getInt("COUSUMOD"));

			// De momento no recupero datos extras de temaMaterial ni tipoMaterial, solo los que se van a mostrar por
			// pantalla
			TipoMaterial tipoMaterial = new TipoMaterial();
			tipoMaterial.setCodigo(rs.getInt("COTIPOMAT"));
			tipoMaterial.setAbreviaturaCastellano(rs.getString("ABREVIATURA"));
			tipoMaterial.setAbreviaturaEuskera(rs.getString("ABREVIATURA_EU"));

			material.setTipoMaterial(tipoMaterial);

			// El tipo de material debería ser el mismo el de la tabla temamaterial que el de tipomaterial??
			TemaMaterial temaMaterial = new TemaMaterial();
			temaMaterial.setCodigoTemaMaterial(rs.getInt("COTEMAMAT"));
			temaMaterial.setCodigoTipoMaterial(rs.getInt("COTIPOMAT"));
			temaMaterial.setNombreCastellano(rs.getString("NOTEMAMAT"));
			temaMaterial.setNombreEuskera(rs.getString("NOTEMAMAT_EU"));

			material.setTema(temaMaterial);

			return material;
		}
	}

	/**
	 * Método para establecer el datasource.
	 * 
	 * @param dataSource DataSource
	 */
	@Resource()
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findByCriteria(CriteriosBusqueda)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Material> findByCriteria(CriteriosBusqueda filtro) {

		// De momento no recupero datos extras de temaMaterial ni tipoMaterial, solo los que se van a mostrar por
		// pantalla
		StringBuilder query = new StringBuilder(
				"SELECT COMATERIAL,M.COTIPOMAT,COOFIORIGEN,DESCATALOGADO,FEDESCATALOGADO,TITULO,TITULO_EU,MONOLINGUE,")
				.append("ANIOEDICION,M.COTEMAMAT,FORMATO,FECHAMOD,COUSUMOD, TIM.ABREVIATURA, TIM.ABREVIATURA_EU,")
				.append("TM.NOTEMAMAT, TM.NOTEMAMAT_EU  ").append(this.fromFindByCriteria);

		List<Object> argumentos = new ArrayList<Object>();
		StringBuilder whereSql = new StringBuilder();
		// Cuidado, si cambia algo del filtro, hay que cambiar en el método findByCountCriteria que devuelve el contador
		// de la consulta para la paginación.
		if (filtro.getTexto() != null && !"".equals(filtro.getTexto())) {
			whereSql.append(" WHERE (UPPER(TITULO) LIKE UPPER(?) or UPPER(TITULO_EU) LIKE UPPER(?)) ");
			argumentos.add("%" + filtro.getTexto() + "%");
			argumentos.add("%" + filtro.getTexto() + "%");
		}
		if (filtro.getTipoMaterial() != null && !"".equals(filtro.getTipoMaterial())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" TIM.COTIPOMAT = ? ");
			argumentos.add(filtro.getTipoMaterial());
		}
		if (filtro.getTema() != null && !"".equals(filtro.getTema())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND"))
					.append(" (TM.COTEMAMAT = ? or exists (SELECT MT.COTEMAMAT FROM MATERIALTEMA MT WHERE MT.COMATERIAL=M.COMATERIAL AND MT.COTEMAMAT = ?)) ");
			argumentos.add(filtro.getTema());
			argumentos.add(filtro.getTema());
		}
		if (filtro.getIdioma() != null && !"".equals(filtro.getIdioma())) {
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND"))
					.append(" exists (SELECT IA.COIDIOMA FROM ARTICULO A,IDIOMASARTICULO IA WHERE IA.COARTICULO=A.COARTICULO AND A.COMATERIAL=M.COMATERIAL AND IA.COIDIOMA = ?) ");
			argumentos.add(filtro.getIdioma());
		}
		if (filtro.getDescatalogado() != null && !"".equals(filtro.getDescatalogado())) {
			// Ojo, aqui es el material el que esta descatalogado
			whereSql.append((whereSql.length() == 0 ? " WHERE " : " AND")).append(" M.DESCATALOGADO=? ");
			argumentos.add(filtro.getDescatalogado());
		}

		// TODO si se utiliza este método Hay que meter paginacion

		query.append(whereSql).append("ORDER BY TITULO");
		if (new Locale("eu").equals(LocaleContextHolder.getLocale())) {
			query.append("_EU"); // Para que ordene por el nombre en euskera
		}

		List<Material> materiales;
		if (!"".equals(whereSql)) {
			materiales = this.jdbcTemplate.query(query.toString(), new MaterialRowMapper(), argumentos.toArray());
		} else {
			materiales = this.jdbcTemplate.query(query.toString(), new MaterialRowMapper());
		}

		return materiales;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findByCriteria(CriteriosBusqueda)
	 */
	@Transactional(readOnly = true)
	@Override()
	public int findByCountCriteria(CriteriosBusqueda filtro) {

		// De momento no recupero datos extras de temaMaterial ni tipoMaterial, solo los que se van a mostrar por
		// pantalla
		StringBuilder query = new StringBuilder("SELECT COUNT(COMATERIAL) ").append(this.fromFindByCriteria);

		List<Object> argumentos = new ArrayList<Object>();
		StringBuilder whereSql = new StringBuilder();
		// Cuidado, si cambia algo del filtro en el método findByCriteria hay que cambiarlo también aquí.
		if (filtro.getTexto() != null && !"".equals(filtro.getTexto())) {
			whereSql.append(" WHERE (UPPER(TITULO) LIKE UPPER(?) or UPPER(TITULO_EU) LIKE UPPER(?)) ");
			argumentos.add("%" + filtro.getTexto() + "%");
			argumentos.add("%" + filtro.getTexto() + "%");
		}
		if (filtro.getTipoMaterial() != null && !"".equals(filtro.getTipoMaterial())) {
			whereSql.append(("".equals(whereSql) ? " WHERE " : " AND")).append(" TIM.COTIPOMAT = ? ");
			argumentos.add(filtro.getTipoMaterial());
		}
		if (filtro.getTema() != null && !"".equals(filtro.getTema())) {
			whereSql.append(("".equals(whereSql) ? " WHERE " : " AND"))
					.append(" (TM.COTEMAMAT = ? or exists (SELECT MT.COTEMAMAT FROM MATERIALTEMA MT WHERE MT.COMATERIAL=M.COMATERIAL AND MT.COTEMAMAT = ?)) ");
			argumentos.add(filtro.getTema());
			argumentos.add(filtro.getTema());
		}
		if (filtro.getIdioma() != null && !"".equals(filtro.getIdioma())) {
			whereSql.append(("".equals(whereSql) ? " WHERE " : " AND"))
					.append(" exists (SELECT IA.COIDIOMA FROM ARTICULO A,IDIOMASARTICULO IA WHERE IA.COARTICULO=A.COARTICULO AND A.COMATERIAL=M.COMATERIAL AND IA.COIDIOMA = ?) ");
			argumentos.add(filtro.getIdioma());
		}

		int numRegistros;
		if (!"".equals(whereSql)) {
			numRegistros = this.jdbcTemplate.queryForObject(query.toString(), argumentos.toArray(), Integer.class);
		} else {
			numRegistros = this.jdbcTemplate.queryForObject(query.toString(), Integer.class);
		}
		return numRegistros;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findByCodigo(int, boolean, boolean)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Material findByCodigo(int codigoMaterial, boolean incluirTemasSecundarios, boolean incluirInfoArticulo) {
		StringBuilder query = new StringBuilder(
				"SELECT COMATERIAL,M.COTIPOMAT,DESCATALOGADO,FEDESCATALOGADO,TITULO,TITULO_EU,MONOLINGUE,")
				.append("ANIOEDICION,M.COTEMAMAT,FORMATO,FECHAMOD,COUSUMOD, TIM.ABREVIATURA, TIM.ABREVIATURA_EU,")
				.append("TM.NOTEMAMAT, TM.NOTEMAMAT_EU ")
				.append("FROM MATERIAL M LEFT JOIN TEMAMATERIAL TM ON  M.COTEMAMAT = TM.COTEMAMAT LEFT JOIN TIPOMATERIAL TIM ON M.COTIPOMAT = TIM.COTIPOMAT ")
				.append("WHERE COMATERIAL = ?");

		List<Map<String, Object>> rows = this.jdbcTemplate.queryForList(query.toString(), codigoMaterial);
		Material material = new Material();

		for (Map<String, Object> row : rows) { // TODO se puede convertir en if, Solo tiene que devolver uno

			material.setCodigoMaterial(((BigDecimal) row.get("COMATERIAL")).intValue());
			material.setCodigoTipoMaterial(((BigDecimal) row.get("COTIPOMAT")).intValue());
			material.setCodigoOficinaOrigen((String) row.get("COOFIORIGEN"));
			material.setDescatalogado((String) row.get("DESCATALOGADO"));
			material.setFechaDescatalogado((Date) row.get("FEDESCATALOGADO"));
			material.setTituloCastellano((String) row.get("TITULO"));
			material.setTituloEuskera((String) row.get("TITULO_EU"));
			material.setMonolingue((String) row.get("MONOLINGUE"));
			material.setAnioEdicion((String) row.get("ANIOEDICION"));
			if (row.get("COTEMAMAT") != null) {
				material.setCodigoTema(((BigDecimal) row.get("COTEMAMAT")).intValue());
			}
			material.setFormato((String) row.get("FORMATO"));
			material.setFechaModificacion((Date) row.get("FECHAMOD"));
			material.setCodigoUsuarioModificacion(((BigDecimal) row.get("COUSUMOD")).intValue());

			// De momento no recupero datos extras de temaMaterial ni tipoMaterial, solo los que se van a mostrar por
			// pantalla
			TipoMaterial tipoMaterial = new TipoMaterial();
			tipoMaterial.setCodigo(((BigDecimal) row.get("COTIPOMAT")).intValue());
			tipoMaterial.setAbreviaturaCastellano((String) row.get("ABREVIATURA"));
			tipoMaterial.setAbreviaturaEuskera((String) row.get("ABREVIATURA_EU"));

			material.setTipoMaterial(tipoMaterial);

			// El tipo de material debería ser el mismo el de la tabla temamaterial que el de tipomaterial??
			TemaMaterial temaMaterial = null;
			if (row.get("COTEMAMAT") != null) {
				temaMaterial = new TemaMaterial();
				temaMaterial.setCodigoMaterial(((BigDecimal) row.get("COMATERIAL")).intValue());
				temaMaterial.setCodigoTemaMaterial(((BigDecimal) row.get("COTEMAMAT")).intValue());
				temaMaterial.setCodigoTipoMaterial(((BigDecimal) row.get("COTIPOMAT")).intValue());
				temaMaterial.setNombreCastellano((String) row.get("NOTEMAMAT"));
				temaMaterial.setNombreEuskera((String) row.get("NOTEMAMAT_EU"));

				material.setTema(temaMaterial);
			}

			// Se buscan los temas secundarios si procede, se añade el principal también a la lista.
			if (incluirTemasSecundarios && material.getCodigoTipoMaterial() == Constants.TIPO_MATERIAL_FOLLETO) {
				List<TemaMaterial> temasLst = new ArrayList<TemaMaterial>();
				// Se coge el tema principal, se indica que lo es y se añade a la lista de temas.
				if (temaMaterial != null) {
					temaMaterial.setPrincipal(true);
					temasLst.add(temaMaterial);
				}
				temasLst.addAll(findTemasSecundariosByMaterial(codigoMaterial));
				material.setTemasLst(temasLst);

			}
			// Se buscan los articulos con sus idioma si procede
			if (incluirInfoArticulo) {
				if (material.getCodigoTipoMaterial() == Constants.TIPO_MATERIAL_FOLLETO) {
					material.setArticulosLst(findInfoArticulosByMaterialFOLLETO(codigoMaterial));
				} else {
					material.setArticuloPromocional(findInfoArticuloByMaterialPROMOCIONAL(codigoMaterial));
				}
			}
		}
		return material;
	}

	/**
	 * Obtiene los idiomas para el material.
	 * 
	 * @param codigoMaterial Código del material.
	 * 
	 * @deprecated De momento no se usa
	 * 
	 * @return List {@link Idioma} la lista de los idiomas del material, si es que los hubiera.
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findIdiomasByMaterial(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	@Deprecated()
	public List<Idioma> findIdiomasByMaterial(int codigoMaterial) { // Si se vuelve a usar quitar el deprecated

		StringBuilder query = new StringBuilder(
				"SELECT I.COIDIOMA, I.NOIDIOMA, I.NOIDIOMA_EU FROM ARTICULO A INNER JOIN IDIOMASARTICULO IA ON A.COARTICULO=IA.COARTICULO ")
				.append(" INNER JOIN IDIOMAS I ON I.COIDIOMA=IA.COIDIOMA WHERE A.COMATERIAL=? ORDER BY NOIDIOMA");
		if (new Locale("eu").equals(LocaleContextHolder.getLocale())) {
			query.append("_EU"); // Para que ordene por el nombre en euskera
		}
		return this.jdbcTemplate.query(query.toString(), new IdiomaDaoImpl.IdiomaRowMapper(), codigoMaterial);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findInfoArticulosByMaterialFOLLETO(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<Articulo> findInfoArticulosByMaterialFOLLETO(int codigoMaterial) {

		List<Map<String, Object>> rows = this.jdbcTemplate
				.queryForList(
						"SELECT A.COARTICULO,A.STOCK, A.STOCKMIN,A.NUMINIMO, A.URLPORTADA, A.URLPDF, A.DESCATALOGADO FROM ARTICULO A WHERE A.COMATERIAL=? ORDER BY COARTICULO",
						codigoMaterial);
		List<Articulo> articulosLst = new ArrayList<Articulo>();
		// int codigoArticuloAnterior = 0;
		Articulo articulo = null;
		for (Map<String, Object> row : rows) {
			articulo = new Articulo();
			articulo.setCodArticulo(((BigDecimal) row.get("COARTICULO")).intValue());

			if (row.get("NUMINIMO") != null) {
				articulo.setNumMinimo(((BigDecimal) row.get("NUMINIMO")).intValue());
			}

			if (row.get("STOCK") != null) {
				articulo.setStock(((BigDecimal) row.get("STOCK")).intValue());
			}

			if (row.get("STOCKMIN") != null) {
				articulo.setStockMin(((BigDecimal) row.get("STOCKMIN")).intValue());
			}

			articulo.setUrlPortada((String) row.get("URLPORTADA"));
			articulo.setUrlPDF((String) row.get("URLPDF"));
			articulo.setDescatalogado((String) row.get("DESCATALOGADO"));

			this.articuloDao.actualizaIdiomasEnArticulo(articulo);
			articulosLst.add(articulo);
		}
		return articulosLst;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findInfoArticulosByMaterialPROMOCIONAL(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public Articulo findInfoArticuloByMaterialPROMOCIONAL(int codigoMaterial) {
		StringBuilder query = new StringBuilder(
				"SELECT A.COARTICULO,A.COMATERIAL,A.URLPDF,A.FECHAMOD,A.COUSUARIOMOD,A.COALMACEN,")
				.append("A.DESCATALOGADO,A.STOCKMIN,A.STOCK,A.FEINVENTARIO,A.FEES,A.COFORMAENVIO,A.MODOPETICION,A.NUMINIMO,A.NUEJEMPLARES,A.URLPORTADA ")
				.append("FROM ARTICULO A WHERE A.COMATERIAL=?");
		List<Articulo> articuloLst = this.jdbcTemplate.query(query.toString(), new ArticuloDaoImpl.ArticuloRowMapper(),
				codigoMaterial);

		if (articuloLst != null && articuloLst.size() == 1) {
			return articuloLst.get(0);
		} else {
			return new Articulo();
		}

		// TODO nuevo mapper con menos infor,creo que sobra urls
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findTemasSecundariosByMaterial(int)
	 */
	@Transactional(readOnly = true)
	@Override()
	public List<TemaMaterial> findTemasSecundariosByMaterial(int codigoMaterial) {
		StringBuilder query = new StringBuilder(
				"SELECT MT.COMATERIAL, TM.COTEMAMAT, COTIPOMAT, NOTEMAMAT, NOTEMAMAT_EU FROM MATERIALTEMA MT INNER JOIN TEMAMATERIAL TM ON MT.COTEMAMAT=TM.COTEMAMAT WHERE COMATERIAL = ? ORDER BY NOTEMAMAT");
		if (new Locale("eu").equals(LocaleContextHolder.getLocale())) {
			query.append("_EU"); // Para que ordene por el nombre en euskera
		}
		return this.jdbcTemplate.query(query.toString(), new TemaMaterialDaoImpl.TemaMaterialRowMapper(),
				codigoMaterial);

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#updateTemaMaterial(int,int)
	 */
	@Transactional()
	@Override()
	public int updateTemaMaterial(TemaMaterial temaMaterial) {
		Object[] insertObj = { temaMaterial.getCodigoTemaMaterial(), temaMaterial.getCodigoMaterial() };
		return this.jdbcTemplate.update("UPDATE MATERIAL SET COTEMAMAT=? WHERE COMATERIAL=?", insertObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#findTemasSecundariosByMaterial(int,int)
	 */
	@Transactional()
	@Override()
	public int insertTemaMaterial(TemaMaterial temaMaterial) {
		Object[] insertObj = { temaMaterial.getCodigoMaterial(), temaMaterial.getCodigoTemaMaterial() };
		return this.jdbcTemplate.update("INSERT INTO MATERIALTEMA (COMATERIAL,COTEMAMAT) VALUES (?,?)", insertObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#deleteTemaMaterialPpal(int)
	 */
	@Transactional()
	@Override()
	public int deleteTemaMaterialPpal(int codMaterial) {
		Object[] insertObj = { codMaterial };
		return this.jdbcTemplate.update("UPDATE MATERIAL SET COTEMAMAT=NULL WHERE COMATERIAL=?", insertObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#deleteTemaMaterial(int,int)
	 */
	@Transactional()
	@Override()
	public int deleteTemaMaterial(TemaMaterial temaMaterial) {
		Object[] insertObj = { temaMaterial.getCodigoMaterial(), temaMaterial.getCodigoTemaMaterial() };
		return this.jdbcTemplate.update("DELETE FROM MATERIALTEMA WHERE COMATERIAL = ? AND COTEMAMAT = ?", insertObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#insertarMaterial(Material)
	 */
	@Transactional()
	@Override()
	public Material insertarMaterial(Material material) {

		material.setCodigoMaterial(this.jdbcTemplate
				.queryForObject("SELECT U74A47Q00.NEXTVAL FROM dual", Integer.class));

		Object[] insertObj = { material.getCodigoMaterial(), material.getCodigoTipoMaterial(),
				Constants.CODIGO_OFICINA_GV, material.getDescatalogado(), material.getFechaDescatalogado(),
				material.getTituloCastellano(), material.getTituloEuskera(), material.getMonolingue(),
				material.getAnioEdicion(), material.getFormato(), material.getFechaModificacion(),
				material.getCodigoUsuarioModificacion() };

		this.jdbcTemplate
				.update("INSERT INTO MATERIAL (COMATERIAL,COTIPOMAT,COOFIORIGEN,DESCATALOGADO,FEDESCATALOGADO,TITULO,TITULO_EU,MONOLINGUE,ANIOEDICION,FORMATO,FECHAMOD,COUSUMOD) VALUES (?,?,?,?,?,?,?,?,?,?,?,?)",
						insertObj);

		// if (numAfectadas > 0) {
		return material;
		// }
		// return 0;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#actualizarMaterial(Material)
	 */
	@Transactional()
	@Override()
	public Material actualizarMaterial(Material material) {

		Object[] updateObj = { material.getDescatalogado(), material.getFechaDescatalogado(),
				material.getTituloCastellano(), material.getTituloEuskera(), material.getMonolingue(),
				material.getAnioEdicion(), material.getFormato(), material.getFechaModificacion(),
				material.getCodigoUsuarioModificacion(), material.getCodigoMaterial() };

		this.jdbcTemplate
				.update("UPDATE MATERIAL SET DESCATALOGADO = ?,FEDESCATALOGADO = ?,TITULO = ?,TITULO_EU = ?,MONOLINGUE = ?,ANIOEDICION =?,FORMATO = ?,FECHAMOD = ?,COUSUMOD = ? WHERE COMATERIAL = ?",
						updateObj);

		return material;

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#updateTipoIdiomaMaterial(int,int)
	 */
	@Transactional()
	@Override()
	public int updateTipoIdiomaMaterial(int codMaterial, String monolingue) {
		Object[] updateObj = { monolingue, codMaterial };
		return this.jdbcTemplate.update("UPDATE MATERIAL SET MONOLINGUE=? WHERE COMATERIAL=?", updateObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#actualizarDescatalogadoArticulosMaterial(String,int)
	 */
	@Transactional()
	@Override()
	public int actualizarDescatalogadoArticulosMaterial(String descatalogado, int codigoMaterial) {
		Object[] updateObj = { descatalogado, codigoMaterial };
		return this.jdbcTemplate.update("UPDATE ARTICULO SET DESCATALOGADO = ? WHERE COMATERIAL = ?", updateObj);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.ejie.u74a.dao.MaterialDao#existeIdiomaEnMaterial(java.lang.String, int)
	 */
	@Override()
	public boolean existeIdiomaEnMaterial(String codigoIdioma, Integer codigoMaterial, Integer codigoArticulo) {
		List<Map<String, Object>> resultado = null;

		StringBuilder query = new StringBuilder(
				"SELECT I.COIDIOMA FROM ARTICULO A INNER JOIN IDIOMASARTICULO IA ON A.COARTICULO = IA.COARTICULO ")
				.append(" INNER JOIN IDIOMAS I ON I.COIDIOMA = IA.COIDIOMA WHERE A.COMATERIAL = ? AND IA.COIDIOMA = ? ");

		if (codigoArticulo != null) { // Para permitir la modificación de los datos del artículo
			query.append(" AND IA.COARTICULO <> ?");
			resultado = this.jdbcTemplate.queryForList(query.toString(), codigoMaterial, codigoIdioma, codigoArticulo);
		} else {
			resultado = this.jdbcTemplate.queryForList(query.toString(), codigoMaterial, codigoIdioma);
		}

		if (resultado.size() > 0) {
			return true;
		}

		return false;
	}

}
