package com.ejie.u74a.dao;

import java.util.List;

import com.ejie.u74a.model.Articulo;
import com.ejie.u74a.model.CriteriosBusqueda;
import com.ejie.u74a.model.Idioma;
import com.ejie.u74a.model.Material;
import com.ejie.u74a.model.TemaMaterial;

/**
 * Interfaz de acceso a la base de datos para obtener datos de materiales. Tabla: MATERIAL
 * 
 *  
 */
public interface MaterialDao {

	/**
	 * Obtiene los materiales dependiendo del filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return lista con los materiales
	 */
	List<Material> findByCriteria(CriteriosBusqueda filtro);

	/**
	 * Obtiene el número de los materiales que cumplen el filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return int número de los materiales que cumplen el filtro aplicado.
	 */
	int findByCountCriteria(CriteriosBusqueda filtro);

	/**
	 * Busca los datos de un material en función del código proporcionado.
	 * 
	 * @param codigoMaterial El código del material.
	 * @param incluirTemasSecundarios true: Indica si en la consulta se deben obtener también los temas secundarios del
	 *            material si los hubiera, false: para no incluirlos.
	 * @param incluirInfoArticulo true:Indica si en la consulta se deben obtener también los articulos e idiomas del
	 *            material si los hubiera, false: para no incluirlos.
	 * @return Objeto {@link Material} con los datos del material correspondiente al código.
	 */
	Material findByCodigo(int codigoMaterial, boolean incluirTemasSecundarios, boolean incluirInfoArticulo);

	/**
	 * Obtiene los idiomas para el material.
	 * 
	 * @param codigoMaterial Código del material.
	 * 
	 * @deprecated De momento no se usa
	 * 
	 * @return List {@link Idioma} la lista de los idiomas del material, si es que los hubiera.
	 */
	@Deprecated()
	List<Idioma> findIdiomasByMaterial(int codigoMaterial);

	/**
	 * Obtiene información mínima de artículo - idioma para mostrar en la ficha de material para tipo FOLLETO.
	 * 
	 * @param codigoMaterial Código del material.
	 * @return List {@link Articulo} la lista de los articulos con sus idiomas, stock y mínimo.
	 */
	List<Articulo> findInfoArticulosByMaterialFOLLETO(int codigoMaterial);

	/**
	 * Obtiene información de articulo para mostrar en la ficha de material para tipo MATERIAL PROMOCIONAL.
	 * 
	 * @param codigoMaterial Código del material.
	 * @return {@link Articulo} El artículo con stock, mínimo, modo petición.
	 */
	Articulo findInfoArticuloByMaterialPROMOCIONAL(int codigoMaterial);

	/**
	 * Obtiene los temas secundarios para el material.
	 * 
	 * @param codigoMaterial Código del material.
	 * 
	 * @return List {@link TemaMaterial} la lista de los temas secundarios del material, si es que los hubiera.
	 */
	List<TemaMaterial> findTemasSecundariosByMaterial(int codigoMaterial);

	/**
	 * Guarda el tema asociado con el material como principal.
	 * 
	 * @param temaMaterial objeto TemaMaterial *
	 * @return Devuelve el número de filas afectadas.
	 */
	int updateTemaMaterial(TemaMaterial temaMaterial);

	/**
	 * Guarda el tema asociado con el material como secundario.
	 * 
	 * @param temaMaterial objeto TemaMaterial
	 * @return Devuelve el número de filas afectadas.
	 */
	int insertTemaMaterial(TemaMaterial temaMaterial);

	/**
	 * Borra el tema asociado con el material como principal.
	 * 
	 * @param codMaterial El código del material.
	 * @return Devuelve el número de filas afectadas.
	 */
	int deleteTemaMaterialPpal(int codMaterial);

	/**
	 * Borra el tema asociado con el material como secundario.
	 * 
	 * @param temaMaterial objeto TemaMaterial
	 * @return Devuelve el número de filas afectadas.
	 */
	int deleteTemaMaterial(TemaMaterial temaMaterial);

	/**
	 * Inserta un nuevo material.
	 * 
	 * @param material objeto {@link Material} con la información.
	 * @return Devuelve el código de material si ha ido bien.
	 */
	Material insertarMaterial(Material material);

	/**
	 * Actualiza los datos de un material.
	 * 
	 * @param material objeto {@link Material} con la información.
	 * @return Devuelve el número de filas afectadas.
	 */
	Material actualizarMaterial(Material material);

	/**
	 * Guarda el nuevo tipo de idioma para el material.
	 * 
	 * @param codigoMaterial El código del material.
	 * @param monolingue S para indicar que es monolingue, en cualquier otro caso se informará como N.
	 * @return Devuelve el número de filas afectadas.
	 */
	int updateTipoIdiomaMaterial(int codigoMaterial, String monolingue);

	/**
	 * Actualizar el campo descatalogado de los articulos de un material.
	 * 
	 * @param descatalogado valor para el campo descatalogado
	 * @param codigoMaterial El código de material
	 * @return Devuelve el número de filas afectadas.
	 */
	int actualizarDescatalogadoArticulosMaterial(String descatalogado, int codigoMaterial);

	/**
	 * Comprueba si el material ya tiene un artículo con ese idioma
	 * 
	 * @param codigoIdioma El código del idioma
	 * @param codigoMaterial El código del material.
	 * @param codigoArticulo El código del articulo.
	 * 
	 * @return Devuelve true si el idioma ya existe para ese material.
	 */
	boolean existeIdiomaEnMaterial(String codigoIdioma, Integer codigoMaterial, Integer codigoArticulo);

}
