package com.ejie.u74a.dao;

import java.util.Date;
import java.util.List;

import com.ejie.u74a.model.Articulo;
import com.ejie.u74a.model.CriteriosBusqueda;
import com.ejie.u74a.model.Idioma;

/**
 * Interfaz de acceso a la base de datos para obtener datos de artículos. Tabla: ARTICULO
 * 
 *  
 */
public interface ArticuloDao {

	/**
	 * Obtiene los artículos asociados a una lista de códigos de artículos.
	 * 
	 * @param codigoArticulosLst lista con los códigos de artículos.
	 * 
	 * @return lista con los artículos
	 */
	List<Articulo> findByListaCodigos(List<Integer> codigoArticulosLst); // findByMaterial(List<Integer>
																			// codigoMaterialesLst);

	/**
	 * Obtiene los artículos que cumplen el filtro proporcionado.
	 * 
	 * @param filtro {@link CriteriosBusqueda} objeto con el filtro a cumplir.
	 * 
	 * @return List<Articulo> Devuelve el lisatdo de artículos que cumplen con el filtro.
	 */
	List<Articulo> findByCriteria(CriteriosBusqueda filtro);

	/**
	 * Obtiene el número de los artículos que cumplen del filtro aplicado.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return int número de los artículos que cumplen el filtro aplicado.
	 */
	int findByCountCriteria(CriteriosBusqueda filtro);

	/**
	 * Obtiene los artículos que cumplen el filtro proporcionado para la gestion de stocks.
	 * 
	 * @param filtro {@link CriteriosBusqueda} objeto con el filtro a cumplir.
	 * 
	 * @return List<Articulo> Devuelve el lisatdo de artículos que cumplen con el filtro.
	 */
	List<Articulo> findByCriteriaGestionStock(CriteriosBusqueda filtro);

	/**
	 * Obtiene el número de los artículos que cumplen del filtro aplicadopara la gestion de stocks.
	 * 
	 * @param filtro Clase con los filtros de búsqueda.
	 * 
	 * @return int número de los artículos que cumplen el filtro aplicado.
	 */
	int findByCountCriteriaGestionStock(CriteriosBusqueda filtro);

	/**
	 * Busca los idiomas para el artículo y actualiza en el articulo la descripción correspondiente, si no tiene
	 * idiomas: sin idioma, si tiene un idioma: el nombre del idioma, y si tiene más de un idioma: multilingüe.
	 * 
	 * @param articulo objeto @ link Articulo} a actualizar.
	 */
	void actualizaIdiomasEnArticulo(Articulo articulo);

	/**
	 * Busca los idiomas para el artículo devuelve la descripción correspondiente, si no tiene idiomas: sin idioma, si
	 * tiene un idioma: el nombre del idioma, y si tiene más de un idioma: multilingüe.
	 * 
	 * @param codArticulo El código del artículo.
	 * @return Devuelve la descripcion correspondiente a los idiomas del artículo.
	 */
	String getDescripcionIdiomasArticulo(int codArticulo);

	/**
	 * Obtiene los idiomas para el articulo.
	 * 
	 * @param codigoArticulo Código del artículo.
	 * 
	 * @return List {@link Idioma} la lista de los idiomas del artículo, si es que los hubiera.
	 */
	List<Idioma> findIdiomasByArticulo(int codigoArticulo);

	/**
	 * Obtiene los datos del artículo correspondientes al codigo proporcionado.
	 * 
	 * @param codigoArticulo Código del artículo.
	 * 
	 * @return objeto {@link Articulo} con los datos del artículo.
	 */
	Articulo findByCodigo(int codigoArticulo);

	/**
	 * Obtiene los datos del artículo correspondientes al codigo proporcionado con información extra.
	 * 
	 * @param pCodigoArticulo Código del artículo.
	 * 
	 * @return objeto {@link Articulo} con los datos del artículo.
	 */
	Articulo findByCodigoExtra(int pCodigoArticulo);

	/**
	 * Inserta un nuevo artículo.
	 * 
	 * @param articulo objeto {@link Articulo} con la información.
	 * @return Devuelve el código de artículo si ha ido bien.
	 */
	Articulo insertarArticulo(Articulo articulo);

	/**
	 * Inserta un nuevo idioma para un artículo.
	 * 
	 * @param codigoArticulo el código del artículo
	 * @param codigoIdioma el código del idioma
	 * @return Devuelve el número de filas afectadas.
	 */
	int insertarIdiomaArticulo(int codigoArticulo, String codigoIdioma);

	/**
	 * Borrar los idiomas de un artículo.
	 * 
	 * @param codigoArticulo El código del artículo
	 * @return Devuelve el número de filas afectadas.
	 */
	int borrarIdiomasArticulo(int codigoArticulo);

	/**
	 * Borrar los datos de un artículo para el código proporcionado.
	 * 
	 * @param codigoArticulo El código del artículo
	 * @return Devuelve el número de filas afectadas.
	 */
	int borrarArticulo(int codigoArticulo);

	/**
	 * Actualiza los datos de un artículo.
	 * 
	 * @param articulo objeto {@link Articulo} con la información.
	 * @return Devuelve el número de filas afectadas.
	 */
	Articulo actualizarArticulo(Articulo articulo);

	/**
	 * Actualiza el del stock artículo.
	 * 
	 * @param codigoArticulo El código del artículo.
	 * @param salida A true si la cantidad de stock se reduce, false en caso contrario.
	 * @param diferenciaStock La diferencia de stock a aumentar o reducir.
	 * @param codigoUsuarioModificador El código de usuario modificador.
	 * @param fechaModificacion La fecha de modificación.
	 * @return Devuelve el número de filas afectadas.
	 */
	int actualizarStockArticulo(int codigoArticulo, boolean salida, int diferenciaStock, int codigoUsuarioModificador,
			Date fechaModificacion);

	/**
	 * Comprueba si hay peticiones pendientes para el artículo.
	 * 
	 * @param codigoArticulo el código del artículo
	 * @return Devuelve true si hay peticiones pendientes para el artículo.
	 */
	boolean comprobarPeticionesActivasExistentes(int codigoArticulo);

	/**
	 * Borra todos los artículo y sus idiomas para el código de material proporcionado.
	 * 
	 * @param codigoMaterial El código del material.
	 * @return Devuelve el número de filas afectadas.
	 */
	int borrarArticulosDeMaterial(int codigoMaterial);

	/**
	 * Actualiza la información de stock del artículo.
	 * 
	 * @param articulo objeto {@link Articulo} con la información.
	 * @return Devuelve el número de filas afectadas.
	 */
	int actualizarInfoStockArticulo(Articulo articulo);

	/**
	 * Devuelve los datos de stock de un artículo: Stock y Stock mínimo
	 * 
	 * @param codigoArticulo código del artículo
	 * @return objeto Artículo con los datos del stock
	 */
	Articulo getDatosStockArticulo(int codigoArticulo);
}
