package com.ejie.ab78.control.procedimientos;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.ab78.model.procedimientos.Procedimientos;
import com.ejie.ab78.service.procedimientos.ProcedimientosService;
import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;
import com.ejie.x38.dto.JerarquiaDto;
import com.ejie.x38.dto.TableRowDto;

@Controller
@RequestMapping(value = "/procedimientos")

public class ProcedimientosController {

	private static final Logger logger = LoggerFactory.getLogger(ProcedimientosController.class);

	@Autowired
	private ProcedimientosService procedimientosService;

	/*
	 * OPERACIONES CRUD (Create, Read, Update, Delete)
	 * 
	 */

	/**
	 * Operacion CRUD Read. Devuelve el bean correspondiente al identificador
	 * indicado.
	 * 
	 * @param idProcedimiento
	 *            Integer
	 * @return Procedimientos Objeto correspondiente al identificador indicado.
	 */
	@RequestMapping(value = "/{idProcedimiento}", method = RequestMethod.GET)
	public @ResponseBody Procedimientos get(@PathVariable Integer idProcedimiento) {
		Procedimientos procedimientos = new Procedimientos();
		procedimientos.setIdProcedimiento(idProcedimiento);
		procedimientos = this.procedimientosService.find(procedimientos);
		ProcedimientosController.logger.info("[GET - findBy_PK] : Obtener Procedimientos por PK");
		return procedimientos;
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterProcedimientos
	 *            Procedimientos Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Procedimientos> Lista de objetos correspondientes a la
	 *         busqueda realizada.
	 */
	@RequestMapping(method = RequestMethod.GET)
	public @ResponseBody List<Procedimientos> getAll(@ModelAttribute Procedimientos filterProcedimientos) {
		ProcedimientosController.logger.info("[GET - find_ALL] : Obtener Procedimientos por filtro");

		return this.procedimientosService.findAll(filterProcedimientos, null);
	}

	@RequestMapping(value = "/combo/censo", method = RequestMethod.GET)
	public @ResponseBody List<Procedimientos> getAllCenso(@ModelAttribute Procedimientos filterProcedimientos) {
		ProcedimientosController.logger.info("[GET - find_ALL] : Obtener Procedimientos por filtro");
		return this.procedimientosService.findAllCenso(filterProcedimientos);
	}

	@RequestMapping(value = "/combo/tablaMaestro/{idDireccion}", method = RequestMethod.GET)
	public @ResponseBody List<Procedimientos> getAllTablaMaestro(@ModelAttribute Procedimientos filterProcedimientos, @PathVariable Integer idDireccion) {
		ProcedimientosController.logger.info("[GET - find_ALL] : Obtener Procedimientos por filtro");
		
		return this.procedimientosService.findAllTablaMaestro(filterProcedimientos, idDireccion, false);
		
	}
	
	@RequestMapping(value = "/combo/tablaMaestro", method = RequestMethod.GET)
	public @ResponseBody List<Procedimientos> getAllTablaMaestroSinDireccion(@ModelAttribute Procedimientos filterProcedimientos) {
		ProcedimientosController.logger.info("[GET - find_ALL] : Obtener Procedimientos por filtro");
		
		return this.procedimientosService.findAllTablaMaestro(filterProcedimientos, null, false);
		
	}

	/**
	 * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
	 * indicado.
	 *
	 * @param procedimientos
	 *            Procedimientos Bean que contiene la informacion con la que se
	 *            va a crear el nuevo registro.
	 * @return Procedimientos Bean resultante del proceso de creacion.
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody Procedimientos add(@RequestBody Procedimientos procedimientos) {
		Procedimientos procedimientosAux = this.procedimientosService.add(procedimientos);
		ProcedimientosController.logger.info("[POST] : Procedimientos insertado correctamente");
		return procedimientosAux;
	}

	/**
	 * Operacion CRUD Delete. Borrado del registro correspondiente al
	 * identificador especificado.
	 *
	 * @param idProcedimiento
	 *            Integer Identificador del objeto que se desea eliminar.
	 * @return Procedimientos Bean eliminado.
	 */
	@RequestMapping(value = "/{idProcedimiento}", method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody Procedimientos remove(@PathVariable Integer idProcedimiento) {
		Procedimientos procedimientos = new Procedimientos();
		procedimientos.setIdProcedimiento(idProcedimiento);
		this.procedimientosService.remove(procedimientos);
		ProcedimientosController.logger.info("[DELETE] : Procedimientos borrado correctamente");
		return procedimientos;
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE
	 * 
	 */

	/**
	 * Metodo de presentacion del RUP_TABLE.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "/maint", method = RequestMethod.GET)
	public String getFormEdit(Model model) {
		ProcedimientosController.logger.info("[GET - View] : procedimientos");
		return "procedimientos";
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterProcedimientos
	 *            Procedimientos Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<Procedimientos> Dto que contiene el resultado
	 *         del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<Procedimientos> filter(
			@RequestJsonBody(param = "filter") Procedimientos filterProcedimientos,
			@RequestJsonBody TableRequestDto TableRequestDto) {
		ProcedimientosController.logger.info("[POST - filter] : Obtener Procedimientoss");
		return this.procedimientosService.filter(filterProcedimientos, TableRequestDto, false);
	}

	/**
	 * Operacion de busqueda del componente RUP_TABLE.
	 * 
	 * @param filterProcedimientos
	 *            Procedimientos Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param searchProcedimientos
	 *            Procedimientos Bean que contiene los parametros de busqueda a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la búsqueda.
	 * @return TableRowDto<Procedimientos> Dto que contiene el resultado de la
	 *         busqueda realizada por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/search", method = RequestMethod.POST)
	public @ResponseBody List<TableRowDto<Procedimientos>> search(
			@RequestJsonBody(param = "filter") Procedimientos filterProcedimientos,
			@RequestJsonBody(param = "search") Procedimientos searchProcedimientos,
			@RequestJsonBody TableRequestDto TableRequestDto) {
		ProcedimientosController.logger.info("[POST - search] : Buscar Procedimientoss");
		return this.procedimientosService.search(filterProcedimientos, searchProcedimientos, TableRequestDto, false);
	}

	/**
	 * Borrado multiple de registros
	 * 
	 * @param filterProcedimientos
	 *            Procedimientos Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la busqueda.
	 * @return List<String> Lista de los identificadores de los registros
	 *         eliminados.
	 * 
	 */
	@RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody List<String> removeMultiple(
			@RequestJsonBody(param = "filter") Procedimientos filterProcedimientos,
			@RequestJsonBody TableRequestDto TableRequestDto) {
		ProcedimientosController.logger
				.info("[POST - search] : [POST - removeMultiple] : Eliminar multiples Procedimientoss");
		this.procedimientosService.removeMultiple(filterProcedimientos, TableRequestDto, false);
		ProcedimientosController.logger.info("All entities correctly deleted!");

		return TableRequestDto.getMultiselection().getSelectedIds();
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE - JERARQUIA
	 */

	/**
	 * Operacion de filtrado del componente RUP_TABLE para presentar los
	 * registros mediante visualizacion jerarquica.
	 * 
	 * @param filterProcedimientos
	 *            Procedimientos Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<Procedimientos>> Dto que contiene
	 *         el resultado del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquia/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Procedimientos>> jerarquia(
			@RequestJsonBody(param = "filter") Procedimientos filterProcedimientos,
			@RequestJsonBody TableRequestDto TableRequestDto) {
		ProcedimientosController.logger.info("[POST - jerarquia] : Obtener Procedimientoss jerarquia");
		return this.procedimientosService.jerarquia(filterProcedimientos, TableRequestDto, false);
	}

	/**
	 * Recupera los hijos de los registros desplegados en la visualizacion
	 * jerarquica.
	 * 
	 * @param filterProcedimientos
	 *            Procedimientos Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<Procedimientos>> Dto que contiene
	 *         el resultado del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquiaChildren", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Procedimientos>> jerarquiaChildren(
			@RequestJsonBody(param = "filter") Procedimientos filterProcedimientos,
			@RequestJsonBody TableRequestDto TableRequestDto) {
		ProcedimientosController.logger.info("[POST - jerarquia] : Obtener Procedimientoss jerarquia - Hijos");
		return this.procedimientosService.jerarquiaChildren(filterProcedimientos, TableRequestDto);
	}

}
