package com.ejie.ab78.control.maestros;

import java.util.List;

import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.ab78.comun.Constantes;
import com.ejie.ab78.comun.UtilUsuario;
import com.ejie.ab78.model.maestros.Motivo;
import com.ejie.ab78.model.usuario.Usuario;
import com.ejie.ab78.service.maestros.MotivosService;
import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JerarquiaDto;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;
import com.ejie.x38.dto.TableRowDto;

/**
 * MotivosController  
 * 
 *  
 */

@Controller
@RequestMapping(value = "/maestros/motivos")

public class MotivosController {

	private static final Logger logger = LoggerFactory.getLogger(MotivosController.class);

	@Autowired
	private MotivosService motivosService;

	/*
	 * OPERACIONES CRUD (Create, Read, Update, Delete)
	 * 
	 */

	/**
	 * Operacion CRUD Read. Devuelve el bean correspondiente al identificador
	 * indicado.
	 * 
	 * @param idMotivo
	 *            Long
	 * @return RelacionMotivoDen Objeto correspondiente al identificador
	 *         indicado.
	 */
	@RequestMapping(value = "/{idMotivo}", method = RequestMethod.GET)
	public @ResponseBody Motivo get(@PathVariable Long idMotivo) {
		Motivo motivo = new Motivo();
		motivo.setIdMotivo(idMotivo);
		motivo = this.motivosService.find(motivo);
		MotivosController.logger.info("[GET - findBy_PK] : Obtener RelacionMotivoDen por PK");
		return motivo;
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterMotivo
	 *            RelacionMotivoDen Objeto que contiene los parametros de
	 *            filtrado utilizados en la busqueda.
	 * @return List<RelacionMotivoDen> Lista de objetos correspondientes a la
	 *         busqueda realizada.
	 */
	@RequestMapping(value = "/motivosGetAll", method = RequestMethod.GET)
	public @ResponseBody List<Motivo> getAll(@ModelAttribute Motivo filterMotivo) {
		MotivosController.logger.info("[GET - find_ALL] : Obtener RelacionMotivoDen por filtro");
		return this.motivosService.findAll(filterMotivo, null);
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterMotivo
	 *            RelacionMotivoDen Objeto que contiene los parametros de
	 *            filtrado utilizados en la busqueda.
	 * @return List<RelacionMotivoDen> Lista de objetos correspondientes a la
	 *         busqueda realizada.
	 */
	@RequestMapping(value = "/motivosGetAll/{idProcedimiento}", method = RequestMethod.GET)
	public @ResponseBody List<Motivo> getAll(@ModelAttribute Motivo filterMotivo,
			@PathVariable("idProcedimiento") Long idProcedimiento) {
		MotivosController.logger.info("[GET - find_ALL] : Obtener RelacionMotivoDen por filtro");
		filterMotivo.setIdProcedimiento(idProcedimiento);
		return this.motivosService.findAllFilterProcedimiento(filterMotivo, null);
	}

	/**
	 * Operacion CRUD Edit. Modificacion del bean indicado.
	 *
	 * @param motivo
	 *            RelacionMotivoDen Bean que contiene la informacion a
	 *            modificar.
	 * @return RelacionMotivoDen Bean resultante de la modificacion.
	 */
	@RequestMapping(method = RequestMethod.PUT)
	public @ResponseBody Motivo edit(@RequestBody Motivo motivo) {
		Motivo motivoAux = this.motivosService.update(motivo);
		MotivosController.logger.info("[PUT] : RelacionMotivoDen actualizado correctamente");
		return motivoAux;
	}

	/**
	 * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
	 * indicado.
	 *
	 * @param motivo
	 *            RelacionMotivoDen Bean que contiene la informacion con la que
	 *            se va a crear el nuevo registro.
	 * @return RelacionMotivoDen Bean resultante del proceso de creacion.
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody Motivo add(@RequestBody Motivo motivo) {
		Motivo motivoAux = this.motivosService.add(motivo);
		MotivosController.logger.info("[POST] : RelacionMotivoDen insertado correctamente");
		return motivoAux;
	}

	/**
	 * Operacion CRUD Delete. Borrado del registro correspondiente al
	 * identificador especificado.
	 *
	 * @param idMotivo
	 *            Long Identificador del objeto que se desea eliminar.
	 * @return RelacionMotivoDen Bean eliminado.
	 */
	@RequestMapping(value = "/{idMotivo}", method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody Motivo remove(@PathVariable Long idMotivo) {
		Motivo motivo = new Motivo();
		motivo.setIdMotivo(idMotivo);
		this.motivosService.remove(motivo);
		MotivosController.logger.info("[DELETE] : RelacionMotivoDen borrado correctamente");
		return motivo;
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE
	 * 
	 */

	/**
	 * Metodo de presentacion del RUP_TABLE.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "/maint", method = RequestMethod.GET)
	public String getFormEdit(Model model, HttpSession httpSession) {
		MotivosController.logger.info("[GET - View] : motivo");

		String username = UtilUsuario.getUserConnected();
		Usuario usuario = new Usuario();
		usuario.setNombreUsuario(username);
		model.addAttribute("usuario", usuario);

		if (httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE) != null) {
			Object folderIdUltimoExpediente = httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE);

			model.addAttribute("ultimoExpediente", folderIdUltimoExpediente);
		}
		if (httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE_MIGRADO) != null) {
			Object IdDenunciaUltimoExpediente = httpSession
					.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE_MIGRADO);

			model.addAttribute("IdDenunciaUltimoExpediente", IdDenunciaUltimoExpediente);
		}

		return "motivos";
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterMotivo
	 *            RelacionMotivoDen Bean que contiene los parametros de filtrado
	 *            a emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<RelacionMotivoDen> Dto que contiene el resultado
	 *         del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<Motivo> filter(@RequestJsonBody(param = "filter") Motivo filterMotivo,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		MotivosController.logger.info("[POST - filter] : Obtener Motivos");
		return this.motivosService.filter(filterMotivo, tableRequestDto, false);
	}

	/**
	 * Operacion de busqueda del componente RUP_TABLE.
	 * 
	 * @param filterMotivo
	 *            RelacionMotivoDen Bean que contiene los parametros de filtrado
	 *            a emplear.
	 * @param searchMotivo
	 *            RelacionMotivoDen Bean que contiene los parametros de busqueda
	 *            a emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la búsqueda.
	 * @return TableRowDto<RelacionMotivoDen> Dto que contiene el resultado de
	 *         la busqueda realizada por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/search", method = RequestMethod.POST)
	public @ResponseBody List<TableRowDto<Motivo>> search(@RequestJsonBody(param = "filter") Motivo filterMotivo,
			@RequestJsonBody(param = "search") Motivo searchMotivo, @RequestJsonBody TableRequestDto tableRequestDto) {
		MotivosController.logger.info("[POST - search] : Buscar Motivos");
		return this.motivosService.search(filterMotivo, searchMotivo, tableRequestDto, false);
	}

	/**
	 * Borrado multiple de registros
	 * 
	 * @param filterMotivo
	 *            RelacionMotivoDen Bean que contiene los parametros de filtrado
	 *            a emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la busqueda.
	 * @return List<String> Lista de los identificadores de los registros
	 *         eliminados.
	 * 
	 */
	@RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody List<String> removeMultiple(@RequestJsonBody(param = "filter") Motivo filterMotivo,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		MotivosController.logger.info("[POST - search] : [POST - removeMultiple] : Eliminar multiples Motivos");
		this.motivosService.removeMultiple(filterMotivo, tableRequestDto, false);
		MotivosController.logger.info("All entities correctly deleted!");

		return tableRequestDto.getMultiselection().getSelectedIds();
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE - JERARQUIA
	 */

	/**
	 * Operacion de filtrado del componente RUP_TABLE para presentar los
	 * registros mediante visualizacion jerarquica.
	 * 
	 * @param filterMotivo
	 *            RelacionMotivoDen Bean que contiene los parametros de filtrado
	 *            a emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<RelacionMotivoDen>> Dto que
	 *         contiene el resultado del filtrado realizado por el componente
	 *         RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquia/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Motivo>> jerarquia(
			@RequestJsonBody(param = "filter") Motivo filterMotivo, @RequestJsonBody TableRequestDto tableRequestDto) {
		MotivosController.logger.info("[POST - jerarquia] : Obtener Motivos jerarquia");
		return this.motivosService.jerarquia(filterMotivo, tableRequestDto, false);
	}

	/**
	 * Recupera los hijos de los registros desplegados en la visualizacion
	 * jerarquica.
	 * 
	 * @param filterMotivo
	 *            RelacionMotivoDen Bean que contiene los parametros de filtrado
	 *            a emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<RelacionMotivoDen>> Dto que
	 *         contiene el resultado del filtrado realizado por el componente
	 *         RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquiaChildren", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Motivo>> jerarquiaChildren(
			@RequestJsonBody(param = "filter") Motivo filterMotivo, @RequestJsonBody TableRequestDto tableRequestDto) {
		MotivosController.logger.info("[POST - jerarquia] : Obtener Motivos jerarquia - Hijos");
		return this.motivosService.jerarquiaChildren(filterMotivo, tableRequestDto);
	}

	/**
	 * EXPORTERS
	 */

	@RequestMapping(value = "/clipboardReport", method = RequestMethod.POST)
	protected @ResponseBody List<Motivo> getClipboardReport(@RequestJsonBody(param = "filter") Motivo filterMotivo,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		MotivosController.logger.info("[POST - clipboardReport] : : Copiar multiples usuarios");
		return this.motivosService.getMultiple(filterMotivo, tableRequestDto, false);
	}

}
