package com.ejie.ab78.control.maestros;

import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JerarquiaDto;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;
import com.ejie.x38.dto.TableRowDto;
import java.util.List;

import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.ab78.comun.Constantes;
import com.ejie.ab78.model.maestros.Irregularidad;
import com.ejie.ab78.service.maestros.IrregularidadesService;

/**
 * IrregularidadsController  
 * 
 *  
 */

@Controller
@RequestMapping(value = "/maestros/irregularidades")
public class IrregularidadesController {

	private static final Logger logger = LoggerFactory.getLogger(IrregularidadesController.class);

	@Autowired
	private IrregularidadesService irregularidadesService;

	/*
	 * OPERACIONES CRUD (Create, Read, Update, Delete)
	 * 
	 */

	/**
	 * Operacion CRUD Read. Devuelve el bean correspondiente al identificador
	 * indicado.
	 * 
	 * @param idIrregularidad
	 *            Long
	 * @return Irregularidad Objeto correspondiente al identificador indicado.
	 */
	@RequestMapping(value = "/{idIrregularidad}", method = RequestMethod.GET)
	public @ResponseBody Irregularidad get(@PathVariable Long idIrregularidad) {
		Irregularidad irregularidad = new Irregularidad();
		irregularidad.setIdIrregularidad(idIrregularidad);
		irregularidad = this.irregularidadesService.find(irregularidad);
		IrregularidadesController.logger.info("[GET - findBy_PK] : Obtener Irregularidad por PK");
		return irregularidad;
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterIrregularidad
	 *            Irregularidad Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Irregularidad> Lista de objetos correspondientes a la
	 *         busqueda realizada.
	 */
	@RequestMapping(method = RequestMethod.GET)
	public @ResponseBody List<Irregularidad> getAll(@ModelAttribute Irregularidad filterIrregularidad) {
		IrregularidadesController.logger.info("[GET - find_ALL] : Obtener Irregularidad por filtro");
		filterIrregularidad.setBaja(0);
		return this.irregularidadesService.findAll(filterIrregularidad, null);
	}

	/**
	 * Operacion CRUD Edit. Modificacion del bean indicado.
	 *
	 * @param irregularidad
	 *            Irregularidad Bean que contiene la informacion a modificar.
	 * @return Irregularidad Bean resultante de la modificacion.
	 */
	@RequestMapping(method = RequestMethod.PUT)
	public @ResponseBody Irregularidad edit(@RequestBody Irregularidad irregularidad) {
		Irregularidad irregularidadAux = this.irregularidadesService.update(irregularidad);
		IrregularidadesController.logger.info("[PUT] : Irregularidad actualizado correctamente");
		return irregularidadAux;
	}

	/**
	 * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
	 * indicado.
	 *
	 * @param irregularidad
	 *            Irregularidad Bean que contiene la informacion con la que se
	 *            va a crear el nuevo registro.
	 * @return Irregularidad Bean resultante del proceso de creacion.
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody Irregularidad add(@RequestBody Irregularidad irregularidad) {
		Irregularidad irregularidadAux = this.irregularidadesService.add(irregularidad);
		IrregularidadesController.logger.info("[POST] : Irregularidad insertado correctamente");
		return irregularidadAux;
	}

	/**
	 * Operacion CRUD Delete. Borrado del registro correspondiente al
	 * identificador especificado.
	 *
	 * @param idIrregularidad
	 *            Long Identificador del objeto que se desea eliminar.
	 * @return Irregularidad Bean eliminado.
	 */
	@RequestMapping(value = "/{idIrregularidad}", method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody Irregularidad remove(@PathVariable Long idIrregularidad) {
		Irregularidad irregularidad = new Irregularidad();
		irregularidad.setIdIrregularidad(idIrregularidad);
		this.irregularidadesService.remove(irregularidad);
		IrregularidadesController.logger.info("[DELETE] : Irregularidad borrado correctamente");
		return irregularidad;
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE
	 * 
	 */

	/**
	 * Metodo de presentacion del RUP_TABLE.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "/maint", method = RequestMethod.GET)
	public String getFormEdit(Model model, HttpSession httpSession) {
		IrregularidadesController.logger.info("[GET - View] : ab7804s00");

		if (httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE) != null) {
			Object folderIdUltimoExpediente = httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE);

			model.addAttribute("ultimoExpediente", folderIdUltimoExpediente);
		}
		if (httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE_MIGRADO) != null) {
			Object IdDenunciaUltimoExpediente = httpSession
					.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE_MIGRADO);

			model.addAttribute("IdDenunciaUltimoExpediente", IdDenunciaUltimoExpediente);
		}

		return "irregularidades";
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterIrregularidad
	 *            Irregularidad Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<Irregularidad> Dto que contiene el resultado del
	 *         filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<Irregularidad> filter(
			@RequestJsonBody(param = "filter") Irregularidad filterIrregularidad,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		IrregularidadesController.logger.info("[POST - filter] : Obtener Ab7804s00s");
		return this.irregularidadesService.filter(filterIrregularidad, tableRequestDto, false);
	}

	/**
	 * Operacion de busqueda del componente RUP_TABLE.
	 * 
	 * @param filterIrregularidad
	 *            Irregularidad Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param searchAb7804s00
	 *            Irregularidad Bean que contiene los parametros de busqueda a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la búsqueda.
	 * @return TableRowDto<Irregularidad> Dto que contiene el resultado de la
	 *         busqueda realizada por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/search", method = RequestMethod.POST)
	public @ResponseBody List<TableRowDto<Irregularidad>> search(
			@RequestJsonBody(param = "filter") Irregularidad filterIrregularidad,
			@RequestJsonBody(param = "search") Irregularidad searchAb7804s00,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		IrregularidadesController.logger.info("[POST - search] : Buscar Ab7804s00s");
		return this.irregularidadesService.search(filterIrregularidad, searchAb7804s00, tableRequestDto, false);
	}

	/**
	 * Borrado multiple de registros
	 * 
	 * @param filterIrregularidad
	 *            Irregularidad Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la busqueda.
	 * @return List<String> Lista de los identificadores de los registros
	 *         eliminados.
	 * 
	 */
	@RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody List<String> removeMultiple(
			@RequestJsonBody(param = "filter") Irregularidad filterIrregularidad,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		IrregularidadesController.logger
				.info("[POST - search] : [POST - removeMultiple] : Eliminar multiples Ab7804s00s");
		this.irregularidadesService.removeMultiple(filterIrregularidad, tableRequestDto, false);
		IrregularidadesController.logger.info("All entities correctly deleted!");

		return tableRequestDto.getMultiselection().getSelectedIds();
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE - JERARQUIA
	 */

	/**
	 * Operacion de filtrado del componente RUP_TABLE para presentar los
	 * registros mediante visualizacion jerarquica.
	 * 
	 * @param filterIrregularidad
	 *            Irregularidad Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<Irregularidad>> Dto que contiene el
	 *         resultado del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquia/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Irregularidad>> jerarquia(
			@RequestJsonBody(param = "filter") Irregularidad filterIrregularidad,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		IrregularidadesController.logger.info("[POST - jerarquia] : Obtener Ab7804s00s jerarquia");
		return this.irregularidadesService.jerarquia(filterIrregularidad, tableRequestDto, false);
	}

	/**
	 * Recupera los hijos de los registros desplegados en la visualizacion
	 * jerarquica.
	 * 
	 * @param filterIrregularidad
	 *            Irregularidad Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<Irregularidad>> Dto que contiene el
	 *         resultado del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquiaChildren", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Irregularidad>> jerarquiaChildren(
			@RequestJsonBody(param = "filter") Irregularidad filterIrregularidad,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		IrregularidadesController.logger.info("[POST - jerarquia] : Obtener Ab7804s00s jerarquia - Hijos");
		return this.irregularidadesService.jerarquiaChildren(filterIrregularidad, tableRequestDto);
	}

	/**
	 * EXPORTERS
	 */

	@RequestMapping(value = "/clipboardReport", method = RequestMethod.POST)
	protected @ResponseBody List<Irregularidad> getClipboardReport(
			@RequestJsonBody(param = "filter") Irregularidad filterIrregularidad,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		IrregularidadesController.logger.info("[POST - clipboardReport] : : Copiar multiples usuarios");
		return this.irregularidadesService.getMultiple(filterIrregularidad, tableRequestDto, false);
	}

}
