package com.ejie.ab78.control.maestros;

import java.util.List;

import javax.servlet.http.HttpSession;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.ejie.ab78.comun.Constantes;
import com.ejie.ab78.comun.UtilUsuario;
import com.ejie.ab78.model.maestros.Decision;
import com.ejie.ab78.model.usuario.Usuario;
import com.ejie.ab78.service.maestros.DecisionesService;
import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.JerarquiaDto;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;
import com.ejie.x38.dto.TableRowDto;

/**
 * DecisionesController  
 * 
 *  
 */

@Controller
@RequestMapping(value = "/maestros/decisiones")
public class DecisionesController {

	private static final Logger logger = LoggerFactory.getLogger(DecisionesController.class);

	@Autowired
	private DecisionesService decisionesService;

	/*
	 * OPERACIONES CRUD (Create, Read, Update, Delete)
	 * 
	 */

	/**
	 * Operacion CRUD Read. Devuelve el bean correspondiente al identificador
	 * indicado.
	 * 
	 * @param idDecision
	 *            Long
	 * @return Decision Objeto correspondiente al identificador indicado.
	 */
	@RequestMapping(value = "/{idDecision}", method = RequestMethod.GET)
	public @ResponseBody Decision get(@PathVariable Long idDecision) {
		Decision decision = new Decision();
		decision.setIdDecision(idDecision);
		decision = this.decisionesService.find(decision);
		DecisionesController.logger.info("[GET - findBy_PK] : Obtener Decision por PK");
		return decision;
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterDecision
	 *            Decision Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Decision> Lista de objetos correspondientes a la busqueda
	 *         realizada.
	 */
	@RequestMapping(method = RequestMethod.GET)
	public @ResponseBody List<Decision> getAll(@ModelAttribute Decision filterDecision) {
		DecisionesController.logger.info("[GET - find_ALL] : Obtener Decision por filtro");
		return this.decisionesService.findAll(filterDecision, null);
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterDecision
	 *            Decision Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Decision> Lista de objetos correspondientes a la busqueda
	 *         realizada.
	 */
	@RequestMapping(value = "/no-formales", method = RequestMethod.GET)
	public @ResponseBody List<Decision> getAllNoFormales(@ModelAttribute Decision filterDecision) {
		DecisionesController.logger.info("[GET - find_ALL] : Obtener Decision por filtro");
		filterDecision.setEsNoFormal(Constantes.DECISION_ES_NO_FORMAL);
		filterDecision.setBaja(0);
		return this.decisionesService.findAll(filterDecision, null);
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterDecision
	 *            Decision Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Decision> Lista de objetos correspondientes a la busqueda
	 *         realizada.
	 */
	@RequestMapping(value = "/formales", method = RequestMethod.GET)
	public @ResponseBody List<Decision> getAllFormales(@ModelAttribute Decision filterDecision) {
		DecisionesController.logger.info("[GET - find_ALL] : Obtener Decision por filtro");
		filterDecision.setEsNoFormal(Constantes.DECISION_ES_FORMAL);
		filterDecision.setBaja(0);
		List<Decision> lista = this.decisionesService.findAll(filterDecision, null);
		Decision sinDecision = new Decision();
		sinDecision.setIdDecision(0L);
		sinDecision.setDescTipoEs("(vacio)");
		sinDecision.setDescTipoEu("(hutsik)");
		lista.add(sinDecision);
		return lista;
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterDecision
	 *            Decision Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda. idProcedimiento Identificador del
	 *            procedimiento para filtrar decisiones
	 * @return List<Decision> Lista de objetos correspondientes a la busqueda
	 *         realizada.
	 */
	@RequestMapping(value = "/formales/{idProcedimiento}", method = RequestMethod.GET)
	public @ResponseBody List<Decision> getAllFormalesFilter(@ModelAttribute Decision filterDecision,
			@PathVariable("idProcedimiento") Long idProcedimiento) {
		DecisionesController.logger.info("[GET - find_ALL] : Obtener Decision por filtro");
		filterDecision.setEsNoFormal(Constantes.DECISION_ES_FORMAL);
		filterDecision.setBaja(0);
		filterDecision.setIdProcedimiento(idProcedimiento);
		return this.decisionesService.findAllFilterProcedimiento(filterDecision, null);
	}

	/**
	 * Operacion CRUD Edit. Modificacion del bean indicado.
	 *
	 * @param decision
	 *            Decision Bean que contiene la informacion a modificar.
	 * @return Decision Bean resultante de la modificacion.
	 */
	@RequestMapping(method = RequestMethod.PUT)
	public @ResponseBody Decision edit(@RequestBody Decision decision) {
		Decision decisionAux = this.decisionesService.update(decision);
		DecisionesController.logger.info("[PUT] : Decision actualizado correctamente");
		return decisionAux;
	}

	/**
	 * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
	 * indicado.
	 *
	 * @param decision
	 *            Decision Bean que contiene la informacion con la que se va a
	 *            crear el nuevo registro.
	 * @return Decision Bean resultante del proceso de creacion.
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody Decision add(@RequestBody Decision decision) {
		Decision decisionAux = this.decisionesService.add(decision);
		DecisionesController.logger.info("[POST] : Decision insertado correctamente");
		return decisionAux;
	}

	/**
	 * Operacion CRUD Delete. Borrado del registro correspondiente al
	 * identificador especificado.
	 *
	 * @param idDecision
	 *            Long Identificador del objeto que se desea eliminar.
	 * @return Decision Bean eliminado.
	 */
	@RequestMapping(value = "/{idDecision}", method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody Decision remove(@PathVariable Long idDecision) {
		Decision decision = new Decision();
		decision.setIdDecision(idDecision);
		this.decisionesService.remove(decision);
		DecisionesController.logger.info("[DELETE] : Decision borrado correctamente");
		return decision;
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE
	 * 
	 */

	/**
	 * Metodo de presentacion del RUP_TABLE.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "/maint", method = RequestMethod.GET)
	public String getFormEdit(Model model, HttpSession httpSession) {
		DecisionesController.logger.info("[GET - View] : Decision");

		String username = UtilUsuario.getUserConnected();
		Usuario usuario = new Usuario();
		usuario.setNombreUsuario(username);
		model.addAttribute("usuario", usuario);

		if (httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE) != null) {
			Object folderIdUltimoExpediente = httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE);

			model.addAttribute("ultimoExpediente", folderIdUltimoExpediente);
		}
		if (httpSession.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE_MIGRADO) != null) {
			Object IdDenunciaUltimoExpediente = httpSession
					.getAttribute(Constantes.SESSIONKEY_ULTIMO_EXPEDIENTE_MIGRADO);

			model.addAttribute("IdDenunciaUltimoExpediente", IdDenunciaUltimoExpediente);
		}

		return "decisiones";
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterDecision
	 *            Decision Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<Decision> Dto que contiene el resultado del
	 *         filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<Decision> filter(@RequestJsonBody(param = "filter") Decision filterDecision,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		DecisionesController.logger.info("[POST - filter] : Obtener Decisions");
		return this.decisionesService.filter(filterDecision, tableRequestDto, false);
	}

	/**
	 * Operacion de busqueda del componente RUP_TABLE.
	 * 
	 * @param filterDecision
	 *            Decision Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param searchAb7802s00
	 *            Decision Bean que contiene los parametros de busqueda a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la búsqueda.
	 * @return TableRowDto<Decision> Dto que contiene el resultado de la
	 *         busqueda realizada por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/search", method = RequestMethod.POST)
	public @ResponseBody List<TableRowDto<Decision>> search(@RequestJsonBody(param = "filter") Decision filterDecision,
			@RequestJsonBody(param = "search") Decision searchAb7802s00,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		DecisionesController.logger.info("[POST - search] : Buscar Ab7802s00s");
		return this.decisionesService.search(filterDecision, searchAb7802s00, tableRequestDto, false);
	}

	/**
	 * Borrado multiple de registros
	 * 
	 * @param filterAb7802s00
	 *            Decision Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la busqueda.
	 * @return List<String> Lista de los identificadores de los registros
	 *         eliminados.
	 * 
	 */
	@RequestMapping(value = "/deleteAll", method = RequestMethod.POST)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody List<String> removeMultiple(@RequestJsonBody(param = "filter") Decision filterAb7802s00,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		DecisionesController.logger.info("[POST - search] : [POST - removeMultiple] : Eliminar multiples Ab7802s00s");
		this.decisionesService.removeMultiple(filterAb7802s00, tableRequestDto, false);
		DecisionesController.logger.info("All entities correctly deleted!");

		return tableRequestDto.getMultiselection().getSelectedIds();
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE - JERARQUIA
	 */

	/**
	 * Operacion de filtrado del componente RUP_TABLE para presentar los
	 * registros mediante visualizacion jerarquica.
	 * 
	 * @param filterAb7802s00
	 *            Decision Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<Decision>> Dto que contiene el
	 *         resultado del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquia/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Decision>> jerarquia(
			@RequestJsonBody(param = "filter") Decision filterAb7802s00,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		DecisionesController.logger.info("[POST - jerarquia] : Obtener Ab7802s00s jerarquia");
		return this.decisionesService.jerarquia(filterAb7802s00, tableRequestDto, false);
	}

	/**
	 * Recupera los hijos de los registros desplegados en la visualizacion
	 * jerarquica.
	 * 
	 * @param filterAb7802s00
	 *            Decision Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<JerarquiaDto<Decision>> Dto que contiene el
	 *         resultado del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/jerarquiaChildren", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<JerarquiaDto<Decision>> jerarquiaChildren(
			@RequestJsonBody(param = "filter") Decision filterAb7802s00,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		DecisionesController.logger.info("[POST - jerarquia] : Obtener Ab7802s00s jerarquia - Hijos");
		return this.decisionesService.jerarquiaChildren(filterAb7802s00, tableRequestDto);
	}

	/**
	 * EXPORTERS
	 */

	@RequestMapping(value = "/clipboardReport", method = RequestMethod.POST)
	protected @ResponseBody List<Decision> getClipboardReport(
			@RequestJsonBody(param = "filter") Decision filterDecision,
			@RequestJsonBody TableRequestDto tableRequestDto) {
		DecisionesController.logger.info("[POST - clipboardReport] : : Copiar multiples usuarios");
		return this.decisionesService.getMultiple(filterDecision, tableRequestDto, false);
	}

}
