package com.ejie.ab78.control.expedientes;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.http.HttpStatus;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.SessionAttributes;
import org.springframework.web.servlet.ModelAndView;

import com.ejie.ab78.model.expedientes.Expedientes;
import com.ejie.ab78.model.usuario.Usuario;
import com.ejie.ab78.service.expedientes.ExpedientesService;
import com.ejie.ab78.service.procedimientos.ProcedimientosService;
import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;
import com.ejie.x38.dto.TableRowDto;
import com.ejie.x38.reports.ReportData;

@Controller
@SessionAttributes("usuario")
@RequestMapping(value = "/expedientes")

public class ExpedientesController {

	private static final Logger logger = LoggerFactory.getLogger(ExpedientesController.class);

	@Autowired
	private ExpedientesService expedientesService;

	@Autowired
	private ProcedimientosService procedimientosService;

	@Resource
	private ReloadableResourceBundleMessageSource appMessageSource;

	/*
	 * OPERACIONES CRUD (Create, Read, Update, Delete)
	 * 
	 */

	/**
	 * Operacion CRUD Read. Devuelve el bean correspondiente al identificador
	 * indicado.
	 * 
	 * @param idExpediente
	 *            Integer
	 * @return Expedientes Objeto correspondiente al identificador indicado.
	 */
	@RequestMapping(value = "/{idExpediente}", method = RequestMethod.GET)
	public @ResponseBody Expedientes get(@PathVariable Integer idExpediente) {
		Expedientes expedientes = new Expedientes();
		expedientes.setIdExpediente(idExpediente);
		expedientes = this.expedientesService.find(expedientes);
		ExpedientesController.logger.info("[GET - findBy_PK] : Obtener Expedientes por PK");
		return expedientes;
	}

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterExpedientes
	 *            Expedientes Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Expedientes> Lista de objetos correspondientes a la busqueda
	 *         realizada.
	 */
	@RequestMapping(method = RequestMethod.GET)
	public @ResponseBody List<Expedientes> getAll(@ModelAttribute Expedientes filterExpedientes) {
		ExpedientesController.logger.info("[GET - find_ALL] : Obtener Expedientes por filtro");
		return this.expedientesService.findAll(filterExpedientes, null);
	}

	/**
	 * Operacion CRUD Edit. Modificacion del bean indicado.
	 *
	 * @param expedientes
	 *            Expedientes Bean que contiene la informacion a modificar.
	 * @return Expedientes Bean resultante de la modificacion.
	 */
	@RequestMapping(method = RequestMethod.PUT)
	public @ResponseBody Expedientes edit(@RequestBody Expedientes expedientes) {
		Expedientes expedientesAux = this.expedientesService.update(expedientes);
		ExpedientesController.logger.info("[PUT] : Expedientes actualizado correctamente");
		return expedientesAux;
	}

	/**
	 * Operacion CRUD Edit. Modificacion del bean indicado.
	 *
	 * @param expedientes
	 *            Expedientes Bean que contiene la informacion a modificar.
	 * @return Expedientes Bean resultante de la modificacion.
	 */
	@RequestMapping(value = "/folder", method = RequestMethod.PUT)
	public @ResponseBody Expedientes editFolder(@RequestBody Expedientes expedientes) {
		Expedientes expedientesAux = this.expedientesService.updateFolder(expedientes);
		ExpedientesController.logger.info("[PUT] : Expedientes actualizado correctamente");
		return expedientesAux;
	}

	/**
	 * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
	 * indicado.
	 *
	 * @param expedientes
	 *            Expedientes Bean que contiene la informacion con la que se va
	 *            a crear el nuevo registro.
	 * @return Expedientes Bean resultante del proceso de creacion.
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody Expedientes add(@RequestBody Expedientes expedientes) {
		Expedientes expedientesAux = this.expedientesService.add(expedientes);
		ExpedientesController.logger.info("[POST] : Expedientes insertado correctamente");
		return expedientesAux;
	}

	/**
	 * Operacion CRUD Delete. Borrado del registro correspondiente al
	 * identificador especificado.
	 *
	 * @param idExpediente
	 *            Integer Identificador del objeto que se desea eliminar.
	 * @return Expedientes Bean eliminado.
	 */
	@RequestMapping(value = "/{idExpediente}", method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody Expedientes remove(@PathVariable Integer idExpediente) {
		Expedientes expedientes = new Expedientes();
		expedientes.setIdExpediente(idExpediente);
		this.expedientesService.remove(expedientes);
		ExpedientesController.logger.info("[DELETE] : Expedientes borrado correctamente");
		return expedientes;
	}

	/*
	 * METODOS COMPONENTE RUP_TABLE
	 * 
	 */

	/**
	 * Metodo de presentacion del RUP_TABLE.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "/maint", method = RequestMethod.POST)
	public String getFormEdit(Model model) {
		model.addAttribute("busqExpedientes", new Expedientes());
		// TODO PRELOAD USER INFO PARA ESTABLECER FILTROS GEOGRAFICOS
		ExpedientesController.logger.info("[POST - View] : expedientes");
		return "busqExpedientes";

	}

	@RequestMapping(value = "/maint", method = RequestMethod.GET)
	public String getFormEditURL(Model model) {
		model.addAttribute("busqExpedientes", new Expedientes());
		// TODO PRELOAD USER INFO PARA ESTABLECER FILTROS GEOGRAFICOS

		// ---------------------------------------------------------------------
		// Cogemos de la seguridad el usuario logeado y lo guardamos en el model
		// ----------------------------------------------------------------------
		Authentication aut = SecurityContextHolder.getContext().getAuthentication();
		/** Dejamos el usuario logueado en session */
		Usuario usuario = new Usuario();
		String username = "";
		if (null != aut) {

			username = (String) aut.getPrincipal();
		}

		usuario.setNombreUsuario(username);
		model.addAttribute("usuario", usuario);

		ExpedientesController.logger.info("[GET - View] : expedientes");
		return "busqExpedientes";
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterExpedientes
	 *            Expedientes Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<Expedientes> Dto que contiene el resultado del
	 *         filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<Expedientes> filter(
			@RequestJsonBody(param = "filter") Expedientes filterExpedientes,
			@RequestJsonBody TableRequestDto TableRequestDto) {

		ExpedientesController.logger.info("[POST - filter] : Obtener Expedientess");
		return this.expedientesService.filterTer(filterExpedientes, TableRequestDto, false);

	}

	/**
	 * Método para buscar el Expediente por FOLDER_NUMBER.
	 * 
	 * @param String
	 *            folderNumber
	 * @return <Expedientes>
	 */
	@RequestMapping(value = "/buscar/{folderNumber}", method = RequestMethod.GET)
	public @ResponseBody List<Expedientes> buscar(@PathVariable("folderNumber") String folderNumber) {

		ExpedientesController.logger.info("[POST - /buscar/{folderNumber}] : Buscar Expedientes" + folderNumber);

		return this.expedientesService.findExpediente(folderNumber);

	}

	/**
	 * Operacion de busqueda del componente RUP_TABLE.
	 * 
	 * @param filterExpedientes
	 *            Expedientes Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param searchExpedientes
	 *            Expedientes Bean que contiene los parametros de busqueda a
	 *            emplear.
	 * @param TableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en la búsqueda.
	 * @return TableRowDto<Expedientes> Dto que contiene el resultado de la
	 *         busqueda realizada por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/search", method = RequestMethod.POST)
	public @ResponseBody List<TableRowDto<Expedientes>> search(
			@RequestJsonBody(param = "filter") Expedientes filterExpedientes,
			@RequestJsonBody(param = "search") Expedientes searchExpedientes,
			@RequestJsonBody TableRequestDto TableRequestDto) {
		ExpedientesController.logger.info("[POST - search] : Buscar Expedientess");
		return this.expedientesService.search(filterExpedientes, searchExpedientes, TableRequestDto, false);
	}

	@RequestMapping(value = { "/xlsReport", "/xlsxReport" }, method = RequestMethod.POST)
	protected ModelAndView getExcelPOI(@ModelAttribute Expedientes filterExpedientes,
			@ModelAttribute TableRequestDto TableRequestDto, ModelMap modelMap,
			@RequestParam(value = "columns", required = false) String columns, HttpServletRequest request) {

		Locale locale = LocaleContextHolder.getLocale();

		// filterExpedientes.setIdExpediente(initForm.getIdExpediente());
		// Acceso a BD para recuperar datos

		columns = "[['folderNumber','" + appMessageSource.getMessage("excel.expediente", null, locale) + "']"
				+ ",['titulo','" + appMessageSource.getMessage("excel.titulo", null, locale) + "']" + ",['terEmp','"
				+ appMessageSource.getMessage("excel.solicitante", null, locale) + "']" + ",['fechaEntradaHoraStr','"
				+ appMessageSource.getMessage("excel.fechaEntrada", null, locale) + "']" + ",['fechaDocFormalHoraStr','"
				+ appMessageSource.getMessage("excel.fechaDocFormal", null, locale) + "']" + ",['fechaUltAporHoraStr','"
				+ appMessageSource.getMessage("excel.fechaUltApo", null, locale) + "']"
				+ ",['tiposResultado.descripcionCastellano','"
				+ appMessageSource.getMessage("excel.resultado", null, locale) + "']" + "]";

		List<Expedientes> listProgramasPage = this.expedientesService.findAllLikeTer(filterExpedientes, TableRequestDto,
				null);
		TableRequestDto.setPage(null);
		TableRequestDto.setRows(null);
		List<Expedientes> listProgramasAll = this.expedientesService.findAllLikeTer(filterExpedientes, TableRequestDto,
				null);

		// Nombre fichero
		modelMap.put("fileName", "ExpedientesExcel");

		// Datos
		List<Object> reportData = new ArrayList<Object>();
		// Hoja 1
		ReportData<Expedientes> usuarioExcelDataAll = new ReportData<Expedientes>();
		// nombre hoja
		usuarioExcelDataAll.setSheetName(appMessageSource.getMessage("usuarioExcel.totalExpediente", null, locale));
		// cabeceras hoja
		usuarioExcelDataAll.setHeaderNames(ReportData.parseColumns(columns));
		// datos hoja
		usuarioExcelDataAll.setModelData(listProgramasAll);
		reportData.add(usuarioExcelDataAll);
		// Hoja 2
		ReportData<Expedientes> usuarioExcelDataPage = new ReportData<Expedientes>();
		// nombre hoja
		usuarioExcelDataPage.setSheetName(appMessageSource.getMessage("usuarioExcel.pag1Expediente", null, locale));
		// cabeceras hoja
		usuarioExcelDataPage.setHeaderNames(ReportData.parseColumns(columns));
		// datos hoja
		usuarioExcelDataPage.setModelData(listProgramasPage);
		reportData.add(usuarioExcelDataPage);
		modelMap.put("reportData", reportData);

		// Generación del XLS o XLSX
		if (request.getServletPath().indexOf("xlsReport") != -1) {
			return new ModelAndView("xlsReport", modelMap);
		} else {
			return new ModelAndView("xlsxReport", modelMap);
		}

	}

}
