package com.ejie.ab78.dao.listados;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.ab78.comun.Constantes;
import com.ejie.ab78.model.listados.Listados;
import com.ejie.x38.dto.TableManager;
import com.ejie.x38.dto.TableRequestDto;

/**
 * ListadosDaoImpl  
 * 
 *  
 */

@Repository
@Transactional
public class ListadosDaoImpl implements ListadosDao {

	public static final String[] ORDER_BY_WHITE_LIST = new String[] { "IDLISTADO", "IDPROCEDIMIENTO", "NOMBRE",
			"FICHERO", "FECHAREGISTRO", "USUARIOREGISTRO", "FECHAULTMODIFICACION", "USUARIOULTMODIFICACION" };

	private JdbcTemplate jdbcTemplate;

	/*
	 * ROW_MAPPERS
	 */
	private RowMapper<Listados> rwMap = new RowMapper<Listados>() {
		public Listados mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			Listados listado = new Listados();
			listado.setIdListado(resultSet.getInt("IDLISTADO"));
			listado.setNombre(resultSet.getString("NOMBRE"));
			listado.setNombreBusquedas(resultSet.getString("NOMBREBUSQUEDA"));
			return listado;
		}
	};

	private RowMapper<Listados> rwMapKey = new RowMapper<Listados>() {
		public Listados mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			Listados listado = new Listados();
			listado.setIdListado(resultSet.getInt("IDLISTADO"));
			listado.setNombre(resultSet.getString("NOMBRE"));
			listado.setFicheroArray(resultSet.getBytes("FICHERO"));
			listado.setNombreFichero(resultSet.getString("NOMBREFICHERO"));
			return listado;
		}
	};

	private RowMapper<Listados> rwMapRelLisPro = new RowMapper<Listados>() {
		public Listados mapRow(ResultSet resultSet, int rowNum) throws SQLException {

			Listados listados = new Listados();
			listados.setIdListado(resultSet.getInt("IDLISTADO"));
			if (LocaleContextHolder.getLocale().getLanguage().equalsIgnoreCase(Constantes.CASTELLANO)) {
				listados.setNombre(resultSet.getString("DESCRIPCIONCASTELLANO"));
			} else {
				listados.setNombre(resultSet.getString("DESCRIPCIONEUSKERA"));
			}

			return listados;
		}
	};

	/**
	 * Method use to set the datasource.
	 *
	 * @param dataSource
	 *            DataSource
	 * @return
	 */
	@Resource
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * OPERACIONES CRUD
	 */

	/**
	 * Finds a single row in the Listados table.
	 *
	 * @param listados
	 *            Listados
	 * @return Listados
	 */
	@Transactional(readOnly = true)
	public Listados find(Listados listados) {
		String query = "SELECT t1.ID_LISTADO IDLISTADO, t1.NOMBRE_FICHERO NOMBREFICHERO, t1.NOMBRE NOMBRE, t1.FICHERO FICHERO, t1.FECHA_REGISTRO FECHAREGISTRO, t1.USUARIO_REGISTRO USUARIOREGISTRO, t1.FECHA_ULTMODIFICACION FECHAULTMODIFICACION, t1.USUARIO_ULTMODIFICACION USUARIOULTMODIFICACION FROM AB78.LISTADOS t1  WHERE t1.ID_LISTADO = ? ";

		List<Listados> listadosList = this.jdbcTemplate.query(query, this.rwMapKey, listados.getIdListado());
		return (Listados) DataAccessUtils.uniqueResult(listadosList);
	}

	/**
	 * Finds rows in the Listados table using like.
	 *
	 * @param listados
	 *            Listados
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @param startsWith
	 *            Boolean
	 * @return List<Listados>
	 */
	@Transactional(readOnly = true)
	public List<Listados> findAllLike(Listados listados, TableRequestDto TableRequestDto, Boolean startsWith) {
		StringBuilder query = new StringBuilder(
				"SELECT T1.ID_LISTADO IDLISTADO, T1.NOMBRE NOMBRE, (SELECT LISTAGG(CONCAT(T2.NOMBRE,'|')) WITHIN GROUP (ORDER BY T2.NOMBRE) AS CONCATDATA FROM AB78.BUSQUEDAS T2 WHERE T2.ID_LISTADO = T1.ID_LISTADO) as NOMBREBUSQUEDA FROM AB78.LISTADOS T1");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereLikeMap(listados, startsWith);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		if (TableRequestDto != null) {
			query = TableManager.getPaginationQuery(TableRequestDto, query, ListadosDaoImpl.ORDER_BY_WHITE_LIST);
		}

		return (List<Listados>) this.jdbcTemplate.query(query.toString(), this.rwMap, params.toArray());
	}
	/*
	 * MÉTODOS PRIVADOS
	 */

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Listados entity
	 *
	 * @param listados
	 *            Listados Bean with the criteria values to filter by.
	 * @param startsWith
	 *            Boolean
	 * @return Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 */
	// CHECKSTYLE:OFF CyclomaticComplexity - Generación de código de UDA
	private Map<String, Object> getWhereLikeMap(Listados listados, Boolean startsWith) {

		StringBuilder where = new StringBuilder(ListadosDaoImpl.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (listados != null && listados.getIdListado() != null) {
			where.append(" AND t1.ID_LISTADO = ?");
			params.add(listados.getIdListado());
		}
		if (listados != null && listados.getProcedimientos() != null
				&& null != listados.getProcedimientos().getIdProcedimiento()) {
			where.append(" AND t1.ID_PROCEDIMIENTO = ?");
			params.add(listados.getProcedimientos().getIdProcedimiento());
		}
		if (listados != null && listados.getNombre() != null) {
			where.append(" AND UPPER(t1.NOMBRE) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(listados.getNombre().toUpperCase() + "%");
			} else {
				params.add("%" + listados.getNombre().toUpperCase() + "%");
			}
			where.append(" AND t1.NOMBRE IS NOT NULL");
		}
		if (listados != null && listados.getFechaRegistro() != null) {
			where.append(" AND t1.FECHA_REGISTRO = ?");
			params.add(listados.getFechaRegistro());
		}
		if (listados != null && listados.getUsuarioRegistro() != null) {
			where.append(" AND UPPER(t1.USUARIO_REGISTRO) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(listados.getUsuarioRegistro().toUpperCase() + "%");
			} else {
				params.add("%" + listados.getUsuarioRegistro().toUpperCase() + "%");
			}
			where.append(" AND t1.USUARIO_REGISTRO IS NOT NULL");
		}
		if (listados != null && listados.getFechaUltmodificacion() != null) {
			where.append(" AND t1.FECHA_ULTMODIFICACION = ?");
			params.add(listados.getFechaUltmodificacion());
		}
		if (listados != null && listados.getUsuarioUltmodificacion() != null) {
			where.append(" AND UPPER(t1.USUARIO_ULTMODIFICACION) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(listados.getUsuarioUltmodificacion().toUpperCase() + "%");
			} else {
				params.add("%" + listados.getUsuarioUltmodificacion().toUpperCase() + "%");
			}
			where.append(" AND t1.USUARIO_ULTMODIFICACION IS NOT NULL");
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}

	/**
	 * StringBuilder initilization value
	 */
	public static final int STRING_BUILDER_INIT = 4096;

	/**
	 * Finds rows in the Listados table using like.
	 *
	 * @param idProcedimiento
	 *            Integer
	 * @return List<Listados>
	 */
	@Transactional(readOnly = true)
	public List<Listados> findRelLisProByProc(Integer idProcedimiento) {
		StringBuilder query = new StringBuilder();

		query.append(
				"SELECT T1.ID_LISTADO IDLISTADO, T2.DESCRIPCION_CASTELLANO DESCRIPCIONCASTELLANO, T2.DESCRIPCION_EUSKERA DESCRIPCIONEUSKERA");
		query.append(" FROM AB78.LISTADOS T1, AB78.REL_LIS_PRO t2");
		query.append(" WHERE t1.ID_LISTADO     = t2.ID_LISTADO");
		query.append(" AND t2.ID_PROCEDIMIENTO = ?");

		return this.jdbcTemplate.query(query.toString(), this.rwMapRelLisPro, idProcedimiento);
	}
}
