package com.ejie.ab78.dao.expedientes;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.support.DataAccessUtils;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.ab78.comun.Constantes;
import com.ejie.ab78.comun.UtilUsuario;
import com.ejie.ab78.model.Inspeccion;
import com.ejie.ab78.model.expedientes.Expedientes;
import com.ejie.ab78.model.procedimientos.Areas;
import com.ejie.ab78.model.procedimientos.Direcciones;
import com.ejie.ab78.model.procedimientos.Procedimientos;
import com.ejie.ab78.model.terceros.DatosTerceros;
import com.ejie.ab78.model.terceros.TiposTercero;
import com.ejie.x38.dao.RowNumResultSetExtractor;
import com.ejie.x38.dto.JerarquiaDto;
import com.ejie.x38.dto.TableManager;
import com.ejie.x38.dto.TableManagerJerarquia;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableRowDto;

@Repository
@Transactional
public class ExpedientesDaoImpl implements ExpedientesDao {

	public static final String[] ORDER_BY_WHITE_LIST = new String[] { "IDEXPEDIENTE", "IDENTIFICADOR_ESTABLECIMIENTO",
			"IDTERRITORIOHISTORICO", "FOLDERID", "FOLDERNUMBER", "TITULO", "DOCUMENTO", "FECHAREGISTRO",
			"USUARIOREGISTRO", "PROCEDIMIENTOSIDPROCEDIMIENTO", "PROCEDIMIENTOSCODPROCEDUREID",
			"PROCEDIMIENTOSFOLDERID", "PROCEDIMIENTOSFOLDERNUMBER", "PROCEDIMIENTOSTITULOEXPEDIENTE",
			"PROCEDIMIENTOSFECHAREGISTRO", "PROCEDIMIENTOSUSUARIOREGISTRO", "PROCEDIMIENTOSIDDIRECCION",
			"PROCEDIMIENTOSIDAREA", "IMPORTEPRESENTADO", "IMPORTEADMITIDO", "SUBVENCIONCALCULADA", "SUBVENCIONRESUELTA",
			"DESCRIPCIONCASTELLANO", "DESCRIPCIONEUSKERA", "NOMBRETERCERO", "NOMBREEMPRESA", "EXPFECHAENTRADA",
			"EXPFECHACIERRE", "IDRELEXPINS", "IDEXPINSP", "IDINSPECCION", "TIPOINSPECCION", "FECHAREGISTROINSP",
			"USUARIOREGISTROINSP", "IDENTIFICADORINSPECCION", "FOLDERID", "PRODDESCCAST", "PRODDESCEUSK",
			"FOLDERNUMBERORIGEN", "FOLDERIDORIGEN" };

	private JdbcTemplate jdbcTemplate;
	private static final Logger logger = LoggerFactory.getLogger(ExpedientesDaoImpl.class);

	/**
	 * Method use to set the datasource.
	 *
	 * @param dataSource
	 *            DataSource
	 * @return
	 */
	@Resource
	public void setDataSource(DataSource dataSource) {
		this.jdbcTemplate = new JdbcTemplate(dataSource);
	}

	/*
	 * ROW_MAPPERS
	 */
	private RowMapper<Expedientes> rwMap = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {

			Expedientes expediente = new Expedientes();
			expediente.setIdExpediente(resultSet.getInt("IDEXPEDIENTE"));
			expediente.setIdTerritorioHistorico(resultSet.getLong("IDTERRITORIOHISTORICO"));
			expediente.setFolderId(resultSet.getString("FOLDERID"));
			expediente.setFolderNumber(resultSet.getString("FOLDERNUMBER"));
			expediente.setTitulo(resultSet.getString("TITULO"));
			expediente.setIdExpedienteOrigen(resultSet.getInt("IDEXPEDIENTEORIGEN"));

			Procedimientos procedimiento = new Procedimientos();
			procedimiento.setIdProcedimiento(resultSet.getInt("ProcedimientosIDPROCEDIMIENTO"));
			procedimiento.setCodProcedureId(resultSet.getString("ProcedimientosCODPROCEDUREID"));
			procedimiento.setCamposTituloExpediente(resultSet.getString("ProcedimientosTITULOEXPEDIENTE"));
			procedimiento.setDescripcionEs(resultSet.getString("PROCDESCES"));
			procedimiento.setDescripcionEu(resultSet.getString("PROCDESCEU"));
			procedimiento.setInspeccionable(resultSet.getString("INSPECCIONABLE"));
			procedimiento.setTipoDenuncia(resultSet.getInt("TIPODENUNCIA"));
			if (resultSet.wasNull()) {
				procedimiento.setTipoDenuncia(null);
			}

			Direcciones direccion = new Direcciones();
			direccion.setIdDireccion(resultSet.getInt("ProcedimientosIDDIRECCION"));

			Areas area = new Areas();
			area.setIdArea(resultSet.getInt("ProcedimientosIDAREA"));

			procedimiento.setDirecciones(direccion);
			procedimiento.setAreas(area);
			expediente.setProcedimientos(procedimiento);

			return expediente;

		}
	};

	private RowMapper<Expedientes> rwMapExpedientesPendientes = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			return new Expedientes(resultSet.getInt("IDEXPEDIENTE"), resultSet.getLong("IDTERRITORIOHISTORICO"),
					resultSet.getString("FOLDERID"), resultSet.getString("FOLDERNUMBER"), resultSet.getString("TITULO"),
					resultSet.getDate("FECHAREGISTRO"), null,
					new Procedimientos(null, resultSet.getString("ProcedimientosCODPROCEDUREID"), null, null, null,
							null, null, resultSet.getString("DESCCAST"), resultSet.getString("DESCEUSK"),
							resultSet.getString("INSPECCIONABLE")));
		}
	};

	private RowMapper<Inspeccion> rwMapInspecion = new RowMapper<Inspeccion>() {
		@Override
		public Inspeccion mapRow(ResultSet resultSet, int rowNum) throws SQLException {

			return new Inspeccion(resultSet.getInt("ID_REL_EXP_INS"),
					new Expedientes(resultSet.getInt("ID_EXPEDIENTE")), resultSet.getInt("ID_INSPECCION"),
					resultSet.getString("IDENTIFICADOR_INSPECCION"));
		}
	};

	private RowMapper<Date> rwMapDateExp = new RowMapper<Date>() {
		@Override
		public Date mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			return resultSet.getDate(1);
		}
	};

	private RowMapper<Expedientes> rwMapProcedureId = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			Expedientes expedientes = new Expedientes();
			Procedimientos procedimientos = new Procedimientos();
			procedimientos.setIdProcedimiento(resultSet.getInt("IDPROCEDIMIENTO"));
			expedientes.setProcedimientos(procedimientos);

			return expedientes;
		}
	};

	private RowMapper<Expedientes> rwMapExpedienteId = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			Expedientes expedientes = new Expedientes();
			expedientes.setIdExpediente(resultSet.getInt("IDEXPEDIENTE"));
			return expedientes;
		}
	};

	private RowMapper<Expedientes> rwMapEmpTer = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			String folderNumber = "";
			String titulo = "";
			String documento = "";
			String nombre = "";
			String apellido1 = "";
			String apellido2 = "";

			if (null != resultSet.getString("FOLDERNUMBER")) {
				folderNumber = resultSet.getString("FOLDERNUMBER");
			}
			if (null != resultSet.getString("TITULO")) {
				titulo = resultSet.getString("TITULO");
			}

			if (null != resultSet.getString("NOMBRE")) {
				nombre = resultSet.getString("NOMBRE");
			}
			if (null != resultSet.getString("DOCUMENTO")) {
				documento = resultSet.getString("DOCUMENTO");
			}

			if (null != resultSet.getString("APELLIDO1")) {
				apellido1 = resultSet.getString("APELLIDO1");
			}
			if (null != resultSet.getString("APELLIDO2")) {
				apellido2 = resultSet.getString("APELLIDO2");
			}
			Expedientes expedientes = new Expedientes();
			expedientes.setIdExpediente(resultSet.getInt("IDEXPEDIENTE"));
			expedientes.setFolderNumber(folderNumber);
			expedientes.setTitulo(titulo);

			DatosTerceros datosTerceros = new DatosTerceros();

			datosTerceros.setDocumento(documento);
			datosTerceros.setNombre(nombre);
			datosTerceros.setApellido1(apellido1);
			datosTerceros.setApellido2(apellido2);
			expedientes.setDatosTerceros(datosTerceros);

			return expedientes;
		}
	};

	private RowMapper<Expedientes> rwMapPK = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			return new Expedientes(resultSet.getInt("IDEXPEDIENTE"));
		}
	};

	private RowMapper<Expedientes> rwMapFolderNumber = new RowMapper<Expedientes>() {
		@Override
		public Expedientes mapRow(ResultSet resultSet, int rowNum) throws SQLException {
			return new Expedientes(resultSet.getString("FOLDER_NUMBER"));
		}
	};

	/**
	 * Rowmapper para Jerarquía
	 *
	 * @param dataSource
	 *            DataSource
	 * @return
	 */
	private RowMapper<JerarquiaDto<Expedientes>> rwMapJerarquia = new RowMapper<JerarquiaDto<Expedientes>>() {
		@Override
		public JerarquiaDto<Expedientes> mapRow(ResultSet resultSet, int rowNum) throws SQLException {

			Expedientes expedientes = new Expedientes(resultSet.getInt("IDEXPEDIENTE"),
					resultSet.getLong("IDTERRITORIOHISTORICO"), resultSet.getString("FOLDERID"),
					resultSet.getString("FOLDERNUMBER"), resultSet.getString("TITULO"),
					resultSet.getDate("FECHAREGISTRO"), resultSet.getString("USUARIOREGISTRO"),
					new Procedimientos(resultSet.getInt("ProcedimientosIDPROCEDIMIENTO"),
							resultSet.getString("ProcedimientosCODPROCEDUREID"),
							resultSet.getString("ProcedimientosTITULOEXPEDIENTE"),
							resultSet.getDate("ProcedimientosFECHAREGISTRO"),
							resultSet.getString("ProcedimientosUSUARIOREGISTRO"),
							new Direcciones(resultSet.getInt("ProcedimientosIDDIRECCION"), null, null, null, null, null,
									null, null),
							new Areas(resultSet.getInt("ProcedimientosIDAREA"), null, null, null, null, null, null)));

			JerarquiaDto<Expedientes> jerarquia = new JerarquiaDto<Expedientes>();
			jerarquia.setModel(expedientes);
			jerarquia.setLevel(resultSet.getBigDecimal("LEVEL").intValue());
			jerarquia.setParentNodes(resultSet.getString("PARENTNODES"));
			jerarquia.setIsLeaf(Boolean.parseBoolean(resultSet.getString("ISLEAF")));
			jerarquia.setFilter(Boolean.parseBoolean(resultSet.getString("FILTER")));
			return jerarquia;
		}
	};

	/*
	 * OPERACIONES CRUD
	 */

	/**
	 * Inserts a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	public Expedientes add(Expedientes expedientes) {
		String query = "INSERT INTO AB78.EXPEDIENTES (ID_EXPEDIENTE, ID_PROCEDIMIENTO, ID_TERRITORIO_HISTORICO, FOLDER_ID, FOLDER_NUMBER, TITULO, FECHA_REGISTRO, USUARIO_REGISTRO) VALUES (?,?,?,?,?,?,SYSDATE,?)";
		Object getProcedimientosIdProcedimientoAux = null;
		if (expedientes.getProcedimientos() != null && expedientes.getProcedimientos().getIdProcedimiento() != null) {
			getProcedimientosIdProcedimientoAux = expedientes.getProcedimientos().getIdProcedimiento();
		}
		this.jdbcTemplate.update(query, expedientes.getIdExpediente(), getProcedimientosIdProcedimientoAux,
				expedientes.getIdTerritorioHistorico(), expedientes.getFolderId(), expedientes.getFolderNumber(),
				expedientes.getTitulo(), UtilUsuario.getUserConnected());
		return expedientes;
	}

	/**
	 * Updates a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	public Expedientes update(Expedientes expedientes) {
		String query = "UPDATE AB78.EXPEDIENTES SET ID_PROCEDIMIENTO=?, ID_TERRITORIO_HISTORICO=?, FOLDER_ID=?, FOLDER_NUMBER=?, TITULO=?, FECHA_ULTMODIFICACION=SYSDATE, USUARIO_ULTMODIFICACION=? WHERE ID_EXPEDIENTE=?";
		Object getProcedimientosIdProcedimientoAux = null;
		if (expedientes.getProcedimientos() != null && expedientes.getProcedimientos().getIdProcedimiento() != null) {
			getProcedimientosIdProcedimientoAux = expedientes.getProcedimientos().getIdProcedimiento();
		}
		this.jdbcTemplate.update(query, getProcedimientosIdProcedimientoAux, expedientes.getIdTerritorioHistorico(),
				expedientes.getFolderId(), expedientes.getFolderNumber(), expedientes.getTitulo(),
				UtilUsuario.getUserConnected(), expedientes.getIdExpediente());
		return expedientes;
	}

	/**
	 * Updates a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	public Boolean updateOrigen(Expedientes expedientes) {
		String query = "UPDATE AB78.EXPEDIENTES SET ID_EXPEDIENTE_ORIGEN = ?, FECHA_ULTMODIFICACION=SYSDATE, USUARIO_ULTMODIFICACION=? WHERE ID_EXPEDIENTE = ? ";
		int result = this.jdbcTemplate.update(query, expedientes.getIdExpedienteOrigen(),
				UtilUsuario.getUserConnected(), expedientes.getIdExpediente());
		return result == 1;
	}

	/**
	 * Updates a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientesk
	 * @return Expedientes
	 */
	@Override
	public Expedientes updateFolder(Expedientes expedientes) {
		String query = "UPDATE AB78.EXPEDIENTES SET FOLDER_NUMBER=?, FOLDER_ID=?, FECHA_ULTMODIFICACION=SYSDATE, USUARIO_ULTMODIFICACION=? WHERE ID_EXPEDIENTE=?";

		this.jdbcTemplate.update(query, expedientes.getFolderNumber(), expedientes.getFolderId(),
				UtilUsuario.getUserConnected(), expedientes.getIdExpediente());
		return expedientes;
	}

	/**
	 * Updates a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	public Expedientes updateExpedienteFolder(Expedientes expedientes) {
		String query = "UPDATE AB78.EXPEDIENTES SET TITULO=?, FOLDER_NUMBER=?, FECHA_ULTMODIFICACION=SYSDATE, USUARIO_ULTMODIFICACION=? WHERE FOLDER_ID=?";
		logger.info("Inicio updateFolderTitle");
		logger.info("Query = " + query);
		logger.info("PARAMETRO TITULO = " + expedientes.getTitulo());
		logger.info("PARAMETRO FECHA_REGISTRO = SYSDATE");
		logger.info("PARAMETRO USUARIO_REGISTRO = " + expedientes.getUsuarioRegistro());
		logger.info("PARAMETRO FOLDER_ID = " + expedientes.getFolderId());
		this.jdbcTemplate.update(query, expedientes.getTitulo(), expedientes.getFolderNumber(),
				UtilUsuario.getUserConnected(), expedientes.getFolderId());
		return expedientes;
	}

	/**
	 * Updates a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	public Expedientes updateTituloMigrados(Expedientes expedientes) {
		String query = "UPDATE AB78.EXPEDIENTES SET TITULO=? WHERE ID_EXPEDIENTE=?";
		this.jdbcTemplate.update(query, expedientes.getTitulo(), expedientes.getIdExpediente());
		return expedientes;
	}

	/**
	 * Finds a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	@Transactional(readOnly = true)
	public Expedientes find(Expedientes expedientes) {
		String query = "SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE, t1.ID_EXPEDIENTE_ORIGEN IDEXPEDIENTEORIGEN, t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO, t1.FOLDER_ID FOLDERID, t1.FOLDER_NUMBER FOLDERNUMBER, t1.TITULO TITULO, t1.FECHA_REGISTRO FECHAREGISTRO, t1.USUARIO_REGISTRO USUARIOREGISTRO, t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO, t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID,  t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE, t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO, t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO, t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION, t2.ID_AREA PROCEDIMIENTOSIDAREA, t2.DESCRIPCION_CASTELLANO PROCDESCES, t2.DESCRIPCION_EUSKERA PROCDESCEU, t2.INSPECCIONABLE, t2.TIPO_DENUNCIA TIPODENUNCIA FROM AB78.EXPEDIENTES t1 , AB78.PROCEDIMIENTOS t2  WHERE t1.ID_EXPEDIENTE = ?   AND t1.ID_PROCEDIMIENTO= t2.ID_PROCEDIMIENTO(+)";

		List<Expedientes> expedientesList = this.jdbcTemplate.query(query, this.rwMap, expedientes.getIdExpediente());
		return DataAccessUtils.uniqueResult(expedientesList);
	}

	/**
	 * Finds a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	@Transactional(readOnly = true)
	public Expedientes findProcedureIdPorExpediente(Expedientes expedientes) {
		String query = "SELECT t1.ID_PROCEDIMIENTO IDPROCEDIMIENTO FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2 WHERE t1.ID_PROCEDIMIENTO = t2.ID_PROCEDIMIENTO(+) AND t1.ID_EXPEDIENTE   = ?";

		List<Expedientes> expedientesList = this.jdbcTemplate.query(query, this.rwMapProcedureId,
				expedientes.getIdExpediente());
		return DataAccessUtils.uniqueResult(expedientesList);
	}

	/**
	 * Finds a single row in the Expedientes table por FolderId.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	@Transactional(readOnly = true)
	public Expedientes findByFolderId(Expedientes expedientes) {
		String query = "SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE, t1.ID_EXPEDIENTE_ORIGEN IDEXPEDIENTEORIGEN, t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO, t1.FOLDER_ID FOLDERID, t1.FOLDER_NUMBER FOLDERNUMBER, t1.TITULO TITULO, t1.FECHA_REGISTRO FECHAREGISTRO, t1.USUARIO_REGISTRO USUARIOREGISTRO, t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO, t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID, t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE, t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO, t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO, t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION, t2.ID_AREA PROCEDIMIENTOSIDAREA, t2.DESCRIPCION_CASTELLANO PROCDESCES, t2.DESCRIPCION_EUSKERA PROCDESCEU, t2.INSPECCIONABLE, t2.TIPO_DENUNCIA TIPODENUNCIA FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2  WHERE t1.FOLDER_ID = ?   AND t1.ID_PROCEDIMIENTO= t2.ID_PROCEDIMIENTO(+)";

		List<Expedientes> expedientesList = this.jdbcTemplate.query(query, this.rwMap, expedientes.getFolderId());
		return DataAccessUtils.uniqueResult(expedientesList);
	}

	/**
	 * Finds a single row in the Expedientes table por FolderId.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Expedientes
	 */
	@Override
	@Transactional(readOnly = true)
	public Expedientes findByFolderNumber(Expedientes expedientes) {
		String query = "SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE FROM AB78.EXPEDIENTES t1 , AB78.PROCEDIMIENTOS t2  WHERE t1.FOLDER_NUMBER = ?";

		List<Expedientes> expedientesList = this.jdbcTemplate.query(query, this.rwMapFolderNumber,
				expedientes.getFolderNumber());
		return DataAccessUtils.uniqueResult(expedientesList);
	}

	/**
	 * Removes a single row in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return
	 */
	@Override
	public void remove(Expedientes expedientes) {
		String query = "DELETE FROM AB78.EXPEDIENTES WHERE ID_EXPEDIENTE=?";
		this.jdbcTemplate.update(query, expedientes.getIdExpediente());
	}

	/**
	 * Finds a list of rows in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @return List<Expedientes>
	 */
	@Override
	@Transactional(readOnly = true)
	public List<Expedientes> findAll(Expedientes expedientes, TableRequestDto TableRequestDto) {
		StringBuilder query = new StringBuilder(
				"SELECT  t1.ID_EXPEDIENTE IDEXPEDIENTE, t1.ID_EXPEDIENTE_ORIGEN IDEXPEDIENTEORIGEN,t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO,t1.FOLDER_ID FOLDERID,t1.FOLDER_NUMBER FOLDERNUMBER,t1.TITULO TITULO,t1.FECHA_REGISTRO FECHAREGISTRO, t1.USUARIO_REGISTRO USUARIOREGISTRO,t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO,t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID,t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE,t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO,t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO,t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION,t2.ID_AREA PROCEDIMIENTOSIDAREA, t2.DESCRIPCION_CASTELLANO PROCDESCES, t2.DESCRIPCION_EUSKERA PROCDESCEU, t2.INSPECCIONABLE, t2.TIPO_DENUNCIA TIPODENUNCIA ");
		query.append("FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(expedientes);
		StringBuilder where = new StringBuilder(" WHERE 1=1 AND t1.ID_PROCEDIMIENTO= t2.ID_PROCEDIMIENTO(+) ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		if (TableRequestDto != null) {
			query = TableManager.getPaginationQuery(TableRequestDto, query, ExpedientesDaoImpl.ORDER_BY_WHITE_LIST);
		}

		return this.jdbcTemplate.query(query.toString(), this.rwMap, params.toArray());
	}

	/*
	 * OPERACIONES RUP_TABLE
	 */

	/**
	 * Counts rows in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @return Long
	 */
	@Override
	@Transactional(readOnly = true)
	public Long findAllCount(Expedientes expedientes) {
		StringBuilder query = new StringBuilder("SELECT COUNT(1) FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2 ");

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereMap(expedientes);
		StringBuilder where = new StringBuilder(" WHERE 1=1 AND t1.ID_PROCEDIMIENTO= t2.ID_PROCEDIMIENTO(+) ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		List<?> params = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.queryForObject(query.toString(), Long.class, params.toArray());
	}

	/**
	 * Finds rows in the Expedientes table using like.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @param startsWith
	 *            Boolean
	 * @return List<Expedientes>
	 */

	@Override
	@Transactional(readOnly = true)
	public List<Expedientes> findAllLikeTer(Expedientes expedientes, TableRequestDto TableRequestDto,
			Boolean startsWith) {
		StringBuilder query = new StringBuilder();
		query.append(
				"SELECT T1.ID_EXPEDIENTE IDEXPEDIENTE, T1.ID_PROCEDIMIENTO IDPROCEDIMIENTO, T1.FOLDER_NUMBER FOLDERNUMBER, T1.TITULO TITULO,  ");
		query.append(" T8.DOCUMENTO DOCUMENTO, T8.NOMBRE NOMBRE, T8.APELLIDO1 APELLIDO1, T8.APELLIDO2 APELLIDO2 ");
		query.append(
				"FROM AB78.EXPEDIENTES T1 INNER JOIN AB78.REL_EXP_TER T2 ON T1.ID_EXPEDIENTE = T2.ID_EXPEDIENTE INNER JOIN AB78.PROCEDIMIENTOS T6 ON T1.ID_PROCEDIMIENTO = T6.ID_PROCEDIMIENTO ");
		query.append(" INNER JOIN AB78.DATOS_TERCERO T8 ON T2.ID_TERCERO = T8.ID_TERCERO AND  T8.TIPO_PARTICIPANTE="
				+ Constantes.PARTICIPANTE_TITULAR);

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereExpediente(expedientes);
		query.append(mapaWhere.get("query"));

		List<?> params = (List<?>) mapaWhere.get("params");

		if (TableRequestDto != null) {
			query = TableManager.getPaginationQuery(TableRequestDto, query, ExpedientesDaoImpl.ORDER_BY_WHITE_LIST);
		}

		return this.jdbcTemplate.query(query.toString(), this.rwMapEmpTer, params.toArray());
	}

	/**
	 * Finds rows in the Expedientes table using like.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @param startsWith
	 *            Boolean
	 * @return List<Expedientes>
	 */

	@Override
	@Transactional(readOnly = true)
	public List<Expedientes> findAllLike(Expedientes expedientes, TableRequestDto TableRequestDto, Boolean startsWith) {
		StringBuilder query = new StringBuilder();
		query.append(
				"SELECT T1.ID_EXPEDIENTE IDEXPEDIENTE, T1.ID_PROCEDIMIENTO IDPROCEDIMIENTO, T1.FOLDER_NUMBER FOLDERNUMBER, T1.TITULO TITULO, ");
		query.append(" T8.DOCUMENTO DOCUMENTO, T8.NOMBRE NOMBRE, T8.APELLIDO1 APELLIDO1, T8.APELLIDO2 APELLIDO2 ");
		query.append(
				"FROM AB78.EXPEDIENTES T1 INNER JOIN AB78.REL_EXP_TER T2 ON T1.ID_EXPEDIENTE = T2.ID_EXPEDIENTE INNER JOIN AB78.PROCEDIMIENTOS T6 ON T1.ID_PROCEDIMIENTO = T6.ID_PROCEDIMIENTO ");
		query.append(" INNER AB78.JOIN DATOS_TERCERO T8 ON T2.ID_TERCERO = T8.ID_TERCERO AND  T8.TIPO_PARTICIPANTE="
				+ Constantes.PARTICIPANTE_TITULAR);

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereExpediente(expedientes);
		query.append(mapaWhere.get("query"));

		List<?> params = (List<?>) mapaWhere.get("params");

		if (TableRequestDto != null) {
			query = TableManager.getPaginationQuery(TableRequestDto, query, ExpedientesDaoImpl.ORDER_BY_WHITE_LIST);
		}

		return this.jdbcTemplate.query(query.toString(), this.rwMapEmpTer, params.toArray());
	}

	/**
	 * Counts rows in the Expedientes table using like.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	@Override
	@Transactional(readOnly = true)
	public Long findAllLikeCountTer(Expedientes expedientes, Boolean startsWith) {

		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM AB78.EXPEDIENTES T1 INNER JOIN AB78.REL_EXP_TER T2 ON T1.ID_EXPEDIENTE = T2.ID_EXPEDIENTE INNER JOIN AB78.PROCEDIMIENTOS T6 ON T1.ID_PROCEDIMIENTO = T6.ID_PROCEDIMIENTO INNER JOIN AB78.DATOS_TERCERO T8 ON T2.ID_TERCERO = T8.ID_TERCERO AND T8.TIPO_PARTICIPANTE="
						+ Constantes.PARTICIPANTE_TITULAR);

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereExpediente(expedientes);
		query.append(mapaWhere.get("query"));
		List<?> params = (List<?>) mapaWhere.get("params");
		return this.jdbcTemplate.queryForObject(query.toString(), Long.class, params.toArray());
	}

	/**
	 * Counts rows in the Expedientes table using like.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param startsWith
	 *            Boolean
	 * @return Long
	 */
	@Override
	@Transactional(readOnly = true)
	public Long findAllLikeCount(Expedientes expedientes, Boolean startsWith) {

		StringBuilder query = new StringBuilder(
				"SELECT COUNT(1) FROM AB78.EXPEDIENTES T1 INNER JOIN AB78.REL_EXP_TER T2 ON T1.ID_EXPEDIENTE = T2.ID_EXPEDIENTE INNER JOIN AB78.PROCEDIMIENTOS T6 ON T1.ID_PROCEDIMIENTO = T6.ID_PROCEDIMIENTO INNER JOIN AB78.DATOS_TERCERO T8 ON T2.ID_TERCERO = T8.ID_TERCERO AND T8.TIPO_PARTICIPANTE="
						+ Constantes.PARTICIPANTE_TITULAR);

		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereExpediente(expedientes);
		query.append(mapaWhere.get("query"));
		List<?> params = (List<?>) mapaWhere.get("params");
		return this.jdbcTemplate.queryForObject(query.toString(), Long.class, params.toArray());
	}

	/**
	 * Finds a list of rows in the Expedientes table.
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @return List<Expedientes>
	 */
	@Override
	@Transactional(readOnly = true)
	public List<Expedientes> findLupaExpediente(String folderNumber) {
		StringBuilder query = new StringBuilder(
				"SELECT T1.ID_EXPEDIENTE IDEXPEDIENTE FROM AB78.EXPEDIENTES T1 WHERE UPPER(t1.FOLDER_NUMBER) LIKE UPPER('%"
						+ folderNumber + "%')");
		return this.jdbcTemplate.query(query.toString(), this.rwMapExpedienteId);
	}

	/**
	 * Reorder the data list of Expedientes selected for rup_table
	 *
	 * @param expedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @param startsWith
	 *            Boolean
	 * @return List<TableRowDto<Expedientes>>
	 */
	@Override
	public List<TableRowDto<Expedientes>> reorderSelection(Expedientes expedientes, TableRequestDto TableRequestDto,
			Boolean startsWith) {
		// SELECT
		StringBuilder query = new StringBuilder(
				"SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE,t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO,t1.FOLDER_ID FOLDERID,t1.FOLDER_NUMBER FOLDERNUMBER,t1.TITULO TITULO,t1.FECHA_REGISTRO FECHAREGISTRO,t1.USUARIO_REGISTRO USUARIOREGISTRO,t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO,t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID,t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE,t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO,t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO,t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION,t2.ID_AREA PROCEDIMIENTOSIDAREA ");
		// FROM
		query.append("FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2 ");
		// FILTRADO
		// Where clause & Params
		Map<String, ?> mapaWhere = this.getWhereLikeMap(expedientes, startsWith);
		StringBuilder where = new StringBuilder(" WHERE 1=1 AND t1.ID_PROCEDIMIENTO= t2.ID_PROCEDIMIENTO(+) ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		// Parámetros de filtrado
		@SuppressWarnings("unchecked")
		List<Object> filterParamList = (List<Object>) mapaWhere.get("params");

		// SQL para la reordenación
		StringBuilder sbReorderSelectionSQL = TableManager.getReorderQuery(query, TableRequestDto, Expedientes.class,
				filterParamList, "ID_EXPEDIENTE");

		return this.jdbcTemplate.query(sbReorderSelectionSQL.toString(),
				new RowNumResultSetExtractor<Expedientes>(this.rwMapPK, TableRequestDto), filterParamList.toArray());
	}

	/**
	 * Search method for rup_table
	 *
	 * @param filterParams
	 *            Expedientes
	 * @param searchParams
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @param startsWith
	 *            Boolean
	 * @return List<TableRowDto<Expedientes>>
	 */
	@Override
	public List<TableRowDto<Expedientes>> search(Expedientes filterParams, Expedientes searchParams,
			TableRequestDto TableRequestDto, Boolean startsWith) {
		// SELECT
		StringBuilder query = new StringBuilder(
				"SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE,t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO,t1.FOLDER_ID FOLDERID,t1.FOLDER_NUMBER FOLDERNUMBER,t1.TITULO TITULO,t1.FECHA_REGISTRO FECHAREGISTRO,t1.USUARIO_REGISTRO USUARIOREGISTRO,t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO,t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID,t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE,t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO,t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO,t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION,t2.ID_AREA PROCEDIMIENTOSIDAREA ");
		// FROM
		query.append("FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2 ");
		// TABLAS_ALIAS
		List<String> from_alias = new ArrayList<String>();
		from_alias.add(" t1 ");
		from_alias.add(" t2 ");

		// FILTRADO
		Map<String, ?> mapaWhereFilter = this.getWhereLikeMap(filterParams, startsWith);
		StringBuilder where = new StringBuilder(" WHERE 1=1 AND t1.ID_PROCEDIMIENTO= t2.ID_PROCEDIMIENTO(+) ");
		where.append(mapaWhereFilter.get("query"));
		query.append(where);
		// Filter params
		@SuppressWarnings("unchecked")
		List<Object> filterParamList = (List<Object>) mapaWhereFilter.get("params");

		// BUSQUEDA
		Map<String, Object> mapaWhereSearch = this.getWhereLikeMap(searchParams, startsWith);
		String searchSQL = ((StringBuilder) mapaWhereSearch.get("query")).toString();
		// Search params
		@SuppressWarnings("unchecked")
		List<Object> searchParamList = (List<Object>) mapaWhereSearch.get("params");

		// SQL
		StringBuilder sbReorderSelectionSQL = TableManager.getSearchQuery(query, TableRequestDto, Expedientes.class,
				filterParamList, searchSQL, searchParamList, from_alias, "ID_EXPEDIENTE");

		return this.jdbcTemplate.query(sbReorderSelectionSQL.toString(),
				new RowNumResultSetExtractor<Expedientes>(this.rwMapPK, TableRequestDto), filterParamList.toArray());
	}

	/**
	 * Remove multiple method for rup_table
	 *
	 * @param filterexpedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @param startsWith
	 *            Boolean
	 */
	@Override
	public void removeMultiple(Expedientes filterexpedientes, TableRequestDto TableRequestDto, Boolean startsWith) {
		// SELECT
		/** TODO: select por clave */
		StringBuilder query = new StringBuilder(
				"SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE,t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO,t1.FOLDER_ID FOLDERID,t1.FOLDER_NUMBER FOLDERNUMBER,t1.TITULO TITULO,t1.FECHA_REGISTRO FECHAREGISTRO,t1.USUARIO_REGISTRO USUARIOREGISTRO,t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO,t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID,t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE,t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO,t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO,t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION,t2.ID_AREA PROCEDIMIENTOSIDAREA ");
		// FROM
		query.append("FROM AB78.EXPEDIENTES t1, AB78.PROCEDIMIENTOS t2 ");

		// WHERE
		Map<String, Object> mapaWhere = this.getWhereLikeMap(filterexpedientes, startsWith);
		StringBuilder where = new StringBuilder(" WHERE 1=1 ");
		where.append(mapaWhere.get("query"));
		query.append(where);

		@SuppressWarnings("unchecked")
		List<Object> params = (List<Object>) mapaWhere.get("params");

		StringBuilder sbRemoveMultipleSQL = TableManager.getRemoveMultipleQuery(TableRequestDto, Expedientes.class,
				query, params, "ID_EXPEDIENTE");
		this.jdbcTemplate.update(sbRemoveMultipleSQL.toString(), params.toArray());
	}

	/**
	 * Filter in the hierarchical Expedientes table.
	 *
	 * @param filterExpedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @return List<JerarquiaDto<Expedientes>>
	 */
	@Override
	public List<JerarquiaDto<Expedientes>> findAllLikeJerarquia(Expedientes filterExpedientes,
			TableRequestDto TableRequestDto) {
		// SELECT
		StringBuilder query = new StringBuilder(
				"SELECT t1.ID_EXPEDIENTE IDEXPEDIENTE,t1.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO,t1.FOLDER_ID FOLDERID,t1.FOLDER_NUMBER FOLDERNUMBER,t1.TITULO TITULO,t1.FECHA_REGISTRO FECHAREGISTRO,t1.USUARIO_REGISTRO USUARIOREGISTRO,t2.ID_PROCEDIMIENTO PROCEDIMIENTOSIDPROCEDIMIENTO,t2.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID,t2.CAMPOS_TITULO_EXPEDIENTE ProcedimientosTITULOEXPEDIENTE,t2.FECHA_REGISTRO PROCEDIMIENTOSFECHAREGISTRO,t2.USUARIO_REGISTRO PROCEDIMIENTOSUSUARIOREGISTRO,t2.ID_DIRECCION PROCEDIMIENTOSIDDIRECCION,t2.ID_AREA PROCEDIMIENTOSIDAREA ");
		// FROM
		// StringBuilder from = new StringBuilder("FROM EXPEDIENTES t1
		// ,PROCEDIMIENTOS t2 ");
		List<String> from = new ArrayList<String>();
		from.add("AB78.EXPEDIENTES");
		from.add("AB78.PROCEDIMIENTOS");
		// TABLAS_ALIAS
		List<String> from_alias = new ArrayList<String>();
		from_alias.add(" t1 ");
		from_alias.add(" t2 ");

		// JOINS TABLAS
		/*
		 * Ejemplo de como indicar joins entre las tablas de la query
		 */
		/*
		 * StringBuilder joins = new StringBuilder("");
		 * joins.append("AND T1.ID=T2.ID_T1");
		 */

		// CONDICIONES (negocio)
		/*
		 * Ejemplo de como incluir condiciones de negocio en la consulta de la
		 * jerarquía
		 */
		/*
		 * StringBuilder businessFilters = new StringBuilder(); List<Object>
		 * businessParams = new ArrayList<Object>();
		 * businessFilters.append("   AND t1.EJIE = ?  ");
		 * businessParams.add("1");
		 */

		// FILTRO
		Map<String, ?> mapaWhere = this.getWhereLikeMap(filterExpedientes, false);

		// JERARQUIA
		// Especificar la columna padre correspondiente en lugar de la generada
		// por defecto (ID_PADRE)
		query = TableManagerJerarquia.getQuery(TableRequestDto, query, mapaWhere, "ID_EXPEDIENTE", "ID_PADRE", "NOMBRE",
				from, from_alias);

		/*
		 * Ejemplo de llamada en caso de indicar joins entre tablas y
		 * condiciones de negocio
		 */
		// query = TableManagerJerarquia.getQuery(TableRequestDto, query,
		// mapaWhere, "ID", "ID_PADRE", "NOMBRE", from, from_alias, joins,
		// businessFilters, businessParams);

		// PAGINACION
		if (TableRequestDto != null) {
			query = TableManagerJerarquia.getPaginationQuery(TableRequestDto, query,
					ExpedientesDaoImpl.ORDER_BY_WHITE_LIST);
		}

		List<?> params = (List<?>) mapaWhere.get("params");
		return this.jdbcTemplate.query(query.toString(), this.rwMapJerarquia, params.toArray());
	}

	/**
	 * Count rows in hierarchy
	 *
	 * @param filterExpedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @return Long
	 */
	@Override
	public Long findAllLikeCountJerarquia(Expedientes filterExpedientes, TableRequestDto TableRequestDto) {
		// FROM
		// StringBuilder from = new StringBuilder("FROM EXPEDIENTES t1
		// ,PROCEDIMIENTOS t2 ");
		List<String> from = new ArrayList<String>();
		from.add("AB78.EXPEDIENTES");
		from.add("AB78.PROCEDIMIENTOS");

		// TABLAS_ALIAS
		List<String> from_alias = new ArrayList<String>();
		from_alias.add(" t1 ");
		from_alias.add(" t2 ");

		// JOINS TABLAS
		/*
		 * Ejemplo de como indicar joins entre las tablas de la query
		 */
		/*
		 * StringBuilder joins = new StringBuilder("");
		 * joins.append("AND T1.ID=T2.ID_T1");
		 */

		// CONDICIONES (negocio)
		/*
		 * Ejemplo de como incluir condiciones de negocio en la consulta de la
		 * jerarquía
		 */
		/*
		 * StringBuilder businessFilters = new StringBuilder(); List<Object>
		 * businessParams = new ArrayList<Object>();
		 * businessFilters.append("   AND t1.EJIE = ?  ");
		 * businessParams.add("1");
		 */

		// FILTRO
		Map<String, ?> mapaWhere = this.getWhereLikeMap(filterExpedientes, false);

		// JERARQUIA
		// Especificar la columna padre correspondiente en lugar de la generada
		// por defecto (ID_PADRE)
		StringBuilder query = TableManagerJerarquia.getQueryCount(TableRequestDto, mapaWhere, "ID_EXPEDIENTE",
				"ID_PADRE", from, from_alias);

		/*
		 * Ejemplo de llamada en caso de indicar joins entre tablas y
		 * condiciones de negocio
		 */
		// StringBuilder query =
		// TableManagerJerarquia.getQueryCount(TableRequestDto, mapaWhere,
		// "ID", "ID_PADRE", from, from_alias, joins, businessFilters,
		// businessParams);

		List<?> params = (List<?>) mapaWhere.get("params");
		return this.jdbcTemplate.queryForObject(query.toString(), Long.class, params.toArray());
	}

	@Override
	@Transactional(readOnly = true)
	public List<Expedientes> findExpedientesPendientesByExpediente(Integer idExpediente) {
		StringBuilder query = new StringBuilder();
		query.append("SELECT t02.ID_EXPEDIENTE IDEXPEDIENTE, ");
		query.append("  t02.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO, ");
		query.append("  t02.FOLDER_ID FOLDERID, ");
		query.append("  t02.FOLDER_NUMBER FOLDERNUMBER, ");
		query.append("  t02.FECHA_REGISTRO FECHAREGISTRO, ");
		query.append("  t02.TITULO TITULO, ");
		query.append("  t01.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID, ");
		query.append(" t01.DESCRIPCION_CASTELLANO DESCCAST, ");
		query.append(" t01.DESCRIPCION_EUSKERA DESCEUSK, ");
		query.append(" t01.INSPECCIONABLE INSPECCIONABLE ");
		query.append("FROM AB78.REL_EXP_TER t34, ");
		query.append("  AB78.DATOS_DIRECCIONES t36, ");
		query.append("  AB78.EXPEDIENTES t02, ");
		query.append("  AB78.PROCEDIMIENTOS t01 ");
		query.append(" WHERE t34.ID_TERCERO = t36.ID_TERCERO ");
		query.append(" AND t34.ID_EXPEDIENTE = t02.ID_EXPEDIENTE ");
		query.append(" AND t02.ID_PROCEDIMIENTO = t01.ID_PROCEDIMIENTO ");
		query.append(
				" AND t02.id_expediente NOT IN (SELECT relFec.id_expediente FROM AB78.REL_EXP_FECH relFec, AB78.FECHAS_EXPEDIENTE FEC WHERE relFec.ID_FECHA_EXPEDIENTE = FEC.ID_FECHA_EXPEDIENTE AND FEC.ETIQUETA_FECHA = 'FECHA_CIERRE' ) ");
		query.append(" AND t01.COD_PROCEDURE_ID IN ('").append(Constantes.PROCEDIMIENTO_INICIO_ACTIVIDAD).append("','")
				.append(Constantes.PROCEDIMIENTO_MODIFICACIONES_ACTIVIDAD).append("','")
				.append(Constantes.PROCEDIMIENTO_CESE_ACTIVIDAD).append("','")
				.append(Constantes.PROCEDIMIENTO_DISPENSAS_ACTIVIDAD).append("','")
				.append(Constantes.PROCEDIMIENTO_CONSULTAS_ACTIVIDAD).append("','")
				.append(Constantes.PROCEDIMIENTO_ADAPTACION_DECRETO_ACTIVIDAD).append("','")
				.append(Constantes.PROCEDIMIENTO_COMPROBACION_DE_OFICIO).append("') ");
		// query.append(" AND t34.ID_TIPO_TERCERO = 1 ");
		query.append(" AND t36.ID_TIPO_DIRECCION = 2 ");
		query.append(" AND t36.IDENTIFICADOR_ESTABLECIMIENTO IN ");
		query.append("  (SELECT t36.IDENTIFICADOR_ESTABLECIMIENTO ");
		query.append("  FROM AB78.REL_EXP_TER t34, ");
		query.append("    AB78.DATOS_DIRECCIONES t36 ");
		query.append("  WHERE t34.ID_EXPEDIENTE = ?");
		query.append("  AND t34.ID_TERCERO = t36.ID_TERCERO");
		// query.append(" AND t34.ID_TIPO_TERCERO = 1");
		query.append("  AND t36.ID_TIPO_DIRECCION = 2");
		query.append(" ) ");
		query.append("ORDER BY FECHAREGISTRO DESC");

		// Where clause & Params
		List<Object> params = new ArrayList<Object>();

		params.add(idExpediente);

		return this.jdbcTemplate.query(query.toString(), this.rwMapExpedientesPendientes, params.toArray());
	}

	@Override
	public Boolean hasInspectExpedient(Integer idExpediente) {
		String query = "SELECT COUNT(ID_EXPEDIENTE) FROM AB78.INSPECCIONES_EXPEDIENTE WHERE ID_EXPEDIENTE = ? ";
		Long numInspect = this.jdbcTemplate.queryForObject(query, Long.class, idExpediente);
		return numInspect > 0;
	}

	@Override
	public Boolean hasInspeccion(Integer idExpediente) {
		String query = "SELECT COUNT(1) FROM AB78.INSPECCIONES_EXPEDIENTE WHERE ID_EXPEDIENTE = ? ";
		Long numInspect = this.jdbcTemplate.queryForObject(query, Long.class, idExpediente);
		return numInspect > 0;
	}

	@Override
	public boolean checkDateExpediente(Expedientes expediente, String etiqueta) {
		String query = "SELECT COUNT(ID_EXPEDIENTE) FROM AB78.REL_EXP_FECH RELFEC, AB78.FECHAS_EXPEDIENTE FEC WHERE RELFEC.ID_FECHA_EXPEDIENTE =FEC.ID_FECHA_EXPEDIENTE AND FEC.ETIQUETA_FECHA = ? AND ID_EXPEDIENTE = ? ";
		Long numInspect = this.jdbcTemplate.queryForObject(query, Long.class, etiqueta, expediente.getIdExpediente());
		return numInspect > 0;
	}

	@Override
	public Inspeccion getInspectExpedient(Integer idExpediente) {
		String query = "SELECT ID_REL_EXP_INS, ID_EXPEDIENTE, ID_INSPECCION, IDENTIFICADOR_INSPECCION FROM AB78.INSPECCIONES_EXPEDIENTE WHERE ID_EXPEDIENTE = ? ";
		List<Object> params = new ArrayList<Object>();
		params.add(idExpediente);
		List<Inspeccion> listInps = this.jdbcTemplate.query(query, this.rwMapInspecion, params.toArray());
		if (listInps.isEmpty()) {
			return new Inspeccion();
		}
		return listInps.get(0);
	}

	@Override
	public Inspeccion getInspectByFolderId(String folderId) {
		String query = "SELECT REL.ID_REL_EXP_INS, EXP.ID_EXPEDIENTE, REL.IDENTIFICADOR_INSPECCION, REL.ID_INSPECCION, EXP.TITULO, EXP.FOLDER_ID FROM AB78.INSPECCIONES_EXPEDIENTE REL, AB78.EXPEDIENTES EXP WHERE REL.ID_EXPEDIENTE = EXP.ID_EXPEDIENTE AND EXP.FOLDER_ID = ?";
		List<Object> params = new ArrayList<Object>();
		params.add(folderId);
		List<Inspeccion> listInps = this.jdbcTemplate.query(query, this.rwMapInspecion, params.toArray());
		if (listInps.isEmpty()) {
			return new Inspeccion();
		}
		return listInps.get(0);
	}

	@Override
	public Inspeccion getInspect(String identInspeccion) {
		String query = "SELECT ID_REL_EXP_INS, ID_EXPEDIENTE, ID_INSPECCION, IDENTIFICADOR_INSPECCION FROM AB78.INSPECCIONES_EXPEDIENTE WHERE IDENTIFICADOR_INSPECCION = ? ";
		List<Object> params = new ArrayList<Object>();
		params.add(identInspeccion);
		List<Inspeccion> listInps = this.jdbcTemplate.query(query, this.rwMapInspecion, params.toArray());
		if (listInps.isEmpty()) {
			return new Inspeccion();
		}
		return listInps.get(0);
	}

	@Override
	@Transactional(readOnly = true)
	public List<Expedientes> findAllExpedientesOrigen(String identificadorEstablecimiento) {
		StringBuilder query = new StringBuilder();
		query.append("SELECT DISTINCT t02.ID_EXPEDIENTE IDEXPEDIENTE, ");
		query.append("  t02.ID_TERRITORIO_HISTORICO IDTERRITORIOHISTORICO, ");
		query.append("  t02.FOLDER_ID FOLDERID, ");
		query.append("  t02.FOLDER_NUMBER FOLDERNUMBER, ");
		query.append("  t02.FECHA_REGISTRO FECHAREGISTRO, ");
		query.append("  t02.TITULO TITULO, ");
		query.append("  t01.COD_PROCEDURE_ID PROCEDIMIENTOSCODPROCEDUREID, ");
		query.append(" t01.DESCRIPCION_CASTELLANO DESCCAST, ");
		query.append(" t01.DESCRIPCION_EUSKERA DESCEUSK,t01.INSPECCIONABLE INSPECCIONABLE ");
		query.append(" FROM AB78.REL_EXP_TER t34, ");
		query.append("  AB78.DATOS_DIRECCIONES t36, ");
		query.append("  AB78.EXPEDIENTES t02, ");
		query.append("  AB78.FECHAS_EXPEDIENTE T4, ");
		query.append("  AB78.REL_EXP_FECH T5, ");
		query.append("  AB78.PROCEDIMIENTOS t01 ");
		query.append(" WHERE t34.ID_TERCERO = t36.ID_TERCERO ");
		query.append(" AND t34.ID_EXPEDIENTE = t02.ID_EXPEDIENTE ");
		query.append(" AND t5.id_expediente = t02.id_expediente ");
		query.append(" AND t02.ID_PROCEDIMIENTO = t01.ID_PROCEDIMIENTO ");
		query.append(" AND T5.ID_FECHA_EXPEDIENTE = T4.ID_FECHA_EXPEDIENTE  ");
		query.append(" AND T34.ID_EXPEDIENTE = T02.ID_EXPEDIENTE ");
		query.append(" AND T4.ETIQUETA_FECHA = 'FECHA_CIERRE' ");

		query.append("AND t01.COD_PROCEDURE_ID IN ('").append(Constantes.PROCEDIMIENTO_INICIO_ACTIVIDAD).append("', '")
				.append(Constantes.PROCEDIMIENTO_MODIFICACIONES_ACTIVIDAD).append("', '")
				.append(Constantes.PROCEDIMIENTO_COMPROBACION_DE_OFICIO).append("', '")
				.append(Constantes.PROCEDIMIENTO_ADAPTACION_DECRETO_ACTIVIDAD).append("') ");
		query.append(" AND (t34.ID_TIPO_TERCERO = 1 OR t34.ID_TIPO_TERCERO = 3) ");
		query.append(" AND t36.ID_TIPO_DIRECCION = 2 ");
		query.append(" AND t36.IDENTIFICADOR_ESTABLECIMIENTO = ? ");
		query.append(" ORDER BY FECHAREGISTRO DESC ");

		// Where clause & Params
		List<Object> params = new ArrayList<Object>();

		params.add(identificadorEstablecimiento);

		return this.jdbcTemplate.query(query.toString(), this.rwMapExpedientesPendientes, params.toArray());
	}

	/**
	 * Find dependent hierarchical rows
	 *
	 * @param filterExpedientes
	 *            Expedientes
	 * @param TableRequestDto
	 *            TableRequestDto
	 * @return List<TableRowDto<Expedientes>>
	 */
	@Override
	public List<TableRowDto<Expedientes>> findAllChild(Expedientes filterExpedientes, TableRequestDto TableRequestDto) {

		// FROM
		// StringBuilder from = new StringBuilder("FROM EXPEDIENTES t1
		// ,PROCEDIMIENTOS t2 ");
		List<String> from = new ArrayList<String>();
		from.add("AB78.EXPEDIENTES");
		from.add("AB78.PROCEDIMIENTOS");

		// TABLAS_ALIAS
		List<String> from_alias = new ArrayList<String>();
		from_alias.add(" t1 ");
		from_alias.add(" t2 ");

		// JOINS TABLAS
		/*
		 * Ejemplo de como indicar joins entre las tablas de la query
		 */
		/*
		 * StringBuilder joins = new StringBuilder("");
		 * joins.append("AND T1.ID=T2.ID_T1");
		 */

		// CONDICIONES (negocio)
		/*
		 * Ejemplo de como incluir condiciones de negocio en la consulta de la
		 * jerarquía
		 */
		/*
		 * StringBuilder businessFilters = new StringBuilder(); List<Object>
		 * businessParams = new ArrayList<Object>();
		 * businessFilters.append("   AND t1.EJIE = ?  ");
		 * businessParams.add("1");
		 */

		// FILTRO
		Map<String, ?> mapaWhere = this.getWhereLikeMap(filterExpedientes, false);

		// MULTISELECCION
		// Especificar la columna padre correspondiente en lugar de la generada
		// por defecto (ID_PADRE)
		StringBuilder query = TableManagerJerarquia.getQueryChildren(TableRequestDto, mapaWhere, "ID_EXPEDIENTE",
				"ID_PADRE", from, from_alias);

		/*
		 * Ejemplo de llamada en caso de indicar joins entre tablas y
		 * condiciones de negocio
		 */
		// StringBuilder query =
		// TableManagerJerarquia.getQueryChildren(TableRequestDto, mapaWhere,
		// "ID", "ID_PADRE", from, from_alias, joins, businessFilters,
		// businessParams);

		List<?> params = (List<?>) mapaWhere.get("params");

		return this.jdbcTemplate.query(query.toString(),
				new RowNumResultSetExtractor<Expedientes>(this.rwMapPK, "ID_EXPEDIENTE"), params.toArray());
	}

	/*
	 * MÉTODOS PRIVADOS
	 */

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Expedientes entity
	 *
	 * @param expedientes
	 *            Expedientes Bean with the criteria values to filter by.
	 * @return Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 */
	// CHECKSTYLE:OFF CyclomaticComplexity - Generación de código de UDA
	private Map<String, ?> getWhereMap(Expedientes expedientes) {

		StringBuilder where = new StringBuilder(ExpedientesDaoImpl.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (expedientes != null && expedientes.getIdExpediente() != null) {
			where.append(" AND t1.ID_EXPEDIENTE = ?");
			params.add(expedientes.getIdExpediente());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getIdProcedimiento() != null) {
			where.append(" AND t2.ID_PROCEDIMIENTO = ?");
			params.add(expedientes.getProcedimientos().getIdProcedimiento());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getDirecciones() != null
				&& expedientes.getProcedimientos().getDirecciones().getIdDireccion() != null) {
			where.append(" AND t2.ID_DIRECCION = ?");
			params.add(expedientes.getProcedimientos().getDirecciones().getIdDireccion());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getAreas() != null
				&& expedientes.getProcedimientos().getAreas().getIdArea() != null) {
			where.append(" AND t2.ID_AREA = ?");
			params.add(expedientes.getProcedimientos().getAreas().getIdArea());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getCodProcedureId() != null) {
			where.append(" AND t2.COD_PROCEDURE_ID = ?");
			params.add(expedientes.getProcedimientos().getCodProcedureId());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getCamposTituloExpediente() != null) {
			where.append(" AND t2.CAMPOS_TITULO_EXPEDIENTE = ?");
			params.add(expedientes.getProcedimientos().getCamposTituloExpediente());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getFechaRegistro() != null) {
			where.append(" AND t2.FECHA_REGISTRO = ?");
			params.add(expedientes.getProcedimientos().getFechaRegistro());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getUsuarioRegistro() != null) {
			where.append(" AND t2.USUARIO_REGISTRO = ?");
			params.add(expedientes.getProcedimientos().getUsuarioRegistro());
		}
		if (expedientes != null && expedientes.getIdTerritorioHistorico() != null) {
			where.append(" AND t1.ID_TERRITORIO_HISTORICO = ?");
			params.add(expedientes.getIdTerritorioHistorico());
		}
		if (expedientes != null && expedientes.getFolderId() != null) {
			where.append(" AND t1.FOLDER_ID = ?");
			params.add(expedientes.getFolderId());
		}
		if (expedientes != null && expedientes.getFolderNumber() != null) {
			where.append(" AND t1.FOLDER_NUMBER = ?");
			params.add(expedientes.getFolderNumber());
		}
		if (expedientes != null && expedientes.getTitulo() != null) {
			where.append(" AND t1.TITULO = ?");
			params.add(expedientes.getTitulo());
		}
		if (expedientes != null && expedientes.getFechaRegistro() != null) {
			where.append(" AND t1.FECHA_REGISTRO = ?");
			params.add(expedientes.getFechaRegistro());
		}
		if (expedientes != null && expedientes.getUsuarioRegistro() != null) {
			where.append(" AND t1.USUARIO_REGISTRO = ?");
			params.add(expedientes.getUsuarioRegistro());
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}
	// CHECKSTYLE:ON CyclomaticComplexity - Generación de código de UDA

	/**
	 * Returns a map with the needed value to create the conditions to filter by
	 * the Expedientes entity
	 *
	 * @param expedientes
	 *            Expedientes Bean with the criteria values to filter by.
	 * @param startsWith
	 *            Boolean
	 * @return Map created with two keys key query stores the sql query syntax
	 *         key params stores the parameter values to be used in the
	 *         condition sentence.
	 */
	// CHECKSTYLE:OFF CyclomaticComplexity - Generación de código de UDA
	private Map<String, Object> getWhereLikeMap(Expedientes expedientes, Boolean startsWith) {

		StringBuilder where = new StringBuilder(ExpedientesDaoImpl.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		if (expedientes != null && expedientes.getIdExpediente() != null) {
			where.append(" AND t1.ID_EXPEDIENTE = ?");
			params.add(expedientes.getIdExpediente());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getIdProcedimiento() != null) {
			where.append(" AND t2.ID_PROCEDIMIENTO = ?");
			params.add(expedientes.getProcedimientos().getIdProcedimiento());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getDirecciones() != null
				&& expedientes.getProcedimientos().getDirecciones().getIdDireccion() != null) {
			where.append(" AND t2.ID_DIRECCION = ?");
			params.add(expedientes.getProcedimientos().getDirecciones().getIdDireccion());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getAreas() != null
				&& expedientes.getProcedimientos().getAreas().getIdArea() != null) {
			where.append(" AND t2.ID_AREA = ?");
			params.add(expedientes.getProcedimientos().getAreas().getIdArea());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getCodProcedureId() != null) {
			where.append(" AND UPPER(t2.COD_PROCEDURE_ID) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getProcedimientos().getCodProcedureId().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getProcedimientos().getCodProcedureId().toUpperCase() + "%");
			}
			where.append(" AND t2.COD_PROCEDURE_ID IS NOT NULL");
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getCamposTituloExpediente() != null) {
			where.append(" AND UPPER(t2.CAMPOS_TITULO_EXPEDIENTE) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getProcedimientos().getCamposTituloExpediente().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getProcedimientos().getCamposTituloExpediente().toUpperCase() + "%");
			}
			where.append(" AND t2.CAMPOS_TITULO_EXPEDIENTE IS NOT NULL");
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getFechaRegistro() != null) {
			where.append(" AND t2.FECHA_REGISTRO = ?");
			params.add(expedientes.getProcedimientos().getFechaRegistro());
		}
		if (expedientes != null && expedientes.getProcedimientos() != null
				&& expedientes.getProcedimientos().getUsuarioRegistro() != null) {
			where.append(" AND UPPER(t2.USUARIO_REGISTRO) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getProcedimientos().getUsuarioRegistro().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getProcedimientos().getUsuarioRegistro().toUpperCase() + "%");
			}
			where.append(" AND t2.USUARIO_REGISTRO IS NOT NULL");
		}
		if (expedientes != null && expedientes.getIdTerritorioHistorico() != null) {
			where.append(" AND t1.ID_TERRITORIO_HISTORICO = ?");
			params.add(expedientes.getIdTerritorioHistorico());
		}
		if (expedientes != null && expedientes.getFolderId() != null) {
			where.append(" AND UPPER(t1.FOLDER_ID) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getFolderId().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getFolderId().toUpperCase() + "%");
			}
			where.append(" AND t1.FOLDER_ID IS NOT NULL");
		}
		if (expedientes != null && expedientes.getFolderNumber() != null) {
			where.append(" AND UPPER(t1.FOLDER_NUMBER) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getFolderNumber().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getFolderNumber().toUpperCase() + "%");
			}
			where.append(" AND t1.FOLDER_NUMBER IS NOT NULL");
		}
		if (expedientes != null && expedientes.getTitulo() != null) {
			where.append(" AND UPPER(t1.TITULO) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getTitulo().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getTitulo().toUpperCase() + "%");
			}
			where.append(" AND t1.TITULO IS NOT NULL");
		}
		if (expedientes != null && expedientes.getFechaRegistro() != null) {
			where.append(" AND t1.FECHA_REGISTRO = ?");
			params.add(expedientes.getFechaRegistro());
		}
		if (expedientes != null && expedientes.getUsuarioRegistro() != null) {
			where.append(" AND UPPER(t1.USUARIO_REGISTRO) like ? ESCAPE  '\\'");
			if (startsWith) {
				params.add(expedientes.getUsuarioRegistro().toUpperCase() + "%");
			} else {
				params.add("%" + expedientes.getUsuarioRegistro().toUpperCase() + "%");
			}
			where.append(" AND t1.USUARIO_REGISTRO IS NOT NULL");
		}

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		return mapWhere;
	}

	private Map<String, ?> getWhereExpediente(Expedientes expedientes) {

		StringBuilder where = new StringBuilder(ExpedientesDaoImpl.STRING_BUILDER_INIT);
		List<Object> params = new ArrayList<Object>();

		Map<String, Object> mapWhere = new HashMap<String, Object>();
		where.append(" WHERE 1=1");
		mapWhere.put("query", where);
		mapWhere.put("params", params);

		// SEGURIDAD
		List<GrantedAuthority> list = new ArrayList<GrantedAuthority>(
				SecurityContextHolder.getContext().getAuthentication().getAuthorities());

		boolean esComercio = false;
		boolean esTurismo = false;

		for (int i = 0; i < list.size(); i++) {

			if ("ROLE_AB78-IN-0001".equals(list.get(i).getAuthority())) {
				esComercio = true;
				esTurismo = true;
			}
			if ("ROLE_AB78-IN-0002".equals(list.get(i).getAuthority())) {
				esTurismo = true;
			}
			if ("ROLE_AB78-IN-0003".equals(list.get(i).getAuthority())) {
				esTurismo = true;
			}
			if ("ROLE_AB78-IN-0004".equals(list.get(i).getAuthority())) {
				esComercio = true;
			}
			if ("ROLE_AB78-IN-0005".equals(list.get(i).getAuthority())) {
				esComercio = true;
			}

		}

		where.append(" AND (t6.ID_DIRECCION = ? OR t6.ID_DIRECCION = ?) ");

		if (esComercio && esTurismo) {
			params.add(Constantes.ID_TIPO_DIRECCION_ESTABLECIMIENTO_EXPEDIENTE);
			params.add(Constantes.COD_DIRECCION_TURISMO);
		} else if (esComercio) {
			params.add(Constantes.ID_TIPO_DIRECCION_ESTABLECIMIENTO_EXPEDIENTE);
			params.add(Constantes.ID_TIPO_DIRECCION_ESTABLECIMIENTO_EXPEDIENTE);
		} else {
			params.add(Constantes.COD_DIRECCION_TURISMO);
			params.add(Constantes.COD_DIRECCION_TURISMO);
		}

		TiposTercero tiposTercero = new TiposTercero();
		tiposTercero.setIdTipoTercero(com.ejie.ab78.comun.Constantes.SOLICITANTE);

		if (expedientes != null && expedientes.getProcedimiento() != null
				&& expedientes.getProcedimiento().getIdProcedimiento() != null) {
			where.append(" AND t6.ID_PROCEDIMIENTO = ?");
			params.add(expedientes.getProcedimiento().getIdProcedimiento());
		}

		if (expedientes != null && expedientes.getFolderNumber() != null) {
			where.append(" AND UPPER(t1.FOLDER_NUMBER) LIKE UPPER('%" + expedientes.getFolderNumber() + "%')");
		}

		if (expedientes != null && expedientes.getDatosTerceros() != null
				&& expedientes.getDatosTerceros().getDocumento() != null) {
			where.append(" AND UPPER(t8.DOCUMENTO) = UPPER(?)");
			params.add(expedientes.getDatosTerceros().getDocumento());
		}
		if (expedientes != null && expedientes.getDatosTerceros() != null
				&& expedientes.getDatosTerceros().getNombre() != null) {
			where.append(" AND UPPER(t8.NOMBRE) LIKE UPPER('%" + expedientes.getDatosTerceros().getNombre() + "%')");
		}
		if (expedientes != null && expedientes.getDatosTerceros() != null
				&& expedientes.getDatosTerceros().getApellido1() != null) {
			where.append(
					" AND UPPER(t8.APELLIDO1) LIKE UPPER('%" + expedientes.getDatosTerceros().getApellido1() + "%')");
		}
		if (expedientes != null && expedientes.getDatosTerceros() != null
				&& expedientes.getDatosTerceros().getApellido2() != null) {
			where.append(
					" AND UPPER(t8.APELLIDO2) LIKE UPPER('%" + expedientes.getDatosTerceros().getApellido2() + "%')");
		}
		if (tiposTercero != null && tiposTercero.getIdTipoTercero() != null) {
			where.append(" AND t2.ID_TIPO_TERCERO = ?");
			params.add(tiposTercero.getIdTipoTercero());
		}
		if (expedientes != null && expedientes.getDatosTerceros() != null
				&& expedientes.getDatosTerceros().getTipoParticipante() != null) {
			where.append(" AND t8.TIPO_PARTICIPANTE = ?");
			params.add(Constantes.PARTICIPANTE_TITULAR);
		}
		if (expedientes != null && expedientes.getTitulo() != null) {
			where.append(" AND UPPER(t1.TITULO) LIKE UPPER('%" + expedientes.getTitulo() + "%')");
		}
		return mapWhere;
	}

	// CHECKSTYLE:ON CyclomaticComplexity - Generación de código de UDA

	/**
	 * StringBuilder initilization value
	 */
	public static final int STRING_BUILDER_INIT = 4096;

	@Override
	public Date getMaxDateExpediente(Integer idExpediente, String tipoFecha, boolean isEtiqueta) {
		StringBuilder query = new StringBuilder(
				"SELECT MAX(RF.FECHA_EXPEDIENTE) fecha FROM AB78.EXPEDIENTES E, AB78.REL_EXP_FECH RF, AB78.FECHAS_EXPEDIENTE F ");
		query.append("WHERE E.ID_EXPEDIENTE=RF.ID_EXPEDIENTE AND RF.ID_FECHA_EXPEDIENTE = F.ID_FECHA_EXPEDIENTE");
		query.append(" AND E.ID_EXPEDIENTE = ? ");
		if (isEtiqueta) {
			query.append(" AND F.ETIQUETA_FECHA = ? ");
		} else {
			query.append(" AND F.PLATEA_TYPE = ? ");
		}
		query.append("GROUP BY F.PLATEA_TYPE");
		List<Date> dates = this.jdbcTemplate.query(query.toString(), this.rwMapDateExp, idExpediente, tipoFecha);
		if (dates.isEmpty()) {
			return null;
		}
		return dates.get(0);
	}

	@Override
	public Expedientes updateTerritorioHistorico(Expedientes expediente) {

		String query = "UPDATE AB78.EXPEDIENTES SET ID_TERRITORIO_HISTORICO=?, FECHA_ULTMODIFICACION=SYSDATE, USUARIO_ULTMODIFICACION=? WHERE ID_EXPEDIENTE=?";

		this.jdbcTemplate.update(query, expediente.getIdTerritorioHistorico(), UtilUsuario.getUserConnected(),
				expediente.getIdExpediente());

		return expediente;
	}

	@Override
	public boolean esExpedienteCerrado(Expedientes expediente) {
		String query = "SELECT COUNT(1) FROM REL_EXP_FECH t6142, FECHAS_EXPEDIENTE t6144 WHERE t6142.ID_FECHA_EXPEDIENTE = t6144.ID_FECHA_EXPEDIENTE AND t6144.ETIQUETA_FECHA = ? AND t6142.ID_EXPEDIENTE = ?";

		List<Object> params = new ArrayList<Object>();
		params.add(Constantes.ETIQUETA_FECHA_EXP_CIERRE);
		params.add(expediente.getIdExpediente());

		Long expedientesCount = this.jdbcTemplate.queryForObject(query, params.toArray(), Long.class);

		if (expedientesCount > 0) {
			return true;
		} else {
			return false;
		}
	}

}