package com.ejie.ab59.control.tablasMaestras;

import java.util.Calendar;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.SessionAttributes;

import com.ejie.ab59.clientesWS.ab61WebService.Usuario;
import com.ejie.ab59.comun.Constantes;
import com.ejie.ab59.comun.SeguridadAplicacion;
import com.ejie.ab59.model.procedimientos.Aplicaciones;
import com.ejie.ab59.model.procedimientos.Procedimientos;
import com.ejie.ab59.model.propuestasSancion.RelArtInfProSan;
import com.ejie.ab59.model.tablasMaestras.Infracciones;
import com.ejie.ab59.service.procedimientos.ProcedimientosService;
import com.ejie.ab59.service.tablasMaestras.InfraccionesService;
import com.ejie.x38.control.bind.annotation.RequestJsonBody;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;

/**
 * InfraccionesController  
 * 
 *  
 */
@SessionAttributes("usuario")
@Controller
@RequestMapping(value = "/infracciones")

public class InfraccionesController {

	private static final Logger logger = LoggerFactory.getLogger(InfraccionesController.class);

	@Autowired
	private InfraccionesService infraccionesService;

	@Autowired
	private ProcedimientosService procedimientosService;

	/*
	 * OPERACIONES CRUD (Create, Read, Update, Delete)
	 * 
	 */

	/**
	 * Devuelve una lista de beans correspondientes a los valores de filtrados
	 * indicados en el objeto pasado como parametro.
	 *
	 * @param filterInfracciones
	 *            Infracciones Objeto que contiene los parametros de filtrado
	 *            utilizados en la busqueda.
	 * @return List<Infracciones> Lista de objetos correspondientes a la
	 *         busqueda realizada.
	 */
	@RequestMapping(value = "/cargarCombo", method = RequestMethod.GET)
	public @ResponseBody List<Infracciones> cargarCombo(@ModelAttribute Infracciones filterInfracciones) {
		InfraccionesController.logger.info("[GET - find_ALL] : Obtener Infracciones por filtro");
		return this.infraccionesService.cargarCombo(filterInfracciones);
	}

	/**
	 * Metodo de presentacion del RUP_TABLE.
	 * 
	 * @param model
	 *            Model
	 * @return String
	 */
	@RequestMapping(value = "/maint", method = RequestMethod.GET)
	public String getFormEdit(Model model) {
		InfraccionesController.logger.info("[GET - View] : infracciones");
		String idDireccion = SeguridadAplicacion.precargarDireccion();
		Procedimientos procedimiento = null;
		String idProcedimiento = "";

		if (!"".contentEquals(idDireccion)) {
			procedimiento = this.procedimientosService.obtenerProcedimientos(Integer.parseInt(idDireccion));
		}

		if (procedimiento != null) {
			idProcedimiento = String.valueOf(procedimiento.getIdProcedimiento());
		}

		model.addAttribute("procedimientoPrecargado", idProcedimiento);
		return "infracciones";
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterInfracciones
	 *            Infracciones Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<Infracciones> Dto que contiene el resultado del
	 *         filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<Infracciones> filter(
			@RequestJsonBody(param = "filter") Infracciones filterInfracciones,
			@RequestJsonBody TableRequestDto tableRequestDto) {

		InfraccionesController.logger.info("[POST - filter] : Obtener Calificacioness");
		Procedimientos procedimientoAux = new Procedimientos();

		if (filterInfracciones.getProcedimiento() != null) {
			procedimientoAux = filterInfracciones.getProcedimiento();
		}

		Aplicaciones aplicacion = new Aplicaciones();
		aplicacion.setCodigoAplicacion(Constantes.CODIGO_APLICACION.toUpperCase());
		procedimientoAux.setAplicacion(aplicacion);

		filterInfracciones.setProcedimiento(procedimientoAux);
		filterInfracciones.setBaja(Constantes.ESTADO_NO_BAJA);

		return this.infraccionesService.filter(filterInfracciones, tableRequestDto);
	}

	/**
	 * Operacion CRUD Create. Creacion de un nuevo registro a partir del bean
	 * indicado.
	 *
	 * @param infracciones
	 *            Infracciones Bean que contiene la informacion con la que se va
	 *            a crear el nuevo registro.
	 * @return Infracciones Bean resultante del proceso de creacion.
	 */
	@RequestMapping(method = RequestMethod.POST)
	public @ResponseBody Infracciones add(@RequestBody Infracciones infracciones,
			@ModelAttribute(value = "usuario") Usuario usuario) {

		Calendar cal = Calendar.getInstance();
		infracciones.setFechaRegistro(cal.getTime());
		infracciones.setUsuarioRegistro(usuario.getUsuario());
		infracciones.setBaja(Constantes.ESTADO_NO_BAJA);

		Infracciones infraccionesAux = this.infraccionesService.add(infracciones);
		InfraccionesController.logger.info("[POST] : Infracciones insertado correctamente");
		return infraccionesAux;
	}

	/**
	 * Operacion CRUD Read. Devuelve el bean correspondiente al identificador
	 * indicado.
	 * 
	 * @param idInfraccion
	 *            Integer
	 * @return Infracciones Objeto correspondiente al identificador indicado.
	 */
	@RequestMapping(value = "/{idInfraccion}", method = RequestMethod.GET)
	public @ResponseBody Infracciones get(@PathVariable Integer idInfraccion) {
		Infracciones infracciones = new Infracciones();
		infracciones.setIdInfraccion(idInfraccion);
		infracciones = this.infraccionesService.find(infracciones);
		InfraccionesController.logger.info("[GET - findBy_PK] : Obtener Infracciones por PK");
		return infracciones;
	}

	/**
	 * Operacion CRUD Edit. Modificacion del bean indicado.
	 *
	 * @param infracciones
	 *            Infracciones Bean que contiene la informacion a modificar.
	 * @return Infracciones Bean resultante de la modificacion.
	 */
	@RequestMapping(method = RequestMethod.PUT)
	public @ResponseBody Infracciones edit(@RequestBody Infracciones infracciones,
			@ModelAttribute(value = "usuario") Usuario usuario) {

		Calendar cal = Calendar.getInstance();
		infracciones.setFechaUltmodificacion(cal.getTime());
		infracciones.setUsuarioUltmodificacion(usuario.getUsuario());

		Infracciones infraccionesAux = this.infraccionesService.update(infracciones);
		InfraccionesController.logger.info("[PUT] : Infracciones actualizado correctamente");
		return infraccionesAux;
	}

	/**
	 * Operacion CRUD Delete. Borrado del registro correspondiente al
	 * identificador especificado.
	 *
	 * @param idInfraccion
	 *            Integer Identificador del objeto que se desea eliminar.
	 * @return Infracciones Bean eliminado.
	 */
	@RequestMapping(value = "/{idInfraccion}", method = RequestMethod.DELETE)
	@ResponseStatus(value = HttpStatus.OK)
	public @ResponseBody Infracciones remove(@PathVariable Integer idInfraccion,
			@ModelAttribute(value = "usuario") Usuario usuario) {

		Infracciones infraccion = new Infracciones();

		Calendar cal = Calendar.getInstance();
		infraccion.setFechaUltmodificacion(cal.getTime());
		infraccion.setUsuarioUltmodificacion(usuario.getUsuario());

		infraccion.setIdInfraccion(idInfraccion);
		infraccion.setBaja(Constantes.ESTADO_BAJA);

		this.infraccionesService.remove(infraccion);
		InfraccionesController.logger.info("[DELETE] : Infracción borrada correctamente");
		return infraccion;
	}

	/**
	 * Operacion de filtrado del componente RUP_TABLE.
	 * 
	 * @param filterInfracciones
	 *            RelArtInfProSan Bean que contiene los parametros de filtrado a
	 *            emplear.
	 * @param tableRequestDto
	 *            Dto que contiene los parametros de configuracion propios del
	 *            RUP_TABLE a aplicar en el filtrado.
	 * @return TableResponseDto<RelArtInfProSan> Dto que contiene el resultado
	 *         del filtrado realizado por el componente RUP_TABLE.
	 */
	@RequestMapping(value = "/propuestaSancion/filter", method = RequestMethod.POST)
	public @ResponseBody TableResponseDto<RelArtInfProSan> filterPropuestaSancion(
			@RequestJsonBody(param = "filter") RelArtInfProSan filterInfracciones,
			@RequestJsonBody TableRequestDto tableRequestDto) {

		InfraccionesController.logger.info("[POST - filter] : Obtener Infracciones de la propuesta de sanción");

		return this.infraccionesService.filterInfraccionesByPropuestaSancion(filterInfracciones, tableRequestDto);
	}
}
