package com.ejie.ab59.service.sanciones;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.activation.MimetypesFileTypeMap;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import org.apache.poi.hssf.usermodel.HSSFCellStyle;
import org.apache.poi.hssf.usermodel.HSSFFont;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.Font;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.ui.Model;

import com.ejie.ab59.comun.Constantes;
import com.ejie.ab59.dao.sanciones.SancionesDao;
import com.ejie.ab59.model.expedientes.Expedientes;
import com.ejie.ab59.model.expedientes.RelExpTer;
import com.ejie.ab59.model.expedientes.SubtipoTercero;
import com.ejie.ab59.model.pagos.PagoSan;
import com.ejie.ab59.model.procedimientos.Direcciones;
import com.ejie.ab59.model.sanciones.RelArtInfSan;
import com.ejie.ab59.model.sanciones.Sanciones;
import com.ejie.ab59.service.expedientes.RelExpTerService;
import com.ejie.ab59.service.expedientes.SubtipoTerceroService;
import com.ejie.ab59.service.pagos.PagoSanService;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;

/**
 * SancionesServiceImpl  
 * 
 *  
 */

@Service(value = "sancionesService")
public class SancionesServiceImpl implements SancionesService {

	private static final Logger logger = LoggerFactory.getLogger(SancionesServiceImpl.class);

	@Autowired
	private SancionesDao sancionesDao;

	@Autowired
	private PagoSanService pagoSanService;

	@Resource
	private ReloadableResourceBundleMessageSource appMessageSource;

	@Autowired
	private RelExpTerService relExpTerService;

	@Autowired
	private RelArtInfSanService relArtInfSanService;
	
	
	@Transactional(rollbackFor = Throwable.class)
	public Sanciones add(Sanciones sanciones) {
		return this.sancionesDao.add(sanciones);
	}

	public Sanciones find(Sanciones sanciones) {
		return this.sancionesDao.find(sanciones);
	}

	@Transactional(rollbackFor = Throwable.class)
	public void remove(Sanciones sanciones) {
		this.sancionesDao.remove(sanciones);
	}

	public List<Sanciones> findAll(Sanciones sanciones, TableRequestDto tableRequestDto) {
		return this.sancionesDao.findAll(sanciones, tableRequestDto);
	}

	public TableResponseDto<Sanciones> filter(Sanciones filterSanciones, TableRequestDto tableRequestDto) {

		List<Sanciones> listaSanciones = this.sancionesDao.findAll(filterSanciones, tableRequestDto);
		Long recordNum = this.sancionesDao.findAllCount(filterSanciones != null ? filterSanciones : new Sanciones());

		return new TableResponseDto<Sanciones>(tableRequestDto, recordNum, listaSanciones);
	}

	@Override
	public void getExcel(List<Sanciones> sanciones, HttpServletResponse response) {

		Locale locale = LocaleContextHolder.getLocale();

		boolean castellano = false;
		boolean euskera = false;

		// Configuramos variables de idioma.
		if (locale.toString().equals(Constantes.CASTELLANO)) {
			castellano = true;
		} else if (locale.toString().equals(Constantes.EUSKERA)) {
			euskera = true;
		}

		try {

			// Nombre del archivo
			String fileName = appMessageSource.getMessage("excel.sanciones.sheetName", null, locale) + ".xls";

			// Creamos la hoja
			HSSFWorkbook workbook = new HSSFWorkbook();
			HSSFSheet sheet = workbook
					.createSheet(appMessageSource.getMessage("excel.sanciones.sheetName", null, locale));

			// Definimos los estilos y fuentes de la hoja
			HSSFFont font = workbook.createFont();
			font.setFontName(HSSFFont.FONT_ARIAL);
			font.setFontHeightInPoints((short) 12);
			font.setBoldweight(Font.BOLDWEIGHT_BOLD);

			HSSFCellStyle headerStyle = workbook.createCellStyle();
			headerStyle.setBorderBottom(CellStyle.BORDER_MEDIUM);
			headerStyle.setFont(font);
			headerStyle.setAlignment(CellStyle.ALIGN_CENTER);

			HSSFCellStyle disabledStyle = workbook.createCellStyle();
			disabledStyle.setFillPattern(CellStyle.THIN_FORWARD_DIAG);

			// Creamos la fila cabecera
			HSSFRow rowHead = sheet.createRow(0);

			rowHead.createCell(0)
					.setCellValue(appMessageSource.getMessage("excel.sanciones.numExpediente", null, locale));
			rowHead.createCell(1)
					.setCellValue(appMessageSource.getMessage("excel.sanciones.tituloExpediente", null, locale));
			rowHead.createCell(2)
					.setCellValue(appMessageSource.getMessage("excel.sanciones.importeSancion", null, locale));
			rowHead.createCell(3)
					.setCellValue(appMessageSource.getMessage("excel.sanciones.fechaSancionFirme", null, locale));

			rowHead.getCell(0).setCellStyle(headerStyle);
			rowHead.getCell(1).setCellStyle(headerStyle);
			rowHead.getCell(2).setCellStyle(headerStyle);
			rowHead.getCell(3).setCellStyle(headerStyle);

			// Creamos el resto de filas con sus celdas
			int rowCount = 1;
			for (Sanciones sancion : sanciones) {

				HSSFRow sancionRow = sheet.createRow(rowCount++);

				sancionRow.createCell(0).setCellValue(sancion.getExpediente().getFolderNumber());
				sancionRow.createCell(1).setCellValue(sancion.getExpediente().getTitulo());
				String importeSancion = null;
				if (sancion.getImporteSancion() != null) {
					importeSancion = sancion.getImporteSancion().toString();
				}
				sancionRow.createCell(2).setCellValue(importeSancion);

				if (sancion.getFechaSancionFirme() != null) {

					SimpleDateFormat sdf = new SimpleDateFormat();

					if (castellano) {
						sdf = new SimpleDateFormat(Constantes.FORMATO_FECHA_ES);
					} else if (euskera) {
						sdf = new SimpleDateFormat(Constantes.FORMATO_FECHA_EU);
					}

					String fechaComoCadena = sdf.format(sancion.getFechaSancionFirme());
					sancionRow.createCell(3).setCellValue(fechaComoCadena);
				}
			}

			// Anchura de las columnas
			sheet.autoSizeColumn(0);
			sheet.autoSizeColumn(1);
			sheet.autoSizeColumn(2);
			sheet.autoSizeColumn(3);

			FileOutputStream fileOut = new FileOutputStream(fileName);
			workbook.write(fileOut);
			fileOut.close();

			SancionesServiceImpl.logger.info("[GET - exportarExcel] : Archivo generado");

			// Creamos el archivo a descargar
			File fileToDownload = new File(fileName);
			InputStream in = new FileInputStream(fileToDownload);

			// Recogemos el MIME type del fichero
			String mimeType = new MimetypesFileTypeMap().getContentType(fileName);

			if (mimeType == null) {
				// Si no se encuentra lo seteamos manualmente
				mimeType = "application/octet-stream";
			}
			SancionesServiceImpl.logger.info("[GET - exportarExcel] : MIME type: " + mimeType);

			// Seteamos el contentType de la response
			response.setContentType(mimeType);
			response.setContentLength((int) fileToDownload.length());

			// Descargamos el archivo
			response.setHeader("Content-Disposition", "attachment; filename=\"" + fileToDownload.getName() + "\"");

			// Obtenemos el outputStream de la response
			OutputStream outStream = response.getOutputStream();

			byte[] buffer = new byte[4096];
			int bytesRead = -1;

			while ((bytesRead = in.read(buffer)) != -1) {
				outStream.write(buffer, 0, bytesRead);
			}

			// Cerramos el InputStream y el OutputStream
			in.close();
			outStream.close();

			SancionesServiceImpl.logger.info("[GET - exportarExcel] : Archivo descargado con éxito");

		} catch (Exception e) {
			SancionesServiceImpl.logger.error(e.getMessage());
		}
	}

	@Transactional(rollbackFor = Throwable.class)
	public Sanciones findDatosSancion(Sanciones sanciones) {
		return this.sancionesDao.findDatosSancion(sanciones);
	}

	@Transactional(rollbackFor = Throwable.class)
	public Sanciones updateDatosSancion(Sanciones sanciones) {
		return this.sancionesDao.updateDatosSancion(sanciones);
	}

	public Boolean tieneDireccion(Sanciones sanciones) {
		return this.sancionesDao.tieneDireccion(sanciones);
	}

	@Transactional(readOnly = true)
	public Sanciones obtenerCabeceraSancion(Integer idSancion) {
		return this.sancionesDao.obtenerCabeceraSancion(idSancion);
	}

	@Transactional(readOnly = true)
	public Sanciones findByFolderId(Sanciones sanciones) {
		return this.sancionesDao.findByFolderId(sanciones);
	}

	@Override
	public Sanciones obtenerCabeceraSancionByExpediente(Expedientes expediente) {
		return this.sancionesDao.obtenerCabeceraSancionByExpediente(expediente);
	}

	@Override
	public List<Sanciones> findExpedientesPartesInculpadas(List<String> documentos, Integer idExpediente,
			Direcciones direccion) {
		return this.sancionesDao.findExpedientesPartesInculpadas(documentos, idExpediente, direccion);
	}

	@Override
	public Sanciones updateImportes(Sanciones sancion) {
		return this.sancionesDao.updateImportes(sancion);
	}

	/**
	 * Calcularemos el importe de la resolucion
	 */
	@Override
	public Sanciones calcularImporteResolucion(Sanciones sanciones) {

		logger.info("[calcularImporteResolucion] : Se procede a calcular el importe de resolucion");

		try {

			/**
			 * Obtenemos el pago de la sancion
			 */
			PagoSan pagoSancion = new PagoSan();
			pagoSancion.setSancion(new Sanciones(sanciones.getIdSancion()));
			pagoSancion = this.pagoSanService.find(pagoSancion);

			/**
			 * Calculamos el porcentaje segun los check seleccionados si los
			 * tuviera
			 */
			BigDecimal porcentajeDescuentoTotal = new BigDecimal(0);
			if (pagoSancion != null && pagoSancion.getReconocimientoVoluntario() != null
					&& pagoSancion.getReconocimientoVoluntario().equals(Constantes.CHECKED)) {
				porcentajeDescuentoTotal = porcentajeDescuentoTotal.add(Constantes.DESCUENTO_RECONOCIMIENTO_VOLUNTARIO);
			}

			if (pagoSancion != null && pagoSancion.getPagoVoluntario() != null
					&& pagoSancion.getPagoVoluntario().equals(Constantes.CHECKED)) {
				porcentajeDescuentoTotal = porcentajeDescuentoTotal.add(Constantes.DESCUENTO_PAGO_VOLUNTARIO);
			}

			/**
			 * Obtenemos la sancion
			 */
			sanciones = sancionesDao.find(sanciones);

			/**
			 * Calculamos el importe de resolucion de la infraccion
			 */
			BigDecimal importeResolucionCalculado = (sanciones.getImporteResolucion() != null
					? sanciones.getImporteResolucion() : new BigDecimal(0));
			importeResolucionCalculado = importeResolucionCalculado.subtract(
					importeResolucionCalculado.multiply(porcentajeDescuentoTotal).divide(new BigDecimal(100)));

			/**
			 * Asignamos el nuevo importe de resolucion calculado
			 */
			sanciones.setImporteResolucion(importeResolucionCalculado);
		} catch (Exception e) {
			logger.error("[calcularImporteResolucion] : EXCEPCION calculando el importe de la resolucion", e);
		}

		logger.info("[calcularImporteResolucion] : Importe resolucion calculado");
		return sanciones;
	}
	
	/**
	 * Comprobaremos si el expediente sancionador esta listo para el pago
	 * @param expediente
	 * @param model
	 * @return
	 */
	@Override
	public boolean validarExpedienteSancionadorListoParaElPago(Sanciones sancion, Model model){
		
		logger.info("Inicio de la validacion del expediente para comprobar si esta listo para el pago");
		boolean retorno = false;
		Integer contadorValidacionesCompletadas = 0;
		Integer cuantasValidacionesCompletar = 6;
		try{
		
			/**
			 * Los siguientes pasos para dar un expediente como listo para el pago 
			 * - Tiene al menos una parte inculpada
			 * - Tiene al menos un tipo de infraccin con importe --> miraremos importe resolucion de la tabla de sancion

			 * - Tipo de pago
			 * - Reconocimiento voluntario 
			 * - Pago voluntario
			 * - Envio a carta de pago checkeado
			 */
			List<RelExpTer> listaParticipantesParticipacion = relExpTerService.findParticipaciones(sancion.getExpediente().getIdExpediente(), model);
			if(listaParticipantesParticipacion != null && !listaParticipantesParticipacion.isEmpty()){
				for(int i = 0; i < listaParticipantesParticipacion.size(); i++){	
					if(listaParticipantesParticipacion.get(i) != null && listaParticipantesParticipacion.get(i).getIdSubtipoParticipacion() != null 
							&& listaParticipantesParticipacion.get(i).getIdSubtipoParticipacion().equals(Constantes.PARTE_INCULPADA)){
						
						contadorValidacionesCompletadas++;
						break;
					}
				}
			}
			
			if(sancion.getImporteResolucion() != null && sancion.getImporteResolucion().compareTo(BigDecimal.ZERO) == 1){
				contadorValidacionesCompletadas++;
			}
			
			PagoSan pago = new PagoSan();
			pago.setSancion(sancion);
			pago = this.pagoSanService.find(pago);
			if(pago != null){
				
				if(pago.getTipoPago() != null){
					contadorValidacionesCompletadas++;
				}
				if(pago.getReconocimientoVoluntario() != null){
					contadorValidacionesCompletadas++;
				}
				if(pago.getPagoVoluntario() != null){
					contadorValidacionesCompletadas++;
				}
				if(pago.getEnvioCartaPago() != null && pago.getEnvioCartaPago().equals(Constantes.COMBO_SI)){
					contadorValidacionesCompletadas++;
				}
			}
		}catch (Exception e) {
			logger.error("Excepcion obtenida en la validacion ", e);
			contadorValidacionesCompletadas = 0;
		}
		
		if(contadorValidacionesCompletadas.equals(cuantasValidacionesCompletar)){
			retorno = true;
		}
		
		logger.info("Fin validacion");
		return retorno;
	}
}
