package com.ejie.ab59.service.sanciones;

import java.math.BigDecimal;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.ejie.ab59.comun.Constantes;
import com.ejie.ab59.dao.sanciones.RelArtInfSanDao;
import com.ejie.ab59.dao.sanciones.RelArtInfSanDerDao;
import com.ejie.ab59.dao.sanciones.SancionesDao;
import com.ejie.ab59.dao.tablasMaestras.DerechosDao;
import com.ejie.ab59.model.pagos.PagoSan;
import com.ejie.ab59.model.sanciones.RelArtInfSan;
import com.ejie.ab59.model.sanciones.RelArtInfSanDer;
import com.ejie.ab59.model.sanciones.Sanciones;
import com.ejie.ab59.model.tablasMaestras.Derechos;
import com.ejie.ab59.service.pagos.PagoSanService;
import com.ejie.x38.dto.TableRequestDto;
import com.ejie.x38.dto.TableResponseDto;

/**
 * 
 *  
 *
 */

@Service(value = "relArtInfSanService")
public class RelArtInfSanServiceImpl implements RelArtInfSanService {

	private static final Logger logger = LoggerFactory.getLogger(RelArtInfSanServiceImpl.class);

	@Autowired
	private RelArtInfSanDao relArtInfSanDao;

	@Autowired
	private RelArtInfSanDerDao relArtInfSanDerDao;

	@Autowired
	private DerechosDao derechosDao;

	@Autowired
	private SancionesDao sancionesDao;

	@Autowired
	private PagoSanService pagoSanService;

	@Override
	@Transactional(rollbackFor = Throwable.class)
	public RelArtInfSan add(RelArtInfSan relArtInfSan) {

		/* Agregar registro en REL_ART_INF_SAN */
		relArtInfSan = this.calcularImporteResolucion(relArtInfSan);
		RelArtInfSan relArtInfSanAux = this.relArtInfSanDao.add(relArtInfSan);

		/* Agregar registro en REL_ART_INF_SAN_DER */
		RelArtInfSanDer relArtInfSanDer = new RelArtInfSanDer();
		relArtInfSanDer.setRelArtInfSan(relArtInfSanAux);
		relArtInfSanDer.setUsuarioRegistro(relArtInfSanAux.getUsuarioRegistro());
		String[] listaDerechosInf = relArtInfSanAux.getIdDerechosInf().split("\\#");
		for (int i = 0; i < listaDerechosInf.length; i++) {
			Derechos derechoAux = new Derechos();
			derechoAux.setIdDerecho(Integer.parseInt(listaDerechosInf[i]));
			relArtInfSanDer.setDerecho(derechoAux);

			this.relArtInfSanDerDao.add(relArtInfSanDer);
		}

		/* Actualizar los importes en SANCIONES */
		Sanciones sancion = new Sanciones();
		sancion.setIdSancion(relArtInfSan.getSancion().getIdSancion());
		this.sancionesDao.updateImportes(sancion);

		return relArtInfSanAux;
	}

	@Override
	@Transactional(rollbackFor = Throwable.class)
	public RelArtInfSan update(RelArtInfSan relArtInfSan) {

		/* Actualizar registro en REL_ART_INF_SAN */
		relArtInfSan = this.calcularImporteResolucion(relArtInfSan);
		RelArtInfSan relArtInfSanAux = this.relArtInfSanDao.update(relArtInfSan);

		/* Actualizar registro en REL_ART_INF_SAN_DER */
		// Primero borramos los derechos de la infracción
		RelArtInfSanDer relArtInfSanDer = new RelArtInfSanDer();
		relArtInfSanDer.setRelArtInfSan(relArtInfSanAux);
		this.relArtInfSanDerDao.remove(relArtInfSanDer);

		relArtInfSanDer.setUsuarioRegistro(relArtInfSan.getUsuarioUltModificacion());
		// Luego insertamos los derechos seleccionados
		String[] listaDerechosInf = relArtInfSan.getIdDerechosInf().split("\\#");
		for (int i = 0; i < listaDerechosInf.length; i++) {
			Derechos derechoAux = new Derechos();
			derechoAux.setIdDerecho(Integer.parseInt(listaDerechosInf[i]));
			relArtInfSanDer.setDerecho(derechoAux);

			this.relArtInfSanDerDao.add(relArtInfSanDer);
		}

		/* Actualizar los importes en SANCIONES */
		Sanciones sancion = new Sanciones();
		sancion.setIdSancion(relArtInfSan.getSancion().getIdSancion());
		this.sancionesDao.updateImportes(sancion);

		return relArtInfSanAux;
	}

	@Override
	@Transactional(rollbackFor = Throwable.class)
	public void remove(RelArtInfSan relArtInfSan) {

		/* Eliminar registro en REL_ART_INF_SAN_DER */
		RelArtInfSanDer relArtInfSanDer = new RelArtInfSanDer();
		relArtInfSanDer.setRelArtInfSan(relArtInfSan);
		this.relArtInfSanDerDao.remove(relArtInfSanDer);

		/* Eliminar registro en REL_ART_INF_SAN */
		this.relArtInfSanDao.remove(relArtInfSan);

		// Actualizar los importes en SANCIONES
		Sanciones sancion = new Sanciones();
		sancion.setIdSancion(relArtInfSan.getSancion().getIdSancion());
		this.sancionesDao.updateImportes(sancion);

	}

	@Override
	public TableResponseDto<RelArtInfSan> filter(RelArtInfSan filterRelArtInfSan, TableRequestDto tableRequestDto) {

		List<RelArtInfSan> listaRelArtInfSan = this.relArtInfSanDao.findAll(filterRelArtInfSan, tableRequestDto);

		for (RelArtInfSan relArtInfSan : listaRelArtInfSan) {

			String[] idsDerechos = relArtInfSan.getIdDerechosInf().split("#");
			String descDerechosCas = "";
			String descDerechosEus = "";

			for (String idDerecho : idsDerechos) {
				Derechos derechoAux = this.derechosDao.find(new Derechos(Integer.parseInt(idDerecho)));
				descDerechosCas += derechoAux.getDescripcionCastellano() + "#";
				descDerechosEus += derechoAux.getDescripcionEuskera() + "#";
			}

			if (descDerechosCas != "") {
				descDerechosCas = descDerechosCas.substring(0, descDerechosCas.length() - 1);
			}

			if (descDerechosEus != "") {
				descDerechosEus = descDerechosEus.substring(0, descDerechosEus.length() - 1);
			}

			relArtInfSan.setDescripcionDerechosInfCas(descDerechosCas);
			relArtInfSan.setDescripcionDerechosInfEus(descDerechosEus);

		}

		Long recordNum = this.relArtInfSanDao
				.findAllCount(filterRelArtInfSan != null ? filterRelArtInfSan : new RelArtInfSan());

		return new TableResponseDto<RelArtInfSan>(tableRequestDto, recordNum, listaRelArtInfSan);
	}

	@Override
	public RelArtInfSan find(RelArtInfSan relArtInfSan) {
		return this.relArtInfSanDao.find(relArtInfSan);
	}

	@Override
	public RelArtInfSan totales(RelArtInfSan relArtInfSan) {
		return this.relArtInfSanDao.totales(relArtInfSan);
	}

	@Override
	public RelArtInfSan updatePagos(RelArtInfSan relArtInfSan) {

		relArtInfSan = this.calcularImporteResolucion(relArtInfSan);
		RelArtInfSan relArtInfSanAux = this.relArtInfSanDao.updatePagos(relArtInfSan);

		// Actualizamos los importes en SAANCIONES
		Sanciones sancion = new Sanciones();
		sancion.setIdSancion(relArtInfSan.getSancion().getIdSancion());
		this.sancionesDao.updateImportes(sancion);

		return relArtInfSanAux;
	}

	@Override
	public List<RelArtInfSan> findAll(RelArtInfSan relArtInfSan) {

		return this.relArtInfSanDao.findAll(relArtInfSan, null);
	}

	/**
	 * Calcularemos el importe de la resolucion de una infraccion
	 */
	@Override
	public RelArtInfSan calcularImporteResolucion(RelArtInfSan relArtInfSan) {

		logger.info("[calcularImporteResolucion] : Se procede a calcular el importe de resolucion");

		try {

			/**
			 * Obtenemos el pago de la sancion
			 */
			PagoSan pagoSancion = new PagoSan();
			pagoSancion.setSancion(new Sanciones(relArtInfSan.getSancion().getIdSancion()));
			pagoSancion = this.pagoSanService.find(pagoSancion);

			/**
			 * Calculamos el porcentaje segun los check seleccionados si los
			 * tuviera
			 */
			BigDecimal porcentajeDescuentoTotal = new BigDecimal(0);
			if (pagoSancion != null && pagoSancion.getReconocimientoVoluntario() != null
					&& pagoSancion.getReconocimientoVoluntario().equals(Constantes.CHECKED)) {
				porcentajeDescuentoTotal = porcentajeDescuentoTotal.add(Constantes.DESCUENTO_RECONOCIMIENTO_VOLUNTARIO);
			}

			if (pagoSancion != null && pagoSancion.getPagoVoluntario() != null
					&& pagoSancion.getPagoVoluntario().equals(Constantes.CHECKED)) {
				porcentajeDescuentoTotal = porcentajeDescuentoTotal.add(Constantes.DESCUENTO_PAGO_VOLUNTARIO);
			}

			/**
			 * Caluclamos el importe de resolucion de la infraccion
			 */
			BigDecimal importeResolucionCalculado = (relArtInfSan.getImporteSancionArt() != null
					? relArtInfSan.getImporteSancionArt() : new BigDecimal(0));

			importeResolucionCalculado = importeResolucionCalculado.subtract(
					importeResolucionCalculado.multiply(porcentajeDescuentoTotal).divide(new BigDecimal(100)));

			/**
			 * Asignamos el nuevo importe de resolucion calculado
			 */
			relArtInfSan.setImporteResolucionArt(importeResolucionCalculado);
		} catch (Exception e) {
			logger.error("[calcularImporteResolucion] : EXCEPCION calculando el importe de la resolucion", e);
		}

		logger.info("[calcularImporteResolucion] : Importe resolucion calculado");
		return relArtInfSan;
	}

	/**
	 * Calcularemos el importe de la resolucion de todas las infracciones de una
	 * sancion
	 */
	@Override
	public void actualizarImporteResolucionInfraccionesSancion(Integer idSancion) {

		logger.info(
				"[actualizarImporteResolucionInfraccionesSancion] : Se procede a calcular el importe de resolucion de todas las infracciones de una sancion");

		try {

			if (idSancion != null && !idSancion.equals(0)) {

				/**
				 * Obtenemos todas las infracciones de la sancion
				 */
				RelArtInfSan busquedaInfracciones = new RelArtInfSan();
				busquedaInfracciones.setSancion(new Sanciones(idSancion));
				List<RelArtInfSan> listadoInfracciones = this.relArtInfSanDao.findAll(busquedaInfracciones, null);

				if (listadoInfracciones != null && !listadoInfracciones.isEmpty()) {
					for (int i = 0; i < listadoInfracciones.size(); i++) {
						this.relArtInfSanDao.updatePagos(this.calcularImporteResolucion(listadoInfracciones.get(i)));
					}
				}
			}
		} catch (Exception e) {
			logger.error(
					"[actualizarImporteResolucionInfraccionesSancion] : EXCEPCION calculando el importe de la resolucion",
					e);
		}

		logger.info("[actualizarImporteResolucionInfraccionesSancion] : Importes de resolucion actualizados");

	}

	@Override
	public RelArtInfSan findInfraccionInfringida(Integer idSancion, Integer idInfraccion) {
		return this.relArtInfSanDao.findInfraccionInfringida(idSancion, idInfraccion);
	}
}
