package s12.visorGMaps.actions;

import java.io.IOException;
import java.io.PrintWriter;
import java.net.URLDecoder;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.json.JSONArray;
import org.json.JSONException;

import s12.mapas.dao.S12MapasDAO;
import s12.mapas.dao.S12MapasDaoException;
import s12.mapas.valueObjects.S12Municipio;
import s12.mapas.valueObjects.S12Recurso;
import s12.mapas.valueObjects.S12RecursoGastronomika;
import s12.util.S12Utils;

import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.internacionalizacion.Q70GestorMensajesException;
import es.ejie.frmk.infraestructura.internacionalizacion.Q70GestorMensajesi18n;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;

/**
 *  tica
 *
 * The Class Conexion.
 */
public class S12Conexion extends HttpServlet {

	/** The Constant serialVersionUID. */
	private static final long serialVersionUID = -8074537769359345307L;

	/**
	 * Constante que indica el componente al que pertenece dicha clase. Es
	 * necesario para el uso del log de trazas.
	 */
	private static final String COMPONENTE = "MAPAS";

	private static final int SIZE_STRINGBUFFER = 280;

	/** Constantes para la generacin del POI en formato OV2 */
	private static final int CONVERTIR_LATITUD_LONGITUD = 100000;
	private static final int SIZE_LITTLE_ENDIAN = 13;
	private static final int MARGEN_LONGITUD = 14;
	
	/**
	 * Constructor.
	 */
	public S12Conexion() {
		super();
	}

	/**
	 * The doGet method of the servlet. <br>
	 *
	 * This method is called when a form has its tag value method equals to get.
	 *
	 * @param request the request send by the client to the server
	 * @param response the response send by the server to the client
	 *
	 * @throws ServletException if an error occurred
	 * @throws IOException if an error occurred
	 */
	public void doGet(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {
		this.doPost(request,response);
	}

	/**
	 * The doPost method of the servlet. <br>
	 *
	 * This method is called when a form has its tag value method equals to post.
	 *
	 * @param request the request send by the client to the server
	 * @param response the response send by the server to the client
	 *
	 * @throws ServletException if an error occurred
	 * @throws IOException if an error occurred
	 */
	public void doPost(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException {

		try {
			
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.DEBUG,"S12Conexion::doPost ... INICIO ");
			}
			
			String peticion = request.getParameter("peticion");

			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.DEBUG,"S12Conexion::doPost: peticion: #" + peticion + "#");
			}
			
			// Coordenadas que delimitan la vista del mapa o coordenadas de un recurso
				// Esquina inferior izquierda o coordenadas de un recurso
			String lat1 = request.getParameter("lat1");
			String lng1 = request.getParameter("lng1");
				// Esquina superior derecha
			String lat2 = request.getParameter("lat2");
			String lng2 = request.getParameter("lng2");

			String tipos_recursos = request.getParameter("tipos_recursos");

			String Zoom = request.getParameter("Zoom");

			// Formato de la respuesta [XML | JSON | OV2 | CSV | GeoJSON]
			String tipoDatosRespuesta = request.getParameter("mod");

			String nombreArchivo = request.getParameter("nombreArchivo");
			String formatoarchivo = request.getParameter("formatoarchivo");

			String puntosRuta = request.getParameter("puntosRuta");

			String marcador = request.getParameter("marcador");
			String nombremarcador = request.getParameter("nombremarcador");

			String idioma = request.getParameter("idioma");

			String importancia = request.getParameter("importancia");
			
			String query = request.getParameter("query");


			if (formatoarchivo == null) {
				formatoarchivo = "";
				response.setContentType("text/plain");
			} else {
				StringBuffer nameFichero = new StringBuffer("");

				if (nombreArchivo != null) {
					nameFichero = new StringBuffer(nombreArchivo);
				}
				if (nameFichero.length() == 0) {
					nameFichero = new StringBuffer("Euskadi");
				}
				nameFichero.append(".");
				nameFichero.append(formatoarchivo.toLowerCase());

				if ("KML".equals(formatoarchivo)) {
					response.setContentType("application/vnd.google-earth.kml+xml");
				}
				if ("CSV".equals(formatoarchivo)) {
					response.setContentType("application/csv");
				}
				if ("OV2".equals(formatoarchivo)) {
					response.setContentType("application/octet-stream");
				}

				response.setHeader("Content-Disposition","attachment; filename=" + nameFichero.toString().replaceAll(" ", "_"));
			}


			if ("OV2".equals(formatoarchivo)) {
				ServletOutputStream binaryOutput = response.getOutputStream();

				if ("marcadorPOI".equals(peticion)) {
					byte buffer[] = this.getPOIOV2(nombremarcador, lat1, lng1).array();

					binaryOutput.write(buffer, 0, buffer.length);
				}
			}
			else {
				PrintWriter out = response.getWriter();

				// para cargar los recursos importantes
				if ("importantes".equals(peticion)) {
					out.println(this.getRecursosImportantes(idioma, importancia));
				}

				// para la peticin de archivo con ruta
				if ("ruta".equals(peticion)) {
					out.println(this.getRuta(nombreArchivo, puntosRuta));
				}

				// para cargar los datos de un marcador
				if ("marcador".equals(peticion)) {
					if ("GeoJSON".equals(tipoDatosRespuesta)) {
						out.println(this.getMarcadorGeoJSON(marcador, idioma));
					} else {
						out.println(this.getMarcador(marcador, idioma));
					}
				}

				// para cargar los recursos
				if ("recursos".equals(peticion)) {
					if ("MUNICIPIOS".equalsIgnoreCase(Zoom)) {
						out.println(this.getMunicipios(lat1, lng1, lat2, lng2, tipos_recursos, idioma, tipoDatosRespuesta, nombreArchivo));
					} else {
						out.println(this.getRecursos(lat1, lng1, lat2, lng2, tipos_recursos, idioma, tipoDatosRespuesta, nombreArchivo));
					}
				}
				
				// para cargar los recursos de Euskadi Gastronomika
				if ("recursosGastronomika".equals(peticion)) {
					//TODO: incorporar parametro tipoDatosRespuesta para incluir GeoJSON
					
					if ("MUNICIPIOS".equalsIgnoreCase(Zoom)) {
						out.println(this.getMunicipiosGastronomika(lat1, lng1, lat2, lng2, tipos_recursos, idioma));
					} else {
						out.println(this.getRecursosGastronomika(lat1, lng1, lat2, lng2, tipos_recursos, idioma));
					}
				}

				if ("marcadorPOI".equals(peticion)) {
					out.println(this.getPOICSV(nombremarcador, lat1, lng1));
				}
				
				if ("query".equals(peticion)) {
					S12Utils.trazaLarga(S12Conexion.COMPONENTE, Q70TraceLevel.DEBUG,"S12Conexion::doPost: query: #" + URLDecoder.decode(query, "ISO-8859-1") + "#");
					
					//-CAMBIO- el formato de la query recibida, llegaba como un R01MQueryObject y ahora pasa a ser un JSON
					//out.println(this.getCodigosRecurso(idioma, R01MSearchEngineUtils.decodeURLInQueryObject(URLDecoder.decode(query, "ISO-8859-1"))));
					out.println(this.getCodigosRecurso(idioma, query));
				}

				out.flush();
				out.close();
			}

			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.DEBUG,"S12Conexion::doPost ... FIN ");
			}


		} catch(Exception e){
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::doPost: Exception: " + e.getMessage());
			}
			e.printStackTrace();
			
			for (int i = 0; i < e.getStackTrace().length; i++) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::doPost: StackTrace: " + e.getStackTrace()[i]);
				}
			}
		}
	}

	/**
	 * Se encarga de gestionar una consulta de recursos a travs de metadatos recibidos en un JSON.
	 * 
	 * Devuelve todos los cdigos de recurso que cumplen con las condiciones de la
	 * consulta pasada por parmetro.
	 * 
	 * @param language Idioma
	 * @param queryJSONString String JSON con la informacin de la consulta
	 * 
	 * @return Devuelve una cadena JSON con un array de cdigos de recurso
	 */
	private String getCodigosRecurso(String language, String queryJSONString){
		StringBuffer sRespuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		S12MapasDAO mapasDAO = new S12MapasDAO();
		
		try {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.DEBUG,"S12Conexion::getCodigosRecurso(idioma,json): antes filtrar");
			}
			
			String[] codigosRecursoArray = mapasDAO.filtrarQueryRecursos(language, queryJSONString);
			
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.DEBUG,"S12Conexion::getCodigosRecurso(idioma,json): despues filtrar");
			}

			try {
				sRespuesta.append(new JSONArray(codigosRecursoArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getCodigosRecurso(idioma,json): Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}

		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getCodigosRecurso(idioma,json): Exception: " + e.getMessage());
			}
		}
		
		return sRespuesta.toString();
	}

	/**
	 * Se encarga de gestionar la peticin de informacin sobre recursos.
	 *
	 * Devuelve todos los recursos de los tipos pasados como parmetro
	 * que se encuentren en la zona visible en el mapa delimitada
	 * por las coordenados de lat/long pasadas como parmetros.
	 *
	 * @param lat1 String Latitud de la esquina inferior izquierda de la vista
	 * @param lng1 String Longitud de la esquina inferior izquierda de la vista
	 * @param lat2 String Latitud de la esquina superior derecha de la vista
	 * @param lng2 String Longitud de la esquina superior derecha de la vista
	 * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
	 * 	Formato: tipoRecurso_subTipoRecurso_Categoria. Ejemplo: A1_H_1,B1_1,C2_
	 * @param idioma String Idioma de la vista
	 * @param tipoDatosRespuesta String Formato de la respuesta
	 * @param nombreArchivo String Nombre del archivo de respuesta
	 *
	 * @return StringBuffer
	 */
	private String getRecursos(String lat1, String lng1, String lat2,String lng2,
			String tiposRecursos, String idioma, String tipoDatosRespuesta, String nombreArchivo) {
		StringBuffer sRespuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		S12MapasDAO mapasDAO = new S12MapasDAO();

		if ("GeoJSON".equals(tipoDatosRespuesta)) {
			
		
		} else if ("JSON".equals(tipoDatosRespuesta)) {
			sRespuesta.append("{\"p\"");
			sRespuesta.append(":");
			sRespuesta.append("[");
			sRespuesta.append("{\"n\":");
		} else {
			sRespuesta.append("<?xml version='1.0' encoding='iso-8859-1'?><kml xmlns='http://earth.google.com/kml/2.0'><Document>");
			sRespuesta.append("<name>" + nombreArchivo + "</name>");
		}

		try {
			S12Recurso[] recursosArray = mapasDAO.filtrarRecursos(lat1, lng1, lat2, lng2, tiposRecursos, idioma);

			if ("JSON".equals(tipoDatosRespuesta)) {
				try {
					sRespuesta.append(new JSONArray(recursosArray).toString());
				} catch (JSONException e) {
					if (Q70Factoria.getTraza().isTraceActive()) {
						Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getRecursos: Exception: " + e.getMessage());
					}
					sRespuesta.append("[]");
				}
			} else if ("GeoJSON".equals(tipoDatosRespuesta)) {
				//TODO: rellenar JSON a partir de un array de recursos (no podemos utilizar JSONArray.toString)
				
				
			} else {
				String tipoRecursoActual = "";
				String tipoRecursoUltimo = "";

				for (int i = 0; i < recursosArray.length; i++) {
					tipoRecursoActual = recursosArray[i].getTipoRecurso() + "_" + recursosArray[i].getSubtipoRecurso();

					if (!tipoRecursoActual.equals(tipoRecursoUltimo)) {
						if (!"".equals(tipoRecursoUltimo)) {
							sRespuesta.append("</Folder>");
	  				}

						sRespuesta.append("<Folder>");
						sRespuesta.append("<name>");
						sRespuesta.append(this.getNombreTipoRecurso(recursosArray[i].getTipoRecurso(), recursosArray[i].getSubtipoRecurso(), recursosArray[i].getModalidadRecurso(), recursosArray[i].getIdioma()));
						sRespuesta.append("</name>");

						tipoRecursoUltimo = tipoRecursoActual;
					}

					sRespuesta.append(recursosArray[i].toXMLString());
				}

				sRespuesta.append("</Folder>");
			}

		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getRecursos: Exception: " + e.getMessage());
			}
		}

		if ("JSON".equals(tipoDatosRespuesta)) {
			sRespuesta.append("}");
			sRespuesta.append("]");
			sRespuesta.append("}");
		} else {
			sRespuesta.append("</Document></kml>");
		}

		return sRespuesta.toString();
	}

	/**
	 * Se encarga de gestionar la peticin de informacin sobre municipios.
	 *
	 * Devuelve todos los municipios que se encuentren en la zona visible en el mapa delimitada
	 * por las coordenados de lat/long pasadas como parmetros.
	 *
	 * @param lat1 String Latitud de la esquina inferior izquierda de la vista
	 * @param lng1 String Longitud de la esquina inferior izquierda de la vista
	 * @param lat2 String Latitud de la esquina superior derecha de la vista
	 * @param lng2 String Longitud de la esquina superior derecha de la vista
	 * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
	 * 	Formato: tipoRecurso_subTipoRecurso_Categoria. Ejemplo: A1_H_1,B1_1,C2_
	 * @param idioma String Idioma de la vista
	 * @param tipoDatosRespuesta String Formato de la respuesta
	 * @param nombreArchivo String Nombre del archivo de respuesta
	 *
	 * @return StringBuffer
	 */
	private String getMunicipios(String lat1, String lng1, String lat2,String lng2,
			String tiposRecursos, String idioma, String tipoDatosRespuesta, String nombreArchivo) {
		StringBuffer sRespuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		S12MapasDAO mapasDAO = new S12MapasDAO();

		if ("JSON".equals(tipoDatosRespuesta)) {
			sRespuesta.append("{\"p\"");
			sRespuesta.append(":");
			sRespuesta.append("[");
			sRespuesta.append("{\"n\":");
		} else if ("GeoJSON".equals(tipoDatosRespuesta)) {
			/*
				{
				       "type": "FeatureCollection",
				       "features": [			  
			 */
			sRespuesta.append("{\"type\": \"FeatureCollection\", ");
			sRespuesta.append("\"features\": [");
			
		} else {
			sRespuesta.append("<?xml version='1.0' encoding='iso-8859-1'?><kml xmlns='http://earth.google.com/kml/2.0'><Document>");
			sRespuesta.append("<name>" + nombreArchivo + "</name>");
		}

		try {
			S12Municipio[] municipiosArray = mapasDAO.filtrarMunicipios(lat1, lng1, lat2, lng2, tiposRecursos, idioma);

			if ("JSON".equals(tipoDatosRespuesta)) {
				try {
					sRespuesta.append(new JSONArray(municipiosArray).toString());
				} catch (JSONException e) {
					if (Q70Factoria.getTraza().isTraceActive()) {
						Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMunicipios: Exception: " + e.getMessage());
					}
					sRespuesta.append("[]");
				}
			} else if ("GeoJSON".equals(tipoDatosRespuesta)) {
				for (int i = 0; i < municipiosArray.length; i++) {
					sRespuesta.append(municipiosArray[i].toGeoJSONString());
					if (municipiosArray.length<i+1) {
						sRespuesta.append(",");
					};
				}
			} else {
				for (int i = 0; i < municipiosArray.length; i++) {
					sRespuesta.append(municipiosArray[i].toXMLString());
				}
			}
		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMunicipios: Exception: " + e.getMessage());
			}
		}

		if ("JSON".equals(tipoDatosRespuesta)) {
			sRespuesta.append("}");
			sRespuesta.append("]");
			sRespuesta.append("}");
		} else if ("GeoJSON".equals(tipoDatosRespuesta)) {
			sRespuesta.append("]}");
		} else {
			sRespuesta.append("</Document></kml>");
		}

		return sRespuesta.toString();
	}
	
	/**
	 * Se encarga de gestionar la peticin de informacin sobre recursos de Euskadi Gastronomika.
	 *
	 * Devuelve todos los recursos de los tipos pasados como parmetro
	 * que se encuentren en la zona visible en el mapa delimitada
	 * por las coordenados de lat/long pasadas como parmetros.
	 *
	 * @param lat1 String Latitud de la esquina inferior izquierda de la vista
	 * @param lng1 String Longitud de la esquina inferior izquierda de la vista
	 * @param lat2 String Latitud de la esquina superior derecha de la vista
	 * @param lng2 String Longitud de la esquina superior derecha de la vista
	 * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
	 * 	Formato: tipoRecurso. Ejemplo: Z1,Z2
	 * @param idioma String Idioma de la vista
	 * 
	 * @return StringBuffer
	 */
	private String getRecursosGastronomika(String lat1, String lng1, String lat2,String lng2,
			String tiposRecursos, String idioma) {
		StringBuffer sRespuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		S12MapasDAO mapasDAO = new S12MapasDAO();

		sRespuesta.append("{\"p\"");
		sRespuesta.append(":");
		sRespuesta.append("[");
		sRespuesta.append("{\"n\":");

		try {
			S12RecursoGastronomika[] recursosArray = mapasDAO.filtrarRecursosGastronomika(lat1, lng1, lat2, lng2, tiposRecursos, idioma);

			try {
				sRespuesta.append(new JSONArray(recursosArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getRecursosGastronomika: Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}

		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getRecursosGastronomika: Exception: " + e.getMessage());
			}
		}

		sRespuesta.append("}");
		sRespuesta.append("]");
		sRespuesta.append("}");
		
		return sRespuesta.toString();
	}

	/**
	 * Se encarga de gestionar la peticin de informacin sobre municipios de Euskadi Gastronomika.
	 *
	 * Devuelve todos los municipios que se encuentren en la zona visible en el mapa delimitada
	 * por las coordenados de lat/long pasadas como parmetros.
	 *
	 * @param lat1 String Latitud de la esquina inferior izquierda de la vista
	 * @param lng1 String Longitud de la esquina inferior izquierda de la vista
	 * @param lat2 String Latitud de la esquina superior derecha de la vista
	 * @param lng2 String Longitud de la esquina superior derecha de la vista
	 * @param tiposRecursos String Tipos de recursos a filtrar separados por ','.
	 * 	Formato: tipoRecurso. Ejemplo: Z1,Z2
	 * @param idioma String Idioma de la vista
	 * 
	 * @return StringBuffer
	 */
	private String getMunicipiosGastronomika(String lat1, String lng1, String lat2,String lng2,
			String tiposRecursos, String idioma) {
		StringBuffer sRespuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		S12MapasDAO mapasDAO = new S12MapasDAO();

		sRespuesta.append("{\"p\"");
		sRespuesta.append(":");
		sRespuesta.append("[");
		sRespuesta.append("{\"n\":");
		

		try {
			S12Municipio[] municipiosArray = mapasDAO.filtrarMunicipiosGastronomika(lat1, lng1, lat2, lng2, tiposRecursos, idioma);

			try {
				sRespuesta.append(new JSONArray(municipiosArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMunicipiosGastronomika: Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}
			
		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMunicipiosGastronomika: Exception: " + e.getMessage());
			}
		}

		sRespuesta.append("}");
		sRespuesta.append("]");
		sRespuesta.append("}");

		return sRespuesta.toString();
	}

	/**
	 * Se encarga de gestionar la peticin de informacin de un marcador o un conjunto de marcadores.
	 *
	 * Devuelve los datos del recurso o recursos que cumplen los criterios pasados como parmetro.
	 *
	 * @param codigoRecurso Cdigo del recurso si hay ms de uno separados por ","
	 * @param idioma String Idioma de la vista
	 *
	 * @return StringBuffer
	 */
	private String getMarcador(String codigoRecurso, String idioma) {
		StringBuffer sRespuesta = new StringBuffer();

		S12MapasDAO mapasDAO = new S12MapasDAO();

		sRespuesta.append("{\"n\":[");

		try {
			if (codigoRecurso.indexOf(",") == -1) { // Un solo recurso
				S12Recurso recurso = mapasDAO.obtenerRecurso(codigoRecurso, idioma);

				if (recurso != null) {
					sRespuesta.append("[");
					sRespuesta.append(recurso.toJSONString());
					sRespuesta.append("]");
				}
			} else { // Varios recursos
				S12Recurso[] recursosArray = mapasDAO.obtenerRecursos(codigoRecurso, idioma);

				try {
					sRespuesta.append(new JSONArray(recursosArray).toString());
				} catch (JSONException e) {
					if (Q70Factoria.getTraza().isTraceActive()) {
						Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMarcador: Exception: " + e.getMessage());
					}
					sRespuesta.append("[]");
				}
			}
		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMarcador: Exception: " + e.getMessage());
			}
		}

		sRespuesta.append("]");
		sRespuesta.append("}");

		return sRespuesta.toString();
	}

	/**
	 * Se encarga de gestionar la peticin de informacin de un marcador o un conjunto de marcadores.
	 *
	 * Devuelve los datos del recurso o recursos que cumplen los criterios pasados como parmetro.
	 * El formato de salida es GeoJSON.
	 * 
	 * @param codigoRecurso Cdigo del recurso si hay ms de uno separados por ","
	 * @param idioma String Idioma de la vista
	 *
	 * @return StringBuffer
	 */
	private String getMarcadorGeoJSON(String codigoRecurso, String idioma) {
		StringBuffer sRespuesta = new StringBuffer();

		S12MapasDAO mapasDAO = new S12MapasDAO();

		sRespuesta.append("{\"type\": \"FeatureCollection\", ");
		sRespuesta.append("\"features\": [");

		try {
			if (codigoRecurso.indexOf(",") == -1) { // Un solo recurso
				S12Recurso recurso = mapasDAO.obtenerRecurso(codigoRecurso, idioma);

				if (recurso != null) {
					//sRespuesta.append("[");
					sRespuesta.append(recurso.toGeoJSONString());
					//sRespuesta.append("]");
				}
			} else { // Varios recursos
				S12Recurso[] recursosArray = mapasDAO.obtenerRecursos(codigoRecurso, idioma);
				for (int i = 0; i < recursosArray.length; i++) {
					sRespuesta.append(recursosArray[i].toGeoJSONString());
					if (recursosArray.length>i+1) {
						sRespuesta.append(",");
					};
				}
			}
		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getMarcadorGeoJSON: Exception: " + e.getMessage());
			}
		}
		sRespuesta.append("]}");

		return sRespuesta.toString();
	}
	
	/**
	 * Carga del archivo de properties el nombre de un tipo de recurso, dependiendo del idioma.
	 *
	 * @param psTipoRecurso Tipo de recurso
	 * @param psIdioma Idioma
	 * @param psSubtipoRecurso Subtipo de recurso
	 * @param psCategoriaRecurso Categora del recurso
	 *
	 * @return Nombre en el idioma correspondiente del tipo de recurso pasado como parmetro
	 */
	private String getNombreTipoRecurso(String psTipoRecurso, String psSubtipoRecurso, String psCategoriaRecurso, String psIdioma) {
		String sRespuesta = "";
		StringBuffer tagFicheroConstantes = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		if (psTipoRecurso.equalsIgnoreCase("A1")) {
			if (psSubtipoRecurso.equalsIgnoreCase("K")) {
				tagFicheroConstantes.append("Filtros.Alojamientos.Alojrurales.Agroturismos");
			} else if (psSubtipoRecurso.equalsIgnoreCase("X")) {
				tagFicheroConstantes.append("Filtros.Alojamientos.Alojrurales.Casas");
			} else if (psSubtipoRecurso.equalsIgnoreCase("H") || psSubtipoRecurso.equalsIgnoreCase("6")) {
				tagFicheroConstantes.append("Filtros.Alojamientos.Alojresto.Hoteles");
			} else if (psSubtipoRecurso.equalsIgnoreCase("7")) {
				tagFicheroConstantes.append("Filtros.Alojamientos.Alojresto.Pensiones");
			} else if (psSubtipoRecurso.equalsIgnoreCase("P")) {
				tagFicheroConstantes.append("Filtros.Alojamientos.Alojresto.Campings");
			} else if (psSubtipoRecurso.equalsIgnoreCase("T")) {
				tagFicheroConstantes.append("Filtros.Alojamientos.Alojresto.Apartamentos");
			}
		} else if (psTipoRecurso.equalsIgnoreCase("B1")) {
			if (psSubtipoRecurso.equalsIgnoreCase("1")) {
				tagFicheroConstantes.append("Filtros.Restauracion.Restaurantes");
			} else if (psSubtipoRecurso.equalsIgnoreCase("42")) {
				tagFicheroConstantes.append("Filtros.Restauracion.Sidrerias");
			} else if (psSubtipoRecurso.equalsIgnoreCase("4")) {
				tagFicheroConstantes.append("Filtros.Restauracion.Bodegas");
			}
		} else if (psTipoRecurso.equalsIgnoreCase("C1")) {
			tagFicheroConstantes.append("Filtros.Transporte.Aeropuertos");
		} else if (psTipoRecurso.equalsIgnoreCase("C2")) {
			tagFicheroConstantes.append("Filtros.Transporte.Compaias");
		} else if (psTipoRecurso.equalsIgnoreCase("C3")) {
			tagFicheroConstantes.append("Filtros.Transporte.Estaciones");
		} else if (psTipoRecurso.equalsIgnoreCase("C6")) {
			tagFicheroConstantes.append("Filtros.Transporte.Parkings");
		} else if (psTipoRecurso.equalsIgnoreCase("D4")) {
			tagFicheroConstantes.append("Filtros.Rutas");
		} else if (psTipoRecurso.equalsIgnoreCase("D6")) {
			tagFicheroConstantes.append("Filtros.ExperienciasTop");
		} else if (psTipoRecurso.equalsIgnoreCase("D7")) {
			tagFicheroConstantes.append("Filtros.Rutas.ViasVerdes");
		} else if (psTipoRecurso.equalsIgnoreCase("E1")) {
			tagFicheroConstantes.append("Filtros.InfoTurismo.Oficinas");
		} else if (psTipoRecurso.equalsIgnoreCase("F1")) {
			tagFicheroConstantes.append("Filtros.PlanesEuskadi");
		} else if (psTipoRecurso.equalsIgnoreCase("G1")) {
			if (psSubtipoRecurso.equalsIgnoreCase("0001")) {
				tagFicheroConstantes.append("Filtros.Naturaleza.Playas");
			} else {
				tagFicheroConstantes.append("Filtros.Naturaleza.Embalses");
			}
		} else if (psTipoRecurso.equalsIgnoreCase("G2")) {
			tagFicheroConstantes.append("Filtros.Naturaleza.EspaciosNaturales");
		} else if (psTipoRecurso.equalsIgnoreCase("H1")) {
			tagFicheroConstantes.append("Filtros.Cultura.TeatrosCines");
		} else if (psTipoRecurso.equalsIgnoreCase("H4")) {
			if (psSubtipoRecurso.equalsIgnoreCase("0000")) {
				tagFicheroConstantes.append("Filtros.Cultura.Museos");
			} else if (psSubtipoRecurso.equalsIgnoreCase("0004")) {
				tagFicheroConstantes.append("Filtros.Cultura.CentroInterpretacion");
			}
		} else if (psTipoRecurso.equalsIgnoreCase("H5")) {
			tagFicheroConstantes.append("Filtros.Cultura.Patrimonio");
		} else if (psTipoRecurso.equalsIgnoreCase("H6")) {
			tagFicheroConstantes.append("Filtros.Cultura.Puertos");
		} else if (psTipoRecurso.equalsIgnoreCase("H10")) {
			tagFicheroConstantes.append("Filtros.Cultura.Artesanos");
		} else if (psTipoRecurso.equalsIgnoreCase("I2")) {
			if (psSubtipoRecurso.equalsIgnoreCase("1010")) {
				tagFicheroConstantes.append("Filtros.Deporte.BTT");
			} else if (psSubtipoRecurso.equalsIgnoreCase("1003")) {
				tagFicheroConstantes.append("Filtros.Deporte.Golf");
			} else if (psSubtipoRecurso.equalsIgnoreCase("1008")) {
				tagFicheroConstantes.append("Filtros.Deporte.Hipodromos");
			} else if (psSubtipoRecurso.equalsIgnoreCase("1004")) {
				tagFicheroConstantes.append("Filtros.Deporte.PuertosDeportivos");
			} else {
				tagFicheroConstantes.append("Filtros.Deporte.Instalaciones");
			}
		} else if (psTipoRecurso.equalsIgnoreCase("I31")) {
			tagFicheroConstantes.append("Filtros.Deporte.EmpresasAlquiler");
		} else if (psTipoRecurso.equalsIgnoreCase("I32")) {
			tagFicheroConstantes.append("Filtros.TurismoActivo.Empresas");
		} else if (psTipoRecurso.equalsIgnoreCase("J4")) {
			tagFicheroConstantes.append("Filtros.Formacion.Centros");
		} else if (psTipoRecurso.equalsIgnoreCase("K1")) {
			tagFicheroConstantes.append("Filtros.Cultura.Toros");
		} else if (psTipoRecurso.equalsIgnoreCase("K3")) {
			tagFicheroConstantes.append("Filtros.Ocio.Parques");
		} else if (psTipoRecurso.equalsIgnoreCase("K5")) {
			tagFicheroConstantes.append("Filtros.Ocio.Aquariums");
		} else if (psTipoRecurso.equalsIgnoreCase("K8")) {
			tagFicheroConstantes.append("Filtros.Restauracion.BaresPintxos");
		} else if (psTipoRecurso.equalsIgnoreCase("L1")) {
			tagFicheroConstantes.append("Filtros.Ocio.ZonaTiendas");
		} else if (psTipoRecurso.equalsIgnoreCase("L3")) {
			tagFicheroConstantes.append("Filtros.Ocio.Centros");
		} else if (psTipoRecurso.equalsIgnoreCase("M1")) {
			tagFicheroConstantes.append("Filtros.Agenda.Eventos");
		} else if (psTipoRecurso.equalsIgnoreCase("M2")) {
			tagFicheroConstantes.append("Filtros.Ofertas.Alojamiento");
		} else if (psTipoRecurso.equalsIgnoreCase("N1")) {
			tagFicheroConstantes.append("Filtros.CongresosReuniones.Palacios");
		} else if (psTipoRecurso.equalsIgnoreCase("N2")) {
			tagFicheroConstantes.append("Filtros.CongresosReuniones.RecintosFeriales");
		} else if (psTipoRecurso.equalsIgnoreCase("N3")) {
			if (psSubtipoRecurso.equalsIgnoreCase("0009")) {
				tagFicheroConstantes.append("Filtros.CongresosReuniones.ConventionBureau");
			} else if (psSubtipoRecurso.equalsIgnoreCase("0006")) {
				tagFicheroConstantes.append("Filtros.CongresosReuniones.EmpresasServicio");
			}
		} else if (psTipoRecurso.equalsIgnoreCase("N5")) {
			tagFicheroConstantes.append("Filtros.Salud.Termal");
		}
		try {
			if (tagFicheroConstantes.length() > 0) {
				sRespuesta = Q70GestorMensajesi18n.getSingleton().getMessage("s12GoogleMaps", tagFicheroConstantes.toString(), psIdioma).toUpperCase();
			}
		}
		catch (Q70GestorMensajesException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getNombreTipoRecurso: Exception: " + e.getMessage());
			}
		}

		return sRespuesta;
	}

	/**
	 * Compone un buffer de bytes con el recurso en formato OV2 con orden de bytes Little_Endian.
	 * El formato de los bytes es:
	 * 1 BYTE, char, POI status (always '2')
	 * 4 BYTES, long, denotes length of the POI record (nombre.length() + 14).
	 * 4 BYTES, long, longitude * 100000
	 * 4 BYTES, long, latitude * 100000
	 * x BYTES, string, label for POI, x = nombre.length()
	 *
	 * @param nombreRecurso Nombre del recurso
	 * @param lat Latitud
	 * @param lng Longitud
	 *
	 * @return ByteBuffer
	 *
	 * @throws IOException Signals that an I/O exception has occurred.
	 */
	private ByteBuffer getPOIOV2(String nombreRecurso, String lat, String lng) throws IOException {
	  int lengthBuffer = nombreRecurso.length() + S12Conexion.MARGEN_LONGITUD;
	  ByteBuffer buf = ByteBuffer.allocate (lengthBuffer);
	  // Ponemos el orden de bytes a Little_endian porque el archivo ov2 lo requiere as
	  buf.order(ByteOrder.LITTLE_ENDIAN);

	  buf.put((byte)0x02);
	  buf.putInt(lengthBuffer);
	  buf.putInt((int)(Double.parseDouble(lng) * S12Conexion.CONVERTIR_LATITUD_LONGITUD));
	  buf.putInt((int)(Double.parseDouble(lat) * S12Conexion.CONVERTIR_LATITUD_LONGITUD));

	  int i = 0;
	  for (; i < nombreRecurso.length(); i++) {
	  	buf.putChar(i + S12Conexion.SIZE_LITTLE_ENDIAN, nombreRecurso.charAt(i));
	  }
	  buf.put(i + S12Conexion.SIZE_LITTLE_ENDIAN, (byte)0x00);
	  return buf;
	}

	/**
	 * Compone el archivo CSV con el marcador para importar con un programa GPS
	 *
	 * @param nombreRecurso Nombre del recurso
	 * @param lat Latitud
	 * @param lng Longitud
	 *
	 * @return string
	 */
	public StringBuffer getPOICSV(String nombreRecurso, String lat, String lng) {
		StringBuffer respuesta = new StringBuffer("");
		respuesta.append(lng);
		respuesta.append(",");
		respuesta.append(lat);
		respuesta.append(",");
		respuesta.append('"');
		respuesta.append(nombreRecurso);
		respuesta.append('"');

		return respuesta;
	}

	/**
	 * Compone el archivo KML correspondiente a la ruta
	 *
	 * @param nombreRuta Nombre de la ruta
	 * @param puntosRuta Puntos que componen la ruta. Formato: longitud1,latitud1;longitud2,latitud2;...
	 *
	 * @return string
	 */
	public StringBuffer getRuta(String nombreRuta, String puntosRuta) {
		StringBuffer respuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);
		respuesta.append("<?xml version='1.0' encoding='UTF-8'?>");
		respuesta.append("\n<kml xmlns='http://earth.google.com/kml/2.1'>");
		respuesta.append("\n<Document>");
		respuesta.append("\n\t<name>" + nombreRuta + "</name>");
		respuesta.append("\n\t<Placemark>");
		respuesta.append("\n\t\t<name>" + nombreRuta + "</name>");
		respuesta.append("\n\t\t<LineString>");
		respuesta.append("\n\t\t\t<coordinates>");

		String [] arrayPuntos = puntosRuta.split(";");

		for (int i = 0; i < arrayPuntos.length; i++){
			if (!arrayPuntos[i].equals("")){
				respuesta.append("\n\t\t\t\t" + arrayPuntos[i].split(",")[0] + "," + arrayPuntos[i].split(",")[1] + ",0");
			}
		}
		respuesta.append("\n\t\t\t</coordinates>");
		respuesta.append("\n\t\t</LineString>");
		respuesta.append("\n\t</Placemark>");
		respuesta.append("\n</Document>");
		respuesta.append("\n</kml>");
		return respuesta;
	}

	/**
	 * Se encarga de gestionar la peticin de informacin de recursos importantes.
	 *
	 * Devuelve todos los recursos importantes (importancia 1  2) de cualquier tipo
	 *
	 * @param idioma String Idioma de la vista
	 *
	 * @param importancia String importancia del recurso
	 *
	 * @return StringBuffer
	 */
	private String getRecursosImportantes(String idioma, String importancia) {
		StringBuffer sRespuesta = new StringBuffer(S12Conexion.SIZE_STRINGBUFFER);

		S12MapasDAO mapasDAO = new S12MapasDAO();

		sRespuesta.append("{\"p\"");
		sRespuesta.append(":");
		sRespuesta.append("[");
		sRespuesta.append("{\"n\":");


		try {
			S12Recurso[] recursosArray = mapasDAO.filtrarRecursosImportantes(idioma, importancia);

			try {
				sRespuesta.append(new JSONArray(recursosArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getRecursos: Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}

		} catch (S12MapasDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12Conexion.COMPONENTE, Q70TraceLevel.ERROR,"S12Conexion::getRecursosImportantes: Exception: " + e.getMessage());
			}
		}

		sRespuesta.append("}");
		sRespuesta.append("]");
		sRespuesta.append("}");

		return sRespuesta.toString();
	}

}
