/*
 * Created on Jun 15, 2011
 *
 */
package s12.util;

import java.lang.reflect.Array;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.Charset;
import java.sql.Date;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.regex.Pattern;

import javax.servlet.http.Cookie;

import s12.constantes.S12Constantes;
import s12.portalMovil.valueObjects.S12Experiencia;
import s12.portalMovil.valueObjects.S12Ofertas;
import s12.portalMovil.valueObjects.S12RecursoTuristico;

import com.bea.xml.XmlCalendar;

import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;
import es.ejie.frmk.listeners.base.Q70ListenerUtils;

/**
 * Clase con funciones de utilidad
 * 
 *  
 * 
 */
public class S12Utils {

	/** The Constant MAX_LENGTH_NOMBRE_CONT. */
	private final static int MAX_LENGTH_NOMBRE_CONT = 10;

	/** Grados de un ngulo llano. */
	private final static int GRADOS_ANGULO_LLANO = 180;

	/** Kilmetros por grado. */
	private final static double KM_GRADO = 111.194;

	/** Millas por grado. */
	// private final static double MILLAS_GRADO = 69.09;

	/** Nombre de la ficha2 */
	private final static String HTML_FICHA2 = "ficha2";

	/** Nombre de la miniatura28 */
	private final static String HTML_MINIATURA28 = "miniatura28";

	/** Constante typology Eventos */
	public static final String TYPO_EVENTO = "evento";

	/** Constante typology Noticias */
	public static final String TYPO_NOTICIA = "noticia";

	/** Pgina del Portal para las fichas */
	public static final String PAGINA_PORTAL = "12375";

	/** Pgina del Portal para las fichas de Experiencias (D6) */
	public static final String PAGINA_PORTAL_EXPERIENCIAS = "12378";

	/** The Constant MAX_LENGTH_TRAZA */
	private final static int MAX_LENGTH_TRAZA = 255;

	/**
	 * Constructor.
	 */
	private S12Utils() {
	}

	/**
	 * 
	 * Convierte los caracteres especiales a cdigo HTML
	 * 
	 * @param tag Cadena de texto original
	 * 
	 * @return Cadena de texto reemplazada
	 */
	public static String encodeHtmlTag(String tag) {
		if (tag == null) {
			return null;
		}

		int length = tag.length();

		StringBuffer encodedTag = new StringBuffer(2 * length);

		for (int i = 0; i < length; i++) {
			char ch = tag.substring(i, i + 1).charAt(0);

			if (ch >= 'a' && ch <= 'z' || ch >= 'A' && ch <= 'Z' || ch >= '0' && ch <= '9') {
				// safe
				encodedTag.append(ch);
			} else if (Character.isWhitespace(ch)) {
				// paranoid version: whitespaces are unsafe - escape
				// conversion of (int)ch is naive
				encodedTag.append("&#").append((int) ch).append(";");
			} else if (Character.isISOControl(ch)) {
				// paranoid version:isISOControl which are not isWhitespace removed !
				// do nothing do not include in output !
			} else {
				// paranoid version
				// the rest is unsafe, including <127 control chars
				encodedTag.append("&#" + (int) ch + ";");
			}
		}

		return encodedTag.toString();
	}

	/**
	 * Obtener bundle.
	 * 
	 * @param languagueParameter the languague parameter
	 * @param nombreFichero the nombre fichero
	 * 
	 * @return the resource bundle
	 */
	public static ResourceBundle obtenerBundle(String languagueParameter, String nombreFichero) {
		ResourceBundle bundle;
		Locale locale;

		if (languagueParameter != null && "es".equals(languagueParameter)) {
			locale = new Locale(languagueParameter, "ES");
		} else if (languagueParameter != null && "eu".equals(languagueParameter)) {
			locale = new Locale(languagueParameter, "EU");
		} else if (languagueParameter != null && "en".equals(languagueParameter)) {
			locale = new Locale(languagueParameter, "GB");
		} else if (languagueParameter != null && "fr".equals(languagueParameter)) {
			locale = new Locale(languagueParameter, "FR");
		} else if (languagueParameter != null && "de".equals(languagueParameter)) {
			locale = new Locale(languagueParameter, "DE");
		} else {
			locale = new Locale("es", "ES");

		}
		bundle = ResourceBundle.getBundle("/" + nombreFichero, locale);

		return bundle;
	}

	/**
	 * Obtiene la parte de la url del contenido de la forma 0000048241_d6_rec_turismo donde se utiliza el cdigo del
	 * recurso normalizado a 10 caracteres y el tipo de recurso
	 * 
	 * @param idRecurso Cdigo del recurso
	 * @param codTipoRecurso Tipo de recurso
	 * 
	 * @return the nombre contenido
	 */
	public static String getNombreContenido(int idRecurso, String codTipoRecurso) {

		StringBuffer nombreCont = new StringBuffer(String.valueOf(idRecurso));
		while (nombreCont.length() < S12Utils.MAX_LENGTH_NOMBRE_CONT) {
			nombreCont.insert(0, "0");
		}
		nombreCont.append("_");
		nombreCont.append(codTipoRecurso.toLowerCase());
		nombreCont.append("_rec_turismo");
		return nombreCont.toString();
	}

	/**
	 * Obtiene la url del contenido
	 * 
	 * @param idRecurso Cdigo del recurso
	 * @param idTipoRecurso Tipo de recurso
	 * @param idIdioma Idioma
	 * 
	 * @return url del contenido
	 */
	public static String getURLContenido(int idRecurso, String idTipoRecurso, String idIdioma) {
		StringBuffer bufPathContenido = new StringBuffer();
		String typologyRecurso = S12Constantes.getContentType(idTipoRecurso);

		bufPathContenido.append("/contenidos/");
		bufPathContenido.append(typologyRecurso);
		bufPathContenido.append("/");
		bufPathContenido.append(S12Utils.getNombreContenido(idRecurso, idTipoRecurso));
		bufPathContenido.append("/");
		if (idIdioma != null) {
			bufPathContenido.append(idIdioma);
		} else { // Por defecto castellano (elementos multimedia)
			bufPathContenido.append("es");
		}
		bufPathContenido.append("_");
		bufPathContenido.append(idRecurso);
		bufPathContenido.append("/");

		return bufPathContenido.toString();
	}

	/**
	 * Obtiene la url de la ficha antigua.
	 * 
	 * @param idRecurso Cdigo del recurso
	 * @param idTipoRecurso Tipo de recurso
	 * @param idIdioma Idioma
	 * 
	 * @return url de la ficha antigua
	 */
	public static String getFichaAntigua(int idRecurso, String idTipoRecurso, String idIdioma) {
		StringBuffer bufPathContenido = new StringBuffer();
		String typologyRecurso = S12Constantes.getContentType(idTipoRecurso);

		bufPathContenido.append(S12Utils.getURLContenido(idRecurso, idTipoRecurso, idIdioma));

		if (S12Utils.TYPO_EVENTO.equals(typologyRecurso) || S12Utils.TYPO_NOTICIA.equals(typologyRecurso)) {
			if (S12Ofertas.TIPO_RECURSO.equals(idTipoRecurso)) { // si es Ofertas, que apunte a la miniatura28
				bufPathContenido.append(idRecurso).append("-").append(S12Utils.HTML_MINIATURA28).append(".html");
			} else {
				bufPathContenido.append(idRecurso).append(".html");
			}
		} else {
			bufPathContenido.append(idRecurso).append("-").append(S12Utils.HTML_FICHA2).append(".html");
		}

		return bufPathContenido.toString();
	}

	/**
	 * Obtiene la url mvil del recurso
	 * 
	 * @param idRecurso Cdigo del recurso
	 * @param idTipoRecurso Tipo de recurso
	 * @param urlAmigable URL amigable del recurso
	 * @param idIdioma Idioma
	 * 
	 * @return url del contenido
	 */
	public static String getURLMerkur(int idRecurso, String idTipoRecurso, String urlAmigable, String idIdioma) {
		// Dominio
		String dominioMovil = Q70ListenerUtils.getApplicationProperty("url_dominio_movil_" + idIdioma);

		// La pgina es distinta para los recursos de tipo D6 (Experiencias)
		String pagina = S12Experiencia.TIPO_RECURSO.equals(idTipoRecurso) ? S12Utils.PAGINA_PORTAL_EXPERIENCIAS
				: S12Utils.PAGINA_PORTAL;

		StringBuffer bufPathRecurso = new StringBuffer();

		bufPathRecurso.append(dominioMovil);

		if (urlAmigable != null) {
			bufPathRecurso.append(urlAmigable);
			bufPathRecurso.append("mobile/");
			bufPathRecurso.append(pagina);
			bufPathRecurso.append("/?R01HLang=");
			bufPathRecurso.append(idIdioma);
		} else {
			bufPathRecurso.append("mobile/");
			bufPathRecurso.append(pagina);
			bufPathRecurso.append(S12Utils.getFichaAntigua(idRecurso, idTipoRecurso, idIdioma));
			bufPathRecurso.append("/?R01HLang=");
			bufPathRecurso.append(idIdioma);
		}

		return bufPathRecurso.toString();
	}

	/**
	 * Calcula la distancia geodsica entre dos puntos dada su latitud y longitud en coordenadas Google Maps (grados
	 * decimales).
	 * 
	 * @param lat1 Latitud en grados decimales (Google Maps)
	 * @param long1 Longitud en grados decimales (Google Maps)
	 * @param lat2 Latitud en grados decimales (Google Maps)
	 * @param long2 Longitud en grados decimales (Google Maps)
	 * 
	 * @return la distancia entre los dos puntos en Km
	 */
	public static double getDistanciaGeodesica(double lat1, double long1, double lat2, double long2) {
		// Pasamos los grados decimales a radianes
		double rLat1 = lat1 * Math.PI / S12Utils.GRADOS_ANGULO_LLANO;
		double rLong1 = long1 * Math.PI / S12Utils.GRADOS_ANGULO_LLANO;
		double rLat2 = lat2 * Math.PI / S12Utils.GRADOS_ANGULO_LLANO;
		double rLong2 = long2 * Math.PI / S12Utils.GRADOS_ANGULO_LLANO;

		double p = Math.sin(rLat1) * Math.sin(rLat2) + Math.cos(rLat1) * Math.cos(rLat2) * Math.cos(rLong1 - rLong2);
		double d = Math.acos(p) * S12Utils.GRADOS_ANGULO_LLANO / Math.PI; // Valor en grados
		double distancia = d * S12Utils.KM_GRADO; // Distancia en Km
		// double distancia = p * S12Utils.MILLAS_GRADO; // Distancia en Millas
		return distancia;
	}

	/**
	 * Funcin que determina si un punto est dentro de un polgono
	 * 
	 * @param poligono coordenadas del polgono. Formato [[latitud,longitud],...]]
	 * @param latitud latitud del punto
	 * @param longitud longitud del punto
	 * 
	 * @return true si el punto est dentro del polgono
	 */
	public static boolean contienePunto(double[][] poligono, double latitud, double longitud) {
		boolean inPoly = false;

		int numPoints = poligono.length;
		int j = numPoints - 1;

		try {
			for (int i = 0; i < numPoints; i++) {
				double[] vertex1 = poligono[i];
				double[] vertex2 = poligono[j];

				if (vertex1[1] < longitud && vertex2[1] >= longitud || vertex2[1] < longitud && vertex1[1] >= longitud) {
					if (vertex1[0] + (longitud - vertex1[1]) / (vertex2[1] - vertex1[1]) * (vertex2[0] - vertex1[0]) < latitud) {
						inPoly = !inPoly;
					}
				}

				j = i;
			}
		} catch (Exception e) {
			return inPoly;
		}

		return inPoly;
	}

	/**
	 * Aplica la posible paginacin antes de mostrar los resultados
	 * 
	 * @param resultArray array de resultados totales
	 * @param numeroResultados nmero de resultados a mostrar
	 * @param ultimoResultado ltimo resultado mostrado
	 * @param clase tipo de los resultados
	 * 
	 * @return array de resultados paginados
	 */
	@SuppressWarnings("rawtypes")
	public static Object[] getPaginarResultados(Object[] resultArray, int numeroResultados, int ultimoResultado,
			Class clase) {

		Object[] resultArrayPaginado;

		if (numeroResultados > 0 && numeroResultados < resultArray.length && ultimoResultado == 0) {
			resultArrayPaginado = (Object[]) Array.newInstance(clase, numeroResultados);
			System.arraycopy(resultArray, 0, resultArrayPaginado, 0, numeroResultados);
		} else if (ultimoResultado > 0 && ultimoResultado + numeroResultados < resultArray.length) {
			resultArrayPaginado = (Object[]) Array.newInstance(clase, numeroResultados);
			System.arraycopy(resultArray, ultimoResultado, resultArrayPaginado, 0, numeroResultados);
		} else if (ultimoResultado > 0 && ultimoResultado + numeroResultados >= resultArray.length
				&& resultArray.length - ultimoResultado >= 0) {
			resultArrayPaginado = (Object[]) Array.newInstance(clase, resultArray.length - ultimoResultado);
			System.arraycopy(resultArray, ultimoResultado, resultArrayPaginado, 0, resultArray.length - ultimoResultado);
		} else {
			resultArrayPaginado = resultArray;
		}

		return resultArrayPaginado;
	}

	public static Object[] casarOrdenResultados(Object[] resultArray, ArrayList<String> arrCodigos, Class clase) {
		Object[] resultArrayOrdenado;

		// corregimos el ORDER en base a la paginacin.
		// arrRecursos tiene los cdigos de recurso y en el orden en que deben aparecer los resultados.
		// Hay que emparejarlo con el array de datos completos que hemos cargado, para que finalmente aparezcan bien en
		// el JSON.
		int length = resultArray.length;
		resultArrayOrdenado = (Object[]) Array.newInstance(clase, length);

		ArrayList<S12RecursoTuristico> restauracionListOrdenados = new ArrayList<S12RecursoTuristico>();
		for (int j = 0; j < arrCodigos.size(); j++) {
			String corec = arrCodigos.get(j);
			for (int i = 0; i < resultArray.length; i++) {
				if (Integer.parseInt(corec) == ((S12RecursoTuristico) resultArray[i]).getCodigoRecurso()) {
					restauracionListOrdenados.add((S12RecursoTuristico) resultArray[i]);
					break;
				}
			}
		}
		restauracionListOrdenados.toArray(resultArrayOrdenado);
		return resultArrayOrdenado;
	}

	/**
	 * Obtiene la fecha formateada segn el idioma.
	 * 
	 * @param idIdioma Idioma
	 * @param fecha the fecha
	 * 
	 * @return the fecha formateada
	 */
	public static String getFechaFormateada(Date fecha, String idIdioma) {

		//es, en, fr, de --> dd/mm/yyyy
		//eu --> yyyy/mm/dd

		String fechaFrmt;

		SimpleDateFormat df = new SimpleDateFormat();
		if (idIdioma.equals("eu")) {
			df = new SimpleDateFormat("yyyy/MM/dd");
		} else  {
			df = new SimpleDateFormat("dd/MM/yyyy");
		} 
		fechaFrmt = df.format(fecha);

		return fechaFrmt;
	}

	/**
	 * Convierte una fecha en formato DD/MM/AAAA a otra en formato AAAAMMDD.
	 * 
	 * @param fecha fecha con formato DD/MM/AAAA
	 * @return fecha con formato AAAAMMDD
	 */
	public static String parsearFecha(String fecha) {
		String fechaParseada = "";

		try {
			String[] aFecha = fecha.split("/");
			fechaParseada = aFecha[2] + aFecha[1] + aFecha[0];
		} catch (Exception e) {
			// Ante cualquier problema en el formato se devuelve el string vaco
		}

		return fechaParseada;
	}

	/**
	 * Convierte un string codificado en UTF-8 a un string codificado en ISO-8859-1.
	 * 
	 * @param utf8 String codificado en UTF-8
	 * 
	 * @return String codificado en ISO-8859-1
	 */
	public static String utf8toIso88591(String utf8) {
		Charset utf8charset = Charset.forName("UTF-8");
		Charset iso88591charset = Charset.forName("ISO-8859-1");

		ByteBuffer inputBuffer = ByteBuffer.wrap(utf8.getBytes());

		// decode UTF-8
		CharBuffer data = utf8charset.decode(inputBuffer);

		// encode ISO-8859-1
		ByteBuffer outputBuffer = iso88591charset.encode(data);
		byte[] outputData = outputBuffer.array();

		return new String(outputData);
	}

	/**
	 * Modifica el formato de una fecha que recibimos como string
	 * 
	 * @param String Fecha recibida
	 * 
	 * 
	 * @return the ficha antigua
	 */
	public static String setDateFormat(XmlCalendar fecha) {
		String fechaResultado = "";
		SimpleDateFormat sd = new SimpleDateFormat("dd/MM/yyyy");

		fechaResultado = sd.format(fecha.getTime());

		return fechaResultado;
	}

	/**
	 * Transforma una fecha con formato de intercambio bsico YYYYMMDD y la devuelve en el formato de presentacin
	 * correspondiente al idioma.
	 * 
	 * @param fechaFormatoYYYYMMDD
	 * @param idioma Codigo de idioma
	 * @return String con la fecha en formato de presentacin.
	 */
	public static String fechaConFormatoPortal(String fechaFormatoYYYYMMDD, String idioma) {
		String resultado = "";
		try {
			if (fechaFormatoYYYYMMDD == null || "".equals(fechaFormatoYYYYMMDD) || fechaFormatoYYYYMMDD.length() < 8) {
				resultado = fechaFormatoYYYYMMDD;
			}
			// es,en,fr,de coinciden en formato DD/MM/YYYY
			// eu va en formato YYYY/MM/DD
			if (idioma.equals(S12Constantes.IDIOMA_EU)) {
				resultado = fechaFormatoYYYYMMDD.substring(0, 3).concat("/")
						.concat(fechaFormatoYYYYMMDD.substring(4, 5)).concat("/")
						.concat(fechaFormatoYYYYMMDD.substring(6));
			} else {
				resultado = fechaFormatoYYYYMMDD.substring(6, 7).concat("/")
						.concat(fechaFormatoYYYYMMDD.substring(4, 5)).concat("/")
						.concat(fechaFormatoYYYYMMDD.substring(0, 3));
			}
		} catch (Exception e) {
			// si hay cualquier problema en la transformacin devuelve la misma cadena de entrada
			resultado = fechaFormatoYYYYMMDD;
		}
		return resultado;
	}

	/**
	 * Transforma una fecha con formato de presentacin y la devuelve en el formato de intercambio bsico YYYYMMDD
	 * 
	 * @param fechaConFormatoPortal
	 * @param idioma
	 * @return String con la fecha en formato de intercambio bsico
	 */
	public static String fechaFormatoYYYYMMDD(String fechaConFormatoPortal, String idioma) {
		String resultado = "";
		try {
			if (fechaConFormatoPortal == null || "".equals(fechaConFormatoPortal)
					|| fechaConFormatoPortal.length() < 10) {
				// fecha incorrecta
				resultado = fechaConFormatoPortal;
			} else {
				String[] componentesFecha = fechaConFormatoPortal.split("/");
				if (componentesFecha.length != 3) {
					// fecha incorrecta
					resultado = fechaConFormatoPortal;
				}
				// es,en,fr,de coinciden en formato DD/MM/YYYY
				// eu va en formato YYYY/MM/DD
				if (idioma.equals(S12Constantes.IDIOMA_EU)) {
					resultado = componentesFecha[0].concat(componentesFecha[1].concat(componentesFecha[2]));
				} else {
					resultado = componentesFecha[2].concat(componentesFecha[1].concat(componentesFecha[0]));
				}
			}
		} catch (Exception e) {
			// si hay cualquier problema en la transformacin devuelve la misma cadena de entrada
			resultado = fechaConFormatoPortal;
		}
		return resultado;
	}

	/**
	 * Transforma una fecha de tipo Calendar y la devuelve como String en el formato de intercambio bsico YYYYMMDD
	 * 
	 * @param fecha Calendar
	 * @return String con la fecha en formato de intercambio bsico
	 */
	public static String fechaFormatoYYYYMMDD(Calendar fecha) {
		String resultado = "";
		try {

			String dia = Integer.toString(fecha.get(Calendar.DATE));
			if (dia.length() == 1) {
				dia = "0" + dia;
			}

			String mes = Integer.toString(fecha.get(Calendar.MONTH) + 1);
			if (mes.length() == 1) {
				mes = "0" + mes;
			}

			String year = Integer.toString(fecha.get(Calendar.YEAR));
			resultado = year + mes + dia;

		} catch (Exception e) {
			// si hay cualquier problema en la transformacin devuelve cadena vaca
			resultado = "";
		}
		return resultado;
	}

	/**
	 * Saca una traza de log dividiendo la salida en varias lneas cuando la longitud excede del mximo por lnea.
	 * 
	 * @param componente Identifica al componente que enva la traza
	 * @param traceLevel Nivel de traza
	 * @param salidaTraza Texto a sacar en la traza
	 */
	public static void trazaLarga(String componente, Q70TraceLevel traceLevel, String salidaTraza) {
		// si cabe en una linea de traza, sacar y listo
		if (salidaTraza.length() < MAX_LENGTH_TRAZA) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(componente, traceLevel, salidaTraza);
			}
		} else {
			// si es mayor que el tamao mximo para una lnea de traza, lo dividimos en varias:
			String lineaTraza = salidaTraza.substring(0, MAX_LENGTH_TRAZA);
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(componente, traceLevel, lineaTraza);
			}
			salidaTraza = salidaTraza.substring(MAX_LENGTH_TRAZA);
			// repetimos recursivamente
			trazaLarga(componente, traceLevel, salidaTraza);
		}
	}

	// /////////////////////////////////////////// Aadidos de AA80T //////////////////////////////////

	/**
	 * Obtiene el content type en funcin del tipo de recurso.
	 * 
	 * @param tipoRecurso Tipo de recurso
	 * 
	 * @return content type
	 */
	public static String getContentType(String tipoRecurso) {
		if (tipoRecurso == null) {
			return null;
		}

		if (tipoRecurso.length() > 0) {
			String contentTypeKey = tipoRecurso.substring(0, 1).toUpperCase();
			// return Constantes.CONTENT_TYPE_MAP.get(contentTypeKey);
			return S12Constantes.getContentType(contentTypeKey);
		} else {
			return null;
		}
	}

	/**
	 * Obtiene el nombre del contenido en el Portal a partir del cdigo de recurso y del tipo.
	 * 
	 * @param codigoRecurso Cdigo de recurso
	 * @param tipoRecurso Tipo de recurso
	 * 
	 * @return Nombre del contenido en el Gestor de Contenidos
	 */
	public static String getNombreContenido(Long codigoRecurso, String tipoRecurso) {

		StringBuilder nombreContenido = new StringBuilder(Long.toString(codigoRecurso));
		while (nombreContenido.length() < MAX_LENGTH_NOMBRE_CONT) {
			nombreContenido.insert(0, "0");
		}
		nombreContenido.append("_");
		nombreContenido.append(tipoRecurso.toLowerCase());
		nombreContenido.append("_rec_turismo");
		return nombreContenido.toString();
	}

	/**
	 * Obtiene la URL relativa al dominio del contenido.
	 * 
	 * @param codigoRecurso Cdigo de recurso
	 * @param tipoRecurso Tipo de recurso
	 * @param idioma Idioma [es | eu | en | fr | de ]
	 * 
	 * @return URL relativa (sin dominio) del contenido
	 */
	public static String getURLContenido(Long codigoRecurso, String tipoRecurso, String idioma) {

		StringBuilder pathContenido = new StringBuilder();
		String typologyRecurso = getContentType(tipoRecurso);

		pathContenido.append("/contenidos/");
		pathContenido.append(typologyRecurso);
		pathContenido.append("/");
		pathContenido.append(getNombreContenido(codigoRecurso, tipoRecurso));
		pathContenido.append("/");
		pathContenido.append(idioma);
		pathContenido.append("_");
		pathContenido.append(codigoRecurso);

		return pathContenido.toString();
	}

	/**
	 * Obtiene la URL relativa al dominio de la ficha de un recurso.
	 * 
	 * @param codigoRecurso Cdigo de recurso
	 * @param tipoRecurso Tipo de recurso
	 * @param idioma Idioma [es | eu | en | fr | de ]
	 * @param top Indica si el recurso es o no Top
	 * @param importancia Indica la importancia asignada al recurso [ null | 1 - 9 ]
	 * 
	 * @return URL relativa (sin dominio) de la ficha
	 */
	public static String getURLFicha(Long codigoRecurso, String tipoRecurso, String idioma, Boolean top,
			Integer importancia) {

		StringBuilder pathFicha = new StringBuilder();

		// Si la ficha es Top tiene su propia pgina independientemente del tipo
		if (top != null && top) {
			pathFicha.append("/" + S12Constantes.PAGINA_FICHA_TOP + "/");
		}
		// Preguntamos el tipo de recurso, para mostrar la ficha en su pgina correspondiente
		else if (S12Constantes.EXPERIENCIAS.equals(tipoRecurso)) {
			pathFicha.append("/" + S12Constantes.PAGINA_FICHA_EXPERIENCIAS + "/");
		} else if (S12Constantes.RUTAS.equals(tipoRecurso) && importancia != null && 1 == importancia) {
			// Si es una ruta importante tiene su propia pgina
			pathFicha.append("/" + S12Constantes.PAGINA_FICHA_RUTA_IMPORTANTE + "/");
		} else {
			pathFicha.append("/" + S12Constantes.PAGINA_FICHA + "/");
		}
		pathFicha.append(idioma);
		pathFicha.append(getURLContenido(codigoRecurso, tipoRecurso, idioma));

		pathFicha.append("/");
		if (S12Constantes.AGENDA.equals(tipoRecurso) || S12Constantes.OFERTAS.equals(tipoRecurso)
				|| S12Constantes.NOVEDADES.equals(tipoRecurso)) {
			pathFicha.append(codigoRecurso).append(".html");
		} else {
			pathFicha.append(codigoRecurso).append(S12Constantes.HTML_FICHA);
		}

		return pathFicha.toString();
	}

	/**
	 * Obtiene la URL relativa al dominio de la ficha de un recurso a partir de su URL amigable
	 * 
	 * @param urlAmigable URL amigable
	 * @param tipoRecurso Tipo de recurso
	 * @param idioma Idioma [es | eu | en | fr | de ]
	 * @param top Indica si el recurso es o no Top
	 * @param importancia Indica la importancia asignada al recurso [ null | 1 - 9 ]
	 * 
	 * @return URL relativa (sin dominio) de la ficha
	 */
	public static String getURLFicha(String urlAmigable, String tipoRecurso, String idioma, Boolean top,
			Integer importancia) {

		StringBuilder pathFicha = new StringBuilder(urlAmigable);

		// Si la ficha es Top tiene su propia pgina independientemente del tipo
		if (top != null && top) {
			pathFicha.append(S12Constantes.PAGINA_FICHA_TOP + "/");
		}
		// Preguntamos el tipo de recurso, para mostrar la ficha en su pgina correspondiente
		else if (S12Constantes.EXPERIENCIAS.equals(tipoRecurso)) {
			pathFicha.append(S12Constantes.PAGINA_FICHA_EXPERIENCIAS + "/");
		} else if (S12Constantes.RUTAS.equals(tipoRecurso) && importancia == 1) {
			// Si es una ruta importante tiene su propia pgina
			pathFicha.append(S12Constantes.PAGINA_FICHA_RUTA_IMPORTANTE + "/");
		} else {
			pathFicha.append(S12Constantes.PAGINA_FICHA + "/");
		}
		pathFicha.append(idioma).append("/");

		return pathFicha.toString();
	}

	/**
	 * Obtiene el valor idioma relacionado con la URL utilizada, comprobando su subdominio
	 * @param url String con una URL del portal de turismo
	 * @return valor de idioma correspondiente o null si la URL no tiene el formato esperado
	 */
	public static String getIdiomaFromRequestURL(String url) {
		String idioma = null;
		if (url != null){
			String host;
			String subdominio;
			try {
				host = new URL(url).getHost();
				subdominio = host.split(Pattern.quote("."))[0];
				if (S12Constantes.getSubdominiosTurismoMap().containsKey(subdominio)){
					idioma = S12Constantes.getSubdominiosTurismoMap().get(subdominio); 
				}
			} catch (Exception e) {
				// ante cualquier problema se devuelve nulo
				idioma = null;
			}
		}
		return idioma;
	}
	
	/**
	 * Obtiene el valor idioma a partir de la r01euskadiCookie, cuyo valor lleva el idioma como sufijo
	 * @param cookies obtenidas de la request
	 * @return valor de idioma correspondiente o null si se da cualquier problema con la cookie (que no exista o el valor no tenga el formato esperado)
	 */
	public static String getIdiomaFromR01euskadiCookie(Cookie[] cookies){
		String idioma = null;
		if (cookies != null){
			for (int i = 0; i < cookies.length; i++) {
				Cookie cookie = cookies[i];
				if (S12Constantes.R01_EUSKADI_COOKIE.equals(cookie.getName())) {
					//el valor de la cookie lleva el idioma como sufijo (ejemplo: aa30_es)
					try {
						String valorCookie = cookie.getValue();
						idioma = idiomaValido(valorCookie.substring(valorCookie.length()-2,valorCookie.length()));
					} catch (Exception e) {
						// ante cualquier problema con el valor de la cookie se devuelve nulo
						idioma = null;
					}
					break;
				}
			}
		}
		return idioma;
	}

	/**
	 * Comprueba si el valor de idioma recibido est entre los idiomas soportados
	 * @param idioma Valor de idioma a validar
	 * @return boolean
	 */
	public static boolean esIdiomaValido(String idioma) {
		return (S12Constantes.IDIOMA_ES.equals(idioma) 
				|| S12Constantes.IDIOMA_EU.equals(idioma) 
				|| S12Constantes.IDIOMA_EN.equals(idioma) 
				|| S12Constantes.IDIOMA_FR.equals(idioma) 
				|| S12Constantes.IDIOMA_DE.equals(idioma));
	}

	/**
	 * Asegura que el valor de idioma recibido est soportado por la aplicacion, devolviendo su valor o nulo en cualquier otro caso. 
	 * @param idioma
	 * @return String con valor de idioma vlido o nulo 
	 */
	public static String idiomaValido(String idioma) {
		if (esIdiomaValido(idioma)){
			return idioma;
		} else {
			return null;
		}
	}

	/**
	 * Comprueba si el valor de zona recibido est entre los soportados
	 * @param zona Valor de zona (marks) a validar
	 * @return boolean
	 */
	public static boolean esZonaValido(String zona) {
		return (S12Constantes.ZONA_COSTAVASCA.equals(zona) 
				|| S12Constantes.ZONA_VERDENATURAL.equals(zona) 
				|| S12Constantes.ZONA_RIOJAALAVESA.equals(zona) 
				|| S12Constantes.ZONA_BILBAO.equals(zona) 
				|| S12Constantes.ZONA_DONOSTI.equals(zona) 
				|| S12Constantes.ZONA_GASTEIZ.equals(zona));
	}

	/**
	 * Asegura que el valor de zona recibido est soportado por la aplicacion, devolviendo su valor o empty String en cualquier otro caso.
	 * El String vaco sera el correspondiente al valor sin seleccion en combos. 
	 * @param idioma
	 * @return String con valor de zona (marks) vlido o espacio 
	 */
	public static String zonaValido(String zona) {
		if (esZonaValido(zona)){
			return zona;
		} else {
			return S12Constantes.ZONA_NOESPECIFICADA;
		}
	}
}
