package s12.portalMovil.valueObjects;

import java.sql.Types;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpServletRequest;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONString;

import com.ejie.r01m.objects.searchengine.query.R01MSearchedMetaData;
import com.ejie.r01m.utils.searchengine.R01MSearchEngineConstants;

import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;

import s12.constantes.S12Constantes;
import s12.portalMovil.dao.S12PortalMovilDAO;
import s12.portalMovil.dao.S12PortalMovilDaoException;
import s12.util.S12Utils;

/**
 * Clase S12RecintoFerial.
 *
 * Representa una restauracin
 *
 *  
 */
public class S12RecintoFerial extends S12RecursoTuristico implements JSONString {

	private final static int SIZE_STRINGBUFFER = 220;
	
	/** Tipo de recurso */
	public final static String TIPO_RECURSO = "N2";

	/** Telfono asociado al recurso N2. */
	private String telefono;

	/**
	 * Constructor.
	 */
	public S12RecintoFerial() {
	}

	/**
	 * @return Retorna el valor del campo telefono.
	 */
	public String getTelefono() {
		return this.telefono;
	}

	/**
	 * @param telefono Establece el valor del campo telefono.
	 */
	public void setTelefono(String telefono) {
		this.telefono = telefono;
	}

	/**
	 * Funcin que crea los metadatos para recursos de tipo N2.
	 *
	 * @param request request
	 *
	 * @return lista de metadatos
	 */
	public static List<R01MSearchedMetaData> crearMetaDataN2(HttpServletRequest request){

		List<R01MSearchedMetaData> metaData = new ArrayList<R01MSearchedMetaData>();

		// Se establece el templateType
	    metaData.add(new R01MSearchedMetaData(S12Constantes.SF_TEMPLATE_TYPE, R01MSearchEngineConstants.EQUALS_OPERATOR, S12RecintoFerial.TIPO_RECURSO));
	
	    String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_MARKS, R01MSearchEngineConstants.LIKE_OPERATOR, zona));
	    }

		String accesible = request.getParameter(S12Constantes.SF_ACCESIBILITY);
		if (accesible != null) {
			metaData.add(new R01MSearchedMetaData(S12Constantes.SF_ACCESIBILITY, R01MSearchEngineConstants.EQUALS_OPERATOR, "1"));
		}

	    // Filtramos los recursos locales
	    metaData.add(new R01MSearchedMetaData(S12Constantes.SF_GENERAL, R01MSearchEngineConstants.EQUALS_OPERATOR, S12Constantes.GENERAL));
	
	    return metaData;
	}

	/**
	 * Funcin que crea la sql de consulta para recursos de tipo N2.
	 *
	 * @param request request
	 *
	 * @return sql de consulta
	 * @deprecated
	 */
	public static String crearSqlN2(HttpServletRequest request) {
		StringBuffer sql = new StringBuffer(S12RecintoFerial.SIZE_STRINGBUFFER);

		sql.append(S12Constantes.PM_TEMPLATE_TYPE);
		sql.append(" = '");
		sql.append(S12RecintoFerial.TIPO_RECURSO);
		sql.append("' AND ");

		String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
			sql.append(S12Constantes.PM_MARKS);
			sql.append(" LIKE '%");
			sql.append(zona);
			sql.append("%' AND ");
		}

	    String accesible = request.getParameter(S12Constantes.SF_ACCESIBILITY);
	    if (accesible != null) {
	    	sql.append(S12Constantes.PM_ACCESIBILITY);
	  		sql.append(" IS NOT NULL AND ");
	    }

		sql.append(" LANGUAGE = '");
		sql.append(request.getParameter("language"));
		sql.append("'");
	    sql.append(" ORDER BY ");
	    sql.append(S12Constantes.PM_IMPORTANCE);
	    sql.append(" ASC, ");
	    sql.append(S12Constantes.PM_NAME);
	    sql.append(" ASC ");

		return sql.toString();
	}


	/**
	 * Funcin para obtener la lista de recursos de restauracin.
	 *
	 * @param arrRecursos arrRecursos
	 * @param idioma idioma
	 * @param latitud latitud
	 * @param longitud longitud
	 * @param numero_resultados numero_resultados
	 * @param ultimo_resultado ultimo_resultado
	 *
	 * @return lista de recursos de restauracin
	 */
	public static String getRecintoFerialN2(ArrayList<String> arrRecursos, String idioma, double latitud, double longitud, int numero_resultados, int ultimo_resultado) {
		StringBuffer sRespuesta = new StringBuffer();

		S12PortalMovilDAO portalMovilDAO = new S12PortalMovilDAO();

		try {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.DEBUG, "Generando JSON de Recinto Ferial - N2");
			}
			S12RecintoFerial[] recintoFerialArray = portalMovilDAO.getRecintoFerial(arrRecursos, idioma, latitud, longitud);

			try {
				sRespuesta.append(new JSONArray(recintoFerialArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.ERROR, "S12RecintoFerial::getRecintoFerialN2: Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}

		} catch (S12PortalMovilDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.ERROR,"S12RecintoFerial::getRecintoFerialN2: Exception: " + e.getMessage());
			}
		}
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.DEBUG,"FIN Generando JSON de Recinto Ferial - N2");
		}

		return sRespuesta.toString();
	}


	/**
	 * Implementacin de la serializacin JSON del objeto
	 *
	 * @return Serializacin JSON del objeto
	 *
	 * @see org.json.JSONString#toJSONString()
	 */
	public String toJSONString() {
		JSONObject jObject = new JSONObject();
		try {

			jObject.put("name", S12Utils.encodeHtmlTag(this.getNombre()));
			
			if (this.getDistancia() != null) {
				// Formatear la distancia con un decimal
				NumberFormat nf = NumberFormat.getNumberInstance(new Locale(this.getIdioma()));
				DecimalFormat df = (DecimalFormat) nf;
				df.applyPattern("###.#");
				
				jObject.put("distance", df.format(Double.parseDouble(this.getDistancia())));
			}

			if (this.getFotoPrincipal() != null) {
				StringBuffer urlFotoPrincipal = new StringBuffer();
				urlFotoPrincipal.append(S12Utils.getURLContenido(this.getCodigoRecurso(), S12RecintoFerial.TIPO_RECURSO, null));
				urlFotoPrincipal.append("images/");
				urlFotoPrincipal.append(this.getFotoPrincipal());

				jObject.put("image", urlFotoPrincipal);
			}

			JSONObject jObjectLatLong = new JSONObject();
			jObjectLatLong.put("coordX", this.getGmlong());
			jObjectLatLong.put("coordY", this.getGmlat());
			jObject.put("location", jObjectLatLong);

			jObject.put("merkur_url", S12Utils.getURLMerkur(this.getCodigoRecurso(), S12RecintoFerial.TIPO_RECURSO, this.getUrlAmigable(), this.getIdioma()));

			jObject.put("phone_number", this.telefono);

		} catch (JSONException e) {
			e.printStackTrace();
		}

		return jObject.toString();
	}

	/**
	 * Adaptacion del metodo {@link #crearSqlN2(HttpServletRequest) crearSqlN2} que parametriza la consulta para construir un PreparedStatement en el DAO
	 * @param request	request recibida del servlet, tendra los parametros a consultar
	 * @param params	array de objetos con los valores a sustituir
	 * @param tiposSQL	array de tipos SQL para usar en el PreparedStatement
	 * @return	String con la where parametrizada
	 */
	public static String prepararSqlN2(HttpServletRequest request,
			ArrayList<Object> params, ArrayList<Integer> tiposSQL) {
		StringBuffer sql = new StringBuffer(S12RecintoFerial.SIZE_STRINGBUFFER);

		sql.append(S12Constantes.PM_TEMPLATE_TYPE);
		sql.append(" = '");
		sql.append(S12RecintoFerial.TIPO_RECURSO);
		sql.append("' AND ");

	    String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
			sql.append(S12Constantes.PM_MARKS);
	  		sql.append(" LIKE ? AND ");
	  		String expresionLIKE = "%"+zona+"%";
			params.add(expresionLIKE);
			tiposSQL.add(Types.VARCHAR);
		}

	    String accesible = request.getParameter(S12Constantes.SF_ACCESIBILITY);
	    if (accesible != null) {
	    	sql.append(S12Constantes.PM_ACCESIBILITY);
	  		sql.append(" IS NOT NULL AND ");
	    }

		sql.append(" LANGUAGE = ?");
		String language = request.getParameter("language");
		params.add(language);
		tiposSQL.add(Types.VARCHAR);
	    sql.append(" ORDER BY ");
	    sql.append(S12Constantes.PM_IMPORTANCE);
	    sql.append(" ASC, ");
	    sql.append(S12Constantes.PM_NAME);
	    sql.append(" ASC ");

		return sql.toString();
	}

}
