package s12.portalMovil.valueObjects;

import java.sql.Types;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpServletRequest;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONString;

import com.ejie.r01m.objects.searchengine.query.R01MSearchedMetaData;
import com.ejie.r01m.utils.searchengine.R01MSearchEngineConstants;

import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;

import s12.constantes.S12Constantes;
import s12.portalMovil.dao.S12PortalMovilDAO;
import s12.portalMovil.dao.S12PortalMovilDaoException;
import s12.util.S12Utils;

/**
 * Clase S12PlazasToros.
 *
 * Representa un Plaza de Toros
 *
 *  
 */
public class S12PlazasToros extends S12RecursoTuristico implements JSONString {
	
	private final static int SIZE_STRINGBUFFER = 220;
	
	/** Tipo de recurso */
	public final static String TIPO_RECURSO = "K1";

	/** Id Categora asociado al recurso K1. */
	private String idCategoria;

	/** Nombre de la Categora asociado al recurso K1. */
	private String nombreCategoria;

	/**
	 * Constructor.
	 */
	public S12PlazasToros() {
	}

	/**
	 * @return Retorna el valor del campo idCategoria.
	 */
	public String getIdCategoria() {
		return this.idCategoria;
	}

	/**
	 * @param idCategoria Establece el valor del campo idCategoria.
	 */
	public void setIdCategoria(String idCategoria) {
		this.idCategoria = idCategoria;
	}

	/**
	 * @return Retorna el valor del campo nombreCategoria.
	 */
	public String getNombreCategoria() {
		return this.nombreCategoria;
	}

	/**
	 * @param nombreCategoria Establece el valor del campo nombreCategoria.
	 */
	public void setNombreCategoria(String nombreCategoria) {
		this.nombreCategoria = nombreCategoria;
	}

	/**
	 * Funcin que crea los metadatos para recursos de tipo K1.
	 *
	 * @param request request
	 *
	 * @return lista de metadatos
	 */
	public static List<R01MSearchedMetaData> crearMetaDataK1(HttpServletRequest request){

		List<R01MSearchedMetaData> metaData = new ArrayList<R01MSearchedMetaData>();

		// Se establece el templateType
	    metaData.add(new R01MSearchedMetaData(S12Constantes.SF_TEMPLATE_TYPE, R01MSearchEngineConstants.EQUALS_OPERATOR, S12PlazasToros.TIPO_RECURSO));
	
	    String categoria = request.getParameter(S12Constantes.SF_CATEGORY);
	    if(categoria != null && !"".equals(categoria)){
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_CATEGORY, R01MSearchEngineConstants.EQUALS_OPERATOR, categoria));
	    }
	
	    String municipality = request.getParameter(S12Constantes.SF_MUNICIPALITY); //ESTO HAY QUE MIRARLO
	    if (municipality != null && !"".equals(municipality)) {
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_MUNICIPALITY, R01MSearchEngineConstants.LIKE_OPERATOR, municipality.toUpperCase(new Locale("es","es"))));
	    }
	    String municipalityCode = request.getParameter(S12Constantes.SF_MUNICIPALITY_CODE); 
	    if (municipalityCode != null && !"".equals(municipalityCode)) {
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_MUNICIPALITY_CODE, R01MSearchEngineConstants.LIKE_OPERATOR, municipalityCode.toUpperCase(new Locale("es","es"))));
	    }
	    String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_MARKS, R01MSearchEngineConstants.LIKE_OPERATOR, zona));
	    }

		// Filtramos los recursos locales
	    metaData.add(new R01MSearchedMetaData(S12Constantes.SF_GENERAL, R01MSearchEngineConstants.EQUALS_OPERATOR, S12Constantes.GENERAL));
	
	    return metaData;
	}

	/**
	 * Funcin que crea la sql de consulta para recursos de tipo K1.
	 *
	 * @param request request
	 *
	 * @return sql de consulta
	 * @deprecated
	 */
	public static String crearSqlK1(HttpServletRequest request) {
		StringBuffer sql = new StringBuffer(S12PlazasToros.SIZE_STRINGBUFFER);

		sql.append(S12Constantes.PM_TEMPLATE_TYPE);
		sql.append(" = '");
		sql.append(S12PlazasToros.TIPO_RECURSO);
		sql.append("' AND ");


		String categoria = request.getParameter(S12Constantes.SF_CATEGORY);
		if(categoria != null && !"".equals(categoria)){
			sql.append(S12Constantes.PM_CATEGORY);
			sql.append(" = '");
			sql.append(categoria);
			sql.append("' AND ");
		}

		String municipality = request.getParameter(S12Constantes.SF_MUNICIPALITY);
	    if (municipality != null && !"".equals(municipality)) {
	    	sql.append(S12Constantes.PM_MUNICIPALITY);
	  		sql.append(" LIKE '%");
	  		municipality = S12Utils.utf8toIso88591(municipality);
	  		sql.append(municipality.toUpperCase(new Locale("es","es")));
	  		sql.append("%' AND ");
	    }
	
	    String municipalityCode = request.getParameter(S12Constantes.SF_MUNICIPALITY_CODE); 
	    if (municipalityCode != null && !"".equals(municipalityCode)) {
	    	sql.append(S12Constantes.PM_MUNICIPALITY_CODE);
	  		sql.append(" LIKE '%");
	  		sql.append(municipalityCode.toUpperCase(new Locale("es","es")));
	  		sql.append("%' AND ");
	    }
	    
	    String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
			sql.append(S12Constantes.PM_MARKS);
			sql.append(" LIKE '%");
			sql.append(zona);
			sql.append("%' AND ");
		}

		sql.append(" LANGUAGE = '");
		sql.append(request.getParameter("language"));
		sql.append("'");
	    sql.append(" ORDER BY ");
	    sql.append(S12Constantes.PM_IMPORTANCE);
	    sql.append(" ASC, ");
	    sql.append(S12Constantes.PM_NAME);
	    sql.append(" ASC ");

		return sql.toString();
	}


	/**
	 * Funcin para obtener la lista de recursos de Plazas de Toros.
	 *
	 * @param arrRecursos arrRecursos
	 * @param idioma idioma
	 * @param latitud latitud
	 * @param longitud longitud
	 * @param numero_resultados numero_resultados
	 * @param ultimo_resultado ultimo_resultado
	 *
	 * @return lista de recursos de Plazas de Toros
	 */
	public static String getPlazasTorosK1(ArrayList<String> arrRecursos, String idioma, double latitud, double longitud, int numero_resultados, int ultimo_resultado) {
		StringBuffer sRespuesta = new StringBuffer();

		S12PortalMovilDAO portalMovilDAO = new S12PortalMovilDAO();

		try {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.DEBUG, "Generando JSON de Plazas de Toros - K1");
			}
			S12PlazasToros[] plazasTorosArray = portalMovilDAO.getPlazasToros(arrRecursos, idioma, latitud, longitud);

			try {
				sRespuesta.append(new JSONArray(plazasTorosArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.ERROR, "S12PlazasToros::getPlazasTorosK1: Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}

		} catch (S12PortalMovilDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.ERROR,"S12PlazasToros::getPlazasTorosK1: Exception: " + e.getMessage());
			}
		}
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.DEBUG,"FIN Generando JSON de Plazas de Toros - K1");
		}

		return sRespuesta.toString();
	}


	/**
	 * Implementacin de la serializacin JSON del objeto
	 *
	 * @return Serializacin JSON del objeto
	 *
	 * @see org.json.JSONString#toJSONString()
	 */
	public String toJSONString() {
		JSONObject jObject = new JSONObject();
		try {

			jObject.put("name", S12Utils.encodeHtmlTag(this.getNombre()));
			
			if (this.getDistancia() != null) {
				// Formatear la distancia con un decimal
				NumberFormat nf = NumberFormat.getNumberInstance(new Locale(this.getIdioma()));
				DecimalFormat df = (DecimalFormat) nf;
				df.applyPattern("###.#");
				
				jObject.put("distance", df.format(Double.parseDouble(this.getDistancia())));
			}

			if (this.getFotoPrincipal() != null) {
				StringBuffer urlFotoPrincipal = new StringBuffer();
				urlFotoPrincipal.append(S12Utils.getURLContenido(this.getCodigoRecurso(), S12PlazasToros.TIPO_RECURSO, null));
				urlFotoPrincipal.append("images/");
				urlFotoPrincipal.append(this.getFotoPrincipal());

				jObject.put("image", urlFotoPrincipal);
			}

			JSONObject jObjectLatLong = new JSONObject();
			jObjectLatLong.put("coordX", this.getGmlong());
			jObjectLatLong.put("coordY", this.getGmlat());
			jObject.put("location", jObjectLatLong);

			jObject.put("merkur_url", S12Utils.getURLMerkur(this.getCodigoRecurso(), S12PlazasToros.TIPO_RECURSO, this.getUrlAmigable(), this.getIdioma()));

			jObject.put("categoria_id", this.getIdCategoria());
			jObject.put("categoria", S12Utils.encodeHtmlTag(this.getNombreCategoria()));

		} catch (JSONException e) {
			e.printStackTrace();
		}

		return jObject.toString();
	}

	/**
	 * Adaptacion del metodo {@link #crearSqlK1(HttpServletRequest) crearSqlK1} que parametriza la consulta para construir un PreparedStatement en el DAO
	 * @param request	request recibida del servlet, tendra los parametros a consultar
	 * @param params	array de objetos con los valores a sustituir
	 * @param tiposSQL	array de tipos SQL para usar en el PreparedStatement
	 * @return	String con la where parametrizada
	 */
	public static String prepararSqlK1(HttpServletRequest request,
			ArrayList<Object> params, ArrayList<Integer> tiposSQL) {
		StringBuffer sql = new StringBuffer(S12PlazasToros.SIZE_STRINGBUFFER);

		sql.append(S12Constantes.PM_TEMPLATE_TYPE);
		sql.append(" = '");
		sql.append(S12PlazasToros.TIPO_RECURSO);
		sql.append("' AND ");


		String categoria = request.getParameter(S12Constantes.SF_CATEGORY);
		if(categoria != null && !"".equals(categoria)){
			sql.append(S12Constantes.PM_CATEGORY);
			sql.append(" = ? AND ");
			params.add(categoria);
			tiposSQL.add(Types.VARCHAR);
		}

		String municipality = request.getParameter(S12Constantes.SF_MUNICIPALITY);
	    if (municipality != null && !"".equals(municipality)) {
	    	sql.append(S12Constantes.PM_MUNICIPALITY);
	  		sql.append(" LIKE ? AND ");
	  		municipality = S12Utils.utf8toIso88591(municipality);
	  		municipality = municipality.toUpperCase(new Locale("es","es"));
	  		String expresionLIKE = "%"+municipality+"%";
			params.add(expresionLIKE);
			tiposSQL.add(Types.VARCHAR);
	    }
	    String municipalityCode = request.getParameter(S12Constantes.SF_MUNICIPALITY_CODE); 
	    if (municipalityCode != null && !"".equals(municipalityCode)) {
	    	sql.append(S12Constantes.PM_MUNICIPALITY_CODE);
	  		sql.append(" LIKE ? AND ");
	  		municipalityCode = municipalityCode.toUpperCase(new Locale("es","es"));
	  		String expresionLIKE = "%"+municipalityCode+"%";
			params.add(expresionLIKE);
			tiposSQL.add(Types.VARCHAR);
	    }
	    
	    String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
			sql.append(S12Constantes.PM_MARKS);
	  		sql.append(" LIKE ? AND ");
	  		String expresionLIKE = "%"+zona+"%";
			params.add(expresionLIKE);
			tiposSQL.add(Types.VARCHAR);
		}

		sql.append(" LANGUAGE = ?");
		String language = request.getParameter("language");
		params.add(language);
		tiposSQL.add(Types.VARCHAR);
	    sql.append(" ORDER BY ");
	    sql.append(S12Constantes.PM_IMPORTANCE);
	    sql.append(" ASC, ");
	    sql.append(S12Constantes.PM_NAME);
	    sql.append(" ASC ");

		return sql.toString();
	}

}
