package s12.portalMovil.valueObjects;

import java.sql.Types;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.servlet.http.HttpServletRequest;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONString;

import com.ejie.r01m.objects.searchengine.query.R01MSearchedMetaData;
import com.ejie.r01m.utils.searchengine.R01MSearchEngineConstants;

import es.ejie.frmk.Q70Factoria;
import es.ejie.frmk.infraestructura.internacionalizacion.Q70GestorMensajesException;
import es.ejie.frmk.infraestructura.internacionalizacion.Q70GestorMensajesi18n;
import es.ejie.frmk.infraestructura.traza.Q70TraceLevel;

import s12.constantes.S12Constantes;
import s12.portalMovil.dao.S12PortalMovilDAO;
import s12.portalMovil.dao.S12PortalMovilDaoException;
import s12.util.S12Utils;

/**
 * Clase S12Circuito.
 *
 * Representa un Circuito
 *
 *  
 */
public class S12Circuito extends S12RecursoTuristico implements JSONString {

	
	/** Tipo de recurso */
	public final static String TIPO_RECURSO = "D4";

	/** The Constant SIZE_STRINGBUFFER. */
	private final static int SIZE_STRINGBUFFER = 64;

	/** Duracin del circuito */
	private String duracion;

	/** Transporte del circuito */
	private String transporte;

	/** Coordenadas del circuito (etapas) */
	private String gmruta;

	/** Temtica */
	private String idTematica;

	/**
	 * Constructor.
	 */
	public S12Circuito() {
	}

	/**
	 * Retorna el valor del campo duracion.
	 *
	 * @return duracion del circuito
	 */
	public String getDuracion() {
		return this.duracion;
	}

	/**
	 * Establece el valor del campo duracion.
	 *
	 * @param duracion duracion del circuito
	 */
	public void setDuracion(String duracion) {
		this.duracion = duracion;
	}

	/**
	 * Retorna el valor del campo transporte.
	 *
	 * @return transporte del circuito
	 */
	public String getTransporte() {
		return this.transporte;
	}

	/**
	 * Establece el valor del campo transporte.
	 *
	 * @param transporte transporte del circuito.
	 */
	public void setTransporte(String transporte) {
		this.transporte = transporte;
	}


	/**
	 * Retorna el valor del campo gmruta.
	 *
	 * @return gmruta del circuito
	 */
	public String getGmruta() {
		return this.gmruta;
	}

	/**
	 * Establece el valor del campo gmruta.
	 *
	 * @param gmruta del circuito.
	 */
	public void setGmruta(String gmruta) {
		this.gmruta = gmruta;
	}

	/**
	 * Retorna el valor del campo idTematica.
	 *
	 * @return temtica del circuito
	 */
	public String getIdTematica() {
		return this.idTematica;
	}

	/**
	 * Establece el valor del campo idTematica.
	 *
	 * @param idTematica temtica del circuito
	 */
	public void setIdTematica(String idTematica) {
		this.idTematica = idTematica;
	}

	/**
	 * Funcin que crea los metadatos para recursos de tipo D42.
	 *
	 * @param request request
	 *
	 * @return lista de metadatos
	 */
	public static List<R01MSearchedMetaData> crearMetaDataD4(HttpServletRequest request) {

		List<R01MSearchedMetaData> metaData = new ArrayList<R01MSearchedMetaData>();

		// Se establece el templateType
		metaData.add(new R01MSearchedMetaData(S12Constantes.SF_TEMPLATE_TYPE, R01MSearchEngineConstants.EQUALS_OPERATOR, S12Circuito.TIPO_RECURSO));

		// Se establece routeType=2 para que filtre solo los circuitos
		metaData.add(new R01MSearchedMetaData("routeType", R01MSearchEngineConstants.EQUALS_OPERATOR, "2"));

		// Recogida de parmetros del mtodo
		String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_MARKS, R01MSearchEngineConstants.LIKE_OPERATOR, zona));
	    }

		String tematica = request.getParameter(S12Constantes.SF_TYPE);
		if (tematica != null && !"".equals(tematica)) {
			metaData.add(new R01MSearchedMetaData(S12Constantes.SF_TYPE, R01MSearchEngineConstants.LIKE_OPERATOR, tematica));
	    } 

		String transporte = request.getParameter(S12Constantes.SF_TRANSPORT_TYPE);
		if (transporte != null && !"".equals(transporte)) {
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_TRANSPORT_TYPE, R01MSearchEngineConstants.LIKE_OPERATOR, transporte));
	    }

		String duracion = request.getParameter(S12Constantes.SF_DURATION);
		if (duracion != null && !"".equals(duracion)) {
			StringBuffer strDuracion = new StringBuffer();
			if ("0001".equals(duracion)) {
				strDuracion.append("(02,03)");
			} else if ("0002".equals(duracion)) {
				strDuracion.append("(04,05)");
			} else if ("0003".equals(duracion)) {
				strDuracion.append("(06,07)");
			}
	    	metaData.add(new R01MSearchedMetaData(S12Constantes.SF_DURATION, R01MSearchEngineConstants.IN_OPERATOR, strDuracion.toString()));
		}

		String idealPara = request.getParameter(S12Constantes.SF_PARA);
		if (idealPara != null && !"".equals(idealPara)) {
			if ("205".equals(idealPara)) {
				metaData.add(new R01MSearchedMetaData(S12Constantes.SF_PARA_PAREJA, R01MSearchEngineConstants.EQUALS_OPERATOR, "1"));
			} else if ("206".equals(idealPara)) {
				metaData.add(new R01MSearchedMetaData(S12Constantes.SF_PARA_NINIOS, R01MSearchEngineConstants.EQUALS_OPERATOR, "1"));
			} else if ("207".equals(idealPara)) {
				metaData.add(new R01MSearchedMetaData(S12Constantes.SF_PARA_AMIGOS, R01MSearchEngineConstants.EQUALS_OPERATOR, "1"));
			}		
		}

		// Filtramos los recursos locales
	    metaData.add(new R01MSearchedMetaData(S12Constantes.SF_GENERAL, "EQ", S12Constantes.GENERAL));
	
	    return metaData;
	}

	/**
	 * Crear sql d4.
	 *
	 * @param request the request
	 *
	 * @return the string
	 * @deprecated
	 */
	public static String crearSqlD4(HttpServletRequest request) {
		StringBuffer sql = new StringBuffer(S12Circuito.SIZE_STRINGBUFFER);

		sql.append(S12Constantes.PM_TEMPLATE_TYPE);
		sql.append(" = '");
		sql.append(S12Circuito.TIPO_RECURSO);
		sql.append("' AND ");
		
		// Se establece CATEGORIA=2 para que filtre solo los circuitos
		sql.append(S12Constantes.PM_CATEGORY);
		sql.append(" = '2' AND ");

		String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
			sql.append(S12Constantes.PM_MARKS);
			sql.append(" LIKE '%");
			sql.append(zona);
			sql.append("%' AND ");
		}

		String transporte = request.getParameter(S12Constantes.SF_TRANSPORT_TYPE);
		if (transporte != null && !"".equals(transporte)) {
			sql.append(S12Constantes.PM_TRANSPORT_TYPE);
			sql.append(" LIKE '%");
			sql.append(transporte);
			sql.append("%' AND ");
		}

		String duration = request.getParameter(S12Constantes.SF_DURATION);
		if (duration != null && !"".equals(duration)) {
			StringBuffer strDuracion = new StringBuffer();
			if ("0001".equals(duration)) {
				strDuracion.append("02,03");
			} else if ("0002".equals(duration)) {
				strDuracion.append("04,05");
			} else if ("0003".equals(duration)) {
				strDuracion.append("06,07");
			}
			sql.append(S12Constantes.PM_DURATION);
			sql.append(" IN ");
			sql.append(duration);
			sql.append(" AND ");
		}
		
		String idealPara = request.getParameter(S12Constantes.SF_PARA);
		if (idealPara != null && !"".equals(idealPara)) {
			if ("205".equals(idealPara)) {
				sql.append(S12Constantes.PM_COUPLE);
				sql.append(" IS NOT NULL AND ");
			} else if ("206".equals(idealPara)) {
				sql.append(S12Constantes.PM_CHILDREN);
				sql.append(" IS NOT NULL AND ");
			} else if ("207".equals(idealPara)) {
				sql.append(S12Constantes.PM_FRIENDS);
				sql.append(" IS NOT NULL AND ");
			}		
		}

		String tematica = request.getParameter(S12Constantes.SF_TYPE);
		if (tematica != null && !"".equals(tematica)) {
			sql.append(S12Constantes.PM_TYPE);
			sql.append(" LIKE '%");
			sql.append(tematica);
			sql.append("%' AND ");
	  	}

		
		sql.append(" LANGUAGE = '");
		sql.append(request.getParameter("language"));
		sql.append("'");
	    sql.append(" ORDER BY ");
	    sql.append(S12Constantes.PM_IMPORTANCE);
	    sql.append(" ASC, ");
	    sql.append(S12Constantes.PM_NAME);
	    sql.append(" ASC ");

		return sql.toString();
	}


	/**
	 * Funcin para obtener la lista de circuitos.
	 *
	 * @param arrCircuitos array de circuitos
	 * @param idioma idioma
	 * @param latitud latitud
	 * @param longitud longitud
	 * @param numero_resultados numero resultados a devolver (paginacin)
	 * @param ultimo_resultado ultimo resultado devuelto (paginacin)
	 *
	 * @return lista de circuitos
	 */
	public static String getCircuitosD4(ArrayList<String> arrCircuitos, String idioma, double latitud, double longitud, int numero_resultados, int ultimo_resultado) {
		StringBuffer sRespuesta = new StringBuffer();

		S12PortalMovilDAO portalMovilDAO = new S12PortalMovilDAO();

		try {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.DEBUG, "Generando JSON de Circutos - D42");
			}
			S12Circuito[] circuitosArray = portalMovilDAO.getCircuitos(arrCircuitos, idioma, latitud, longitud);

			try {
				sRespuesta.append(new JSONArray(circuitosArray).toString());
			} catch (JSONException e) {
				if (Q70Factoria.getTraza().isTraceActive()) {
					Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.ERROR, "S12Circuito::getCircuitosD42: Exception: " + e.getMessage());
				}
				sRespuesta.append("[]");
			}

		} catch (S12PortalMovilDaoException e) {
			if (Q70Factoria.getTraza().isTraceActive()) {
				Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.ERROR, "S12Circuito::getCircuitosD42: Exception: " + e.getMessage());
			}
		}
		if (Q70Factoria.getTraza().isTraceActive()) {
			Q70Factoria.getTraza().trace(S12RecursoTuristico.COMPONENTE, Q70TraceLevel.DEBUG, "FIN Generando JSON de Circutos - D42");
		}

		return sRespuesta.toString();
	}


	/**
	 * Implementacin de la serializacin JSON del objeto
	 *
	 * @return Serializacin JSON del objeto
	 *
	 * @see org.json.JSONString#toJSONString()
	 */
	public String toJSONString() {
		JSONObject jObject = new JSONObject();
		try {
			jObject.put("name", S12Utils.encodeHtmlTag(this.getNombre()));

			if (this.getFotoPrincipal() != null) {
				StringBuffer urlFotoPrincipal = new StringBuffer();
				urlFotoPrincipal.append(S12Utils.getURLContenido(this.getCodigoRecurso(), S12Circuito.TIPO_RECURSO, null));
				urlFotoPrincipal.append("images/");
				urlFotoPrincipal.append(this.getFotoPrincipal());

				jObject.put("image", urlFotoPrincipal);
			}

			if (this.gmruta != null) {
				// Puntos de la ruta
				String gmRutaTmp = this.gmruta.replaceAll("\\[", "").replaceAll("\\]", "");
				String[] arrPuntos = gmRutaTmp.split(",");

				JSONArray jArrayLatLong = new JSONArray();

				String longitud = "";
				String latitud = "";
				for (int i = 0; i < arrPuntos.length; i++) {
					if ((i % 2) == 0) {
						latitud = arrPuntos[i];
					} else {
						JSONObject jObjectLatLong = new JSONObject();
						longitud = arrPuntos[i];
						jObjectLatLong.put("coordX", longitud);
						jObjectLatLong.put("coordY", latitud);

						jArrayLatLong.put(jObjectLatLong);
					}
				}

				jObject.put("location", jArrayLatLong);
			} else {
				// Punto de origen del circuito
				JSONObject jObjectLatLong = new JSONObject();
				jObjectLatLong.put("coordX", this.getGmlong());
				jObjectLatLong.put("coordY", this.getGmlat());

				jObject.put("location", jObjectLatLong);
				///////////
			}

			jObject.put("merkur_url", S12Utils.getURLMerkur(this.getCodigoRecurso(), S12Circuito.TIPO_RECURSO, this.getUrlAmigable(), this.getIdioma()));

			if (this.getDistancia() != null) {
				// Formatear la distancia con un decimal
				NumberFormat nf = NumberFormat.getNumberInstance(new Locale(this.getIdioma()));
				DecimalFormat df = (DecimalFormat) nf;
				df.applyPattern("###.#");
				
				jObject.put("distance", df.format(Double.parseDouble(this.getDistancia())));
			}

			jObject.put("duracion", this.duracion);
			jObject.put("tematica_id", this.idTematica);

			if (this.idTematica != null) {
				String[] vIdsTematica = this.idTematica.split(",");
				StringBuffer nombreTematica = new StringBuffer();
				for (int i = 0; i < vIdsTematica.length; i++) {
					if (i > 0) {
						nombreTematica.append(",");
					}
					nombreTematica.append(S12Utils.encodeHtmlTag(Q70GestorMensajesi18n.getSingleton().getMessage("s12PortalMovil", "d4." + vIdsTematica[i], this.getIdioma())));					
				}
				
				jObject.put("tematica", nombreTematica.toString());
			}

		} catch (JSONException e) {
			e.printStackTrace();
		} catch (Q70GestorMensajesException e) {
			e.printStackTrace();
		}

		return jObject.toString();
	}

	/**
	 * Adaptacion del metodo {@link #crearSqlD4(HttpServletRequest) crearSqlD4} que parametriza la consulta para construir un PreparedStatement en el DAO
	 * @param request	request recibida del servlet, tendra los parametros a consultar
	 * @param params	array de objetos con los valores a sustituir
	 * @param tiposSQL	array de tipos SQL para usar en el PreparedStatement
	 * @return	String con la where parametrizada
	 */
	public static String prepararSqlD4(HttpServletRequest request,
			ArrayList<Object> params, ArrayList<Integer> tiposSQL) {
		StringBuffer sql = new StringBuffer(S12Circuito.SIZE_STRINGBUFFER);

		sql.append(S12Constantes.PM_TEMPLATE_TYPE);
		sql.append(" = '");
		sql.append(S12Circuito.TIPO_RECURSO);
		sql.append("' AND ");
		
		// Se establece CATEGORIA=2 para que filtre solo los circuitos
		sql.append(S12Constantes.PM_CATEGORY);
		sql.append(" = '2' AND ");

	    String zona = request.getParameter(S12Constantes.SF_MARKS);
		if (zona != null && !"".equals(zona)) {
			sql.append(S12Constantes.PM_MARKS);
	  		sql.append(" LIKE ? AND ");
	  		String expresionLIKE = "%"+zona+"%";
			params.add(expresionLIKE);
			tiposSQL.add(Types.VARCHAR);
		}

		String transporte = request.getParameter(S12Constantes.SF_TRANSPORT_TYPE);
		if (transporte != null && !"".equals(transporte)) {
			sql.append(S12Constantes.PM_TRANSPORT_TYPE);
			sql.append(" = ? AND ");
			params.add(transporte);
			tiposSQL.add(Types.VARCHAR);
		}

		String duration = request.getParameter(S12Constantes.SF_DURATION);
		if (duration != null && !"".equals(duration)) {
			StringBuffer strDuracion = new StringBuffer();
			if ("0001".equals(duration)) {
				strDuracion.append("02,03");
			} else if ("0002".equals(duration)) {
				strDuracion.append("04,05");
			} else if ("0003".equals(duration)) {
				strDuracion.append("06,07");
			}
			sql.append(S12Constantes.PM_DURATION);
			sql.append(" IN ");
			sql.append(duration);
			sql.append(" AND ");
		}
		
		String idealPara = request.getParameter(S12Constantes.SF_PARA);
		if (idealPara != null && !"".equals(idealPara)) {
			if ("205".equals(idealPara)) {
				sql.append(S12Constantes.PM_COUPLE);
				sql.append(" IS NOT NULL AND ");
			} else if ("206".equals(idealPara)) {
				sql.append(S12Constantes.PM_CHILDREN);
				sql.append(" IS NOT NULL AND ");
			} else if ("207".equals(idealPara)) {
				sql.append(S12Constantes.PM_FRIENDS);
				sql.append(" IS NOT NULL AND ");
			}		
		}

		String tematica = request.getParameter(S12Constantes.SF_TYPE);
		if (tematica != null && !"".equals(tematica)) {
			sql.append(S12Constantes.PM_TYPE);
	  		sql.append(" LIKE ? AND ");
	  		String expresionLIKE = "%"+tematica+"%";
			params.add(expresionLIKE);
			tiposSQL.add(Types.VARCHAR);
	  	}

		sql.append(" LANGUAGE = ?");
		String language = request.getParameter("language");
		params.add(language);
		tiposSQL.add(Types.VARCHAR);
	    sql.append(" ORDER BY ");
	    sql.append(S12Constantes.PM_IMPORTANCE);
	    sql.append(" ASC, ");
	    sql.append(S12Constantes.PM_NAME);
	    sql.append(" ASC ");

		return sql.toString();
	}

}
